/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <stdio.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>

#ifdef HAVE_LIBXDPMS
# include <X11/extensions/dpms.h>
/* XFree 3.3.x has'nt prototypes for this ... */
Bool   DPMSQueryExtension(Display*, int*, int*);
Bool   DPMSCapable(Display*);
Status DPMSInfo(Display*, CARD16*, BOOL*);
Status DPMSEnable(Display*);
Status DPMSDisable(Display*);
#endif

#include "sdl_helpers.h"
#include "singit/macros.h"

#define DEBUG(a) g_print a

void sdl_helper_get_position(gint *x, gint *y)
{
	SDL_SysWMinfo info;

	SDL_VERSION(&info.version);
	if (SDL_GetWMInfo(&info) > 0) {
		if (info.subsystem == SDL_SYSWM_X11) {
			Display *display;
			Window window, root;
			unsigned int ww, wh, wb, wd;
			int wx, wy;

			info.info.x11.lock_func();
			display = info.info.x11.display;
			window = info.info.x11.wmwindow;
			if (XGetGeometry(display, window, &root, &wx, &wy, &ww, &wh, &wb, &wd))
			{
				if (x) { *x = wx; }
				if (y) { *y = wy; }
//				g_print("%i / %i\n", wx, wy);
			}
			info.info.x11.unlock_func();
		}
	}
}

void sdl_helper_map(gboolean show)
{
	SDL_SysWMinfo info;

#ifdef CODEDEBUG
	DEBUG(("sdl_helpers.c [sdl_helper_map] : %s : ", ((show) ? "TRUE" : "FALSE")));
#endif

	SDL_VERSION(&info.version);
	if (SDL_GetWMInfo(&info) > 0) {
		if (info.subsystem == SDL_SYSWM_X11) {
			Display *display;
			Window window;
//			Window root;
//			unsigned int ww, wh, wb, wd;
//			int wx, wy;
//			XEvent event;

			info.info.x11.lock_func();
			display = info.info.x11.display;
			window = info.info.x11.wmwindow;

			if (show == FALSE) {
				XUnmapWindow(display, window);
			}
			else {
				XMapRaised(display, window);
/*
				if (XGetGeometry(display, window, &root, &wx, &wy, &ww, &wh, &wb, &wd))
				{
					XMoveWindow (display, window, wx, wy);
				}
*/
			}
			/*
			do {
				XNextEvent(display, &event);
			}
			while (event.type != MapNotify); */

//			XSetInputFocus(display, window, RevertToPointerRoot, CurrentTime);
			info.info.x11.unlock_func();
#	ifdef CODEDEBUG
			DEBUG(("Ok\n"));
#	endif
		}
#	ifdef CODEDEBUG
		else {
			DEBUG(("No SDL-X11 Sub\n"));
		}
#	endif
	}
#	ifdef CODEDEBUG
	else {
		DEBUG(("No Sysinfo\n"));
	}
#	endif
}

/*
	XGetScreenSaver(dpy,&timeout,&interval,
			&prefer_blanking,&allow_exposures);
	XSetScreenSaver(dpy,0,0,DefaultBlanking,DefaultExposures);
*/

typedef struct _BlankingInfo {

	int ss_timeout;
	int ss_interval;
	int ss_prefer_blanking;
	int ss_allow_exposures;

	gboolean ss_enabled;

#ifdef HAVE_LIBXDPMS
	BOOL dpms_capability;
	BOOL dpms_status;
#endif
}
BlankingInfo;

static void fill_blanking_info(BlankingInfo *info, Display *dpy)
{
#ifdef HAVE_LIBXDPMS
	int dpms_dummy;
	CARD16 dpms_state;

	info->dpms_capability =
		(DPMSQueryExtension(dpy, &dpms_dummy, &dpms_dummy) && DPMSCapable(dpy));

	if (info->dpms_capability)
		{ DPMSInfo(dpy, &dpms_state, &info->dpms_status); }
#endif
	XGetScreenSaver(dpy, &info->ss_timeout, &info->ss_interval,
		&info->ss_prefer_blanking, &info->ss_allow_exposures);

	info->ss_enabled = ((info->ss_timeout != 0) && (info->ss_interval != 0));
}

static inline gint enable_blanking(BlankingInfo *info, BlankingInfo *reset, Display *dpy)
{
	gboolean result = BLANK_DISABLE;
#ifdef HAVE_LIBXDPMS
	if ((info->dpms_capability) && (!info->dpms_status)) {
		DPMSEnable(dpy);
		result = BLANK_ENABLE;
	}
#endif
	if ((reset->ss_enabled == TRUE) && (info->ss_enabled == FALSE)) {
		XSetScreenSaver(dpy, reset->ss_timeout, reset->ss_interval,
			reset->ss_prefer_blanking, reset->ss_allow_exposures);
		result = BLANK_ENABLE;
	}

#ifdef CODEDEBUG
	if (result == BLANK_ENABLE) {
		DEBUG(("Enable\n"));
	}
	else {
		DEBUG(("Unchanged (Enabled)\n"));
	}
#endif
	return result;
}

static inline gint disable_blanking(BlankingInfo *info, BlankingInfo *reset, Display *dpy)
{
	gboolean result = BLANK_DISABLE;
#ifdef HAVE_LIBXDPMS
	if ((info->dpms_capability) && (info->dpms_status)) {
		DPMSDisable(dpy);
		result = BLANK_ENABLE;
	}
#endif
	if ((reset->ss_enabled == TRUE) && (info->ss_enabled == TRUE)) {
		XSetScreenSaver(dpy, 0, 0, DefaultBlanking, DefaultExposures);
		result = BLANK_ENABLE;
	}

#ifdef CODEDEBUG
	if (result == BLANK_ENABLE) {
		DEBUG(("Disable\n"));
	}
	else {
		DEBUG(("Unchanged (Disabled)\n"));
	}
#endif
	return result;
}

static inline gint reset_blanking(BlankingInfo *reset, Display *dpy)
{
	gboolean result = BLANK_DISABLE;

	if (reset->ss_timeout != -1) {
		result = BLANK_ENABLE;
#ifdef CODEDEBUG
		DEBUG(("Reset\n"));
#endif
#ifdef HAVE_LIBXDPMS
		if (reset->dpms_capability) {
			if (reset->dpms_status) { DPMSEnable(dpy); }
			else  { DPMSDisable(dpy); }
		}
#endif
		XSetScreenSaver(dpy, reset->ss_timeout, reset->ss_interval,
			reset->ss_prefer_blanking, reset->ss_allow_exposures);
	}
#ifdef CODEDEBUG
	else {
		DEBUG(("Unchanged (Reset)\n"));
	}
#endif
	return result;
}

static inline gint query_blanking(BlankingInfo *info)
{
	gint result;

#ifdef HAVE_LIBXDPMS
	result = (info->ss_enabled || (info->dpms_capability && info->dpms_status));
#else
	result = info->ss_enabled;
#endif

#ifdef CODEDEBUG
	DEBUG(("Query %s\n", (result ? "(Enabled)" : "(Disabled)")));
#endif
	return result;
}

int sdl_helper_blanking(int blank_enum)
{
	static BlankingInfo reset_info = { -1, -1, -1, -1, FALSE
#ifdef HAVE_LIBXDPMS
		,FALSE, FALSE
#endif
	};
	BlankingInfo blank_info;

	int result = BLANK_DISABLE;
	SDL_SysWMinfo info;

#	ifdef CODEDEBUG
	DEBUG(("sdl_helpers.c [sdl_helper_blanking]: "));
#	endif

	SDL_VERSION(&info.version);
	if (SDL_GetWMInfo(&info) > 0) {
		if (info.subsystem == SDL_SYSWM_X11) {
			Display *dpy;
			info.info.x11.lock_func();
			dpy = info.info.x11.display;

			if (reset_info.ss_timeout == -1) {
				if (blank_enum != BLANK_RESET) {
					fill_blanking_info(&reset_info, dpy);
					blank_info = reset_info;
				}
			}
			else { fill_blanking_info(&blank_info, dpy); }

			switch (blank_enum) {
			case BLANK_ENABLE:
				result = enable_blanking(&blank_info, &reset_info, dpy);
				break;
			case BLANK_DISABLE:
				result = disable_blanking(&blank_info, &reset_info, dpy);
				break;
			case BLANK_RESET:
				result = reset_blanking(&reset_info, dpy);
				break;
			case BLANK_QUERY:
				default:
				result = query_blanking(&blank_info);
			}
			info.info.x11.unlock_func();
		}
#	ifdef CODEDEBUG
		else {
			DEBUG(("No SDL-X11 Sub\n"));
		}
#	endif
	}
#	ifdef CODEDEBUG
	else {
		DEBUG(("No Sysinfo\n"));
	}
#	endif

	return result;
}

#define __check_x11_wm_start \
	SDL_SysWMinfo info; \
	\
	SDL_VERSION(&info.version); \
	if (SDL_GetWMInfo(&info) > 0) { \
		if (info.subsystem == SDL_SYSWM_X11) {

#define __check_x11_wm_end \
		} \
	}

void sdl_helper_set_transient(Window parent)
{
	__check_x11_wm_start

	info.info.x11.lock_func();
	XSetTransientForHint(info.info.x11.display, info.info.x11.wmwindow, parent);
	info.info.x11.unlock_func();

	__check_x11_wm_end
}

void sdl_helper_gain_focus(void)
{
	__check_x11_wm_start

	info.info.x11.lock_func();
	XSetInputFocus(info.info.x11.display, info.info.x11.wmwindow, RevertToParent, CurrentTime);
	info.info.x11.unlock_func();

	__check_x11_wm_end
}
