/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef __SINGIT_SONG_H__
#define __SINGIT_SONG_H__

#include <gtk/gtkobject.h>

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef enum _LyricType {

	LT_NONE = 0,
	LT_TEXT,
	LT_TAG,
	LT_MIDI
}
LyricType;

typedef struct _LToken {

	guint line;	// Lyric line
	guint time;	// Play time in milliseconds
	guint pos;	// Used by synchronized lyrics to get the position in the line

}
LToken;

#define TYPE_SINGIT_SONG            (singit_song_get_type ())
#define SINGIT_SONG(obj)            (GTK_CHECK_CAST ((obj), TYPE_SINGIT_SONG, SingitSong))
#define SINGIT_SONG_CLASS(klass)    (GTK_CHECK_CLASS_CAST ((klass), TYPE_SINGIT_SONG, SingitSong))
#define IS_SINGIT_SONG(obj)         (GTK_CHECK_TYPE ((obj), TYPE_SINGIT_SONG))
#define IS_SINGIT_SONG_CLASS(klass) (GTK_CHECK_CLASS_TYPE ((klass), TYPE_SINGIT_SONG))

typedef struct _SingitSong       SingitSong;
typedef struct _SingitSongClass  SingitSongClass;

struct _SingitSong
{
	GtkObject   object;

	GList      *first_token;
	GList      *last_token;
	GList      *active_token;

	// Lyrics text and length
	gchar     **lyrics;
	guint       lyric_lines;

	gchar      *song_filename; 	// Filename of the song
	gchar      *lyric_filename; // Filename of the lyrics text
	guint       song_length;    // Length in ms

	gchar      *delimiter;       // Delimiter type of text
	LyricType   lyric_type;

	gpointer    id3tag;
	gpointer    file_info;

	// Song information
	gchar      *artist;
	gchar      *album;
	gchar      *title;
};

struct _SingitSongClass
{
	GtkObjectClass parent_class;
};

#define tTime(a) (((LToken*) (a->data))->time)
#define tLine(a) (((LToken*) (a->data))->line)
#define tPos(a) (((LToken*) (a->data))->pos)
#define tText(a, b) (((SingitSong *) a)->lyrics[((LToken*) (b->data))->line])

#define singit_song_is_last_token(a) (((SingitSong *) a)->active_token) == last_token)
#define singit_song_is_first_token(a) (((SingitSong *) a)->active_token) == first_token)
#define singit_song_set_length(a, b) (((SingitSong *) a)->song_length) = b

#define singit_song_text_found(a) (((SingitSong *) a)->lyrics != NULL)
#define singit_song_lyrics_found(a) (((SingitSong *) a)->first_token != NULL)

#define TAG(a) ((ID3Tag*) a->id3tag)

GtkType     singit_song_get_type   (void);
GtkObject*  singit_song_new        (gchar* song_filename);

void singit_song_clear(SingitSong* song);
void singit_song_free(SingitSong* song);

// * These calls reference the song object. If the reference count == 0
// * the object is destroyed.
// * Since the same song object is widely used inside the plugin
// * every "user" must get a reference before using the song
SingitSong* singit_song_attach(SingitSong *ssong);
void        singit_song_detach(SingitSong **ssong);

// * Returns the position of the time in the token list compared to the current token
// * Results:
// * 	-1 = time is before the current
// * 	 0 = time is between the current and the next token
// * 	 1 = time is after the next token
gint singit_song_is_time_ok(SingitSong *ssong, gint time);
inline gint inl_singit_song_is_time_ok(SingitSong *ssong, gint time);
inline gboolean singit_song_is_empty_item
	(SingitSong *ssong, GList *item, gboolean check_end);

// * Returns the first time with the line *
gint singit_song_find_line(SingitSong* song, gint line);
// * Returns the first line with the time *
gint singit_song_find_time(SingitSong* song, gint time);

// * Returns the token directly before the time (current token)
// * State is most time the result of "singit_song_is_time_ok"
GList* singit_song_find_current_token
	(SingitSong *ssong, gint time, gint state);
inline GList *inl_singit_song_get_next_token(SingitSong *ssong);
inline GList *inl_singit_song_get_prev_token(SingitSong *ssong);

// * Returns the token with the next lyric line.
// * Set Empty if you accept an empty line
// * Set hops if you want to get the item distance else use NULL
GList* singit_song_find_next_lyric_line
	(SingitSong *ssong, GList *item, gboolean empty, guint *hops);
GList* singit_song_find_prev_lyric_line
	(SingitSong *ssong, GList *item, gboolean empty, guint *hops);

// * Tells you if the lyrics file has changed size or hash
gboolean singit_song_lyrics_changed(SingitSong *ssong);

// * Get the length of the of the text to the next string
// * Result:
// * 	-2 = Error in the lyrics file should not happen in "good" lyrics
// * 	-1 = To the end of line
// * 	> 0 = Length of substring
gint singit_song_get_text_length(SingitSong *ssong);
inline gint inl_singit_song_get_text_length(SingitSong *ssong);

// * Refreshs the id3_tag *
void singit_song_get_id3_tag(SingitSong *ssong, gchar *filename);

// * Checks if every later timetag in the song list has a greater line
// * number or text offset
gint singit_song_check_sync_lyric_consistency(SingitSong *ssong);

// * Adds an overall time to a song (adding < 0 = sub)
void singit_song_modify_overall_time(SingitSong *ssong, gint time);

// * Guess if the lyrics are syncronized by checking every tPos > 0
gboolean singit_song_guess_sync_lyrics(SingitSong *ssong);

#ifdef __cplusplus
}
#endif // __cplusplus

#endif // __SINGIT_SONG_H__
