/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "singit/framerate_counter.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <gtk/gtk.h>
#include <xmms/util.h>

#ifdef HAVE_SDL
#include <SDL/SDL.h>
#include <SDL/SDL_timer.h>
#endif

#include "singit_debug.h"

// #define DEBUG(a,b) g_print a

static GtkObjectClass *parent_class = NULL;

static void singit_framerate_counter_class_init (SingitFramerateCounterClass *klass);
static void singit_framerate_counter_init (SingitFramerateCounter *sfc);

static void singit_framerate_counter_destroy(GtkObject *object);

GtkType singit_framerate_counter_get_type (void)
{
	static GtkType singit_framerate_counter_type = 0;

	if (!singit_framerate_counter_type) {

		static const GtkTypeInfo singit_framerate_counter_info =
		{
			(gchar*) "SingitFramerateCounter",
			sizeof (SingitFramerateCounter),
			sizeof (SingitFramerateCounterClass),
			(GtkClassInitFunc) singit_framerate_counter_class_init,
			(GtkObjectInitFunc) singit_framerate_counter_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_framerate_counter_type = gtk_type_unique
			(GTK_TYPE_OBJECT, &singit_framerate_counter_info);
	}

	return singit_framerate_counter_type;
}

static void singit_framerate_counter_class_init (SingitFramerateCounterClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	object_class->destroy = singit_framerate_counter_destroy;
}

static void singit_framerate_counter_init (SingitFramerateCounter *sfc)
{
	sfc->ringTable = NULL;

	sfc->table_size = 0;
	sfc->table_pos = 0;

	sfc->frame_limit = 0;
	sfc->next_ticks = 0;

	sfc->tick_type = GLIB_TICKS;
}

static void singit_framerate_counter_destroy (GtkObject *object)
{
	SingitFramerateCounter *sfc;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER (object));

	sfc = SINGIT_FRAMERATE_COUNTER (object);

	if (sfc->ringTable != NULL)
		g_free(sfc->ringTable);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

GtkObject *singit_framerate_counter_new(guint table_size)
{
	SingitFramerateCounter *sfc;

#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_new]\n"));
#endif

	g_return_val_if_fail(table_size >= 3, NULL);

	sfc = gtk_type_new(TYPE_SINGIT_FRAMERATE_COUNTER);

	sfc->table_size = table_size;

	sfc->ringTable = g_new(guint32, table_size);

	sfc->ringTable[0] = 0;
	sfc->ringTable[sfc->table_size-1] = 0;

	return GTK_OBJECT(sfc);
}

static guint32 get_ticks(SingitFramerateCounter *sfc)
{
	GTimeVal time;

	switch(sfc->tick_type) {
#ifdef HAVE_SDL
	case SDL_TICKS:
//		if (SDL_WasInit(SDL_INIT_TIMER) == 0)
//			SDL_InitSubSystem(SDL_INIT_TIMER);
		return (SDL_GetTicks());
#endif
	case GLIB_TICKS:
	default:
		g_get_current_time(&time);
		return ((time.tv_sec * 1000) + (time.tv_usec / 1000));
	};
}

void singit_framerate_counter_reset(SingitFramerateCounter *sfc)
{
#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_reset]\n"));
#endif

	g_return_if_fail (sfc != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc));

	sfc->ringTable[0] = 0;
	sfc->ringTable[sfc->table_size-1] = 0;
	sfc->table_pos =  0;
	sfc->next_ticks = 0;
}

#define INTERPIX(sfc) (1000 / sfc->frame_limit)

void singit_framerate_counter_start(SingitFramerateCounter *sfc)
{
	guint32 curTicks, i;

#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_start]\n"));
#endif

	g_return_if_fail (sfc != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc));

	curTicks = get_ticks(sfc);

	sfc->table_pos =  0;

	for (i = 0; i < sfc->table_size; i++)
		sfc->ringTable[i] = curTicks;

	if (sfc->frame_limit > 0)
		sfc->next_ticks = curTicks + INTERPIX(sfc);
}

void singit_framerate_counter_new_frame(SingitFramerateCounter *sfc)
{
	guint32 ticks;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_framerate_counter.c [singit_framerate_counter_new_frame]\n"));
#endif

	g_return_if_fail (sfc != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc));
	g_return_if_fail (sfc->ringTable[0] != 0);

	ticks = get_ticks(sfc);

	sfc->table_pos = (sfc->table_pos + 1) % sfc->table_size;
	sfc->ringTable[sfc->table_pos] = ticks;

	if (sfc->frame_limit > 0) {
		if (ticks < sfc->next_ticks) {
#ifdef CODEDEBUG
			DEBUG(DLV_ALL, ("  framerate sleep: %i\n", (sfc->next_ticks - ticks) * 1000));
#endif
			xmms_usleep((sfc->next_ticks - ticks) * 1000);
			sfc->next_ticks += INTERPIX(sfc);
		}
		else {
			sfc->next_ticks = ticks + INTERPIX(sfc);
		}
	}
}

gfloat singit_framerate_counter_get_value(SingitFramerateCounter *sfc)
{
	guint32 oldPos;

	g_return_val_if_fail (sfc != NULL, 0.0f);
	g_return_val_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc), 0.0f);

#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_get_value]\n"));
#endif

	oldPos = (sfc->table_pos + 1) % sfc->table_size;

	return (float) sfc->table_size
		/ (sfc->ringTable[sfc->table_pos] - sfc->ringTable[oldPos]);
}

void singit_framerate_counter_set_limit(SingitFramerateCounter *sfc, guint limit)
{
	g_return_if_fail (sfc != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc));

#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_set_limit]\n"));
#endif

	if (sfc->frame_limit == limit)
		{ return; }

	sfc->frame_limit = limit;

	if (sfc->frame_limit > 0)
		sfc->next_ticks = get_ticks(sfc) + INTERPIX(sfc);
}

void singit_framerate_counter_set_ticktype(SingitFramerateCounter *sfc, TickType type)
{
	g_return_if_fail (sfc != NULL);
	g_return_if_fail (IS_SINGIT_FRAMERATE_COUNTER(sfc));

#ifdef CODEDEBUG
	DEBUG(5, ("singit_framerate_counter.c [singit_framerate_counter_set_ticktype]\n"));
#endif

	if (sfc->tick_type == type)
		{ return; }

	if (sfc->ringTable[0] != 0)
		g_warning("Changeing TickType while the counter is running may result in invalid framerates.");

	sfc->tick_type = type;
}
