/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include "singit_main_status.h"
#include "singit_main_config.h"

// Since the status is a singleton, keep this global static object
static SingitSingleton       *singit_status = NULL;
static SingitSingletonClass  *parent_class  = NULL;

static void singit_status_class_init (SingitStatusClass *klass);
static void singit_status_init (SingitStatus *sstat);
static void singit_status_destroy(GtkObject *object);
static gboolean singit_status_after_init(SingitSingleton *single);

GtkType singit_status_get_type (void)
{
	static GtkType singit_status_type = 0;

	if (!singit_status_type) {

		static const GtkTypeInfo singit_status_info =
		{
			(gchar*) "SingitStatus",
			sizeof (SingitStatus),
			sizeof (SingitStatusClass),
			(GtkClassInitFunc) singit_status_class_init,
			(GtkObjectInitFunc) singit_status_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_status_type = gtk_type_unique
			(TYPE_SINGIT_SINGLETON, &singit_status_info);
	}

	return singit_status_type;
}

static void singit_status_class_init (SingitStatusClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	SingitSingletonClass *singleton_class =
		(SingitSingletonClass*) klass;

	parent_class = gtk_type_class(TYPE_SINGIT_SINGLETON);

	object_class->destroy = singit_status_destroy;
	singleton_class->after_init = singit_status_after_init;
}

static void singit_status_init(SingitStatus *sstat)
{
        sstat->next_lyric_line = NULL;

	sstat->display_thread = 0;
	sstat->check_thread = 0;

	sstat->song = NULL;

	pthread_mutex_init(&sstat->config_rw_mutex, NULL);
	pthread_mutex_init(&sstat->lyrics_rw_mutex, NULL);

	sstat->kill_threads = FALSE;
	sstat->initialize_plugins = FALSE;

	sstat->config_update = FALSE;
	sstat->playing = FALSE;
	sstat->song_changed = TRUE;
	sstat->text_update = FALSE;

	sstat->singit_sound_precalcs =
		g_malloc(sizeof(SingitSoundPrecalcs));
	sstat->singit_framerate_counter = NULL;

	sstat->config = NULL;

	sstat->old_time = -1;
}

static void singit_status_destroy(GtkObject *object)
{
	SingitStatus *sstat;

	g_return_if_fail(object != NULL);
	g_return_if_fail(IS_SINGIT_STATUS (object));

	sstat = SINGIT_STATUS(object);

	pthread_mutex_destroy(&sstat->config_rw_mutex);
	pthread_mutex_destroy(&sstat->lyrics_rw_mutex);

	if (sstat->singit_sound_precalcs != NULL)
		g_free(sstat->singit_sound_precalcs);

	if (sstat->singit_framerate_counter != NULL)
		gtk_object_destroy(GTK_OBJECT(sstat->singit_framerate_counter));

	singit_config_gen_detach(&sstat->config);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

static gboolean singit_status_after_init(SingitSingleton *single)
{
	g_return_val_if_fail(single != NULL, FALSE);
	g_return_val_if_fail(IS_SINGIT_STATUS(single), FALSE);

	SINGIT_STATUS(single)->config = singit_config_new();
	singit_config_gen_load(SINGIT_STATUS(single)->config);

	return (SINGIT_STATUS(single)->config != NULL);
}

SingitSingleton* singit_status_ref(void)
{
	gboolean init;

	SingitSingleton *status = singit_singleton_construct
		(&singit_status, TYPE_SINGIT_STATUS, &init);
	if (init == TRUE)
		{ return status; }
	return (status != NULL) ? singit_singleton_attach(status) : NULL;
}

void singit_status_unref(SingitSingleton** single)
{
	g_return_if_fail(singit_status != NULL);

	if (single == NULL) {
		SingitSingleton* unref_single = singit_status;
		singit_singleton_detach(&unref_single);
	}
	else {
		g_return_if_fail(*single == singit_status);
		g_return_if_fail(single != &singit_status);
		singit_singleton_detach(single);
	}
}

SingitSingleton* singit_status_noref(void)
{
	if (singit_status == NULL)
		{ return NULL; }

	g_return_val_if_fail(IS_SINGIT_STATUS(singit_status), NULL);

	return singit_status;
}
