/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <string.h>
#include <stdio.h>

#include "singit_debug.h"

#include "mpeg_header_info.h"

#define         DET_BUF_SIZE    1024
/* max = 1728 */
#define         MAXFRAMESIZE    1792

/*
 * Xing header flags
 */

#define FRAMES_FLAG     0x0001
#define BYTES_FLAG      0x0002
#define TOC_FLAG        0x0004
#define VBR_SCALE_FLAG  0x0008

typedef struct _MpegFrame
{
	int stereo;
	int jsbound;
	int single;
	int II_sblimit;
	int down_sample_sblimit;
	int lsf;
	int mpeg25;
	int down_sample;
	int header_change;
	int lay;
	int error_protection;
	int bitrate_index;
	int sampling_frequency;
	int padding;
	int extension;
	int mode;
	int mode_ext;
	int copyright;
	int original;
	int emphasis;
	int framesize;		/* computed framesize */
}
MpegFrame;

int tabsel_123[2][3][16] =
{
	{
		{ 0, 32, 64, 96, 128, 160, 192, 224, 256, 288, 320, 352, 384, 416, 448,},
		{ 0, 32, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320, 384,},
		{ 0, 32, 40, 48, 56, 64, 80, 96, 112, 128, 160, 192, 224, 256, 320,}
	},
	{
		{ 0, 32, 48, 56, 64, 80, 96, 112, 128, 144, 160, 176, 192, 224, 256,},
		{ 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160,},
		{ 0, 8, 16, 24, 32, 40, 48, 56, 64, 80, 96, 112, 128, 144, 160,}
	}
};

long mpeg_header_info_freqs[9] =
	{ 44100, 48000, 32000, 22050, 24000, 16000, 11025, 12000, 8000 };


/*
 * structure to receive extracted Xing header
 */
typedef struct
{
	int frames;		/* total bit stream frames from Xing header data */
	int bytes;		/* total bit stream bytes from Xing header data */
	unsigned char toc[100];	/* "table of contents" */
}
xing_header_t;

static GtkObjectClass *parent_class = NULL;

static void mpeg_header_info_class_init (MPEGHeaderInfoClass *class)
{
	parent_class = gtk_type_class (gtk_widget_get_type ());
}

static void mpeg_header_info_init (MPEGHeaderInfo *mhi)
{
	mhi->filename = NULL;

	mhi->frames = 0;
	mhi->filesize = 0;

	mhi->mpeg_25 = FALSE;
	mhi->mpeg_version = 1;
	mhi->mpeg_layer = 3;

	mhi->bitrate_variable = FALSE;
	mhi->bitrate = 128;
	mhi->samplerate = 44100;
	mhi->channel_mode = 0;
	mhi->emphasis = 0;

	mhi->error_protection = FALSE;
	mhi->copyright = FALSE;
	mhi->original = TRUE;
}

GtkType mpeg_header_info_get_type (void)
{
	static GtkType mpeg_header_info_type = 0;

	if (!mpeg_header_info_type) {

		static const GtkTypeInfo mpeg_header_info_info =
		{
			(gchar*) "MPEGHeaderInfo",
			sizeof (MPEGHeaderInfo),
			sizeof (MPEGHeaderInfoClass),
			(GtkClassInitFunc)  mpeg_header_info_class_init,
			(GtkObjectInitFunc) mpeg_header_info_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL,
		};

		mpeg_header_info_type = gtk_type_unique (gtk_object_get_type(), &mpeg_header_info_info);
	}

	return mpeg_header_info_type;
}

static guint32 convert_to_header(guint8 * buf)
{
	return (buf[0] << 24) + (buf[1] << 16) + (buf[2] << 8) + buf[3];
}

static double mpeg_header_info_compute_bpf(MpegFrame *fr)
{
	double bpf;

	switch (fr->lay)
	{
		case 1:
			bpf = tabsel_123[fr->lsf][0][fr->bitrate_index];
			bpf *= 12000.0 * 4.0;
			bpf /= mpeg_header_info_freqs[fr->sampling_frequency] << (fr->lsf);
			break;
		case 2:
		case 3:
			bpf = tabsel_123[fr->lsf][fr->lay - 1][fr->bitrate_index];
			bpf *= 144000;
			bpf /= mpeg_header_info_freqs[fr->sampling_frequency] << (fr->lsf);
			break;
		default:
			bpf = 1.0;
	}

	return bpf;
}

static int mpeg_header_info_head_check(unsigned long head)
{
	if ((head & 0xffe00000) != 0xffe00000)
		return FALSE;
	if (!((head >> 17) & 3))
		return FALSE;
	if (((head >> 12) & 0xf) == 0xf)
		return FALSE;
	if (!((head >> 12) & 0xf))
		return FALSE;
	if (((head >> 10) & 0x3) == 0x3)
		return FALSE;
	if (((head >> 19) & 1) == 1 && ((head >> 17) & 3) == 3 && ((head >> 16) & 1) == 1)
		return FALSE;
	if ((head & 0xffff0000) == 0xfffe0000)
		return FALSE;

	return TRUE;
}

/*
 * the code a header and write the information
 * into the frame structure
 */
static int mpeg_header_info_decode_header(MpegFrame *fr, unsigned long newhead)
{
	gint ssize = 0;

	if (newhead & (1 << 20))
	{
		fr->lsf = (newhead & (1 << 19)) ? 0x0 : 0x1;
		fr->mpeg25 = 0;
	}
	else
	{
		fr->lsf = 1;
		fr->mpeg25 = 1;
	}
	fr->lay = 4 - ((newhead >> 17) & 3);
	if (fr->mpeg25)
	{
		fr->sampling_frequency = 6 + ((newhead >> 10) & 0x3);
	}
	else
		fr->sampling_frequency = ((newhead >> 10) & 0x3) + (fr->lsf * 3);
	fr->error_protection = ((newhead >> 16) & 0x1) ^ 0x1;

	fr->bitrate_index = ((newhead >> 12) & 0xf);
	fr->padding = ((newhead >> 9) & 0x1);
	fr->extension = ((newhead >> 8) & 0x1);
	fr->mode = ((newhead >> 6) & 0x3);
	fr->mode_ext = ((newhead >> 4) & 0x3);
	fr->copyright = ((newhead >> 3) & 0x1);
	fr->original = ((newhead >> 2) & 0x1);
	fr->emphasis = newhead & 0x3;

	fr->stereo = (fr->mode == MPG_MD_MONO) ? 1 : 2;

	if (!fr->bitrate_index)
		return (0);

	switch (fr->lay)
	{
		case 1:
			fr->framesize = (long) tabsel_123[fr->lsf][0][fr->bitrate_index] * 12000;
			fr->framesize /= mpeg_header_info_freqs[fr->sampling_frequency];
			fr->framesize = ((fr->framesize + fr->padding) << 2) - 4;
			break;
		case 2:
			fr->framesize = (long) tabsel_123[fr->lsf][1][fr->bitrate_index] * 144000;
			fr->framesize /= mpeg_header_info_freqs[fr->sampling_frequency];
			fr->framesize += fr->padding - 4;
			break;
		case 3:
			if (fr->lsf)
				ssize = (fr->stereo == 1) ? 9 : 17;
			else
				ssize = (fr->stereo == 1) ? 17 : 32;
			if (fr->error_protection)
				ssize += 2;
			fr->framesize = (long) tabsel_123[fr->lsf][2][fr->bitrate_index] * 144000;
			fr->framesize /= mpeg_header_info_freqs[fr->sampling_frequency] << (fr->lsf);
			fr->framesize = fr->framesize + fr->padding - 4;
			break;
		default:
			return (0);
	}
	if(fr->framesize > MAXFRAMESIZE)
		return 0;
	return 1;
}

gboolean mpeg_header_info_detect_by_content(gchar *filename)
{
	FILE *file;
	guchar tmp[4];
	guint32 head;
	MpegFrame fr;
	guchar buf[DET_BUF_SIZE];
	gint in_buf, i;

	if((file = fopen(filename, "rb")) == NULL)
		return FALSE;
	if (fread(tmp, 1, 4, file) != 4)
		goto done;
	head = convert_to_header(tmp);
	while(!mpeg_header_info_head_check(head))
	{
		/*
		 * The mpeg-stream can start anywhere in the file,
		 * so we check the entire file
		 */
		/* Optimize this */
		in_buf = fread(buf, 1, DET_BUF_SIZE, file);
		if(in_buf == 0)
			goto done;

		for (i = 0; i < in_buf; i++)
		{
			head <<= 8;
			head |= buf[i];
			if(mpeg_header_info_head_check(head))
			{
				fseek(file, i+1-in_buf, SEEK_CUR);
				break;
			}
		}
	}
	if (mpeg_header_info_decode_header(&fr, head))
	{
		/*
		 * We found something which looks like a MPEG-header.
		 * We check the next frame too, to be sure
		 */

		if (fseek(file, fr.framesize, SEEK_CUR) != 0)
			goto done;
		if (fread(tmp, 1, 4, file) != 4)
			goto done;
		head = convert_to_header(tmp);
		if (mpeg_header_info_head_check(head) && mpeg_header_info_decode_header(&fr, head))
		{
			fclose(file);
			return TRUE;
		}
	}

 done:
	fclose(file);
	return FALSE;
}

static double mpeg_header_info_compute_tpf(MpegFrame *fr)
{
	const int bs[4] = {0, 384, 1152, 1152};
	double tpf;

	tpf = bs[fr->lay];
	tpf /= mpeg_header_info_freqs[fr->sampling_frequency] << (fr->lsf);
	return tpf;
}

/*
 * Handle Xing vbr header
 */

#define GET_INT32BE(b) \
	(i = (b[0] << 24) | (b[1] << 16) | b[2] << 8 | b[3], b += 4, i)

/*
 * Returns zero on fail, non-zero on success
 * xing structure to receive header data (output)
 * buf bitstream input
 */
static int mpeg_header_info_get_xing_header(xing_header_t * xing, unsigned char *buf)
{
	int i, head_flags;
	int id, mode;

	memset(xing, 0, sizeof(xing_header_t));

	/* get selected MPEG header data */
	id = (buf[1] >> 3) & 1;
	mode = (buf[3] >> 6) & 3;
	buf += 4;

	/* Skip the sub band data */
	if (id)
	{
		/* mpeg1 */
		if (mode != 3)
			buf += 32;
		else
			buf += 17;
	}
	else
	{
		/* mpeg2 */
		if (mode != 3)
			buf += 17;
		else
			buf += 9;
	}

	if (strncmp(buf, "Xing", 4))
		return 0;
	buf += 4;

	head_flags = GET_INT32BE(buf);

	if (head_flags & FRAMES_FLAG)
		xing->frames = GET_INT32BE(buf);
	if (xing->frames < 1)
		xing->frames = 1;
	if (head_flags & BYTES_FLAG)
		xing->bytes = GET_INT32BE(buf);

	if (head_flags & TOC_FLAG)
	{
		for (i = 0; i < 100; i++)
			xing->toc[i] = buf[i];
		buf += 100;
	}

#ifdef XING_DEBUG
	for (i = 0; i < 100; i++)
	{
		if ((i % 10) == 0)
			fprintf(stderr, "\n");
		fprintf(stderr, " %3d", xing->toc[i]);
	}
#endif

	return 1;
}

MPEGHeaderInfo* mpeg_header_info_new(gchar *filename)
{
	FILE *file_handle;
	MPEGHeaderInfo* info;
	guint32 head;
	guchar tmp[4], *buf;
	MpegFrame frm;
	gboolean id3_found = FALSE;
	gdouble tpf, bpf;
	gint pos;
	xing_header_t xing_header;
	guint32 len;

	/* check mpeg header */
	g_return_val_if_fail(filename != NULL, NULL);
	if (strncasecmp(filename, "http://", 7) == 0) { return NULL; }

	if (!(file_handle = fopen(filename, "rb"))) { return NULL; }

	if (fread(tmp, 1, 4, file_handle) != 4)
	{
		fclose(file_handle);
		return NULL;
	}

	head = ((guint32) tmp[0] << 24) | ((guint32) tmp[1] << 16) | ((guint32) tmp[2] << 8) | (guint32) tmp[3];
	while (!mpeg_header_info_head_check(head))
	{
		head <<= 8;
		if (fread(tmp, 1, 1, file_handle) != 1)
		{
			fclose(file_handle);
			return NULL;
		}
		head |= tmp[0];
	}

	/* prepare header */
	info = gtk_type_new(TYPE_MPEG_HEADER_INFO);
	info->filename = g_strdup(filename);

	/* decode header */
	if (mpeg_header_info_decode_header(&frm, head))
	{
		buf = g_malloc(frm.framesize + 4);
		fseek(file_handle, -4, SEEK_CUR);
		fread(buf, 1, frm.framesize + 4, file_handle);

		tpf = mpeg_header_info_compute_tpf(&frm);
		bpf = mpeg_header_info_compute_bpf(&frm);

		info->mpeg_25 = frm.mpeg25;
		info->mpeg_version = frm.lsf + 1;
		info->mpeg_layer = frm.lay;

		pos = ftell(file_handle);
		fseek(file_handle, 0, SEEK_END);

		len = info->filesize = ftell(file_handle);

		info->bitrate_variable = ((mpeg_header_info_get_xing_header(&xing_header, buf) != 0) == TRUE);
		if (info->bitrate_variable == TRUE)
		{
			info->frames = xing_header.frames;
			info->bitrate = (gint) ((xing_header.bytes * 8) / (tpf * xing_header.frames * 1000));
			info->length = ((guint) (tpf * xing_header.frames * 1000));
		}
		else {
			info->frames = ((ftell(file_handle) - pos - (id3_found ? 128 : 0)) / mpeg_header_info_compute_bpf(&frm)) + 1;
			info->bitrate = tabsel_123[frm.lsf][frm.lay - 1][frm.bitrate_index];

			fseek(file_handle, -128, SEEK_END);
			fread(tmp, 1, 3, file_handle);
			if (!strncmp(tmp, "TAG", 3)) { len -= 128; }
			info->length = (guint) ((len / bpf) * tpf * 1000);
		}
		info->samplerate = mpeg_header_info_freqs[frm.sampling_frequency];
		info->channel_mode = frm.mode;
		info->error_protection = ((frm.error_protection != 0) == TRUE);
		info->copyright = ((frm.copyright != 0) == TRUE);
		info->original = ((frm.original != 0) == TRUE);
		info->emphasis = frm.emphasis;

		g_free(buf);
	}

	fclose(file_handle);

	return info;
}

void mpeg_header_info_free(MPEGHeaderInfo* mhi)
{
	g_return_if_fail(mhi != NULL);
	g_return_if_fail(IS_MPEG_HEADER_INFO(mhi));

	if (mhi->filename != NULL) { g_free(mhi->filename); }

	g_free(mhi);
}
