#include <signal.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "../include/Y2/Y.h"
#include "../include/Y2/Ylib.h"

#include "../include/string.h"

#include "guiutils.h"
#include "cdialog.h"
#include "fprompt.h"
#include "fb.h"
#include "pdialog.h"
#include "pstepper.h"

#include "yc.h"
#include "config.h"

#include "images/nariative_bg.xpm"
#include "images/title.xpm"
#include "images/icon_yiff_48x48.xpm"
#include "images/icon_yiff_32x32.xpm"
#include "images/icon_wand_32x32.xpm"


static guint manage_timeout_cb_id;


static gboolean load_font_to_style(GtkStyle *style_ptr, const gchar *font_name);

gint YCInit(yc_core_struct *core_ptr, gint argc, gchar **argv);
gint YCManage(gpointer data);
void YCShutdown(yc_core_struct *core_ptr);


/*
 *      Macro used in YCInit() to load fonts to styles.
 *
 *      Returns TRUE if the font was not able to be loaded, otherwise
 *      FALSE on success.
 */
static gboolean load_font_to_style(GtkStyle *style_ptr, const gchar *font_name) 
{
	GdkFont *new_font = gdk_font_load(font_name);

	/* Could not load new font? */
	if(new_font == NULL)
	    return(TRUE);

	/* Unref current font on style structure */
	if(style_ptr->font != NULL)
	    gdk_font_unref(style_ptr->font);

	/* Set newly loaded font to style */
	style_ptr->font = new_font;

	return(FALSE);
}


/*
 *	Initializes the YC core structure.
 *
 *	Returns non-zero on error..
 */
gint YCInit(yc_core_struct *core_ptr, gint argc, gchar **argv)
{
	gint status;
	yc_styles_struct *styles;
	yc_pixmaps_struct *pixmaps;
	yc_struct *yc;
	GtkStyle *style_standard, *style_ptr;
	GdkColor c;
	GdkPixmap *pixmap_gdk;
	GdkBitmap *mask;
	GdkWindow *window = (GdkWindow *)GDK_ROOT_PARENT();


	if(core_ptr == NULL)
	    return(-1);

	styles = &core_ptr->yc_styles;
	pixmaps = &core_ptr->yc_pixmaps;
	yc = &core_ptr->yc;


	/* ******************************************************* */
	/* Styles */
	/* Get standard style from GTK, do not ever destroy it */
	style_standard = gtk_widget_get_default_style();
	styles->standard = style_standard;

	/* Heading 1 */
	style_ptr = ((style_standard == NULL) ?
	    NULL : gtk_style_copy(style_standard)
	);
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-20-*-*-*-*-*-*-*"
	    );
	}
	styles->heading1_text = style_ptr;

	/* Heading 2 */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-18-*-*-*-*-*-*-*"
	    );
	}
	styles->heading2_text = style_ptr;

	/* Heading 3 */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-17-*-*-*-*-*-*-*"
	    );
	}
	styles->heading3_text = style_ptr;

	/* Heading 4 */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-16-*-*-*-*-*-*-*"
	    );
	}
	styles->heading4_text = style_ptr;

	/* Heading 5 */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-14-*-*-*-*-*-*-*"
	    );
	}
	styles->heading5_text = style_ptr;

	/* Heading 6 */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("black", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;

	    load_font_to_style(style_ptr,
		"-adobe-*-regular-r-*-*-12-*-*-*-*-*-*-*"
	    );
	}
	styles->heading6_text = style_ptr;

	/* Nariatives */
	style_ptr = (style_standard != NULL) ?
	    gtk_style_copy(style_standard) : NULL;
	if(style_ptr != NULL)
	{
	    gdk_color_parse("white", &c);
	    style_ptr->fg[GTK_STATE_NORMAL] = c;
	}
	styles->nariative_text = style_ptr;

	/* ******************************************************* */
	/* Pixmaps */
	if((window != NULL) && (style_standard != NULL))
	{
	    /* YIFF nariative strip */
	    pixmap_gdk = gdk_pixmap_create_from_xpm_d(
		window, &mask,
		&style_standard->bg[GTK_STATE_NORMAL],
		(gchar **)nariative_bg_xpm
	    );
	    pixmaps->nariative_bg = pixmap_gdk;
	    pixmaps->nariative_bg_mask = mask;

	    /* YIFF title */
	    pixmap_gdk = gdk_pixmap_create_from_xpm_d(
		window, &mask,
		&style_standard->bg[GTK_STATE_NORMAL],
		(gchar **)title_xpm
	    );
	    pixmaps->title = pixmap_gdk;
	    pixmaps->title_mask = mask;

	    /* YIFF icon (48x48) */
	    pixmap_gdk = gdk_pixmap_create_from_xpm_d(
		window, &mask,
		&style_standard->bg[GTK_STATE_NORMAL],
		(gchar **)icon_yiff_48x48_xpm
	    );
	    pixmaps->icon_yiff_48x48 = NULL;
	    pixmaps->icon_yiff_48x48_mask = NULL;

	    /* YIFF icon (32x32) */
	    pixmap_gdk = gdk_pixmap_create_from_xpm_d(
		window, &mask,
		&style_standard->bg[GTK_STATE_NORMAL],
		(gchar **)icon_yiff_32x32_xpm
	    );
	    pixmaps->icon_yiff_32x32 = NULL;
	    pixmaps->icon_yiff_32x32_mask = NULL;

	    /* Wand icon (32x32) */
	    pixmap_gdk = gdk_pixmap_create_from_xpm_d(
		window, &mask,
		&style_standard->bg[GTK_STATE_NORMAL],
		(gchar **)icon_wand_32x32_xpm
	    );
	    pixmaps->icon_wand_32x32 = pixmap_gdk;
	    pixmaps->icon_wand_32x32_mask = mask; 
	}

	/* ******************************************************* */

	/* Initialize the YIFF Configuration Wizard window widgets */
	status = YCCreate(yc, argc, argv, styles, pixmaps);
	if(status)
	    return(-1);

	YCMap(yc);

	return(0);
}

/*
 *	Manages resources on the given pointer to the core structure.
 *
 *	This is basically the GTK timeout callback function.
 */
gint YCManage(gpointer data)
{
	yc_styles_struct *styles;
	yc_pixmaps_struct *pixmaps;
	yc_struct *yc;
	gchar *buf;
	YEvent event;
	YConnection *recorder;
	pstepper_struct *ps;
	yc_core_struct *core_ptr = (yc_core_struct *)data;
	if(core_ptr == NULL)
	    return(FALSE);

	styles = &core_ptr->yc_styles;
	pixmaps = &core_ptr->yc_pixmaps;
	yc = &core_ptr->yc;  

	/* Manage Y events */
	recorder = yc->recorder;
	if(recorder != NULL)
	{
	    /* Keep handling Y events while there are some in the
	     * queue.
	     */
	    while(YGetNextEvent(recorder, &event, False) > 0)
	    {
		/* Handle Y event by its type */
		switch(event.type)
		{
		  /* A Y sound object has stopped playing */
		  case YSoundObjectKill:
		    if(event.kill.yid == yc->current_sound_play)
		    {
			/* Reset current sound play id to YIDNULL */
			yc->current_sound_play = YIDNULL;
		    }
		    break;

		  /* Y server disconnected on us */
		  case YDisconnect:

		    /* The connection structure is no longer valid
		     * so set it to NULL.
		     */
		    YCloseConnection(yc->recorder, False);
		    yc->recorder = NULL;

		    yc->current_sound_play = YIDNULL;

		    /* Warn about Y server disconnect */
		    buf = g_strdup_printf(
"The Y Server has disconnected with reason code \"%i\".",
			event.disconnect.reason
		    );
		    CDialogGetResponse(
"Y Server Message",
buf,
"The Y Server has disconnected this program.\n\
This may indicate a problem with the Y Server\n\
if this occured abruptly. You may continue\n\
configuration, however if this problem persists\n\
you should contact the Y Server's vendor.",
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    g_free(buf);
		    break;

		  /* Y server disconnected on us */
		  case YShutdown:

		    /* The connection structure is no longer valid
		     * so set it to NULL.
		     */
		    yc->recorder = NULL;

		    yc->current_sound_play = YIDNULL;

		    /* Warn about Y server shutdown */
		    buf = g_strdup_printf(
"The Y Server has shutdown with reason code \"%i\".",
			event.shutdown.reason
		    );
		    CDialogGetResponse(
"Y Server Message",
buf,
"The Y Server has shutdown. This may indicate a\n\
problem with the Y Server if this occured abruptly.\n\
You may continue configuration, however if this\n\
problem persists you should contact the Y Server's\n\
vendor.",
			CDIALOG_ICON_WARNING,
			CDIALOG_BTNFLAG_OK | CDIALOG_BTNFLAG_HELP,
			CDIALOG_BTNFLAG_OK
		    );
		    g_free(buf);
		    break;

		  default:
		    break;
		}

		/* No more connection to Y server? */
		if(yc->recorder == NULL)
		    break;
	    }
	}

	/* Check if yc window has been unmapped */
	ps = yc->ps;
	if((ps != NULL) ? !ps->map_state : TRUE)
	{
	    gint exit_status = 1;

	    if(CDialogIsQuery())
	    {
		exit_status = 0;
	    }
	    else if(FileBrowserIsQuery())
	    {
		exit_status = 0;
	    }
	    else if(PDialogIsQuery())
	    {
		exit_status = 0;
	    }

	    if(exit_status)
	    {
		/* All windows have been unmapped which implies the
		 * program should start exiting now
		 */
		gtk_main_quit();

		/* Remove callback to this function */
		if(manage_timeout_cb_id != (guint)(-1))
		{
		    gtk_timeout_remove(manage_timeout_cb_id);
		    manage_timeout_cb_id = (guint)(-1);
		}

		/* Flush events */
		while(gtk_events_pending() > 0)
		    gtk_main_iteration();
	    }
	}

	return(TRUE);
}

/*
 *	Deallocates all resources on the core structure.
 *
 *	But does not free to core structure itself.
 */
void YCShutdown(yc_core_struct *core_ptr)
{
	yc_styles_struct *styles;
	yc_pixmaps_struct *pixmaps;
	yc_struct *yc;
	GtkWidget *pixmap;
	GdkPixmap *pixmap_gdk;
	GdkBitmap *mask;
	GtkStyle *style_ptr;

	if(core_ptr == NULL)
	    return;

	styles = &core_ptr->yc_styles;
	pixmaps = &core_ptr->yc_pixmaps;
	yc = &core_ptr->yc;


	/* YIFF Configuration Wizard window structure resources.
	 * This will destroy all widgets making up the window and
	 * disconnect from the Y server as needed.
	 */
	YCDestroy(yc);


	/* Pixmaps */
#define UNREF_PIXMAP	\
{ \
 if(pixmap_gdk != NULL) gdk_pixmap_unref(pixmap_gdk); \
 if(mask != NULL) gdk_bitmap_unref(mask); \
 if(pixmap != NULL) gtk_widget_destroy(pixmap); \
}

	pixmap_gdk = pixmaps->nariative_bg;
	mask = pixmaps->nariative_bg_mask;
	pixmap = NULL;
	UNREF_PIXMAP

	pixmap_gdk = pixmaps->title;
	mask = pixmaps->title_mask;
	pixmap = NULL;
	UNREF_PIXMAP

	pixmap_gdk = pixmaps->icon_yiff_48x48;
	mask = pixmaps->icon_yiff_48x48_mask;
	pixmap = NULL;
	UNREF_PIXMAP

	pixmap_gdk = pixmaps->icon_yiff_32x32;
	mask = pixmaps->icon_yiff_32x32_mask;
	pixmap = NULL;
	UNREF_PIXMAP

	pixmap_gdk = pixmaps->icon_wand_32x32;
	mask = pixmaps->icon_wand_32x32_mask;
	pixmap = NULL;
	UNREF_PIXMAP

	memset(pixmaps, 0x00, sizeof(yc_pixmaps_struct));

#undef UNREF_PIXMAP


	/* Styles */
#define UNREF_STYLE	\
{ if(style_ptr != NULL) gtk_style_unref(style_ptr); }

	styles->standard = NULL;	/* Never destroy standard style
					 * which came from GTK.
					 */
	style_ptr = styles->heading1_text;
	styles->heading1_text = NULL;
	UNREF_STYLE

	style_ptr = styles->heading2_text;
	styles->heading2_text = NULL;
	UNREF_STYLE

	style_ptr = styles->heading3_text;
	styles->heading3_text = NULL;
	UNREF_STYLE

	style_ptr = styles->heading4_text;
	styles->heading4_text = NULL;
	UNREF_STYLE

	style_ptr = styles->heading5_text;
	styles->heading5_text = NULL;
	UNREF_STYLE

	style_ptr = styles->heading6_text;
	styles->heading6_text = NULL;
	UNREF_STYLE

	style_ptr = styles->nariative_text;
	styles->nariative_text = NULL;
	UNREF_STYLE

	memset(styles, 0x00, sizeof(yc_styles_struct));

#undef UNREF_STYLE
}




int main(int argc, char **argv)
{
	gint i, status;
	gchar *arg;
	yc_core_struct *core_ptr;


	/* Reset manage function callback id code */  
	manage_timeout_cb_id = (guint)(-1);

	/* Parse basic arguments */
	for(i = 1; i < argc; i++)
	{
	    arg = argv[i];
	    if(arg == NULL)
		continue;

	    /* Help */
	    if(!g_strcasecmp(arg, "--help") ||
	       !g_strcasecmp(arg, "-help") ||
	       !g_strcasecmp(arg, "--h") ||
	       !g_strcasecmp(arg, "-h")
	    )
	    {
		g_print(PROG_HELP_MESG);
		return(0);
	    }
	    /* Version */
	    else if(!g_strcasecmp(arg, "--version") ||
		    !g_strcasecmp(arg, "-version")
	    )
	    {
		g_print("%s Version %s\n", PROG_NAME, PROG_VERSION);
		g_print("%s\n", PROG_COPYRIGHT);
		return(0);
	    }
	}


	/* Initialize */
	signal(SIGINT, YCSignalCB);
	signal(SIGTERM, YCSignalCB);
	signal(SIGSEGV, YCSignalCB);
	signal(SIGPIPE, YCSignalCB);

	gtk_init(&argc, &argv);

	CDialogInit();
	FPromptInit();
	PDialogInit();
	FileBrowserInit();

	core_ptr = (yc_core_struct *)g_malloc0(
	    sizeof(yc_core_struct)
	);
	status = YCInit(core_ptr, argc, argv);
	if(status)
	{
	    YCShutdown(core_ptr);
	    g_free(core_ptr);
	    return(1);
	}

	/* Set timeout (manage) function */
	manage_timeout_cb_id = gtk_timeout_add(
	    250,		/* In milliseconds */
	    YCManage, core_ptr
	);

	/* Enter main gtk loop */
	gtk_main();


	/* Shutdown */
	YCShutdown(core_ptr);
	g_free(core_ptr);

	FileBrowserShutdown();
	PDialogShutdown();
	FPromptShutdown();
	CDialogShutdown();

	return(0);
}
