/*
 * Program:     $RCSfile: client.c,v $  $Revision: 4.12 $
 *
 * Purpose:     UNIX application program that becomes a client for
 *              the Internet "youbin" service.
 *
 * Usage:       See global variable usage.
 *
 * Author:      K.Agusa     agusa@nuie.nagoya-u.ac.jp
 *              S.Yamamoto  yamamoto@nuie.nagoya-u.ac.jp
 *
 * Date:        1993/07/24
 * Modified:    $Date: 1996/05/07 07:01:45 $
 *
 * Copyright:   K.Agusa and S.Yamamoto  1993 - 1994
 *
 * The X Consortium, and any party obtaining a copy of these files from
 * the X Consortium, directly or indirectly, is granted, free ofcharge,
 * a full and unrestricted irrevocable, world-wide, paid up, royalty-free,
 * nonexclusive right and license to deal in this software and documentation
 * files (the "Software"), including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons who receive copies from any such
 * party to do so. This license includes without limitation a license to do
 * the foregoing actions under any patents of the party supplying this
 * software to the X Consortium.
 */

#ifndef lint
static char rcsid[] =
    "$Id: client.c,v 4.12 1996/05/07 07:01:45 yamamoto Exp yamamoto $";
#endif

#include <sys/param.h>
#include <netinet/in.h>
#include <errno.h>
#include <pwd.h>                /* For getpwuid(). */
#include <signal.h>             /* For kill(). */
#include <stdio.h>
#include <sys/file.h>           /* For open(). */

#include <sys/types.h>          /* For open() on Solaris. */
#include <sys/stat.h>           /* For open() on Solaris. */
#include <fcntl.h>              /* For open() on Solaris. */

#if defined(linux)
#include <unistd.h>             /* For getcwd(). on Linux. */
#endif

#include "youbin.h"

/*
 * Compatibility of getting terminal mode.
 * Note, select either USE_SGTTY or USE_TERMIO.
 *
 * USE_SGTTY:       structure sgttyb is available.
 * USE_TERMIO:      structure termio is available.
 */

#if !defined(USE_SGTTY) && !defined(USE_TERMIO)
#ifdef SYSV
#define USE_TERMIO
#else
#define USE_SGTTY
#endif
#endif

#ifdef USE_SGTTY
#include <sgtty.h>              /* For isatty() and sgttyb  
structure. */
#endif

#ifdef USE_TERMIO
#if !(defined(mips) && !defined(sgi))
#include <stdlib.h>             /* For isatty(). */
#endif
#include <termio.h>
#endif

/*
 * Compatibility of getting current working directory pathname.
 *
 * SYSV:        getcwd().
 * Otherwise:   BSD's getwd().
 */

#if (defined(SYSV) || defined(linux))
#define getwd(x)    getcwd(x, MAXPATHLEN)
#endif

static char *usage[] = {
    "NAME",
    "    youbin - client for the Internet \"youbin\" service",
    "SYNOPSIS",
    "    youbin [-f file][-a type][-b head][-d][-h][-m file][-o mode][-p pass][-s host]",
    "OPTIONS",
    "    -f file  Configuration File. Default is ~/.youbinrc",
    "    -a type  Authentification mode. Default is \"rok\".",
    "    -b head  Required Headers in Body,From:,To:,Subject:,cc:,Date:.",
    "    -d       Debug mode.",
    "    -h       Help: print out the usage.",
    "    -m file  Mail check: file checked by sh or csh as mail spool.",
    "    -o mode  outut mode",
    "             0 | none:	 Nothing is outputted.",
    "             1 | size_date: \"size date\\n\" at mail spool update",
    "                                 and at every unit time ( 3 min )",
    "             2 | for_shell: \"Mail_First\\n\" from empty to arrived",
    "                            \"Mail_Appended\\n\"  increasing size", 
    "                            \"Cleared\\n\" change state to empty",
    "             3 | mail_arrive:\"\007New mail for user@%server\007 has arrived:\"",
    "             4 | biff | others: mail_arrive message followed by headders.",
    "             Defaul value is",
    "                   0 with -m option,",
    "                  	1 wtihout -m option if stdout is a pipe,",
    "                   3 without -m option if stdout is a tty.",
    "    -s host  Host name of mail server.",
    "AUTHORS",
    "    K.Agusa     agusa@nuie.nagoya-u.ac.jp",
    "    S.Yamamoto  yamamoto@nuie.nagoya-u.ac.jp",
    NULL
};

/*
 * Global variables.
 */

char    user_name[USER_NAME_LEN + 1];       /* User name.   */
char    *prog_name;                         /* This program name.  
*/
char    print_flag;                         /* Print mode.  */
char    mail_check_name[MAXPATHLEN + 1];    /* For -m flag. */
char    server_name[MAXHOSTNAMELEN + 1];    /* Server name. */

char    *cr;
FILE    *input, *output;                    /* Pipe, stdin or  
stdout. */

/*
 * NONE:        Nothing is printed. ( Can't happen )
 * SIZE_DATE:	size+' '+date
 * FOR_SHELL:   Mail_First, Mail_Appended, Cleared
 * NOTIFY:      "New mail has arrived" is printed.
 * MESS_REQ:    Print some header fields and top of body are printed.
 */
enum    {NONE, SIZE_DATE, FOR_SHELL, NOTIFY, MESS_REQ} print_mode = NONE;

/* Mode flag. */
int     debug_mode = OFF;

int     ppid;                               /* Parent process id */
int     exitcode = EXIT_SUCCESS;

/*
 * External variables.
 */

extern int      errno;

/* For getopt(). */
extern char     *optarg;
extern int      optind, opterr;

/*
 * Local functions.
 */

char            *ctime();

int             youbin_sub();
void            init();
void            callback();
char            *dupopt();
FILE            *get_file();
char            *skipsptab();

int             make_writable_file();
void            send_signal();

char    *head_list = "";
char    *password = "";
char    *config_file = "";
char    *auth = "rok";

int
main(argc, argv)
int     argc;
char    **argv;
{
    int     result;
    int     pwlen;
 
    init(argc, argv );

    if( pwlen = strlen( password )) pwlen++;
    result = youbin_sub(server_name,user_name,auth,password,pwlen,head_list,0,callback,-1);
    if (mail_check_name[0] != '\0') {
        unlink(mail_check_name);
    }
    debug("%s: Exit code: %d\n", prog_name, result);
    return (result);
}

void
init(argc, argv)
int     argc;
char    **argv;
{
    int             c;
    struct passwd   *pwent;
    char            *fname;
    FILE            *fp;
    char            linebuff[MESS_LEN];
    char            *line;
#ifdef USE_SGTTY
    struct sgttyb   gtty;
#else
    struct termio   termio;
#endif
 
    if ((prog_name = strrchr(argv[0], '/')) != NULL) {
        prog_name++;
    } else {
        prog_name = argv[0];
    }

    opterr = 0;
    while ((c = getopt(argc, argv, "f:a:b:dho:m:p:s:")) != EOF) {
        switch (c) {
	  case 'a':
	    auth = optarg;
	    break;
	  case 'b':                         /* Biff compatible. */
            print_mode = MESS_REQ;
	    head_list = optarg;
            break;
          case 'd':                         /* Debug. */
            debug_mode = ON;
            break;
          case 'f':                         /* Configuration file. */
            config_file = optarg;
	    break;
          case 'h':                         /* Help. */
            print_usage(usage);
            exit(EXIT_SUCCESS);
            break;
          case 'm':                         /* Mail check. */
            strncpy(mail_check_name, optarg, MAXPATHLEN );
	    mail_check_name[MAXPATHLEN] = 0;
	    break;
          case 'o':                         /* Output mode. */
            print_flag = *optarg;
	    break;
	  case 's':                         /* Server host name. */
            strncpy(server_name, optarg, MAXHOSTNAMELEN );
            break;
          default:
            exitcode++;
            break;
        }
    }
        
    if((fp = (FILE *)get_file( config_file )) != NULL ){
      while( fgets( linebuff, MESS_LEN, fp ) != NULL ){
	if( *linebuff == '#' ) continue;
	linebuff[ strlen(linebuff) - 1 ] = 0;
	line = skipsptab( linebuff );
	if( *line == 0 ) continue;
	if( strncasecmp( "Password:", line, 9 ) == 0 ){
	  password = dupopt( line + 9 );
	} else if( strncasecmp( "Auth:", line, 5 ) == 0 ){
	  if( strcmp( auth, "rok" ) == 0 ){
	    auth = dupopt( line + 5 );
	  }
	} else if( strncasecmp( "Biff:", line, 5 ) == 0 ){
	  if( print_mode == NONE ){
	    print_mode = MESS_REQ;
	    head_list = dupopt( line + 5 );
	  }
	} else if( strncasecmp( "Mail:", line, 5 ) == 0 ){
	  if( *mail_check_name == 0 ){
	    strncpy(mail_check_name, skipsptab( line + 5 ), MAXPATHLEN );
	    mail_check_name[MAXPATHLEN -1 ] = 0;
	  }
	} else if( strncasecmp( "Print:", line, 6 ) == 0 ){
	  if( print_flag == 0 ){
	    print_flag = *( skipsptab( line + 6 ));
	  }
	} else if( strncasecmp( "Server:", line, 7 ) == 0 ){
	  if( *server_name == 0 ){
	    strncpy( server_name, skipsptab( line + 7 ), MAXHOSTNAMELEN );
	    server_name[MAXHOSTNAMELEN -1] = 0 ;
	  }
	} else if( strncasecmp( "User:", line, 5 ) == 0 ){
	  strncpy( user_name, skipsptab( line + 5 ), USER_NAME_LEN );
	  user_name[USER_NAME_LEN - 1] = 0 ;
	} else {
	  fprintf(stderr, "%s: Unknown options in file..%s\n", prog_name, line );
	  print_usage(usage);
	  exit(EXIT_FAILURE);
	}
      }
    }
    
    if (optind != argc || exitcode != EXIT_SUCCESS) {
        fprintf(stderr, "%s: Unknown options.\n", prog_name);
        print_usage(usage);
        exit(EXIT_FAILURE);
    }
 
    ppid = getppid();                       /* Save parent PID for exiting */

    if (mail_check_name[0] != '\0') {       /* Mail check file for shells. */
        if (make_writable_file(mail_check_name) == FAIL) {
            fprintf(stderr, "%s: No access right to mail check file: %s\n",
                    prog_name, mail_check_name);
            exit(EXIT_FAILURE);
        }
    }

    /* For future extention, set streams. */
    input = stdin;
    output = stdout;
    
    if( print_mode == NONE ){
    	switch( print_flag ){
           case 0: if( mail_check_name[0] != '\0' )
	             print_mode = NONE;
                   else
		     print_mode = isatty(fileno(output)) ? NOTIFY : SIZE_DATE;
		   break;
	   case '0':
	   case'n': print_mode = NONE;
		     break;
	   case '1':
	   case 's': print_mode = SIZE_DATE;
		     break;
	   case '2':
	   case 'f': print_mode = FOR_SHELL;
		     break;
	   case '3':
	   case 'm': print_mode = NOTIFY;
		     break;
	   case '4':
	   case 'b': 
	   default:  print_mode = MESS_REQ;
       }
     }

    /* Get local host name. */
    if (server_name[0] == '\0') {
        gethostname(server_name, sizeof(server_name));
    }
 
    /* Get user name. */
    if(( *user_name == 0 )||( strcmp( auth, "rok") == 0) ){
      if ((pwent = getpwuid(getuid())) != NULL) {
	strcpy(user_name, pwent->pw_name);
      } else {
	fprintf(stderr, "%s: Invalid user\n", prog_name);
	exit(EXIT_FAILURE);
      }
    }
#ifdef USE_SGTTY
    /* This code is available only on BSD. */
    (void)ioctl(fileno(stdout), TIOCGETP, &gtty);
    cr = ((gtty.sg_flags & CRMOD) && !(gtty.sg_flags & RAW)) ? "" : "\r";
#else
    (void)ioctl(fileno(stdout), TCGETA, &termio);
    cr = (((termio.c_oflag & OPOST) && (termio.c_oflag & ONLCR)) ||
          (termio.c_oflag & ONLRET)) ? "" : "\r";
#endif /* not SYSV */
}

char *dupopt( char *buff ){
  char   *dest;
  
  buff = skipsptab( buff );
  if(( dest =(char *)malloc( strlen( buff ) + 1 )) == NULL ){
    fprintf(stderr, "%s: Can not get memory.%s\n", prog_name );
    exit(EXIT_FAILURE);
  }
  strcpy( dest, buff );
  return dest;
}

char *skipsptab( char *data ){
  while( *data == ' ' || *data == '\t' ) data++;
  return data;
}

FILE *get_file( char *config_file ){
  int              c;
  struct passwd   *pwent;
  char            rcfile[MAXPATHLEN+1];
  char            *env = (char *) getenv ("HOME");
  FILE            *fp;

  if( *config_file == '\0' ){
    if (env)
      strcpy ( rcfile, env);
    else if ((pwent = (struct passwd *) getpwuid (getuid ()))
	     && pwent->pw_dir)
      strcpy ( rcfile, pwent->pw_dir);
    else
      return (FILE *)NULL;
    strcat ( rcfile, "/");
    strcat ( rcfile, ".youbinrc");
    return fopen ( rcfile, "r");
  } else {
    if(( fp = (FILE *)fopen( config_file, "r" )) == NULL ){
      fprintf(stderr, "%s: Can not open file..%s\n", prog_name, config_file );
      print_usage(usage);
      exit(EXIT_FAILURE);
    }
    return fp;
  }
}

/*
 * Callback Function
 */

void
callback(size, date, mess)
int         size;
time_t      date;
/* int         date; */
char        *mess;
{
    char            *cp;
    static int      saved_size = 0;
    static time_t   saved_date = 0;
    int		    prev_size;
    char            buff[MESS_LEN + 1];
    int             fid;
 
    /*
     * debug("    biff(): size = %d, date = %d, mess = \"%s\"\n",
     *       size, date, mess);
     */
 
    if (getppid() != ppid) {                      /* Parent process died. */
        send_signal(getpid(), SIGQUIT);
        return;
    }

    if( print_mode == SIZE_DATE ) {	          /* pass through          */
	    fprintf(output, "%d %d\n", size, date);
      	    fflush(output);
    }
    if ( size == saved_size && date == saved_date ) { 
						/*  Spool has not changed. */
	return;
    }

    prev_size = saved_size;
    saved_date = date;				/* Remember the values. */
    saved_size = size;
     
    switch( print_mode ){                       /* print out messages   */
    case FOR_SHELL:
      if( size == 0 ){
	fprintf(output, "Cleared\n");
      } else {
	fprintf(output, prev_size ? "Mail_Appended\n" : "Mail_First\n" );
      }
      fflush(output);
      break;
    case NOTIFY:
    case MESS_REQ:  /* Do not need to distinguish NOTIFY and MESS_REQ. */
      if( size != 0 ){
	fprintf(stdout, "%s\n\007New mail for %s@%s\007 has arrived:%s\n",
                cr, user_name, server_name, cr );
	if( *mess ) {
	  fprintf(stdout, "----%s\n", cr );
	  for (/* Empty. */; *mess != '\0'; mess = cp + 1) {
	    if ((cp = strchr(mess, '\n')) == NULL) {
	      break;
	    }
	    *cp = '\0';
	    fprintf(stdout,"%s%s\n", mess, cr);
	  }
	}
	fflush(stdout);				/* Message out */
      }
      break;
    }
    if (mail_check_name[0] != '\0') {          /* maintain mail check file  */
      if( size == 0 ){                         /* Truncate mail check file. */
	if (truncate( mail_check_name, 0L) == -1) {
	  fprintf(stderr, "%s: Can not truncate file: %s\n",
		  prog_name, mail_check_name);
	  perror("truncate:");
	}
      } else {                                  /* increase mail check file */
	if ((fid = open(mail_check_name, (O_WRONLY | O_APPEND))) == -1) {
	  fprintf(stderr,"%s: Can not re-open mail check file: %s\n",
		  prog_name, mail_check_name);
	  perror("open");
	  exit(EXIT_FAILURE);
	}
	sprintf(buff, "New mail for %s at %s", user_name, ctime(&date));
	/* Note, ctime() adds new line. */
	if (write(fid, buff, strlen(buff)) == -1) {
	  fprintf(stderr,"%s: Can not write to mail check file: %s\n",
		  prog_name, mail_check_name);
	  perror("write");
	  exit(EXIT_FAILURE);
	}
	close(fid);
      }
    }
}

/*
 * Check file.
 * (1) If it exists, it should be writable by the user.
 * (2) If not exists, the directory should be writable by the user.
 *     Then make a file with mode 0600.
 */

int
make_writable_file(name_arg)
char    *name_arg;
{
    char    *cp, name[MAXPATHLEN + 1], pathname[MAXPATHLEN + 1];
    int     fid;
 
    strcpy(name, name_arg);             /* Preserve argument. */

    if (access(name, (F_OK | W_OK)) == 0){
        if (truncate(name, 0L)){
            fprintf(stderr,"%s: Can not truncate file: %s\n", prog_name, name);
            perror("truncate");
            return (FAIL);
        }
        return (OK);                    /* File exists and writable  */
    }
    if (errno != ENOENT){
        perror("access");
        return (FAIL);
    }
    /* File does not exist. */
    if ((cp = strrchr(name, '/')) == NULL){
        /* File on currenct working directory. */
        if ((char*)getwd(pathname) == NULL){
            fprintf(stderr, "%s: Can not get working directry:  %s\n",
                    prog_name, pathname);
            return (FAIL);
        }
        cp = name;
    } else {                            /* File name with prefix. */
        *cp = '\0';
        strcpy(pathname, ((cp == name) ? "/" : name));
        cp++;
    }
    if (access(pathname, W_OK)){
        perror("access");
        return (FAIL);
    }
    strcat(pathname, "/");
    strcat(pathname, cp);
    if ((fid = open(pathname, (O_WRONLY | O_CREAT), 0600)) == -1){
        fprintf(stderr, "%s: Can not open file: %s\n", prog_name, pathname);
        perror("open");
        return (FAIL);
    }
    close(fid);
    if (chown(pathname, getuid(), getgid()) != 0) {
        fprintf(stderr, "%s: Can not change owner: %s\n", prog_name, pathname);
        return (FAIL);
    }
    return (OK);
}

void
send_signal(pid, sig)
int     sig;
{
    if (kill(pid, sig) == -1) {
        fprintf(stderr, "%s: Can not send signal %d to process %d\n",
                prog_name, sig, pid);
        perror("kill");
    }
}
