/** 
 *  Yudit Unicode Editor Source File
 *
 *  GNU Copyright (C) 2002  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2001  Gaspar Sinai <gsinai@yudit.org>  
 *  GNU Copyright (C) 2000  Gaspar Sinai <gsinai@yudit.org>  
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2,
 *  dated June 1991. See file COPYYING for details.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNES FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "swidget/SYuditInput.h"

/**
 * SYuditInput is receiving events from STextEdit
 * It insert glyphs underlined into editorID and 
 * when composition finished it sends it to the editor, after that 
 * it may put half finished compsitions in the buffer.
 * @param name is the input name.
 * @param _eif is the editor interface to modify text.
 * @param ed is the editor.
 */
SYuditInput::SYuditInput (const SString& name, SEditorIF* _eif, SEditor* ed):
     encoder (name), startIndex(0,0), endIndex (0,0)
{
  eif = _eif;
  editor = ed;
}

/**
 * Delete SYuditInput.
 * for normal operaztion use clear before deleting.
 */
SYuditInput::~SYuditInput ()
{
}

/**
 * If SYuditInput is present and working, STextEdit sends the
 * events here first for processing. 
 */
void
SYuditInput::keyPressed (SWindowListener::SKey key, const SString& _s,
            bool ctrl, bool shift, bool meta)
{ 
  SString s = _s;
  if (!eif->isEditable())
  {
    clear();
    editor->keyPressed (key, s, ctrl, shift, meta);
  }
  switch (key)
  {
  case SWindowListener::Key_BackSpace:
  case SWindowListener::Key_Delete:
   if (startIndex!=endIndex)
   {
     SString pre = encoder.preEditBuffer();
     SV_UCS4 pb = encoder.postEditBuffer();
     almostClear();
     (void) encoder.decode ("", false); /* clear */
     SString post = encoder.encode (pb);
     SString a = encoder.encode (SV_UCS4());
     if (a.size()) post.append (a);
     if (pre.size()) post.append (pre);
     if (post.size() > 0) post.truncate(post.size()-1);
     s = post;
   }
   else
   {
     editor->keyPressed (key, s, ctrl, shift, meta);
     return;
   }
   break;
  case SWindowListener::Key_Meta_R:
  case SWindowListener::Key_Meta_L:
  case SWindowListener::Key_Alt_R:
  case SWindowListener::Key_Alt_L:
  case SWindowListener::Key_Escape:
  case SWindowListener::Key_Home:
  case SWindowListener::Key_End:
  case SWindowListener::Key_Prior:
  case SWindowListener::Key_Next:
  case SWindowListener::Key_Up:
  case SWindowListener::Key_Down:
  case SWindowListener::Key_Left:
  case SWindowListener::Key_Right:
  case SWindowListener::Key_Enter:
  case SWindowListener::Key_Return:
  case SWindowListener::Key_F1:
  case SWindowListener::Key_F2:
  case SWindowListener::Key_F3:
  case SWindowListener::Key_F4:
  case SWindowListener::Key_F5:
  case SWindowListener::Key_F6:
  case SWindowListener::Key_F7:
  case SWindowListener::Key_F8:
  case SWindowListener::Key_F9:
  case SWindowListener::Key_F10:
  case SWindowListener::Key_F11:
  case SWindowListener::Key_F12:
    if (startIndex!=endIndex)
    {
      clear();
    }
    editor->keyPressed (key, s, ctrl, shift, meta);
    return;
  default:
    if (s.size()==0 && !(ctrl || meta)) return;
    if (ctrl || meta)
    {
      if (startIndex!=endIndex) clear();
      editor->keyPressed (key, s, ctrl, shift, meta); 
      return;
    }
  }
  if (s.size()==0) return;
  SString remaining = s;
  SV_UCS4 pbefore = encoder.postEditBuffer();
  SV_UCS4 u = encoder.decode (s);
  SV_UCS4 pafter = encoder.postEditBuffer();
  SV_UCS4 pedit;
  /* we just append ? */
  if (u.size() > 0 || pafter.size() != pbefore.size())
  {
    remaining = encoder.preEditBuffer();
    pedit = pafter;
    almostClear ();
    SEncoder utf8;
    editor->keyPressed (SWindowListener::Key_Send, utf8.encode(u), 
           false, false, false); 
  }
  /* process glyphs here */
  if (startIndex==endIndex)
  {
    startIndex = eif->getCaret();
    endIndex = startIndex;
  }
  unsigned int i;
  for (i=0; i<pedit.size(); i++)
  {
    SV_UCS4 u1;
    u1.append (pedit[i]);
    SGlyph g (0, u1, 0, 0, false, 0);
    g.lr = eif->getDirection()=='>';
    g.underlined = true;
    eif->directInsert (g, &endIndex);
  }
  for (i=0; i<remaining.size(); i++)
  {
    SV_UCS4 u1;
    u1.append (SS_UCS4((unsigned char)remaining[i]));
    SGlyph g (0, u1, 0, 0, false, 0);
    g.lr = eif->getDirection()=='>';
    g.underlined = true;
    eif->directInsert (g, &endIndex);
  } 
  preEditSize = remaining.size() + pedit.size();
}

/**
 * clear the input text.
 * @param tosend is true if the remaining characters should
 * be sent out.
 * @return true if it had a text.
 */
bool
SYuditInput::clear(bool tosend)
{
  SString toSend;
  if (preEditSize)
  {
    /* flush */
    SV_UCS4 u = encoder.decode ("", false);
    if (u.size() > 0)
    {
      SEncoder utf8;
      toSend = utf8.encode(u);
    }
//fprintf (stderr, "clear - preedit [%*.*s]\n", SSARGS(toSend));
  }
  /* first clear pre-edit */
  almostClear();
  // FIXME: bug workaround.
  //encoder.clear();
  encoder = SEncoder(encoder.getName());
  if (toSend.size())
  {
     if (tosend)
     {
       editor->keyPressed (SWindowListener::Key_Send, toSend, 
           false, false, false); 
     }
     return true;
  }
  return false;
}

/**
 * clear it.
 */
void
SYuditInput::almostClear()
{
  if (startIndex != endIndex)
  {
//fprintf (stderr, "clear - earse %u %u ->  %u %u\n", 
//    startIndex.line, startIndex.index, endIndex.line, endIndex.index);
    eif->endSelect ();
    eif->caretGoTo (startIndex);
    eif->eraseText (endIndex);
  }
  startIndex = STextIndex (0, 0);
  endIndex = startIndex;
  preEditSize = 0;
}

/**
 * return true if input method works fine.
 */
bool
SYuditInput::isOK()
{
  return encoder.isOK();
}
