/************************************************************************/
/* Module : imageutil.c					                */
/* Purpose: graphics module for mpsql	                                */
/* By     : Keith R. Davis					        */
/* Date   : 9/9/95					                */
/* Notes  : Modified from source  by Douglas Young                      */
/*          Copyright 1994 by Prentice Hall		                */
/************************************************************************/

#include <Xm/Xm.h>		/* motif lib header		*/
#include <Xm/PushB.h>		/* drawing area widget header	*/
#include <Xm/Label.h>           /* label widget header          */
#include <X11/xpm.h>		/* xpm lib header		*/

#include "imageutil.h"		/* image module header	        */

#define MAX(a,b) (a>b ? a : b)
#define PAD 2

/************************************************************************/
/* Function: InstallLabeledPixmap				        */
/* Purpose : places a pixmap & label  in a widgets label property 	*/
/* Params  : w : widget to place pixmap in		                */
/*	     xpmSensitive   : ptr to pixmap data                 	*/
/*           showLabelString : flag to show string desc. w/ image       */
/*                             0 no label / 1 show label                */
/* Returns : nothing			                                */
/************************************************************************/

void InstallLabeledPixmap ( Widget w, char **image, int showLabelString )
{
    XmString       label;
    XmFontList     fontlist;
    GC             gc, inverseGc;
    Dimension      width, height;
    unsigned int   pixmapWidth, pixmapHeight;
    XGCValues      values, stipple_values;
    int            junk, depth;
    Display       *display = XtDisplay ( w );
    unsigned char  alignment;
    XpmAttributes  attributes;
    XpmColorSymbol symbols[5];
    int            totalWidth, totalHeight;
    int            status;
    Pixmap         labelPixmap,
                   labelPixmap2,
                   xpmPixmap,
                   stipple;
    Colormap       cmap;
    static char    stippleBitmap[8] = { 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA, 0x55, 0xAA };

    /* Retrieve the values used by the given widget  */
 
    XtVaGetValues ( w, 
                    XmNlabelString, &label,
                    XmNfontList,    &fontlist,
                    XmNforeground,  &values.foreground, 
                    XmNbackground,  &values.background, 
                    XmNdepth,       &depth, 
                    XmNalignment,   &alignment,
                    XmNcolormap,    &cmap,
                    NULL );
   /*
    * Create two GCs, one to draw the text and copy the pixmaps
    * and another that can be used to erase a pixmap by filling
    * it with the background color of the label widget. Because
    * the normal GC is used by XmStringDraw, which modifies
    * the font attribute of the GC, allocate this GC using 
    * XtAllocateGC() and specify GCFont as modifiable.
    */

    gc = XtAllocateGC ( w, depth, 
                        GCForeground | GCBackground, 
                        &values, GCFont, 0 );
 
    values.foreground = values.background;
 
    inverseGc = XtGetGC ( w, 
                          GCForeground | GCBackground,
                          &values );

   /*
    * Set up the XpmColorSymbol array, binding the name "background"
    * to the actual background color of the widget.
    */
 
    symbols[0].name  = "background";
    symbols[0].value = NULL;
    symbols[0].pixel = values.background;
 
   /*
    * Set the resulting information in the attributes structure
    */
 
    attributes.colorsymbols = symbols;
    attributes.numsymbols   = 1;
 
   /*
    * Specify the visual, colormap and depth 
    * to be used and set the XpmAttributes mask.
    */

    attributes.colormap = cmap;
    attributes.depth    = depth;
    attributes.visual   = DefaultVisual ( display,
                                           DefaultScreen ( display ) );

    attributes.valuemask = XpmColorSymbols | XpmDepth | 
                           XpmColormap | XpmVisual;
 
   /*
    * Create the pixmap of the given image
    */
 
    status = XpmCreatePixmapFromData ( display,
				       DefaultRootWindow ( display ), 
                                       image,  &xpmPixmap,
                                       NULL, &attributes );

   /*
    * Compute the size of the label string and the given pixmap 
   */
 
   if(showLabelString)
     XmStringExtent ( fontlist, label, &width, &height );
   else {
     width = 0; height = 0;
   }
     
   XGetGeometry ( display, xpmPixmap, ( Window * ) &junk,   
                  (int *) &junk, (int *) &junk, 
                  &pixmapWidth, &pixmapHeight,    
                  ( unsigned int *) &junk, ( unsigned int *) &junk );

  /*
   * Compute the sum of the label and pixmap sizes.
   */

   totalHeight = pixmapHeight + height + PAD;
   totalWidth = totalHeight;

  /*
   * Create the final pixmap using the combined size and 
   * fill the pixmap with the background color of the widget 
   */
 

   /* Normal Pixmap */

   labelPixmap = 
               XCreatePixmap ( display, 
                               RootWindowOfScreen ( XtScreen ( w ) ),
                               totalWidth, totalHeight, depth );
  
   XFillRectangle ( display, labelPixmap, 
                    inverseGc, 0, 0, 
                    totalWidth, totalHeight );

   /* Insensitive Pixmap */

   labelPixmap2 = 
               XCreatePixmap ( display, 
                               RootWindowOfScreen ( XtScreen ( w ) ),
                               totalWidth, totalHeight, depth );
 
  /*
   * Copy the Xpm-created pixmap into the larger pixmap and
   * then draw the string below the pixmap.
   */

   XCopyArea ( display, xpmPixmap, labelPixmap, 
               gc, 0, 0, pixmapWidth, pixmapHeight, 
               ( totalWidth - pixmapWidth ) / 2, 
               0 );
  
   if(showLabelString) {
     XmStringDraw ( display, labelPixmap, fontlist, label,
		    gc, 0, pixmapHeight + PAD, totalWidth, 
		    alignment, XmSTRING_DIRECTION_L_TO_R, NULL );
   }

   /* create the stippled insensitive pixmap */

   stipple = XCreateBitmapFromData(display, RootWindowOfScreen ( XtScreen ( w ) ),
				   stippleBitmap, 8, 8);

   /* set the stippling values */
   stipple_values.foreground = values.background;
   stipple_values.fill_style = FillStippled;
   stipple_values.stipple    = stipple;

   XChangeGC ( display, gc, 
	       GCForeground | GCFillStyle | GCStipple, 
	       &stipple_values);

   XCopyArea(display, labelPixmap, labelPixmap2, gc,
	     0, 0, totalWidth, totalHeight, 0, 0);

   XFillRectangle(display, labelPixmap2, gc, 
		  0, 0, totalWidth, totalHeight);

  /*
   * Install the final pixmaps in the widget.
   */
   
   XtVaSetValues ( w,
                   XmNlabelPixmap,            labelPixmap,
		   XmNlabelInsensitivePixmap, labelPixmap2,
                   XmNlabelType,              XmPIXMAP,
                   NULL );
                   
  /*
   * Free the GCs, the initial pixmap, and the string retrieved
   * from the label widget.
   */

   XFreePixmap ( display, xpmPixmap );
   XFreePixmap ( display, stipple );
   XtReleaseGC ( w, gc);
   XtReleaseGC ( w, inverseGc );
   XmStringFree ( label );
}

/************************************************************************/
/* Function: SetupIcon                				        */
/* Purpose : installs pixmap as the apps window manager icon     	*/
/* Params  : shell : app's shell widget 		                */
/*	     xpmDesc : ptr to pixmap data of image to display	        */
/* Returns : nothing			                                */
/************************************************************************/

void SetupIcon ( Widget shell, char **xpmDesc )
{
    Pixmap          pix;
    XpmAttributes   attributes;
    int             status;
    Display        *dpy = XtDisplay ( shell );

   /*
    * Retrieve the depth and colormap used by this widget
    * and store the results in the corresponding field
    * of an XpmAttributes structure.
    */

    XtVaGetValues ( shell,
                    XmNdepth,    &attributes.depth,
                    XmNcolormap, &attributes.colormap,
                    NULL);
   /*
    * Specify the visual to be used and set the XpmAttributes mask.
    */

    attributes.visual = DefaultVisual ( dpy, DefaultScreen ( dpy ) );
    attributes.valuemask = XpmDepth | XpmColormap | XpmVisual;

   /*
    * Create the pixmap
    */

    status = XpmCreatePixmapFromData ( dpy,
                                        DefaultRootWindow ( dpy ), 
                                        xpmDesc, &pix, NULL,
                                        &attributes );

   /*
    * Install the icon_shell's window as the icon window.
    */

    XtVaSetValues(shell, XmNiconPixmap, pix, NULL);
}









