/*
 *   dspparse.c -- Parses input from a .dsp file and sets up a module environment
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

//#define  INCL_BASE
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "dspmgr.h"

#include "dspio.h"
#include "dspparse.h"
#include "dsppcmem.h"
#include "dspstruc.h"
#include "dspfxutl.h"
#include "dspipc.h"
#include "dspstcmp.h"                  // String compare function Prototypes

/****************************************************************************/
/* Parser-specific constants.                                               */
/****************************************************************************/

#define  MAXSTRINGSIZE 256
#define  RIFFHDRSIZE   12
#define  BYTEALIGN     1
#define  FIXUPENTRY    28
#define  TWOBYTES      2
#define  YES           1L

/****************************************************************************/
/* Change GetWordElement and GetDWordElement into macros for speed          */
/* and efficiency                                                           */
/****************************************************************************/

#define GETWORDELEMENT(_arg1, _arg2) _arg2=*((*_arg1)++)
#define GETDWORDELEMENT(_arg1, _arg2) _arg2=*((*_arg1)++)

/****************************************************************************/
/* Define constants for 32bit character tags.  All four-char tags           */
/* read from the DSP are treated as a ULONG for efficiency.                 */
/****************************************************************************/

#define  TAG_NULL      0
#define  TAG_BIOT      0X746F6962
#define  TAG_THDR      0X72646874
#define  TAG_STND      0X646E7473
#define  TAG_ACTV      0X76746361
#define  TAG_ALUO      0X6F756C61
#define  TAG_HWID      0X64697768
#define  TAG_DDS       0X20736464
#define  TAG_DS        0X20207364
#define  TAG_CS        0X20207363
#define  TAG_ISP       0X20707369
#define  TAG_SHDR      0X72646873
#define  TAG_NSTR      0X7274736E
#define  TAG_DSTR      0X72747364
#define  TAG_XTRN      0X6E727478
#define  TAG_PUBS      0X73627570
#define  TAG_IMAG      0X67616D69
#define  TAG_GPC       0X20637067
#define  TAG_ITCB      0X62637469
#define  TAG_IPC       0X20637069
#define  TAG_DMA       0X20616D64
#define  TAG_LDMA      0X616D646C
#define  TAG_CODE      0X65646F63
#define  TAG_DATA      0X61746164
#define  TAG_DSEG      0X67657364
#define  TAG_ISEG      0X67657369
#define  TAG_VSEG      0X67657376
#define  TAG_TSK       0X206B7374
#define  TAG_TOC       0X20636F74
#define  TAG_MHDR      0X7264686D
#define  TAG_OSIN      0X6E69736F
#define  TAG_TINS      0X736E6974
#define  TAG_GPCC      0X63637067
#define  TAG_ITCC      0X63637469
#define  TAG_MOD       0X20646F6D
#define  TAG_OS        0X2020736F
#define  TAG_TASK      0X6B736174
#define  TAG_RIFF      0X46464952
#define  TAG_OVLY      0x796c766f
#define  TAG_HW        0x20207768
#define  TAG_OSHD      0x6468736f
RVER Dspfile;                          /* Local Static Variable read from   */
                                       /* DSP file Version                  */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ParseModule                                            */
/*                                                                          */
/* DISCRIPTIVE NAME: Parse the .DSP file.                                   */
/*                                                                          */
/* FUNCTION:         Take a .DSP file, create a DSP module structure and    */
/*                   return a pointer to the module.                        */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     ParseModule                                            */
/*                                                                          */
/* INPUT:            pszFileName - pointer to the filename.                 */
/*                                                                          */
/* EXIT-NORMAL:      pprmod - pointer to module set.                        */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ParseModule(HFILE hfile,PSZ pszModName,PRMOD *ppRmod)
{

   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulSize;

   ULONG      ulRiffSize = 0;
   ULONG      ulFileRead = 0;
   PSZ        pVirtBuffer;

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspparse::ParseModule entry hfile %x pszModName %s\n",(int)hfile,pszModName);

   if (DSP_NOERROR != (ulRC = ProcessRIFF(hfile, &ulRiffSize, &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessMagic(hfile, &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessTOC(hfile, &ulFileRead, NULL, NULL,
                              ENTIRE_FILE)))
      return (ulRC);

   /*************************************************************************/
   /*   Verify the Module and create the memory for Rmod.                   */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = VerifyAllocMod(hfile, &ulBodySize, ppRmod,
                                             &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessModule(hfile, ulBodySize, ulRiffSize,
                                            ppRmod, &ulFileRead)))
      return (ulRC);

   /* Free the Buffer with the Existing Module Name                         */

   ulSize = strlen((*ppRmod)->MOD_pszName)+1;
   if (DSP_NOERROR != (ulRC = FreePerm((*ppRmod)->MOD_pszName, ulSize)))
      return (ulRC);

   /*************************************************************************/
   /*   Assign the Module Name to the Module.                               */
   /*   When we have pszVirtName for LoadModule API this will               */
   /*   be the code.                                                        */
   /*************************************************************************/

   if (pszModName != NULL) {

      ulSize = strlen(pszModName)+1;
      if (DSP_NOERROR != (ulRC = AllocatePerm(ulSize, (PVOID)&pVirtBuffer)))
         return (ulRC);

      strncpy(pVirtBuffer, pszModName, ulSize);
      (*ppRmod)->MOD_pszName = pVirtBuffer;
   }
   else
      (*ppRmod)->MOD_pszName = NULL;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspparse::ParseModule exit\n");
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ParseTask                                              */
/*                                                                          */
/* DISCRIPTIVE NAME: Parse the .DSP file.                                   */
/*                                                                          */
/* FUNCTION:         Take a .DSP file, create a DSP task structure and      */
/*                   return a pointer to the task.                          */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     ParseTask                                              */
/*                                                                          */
/* INPUT:            hfile       - File handle of DSP file.                 */
/*                   pszRealName - pointer to Task name.                    */
/*                   prmod       - parent module of this task.              */
/*                                                                          */
/* EXIT-NORMAL:      pprtsk - pointer to Task set.                          */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ParseTask(HFILE hfile,PSZ pszRealName, PSZ pszVirtName,
             PRMOD prmod, PRTSK *ppRtsk)
{

   RC         ulRC;
   PRSEG      prseg;
   PRSEG      prseghead;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulRiffSize = 0;
   ULONG      ulFileRead = 0;
   ULONG      ulFileOffset = 0;
   ULONG      aTag;
   PSZ        pVirtBuffer;
   ULONG      ulsize;



   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspparse::ParseTask entry hfile %x pszRealName %s pszVirtName %s prmod %x\n",
	   (int)hfile,pszRealName,pszVirtName,(int)prmod);
   if (DSP_NOERROR != (ulRC = ProcessRIFF(hfile, &ulRiffSize, &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessMagic(hfile, &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessTOC(hfile, &ulFileRead, pszRealName,
                                         &ulFileOffset, FIND_TASK_OFFSET)))
      return (ulRC);

   /*************************************************************************/
   /*   Move file pointer to task offset in DSP file.                       */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)ulFileOffset,
                                            FROMTOP)))
      return (ulRC);

   /*************************************************************************/
   /*   Add this task to the tail of task list.                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AddRTaskTail(&(prmod->MOD_prtskTail), ppRtsk)))
      return (ulRC);

   (*ppRtsk)->TSK_prmod = prmod;

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_TASK != aTag) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ParseTask error ulRC %x 'task' tag not found.\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = TASKSection(hfile, ulBodySize, &prmod, ppRtsk,
                                          FIND_TASK_OFFSET)))
      return (ulRC);

   /* Free the Buffer with the Real Name                                    */
   ulsize = strlen((*ppRtsk)->TSK_pszVirtName)+1;
   if (DSP_NOERROR != (ulRC = FreePerm((*ppRtsk)->TSK_pszVirtName, ulsize)))
      return (ulRC);

   /* Switch the Task's name to the Virtual Name if supplied else NULL      */
   if (pszVirtName != NULL) {

      /* Allocate Perm Buffer for Virtual Name                              */
      ulsize = strlen(pszVirtName)+1;
      if (DSP_NOERROR != (ulRC = AllocatePerm(ulsize, (PVOID)&pVirtBuffer)))
         return (ulRC);

      memcpy(pVirtBuffer, pszVirtName, ulsize);

      /* Update to point to virtual name                                    */

      (*ppRtsk)->TSK_pszVirtName = pVirtBuffer;

   }
   else
      (*ppRtsk)->TSK_pszVirtName = NULL;

   /*************************************************************************/
   /*  Now, we fill segment structures in segment list, start with          */
   /*  code segment list first, then data segment list.                     */
   /*************************************************************************/

   prseghead = (*ppRtsk)->TSK_prsegITail->SEG_prsegNext; // code seg list
   prseg = prseghead;

   do {
      if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)0L, FROMTOP)))
         return (ulRC);
      if (DSP_NOERROR != (ulRC = ParseSeg(hfile, prseg->SEG_pszRealSegName,
                                          *ppRtsk, &prseg, FIND_TASK_OFFSET)))
         return (ulRC);
      prseg = prseg->SEG_prsegNext;
   }  while (prseg != prseghead);      /* enddo                             */

   prseghead = (*ppRtsk)->TSK_prsegDTail->SEG_prsegNext; // data seg list
   prseg = prseghead;

   do {
      if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)0L, FROMTOP)))
         return (ulRC);
      if (DSP_NOERROR != (ulRC = ParseSeg(hfile, prseg->SEG_pszRealSegName,
                                          *ppRtsk, &prseg, FIND_TASK_OFFSET)))
         return (ulRC);
      prseg = prseg->SEG_prsegNext;
   }  while (prseg != prseghead);      /* enddo                             */

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspparse::ParseTask exit\n");
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ParseSeg                                               */
/*                                                                          */
/* DISCRIPTIVE NAME: Parse the .DSP file.                                   */
/*                                                                          */
/* FUNCTION:         Take a .DSP file, create a DSP segment structure and   */
/*                   return a pointer to the segment.                       */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     ParseSeg                                               */
/*                                                                          */
/* INPUT:            hfile       - File handle of DSP file.                 */
/*                   pszRealName - pointer to Task name.                    */
/*                   prtsk       - parent task of this segment.             */
/*                                                                          */
/* EXIT-NORMAL:      ppRseg - pointer to segment set.                       */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ParseSeg(HFILE hfile, PSZ pszRealName, PRTSK prtsk,
            PRSEG *ppRseg, USHORT usReadFlag)
{
   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulRiffSize = 0;
   ULONG      ulFileRead = 0;
   ULONG      ulFileOffset = 0;
   ULONG      aTag;

   MW_SYSLOG_4(TRACE_MANAGER_CORE,"dspparse::ParseSeg entry hfile %x pszRealName %s prtsk %x\n",
	   (int)hfile,pszRealName,(int)prtsk);
   if (DSP_NOERROR != (ulRC = ProcessRIFF(hfile, &ulRiffSize, &ulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ProcessMagic(hfile, &ulFileRead)))
      return (ulRC);

  /**************************************************************************/
  /* If ParseSeg called from ParseTask then we do not need to add seg       */
  /* to tail of segment list.                                               */
  /**************************************************************************/

   if (DSP_NOERROR != (ulRC = ProcessTOC(hfile, &ulFileRead, pszRealName,
                                         &ulFileOffset, FIND_SEGMENT_OFFSET)))
      return (ulRC);

  /**************************************************************************/
  /*   Move file pointer to segment offset in DSP file.                     */
  /**************************************************************************/

   if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)ulFileOffset,
                                            FROMTOP)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_CODE == aTag) {
      if (usReadFlag == FIND_SEGMENT_OFFSET) {
         if (DSP_NOERROR != (ulRC = AddRSegmentTail(&(prtsk->TSK_prsegITail),
                                                      ppRseg)))
            return (ulRC);

         (*ppRseg)->SEG_prsegtmp->SEG_hfile = hfile;
         (*ppRseg)->SEG_usflg |= SEG_INST;
      }
   }
   else
      if (TAG_DATA == aTag) {
         if (usReadFlag == FIND_SEGMENT_OFFSET) {
            if (DSP_NOERROR !=
                  (ulRC = AddRSegmentTail(&(prtsk->TSK_prsegDTail), ppRseg)))
               return (ulRC);

            (*ppRseg)->SEG_prsegtmp->SEG_hfile = hfile;
            (*ppRseg)->SEG_usflg |= SEG_DATA;
         }
      }
      else {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ParseSeg error ulRC %x 'data' tag not found\n",DSP_IO_INV_DSPFILE);
         return (DSP_IO_INV_DSPFILE);
      }

   (*ppRseg)->SEG_prtsk = prtsk;

   if (DSP_NOERROR != (ulRC = SEGSection(hfile, aTag,  // segment type
      ulBodySize, &ulFileOffset,       // needed for offset in DSP file
      &(prtsk->TSK_prmod),             // of binary image
      ppRseg, usReadFlag)))
      return (ulRC);

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspparse::ParseSeg exit\n");
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ProcessModule                                          */
/*                                                                          */
/* DISCRIPTIVE NAME: Fill in the information from the body into the module. */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     ProcessModule                                          */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   ulBodySize - Size of the Module.                       */
/*                                                                          */
/* EXIT-NORMAL:      File Read/Write pointer set to Next   section.         */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessModule(HFILE hfile, ULONG ulModSize, ULONG ulRiffSize,
                 PRMOD *ppRmod, PULONG pulFileRead)
{

   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      aTag;
   PRTSK      prTskNode;
   PBST       pbstDS,pbstIS,pbstDSLast,pbstISLast,pbstTemp;
   PRSEG      prSegNode;
   BOOL       bFound,bFirstPassIS,bFirstPassDS;
   INT        cmp;

   MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ProcessModule entry hfile %x\n",(int)hfile);

   if (DSP_NOERROR != (ulRC = MODSection(hfile, ulModSize, ppRmod,
                                         pulFileRead)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   while (TAG_OS == aTag) {

      if (DSP_NOERROR != (ulRC = OSSection(hfile, ulBodySize, ppRmod)))
         return (ulRC);

      *pulFileRead += ulTagSize;
      if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                            &ulTagSize)))
         return (ulRC);

   }                                   /* endwhile                          */
   while (TAG_TASK == aTag) {

      if (DSP_NOERROR != (ulRC = TASKSection(hfile, ulBodySize, ppRmod,
                                             NULL, ENTIRE_FILE)))
         return (ulRC);

      *pulFileRead += ulTagSize;
      if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                            &ulTagSize)))
         return (ulRC);

   }                                   /* endwhile                          */

   /*************************************************************************/
   /*  verify that a 'task' chunk was found for each 'tins'                 */
   /*************************************************************************/

   prTskNode = (*ppRmod)->MOD_prtskTail;
   do {
      if ((prTskNode->TSK_usflgType&TSK_USEMASK) == 0xFFFF)
         return (DSP_IO_INV_DSPFILE);
      prTskNode = prTskNode->TSK_prtskNext;
   }  while (prTskNode != (*ppRmod)->MOD_prtskTail);

   /*************************************************************************/
   /* Build a binary sorted tree structure of all segments in the           */
   /* module for use by the SEGSection function.  This is used only         */
   /* for a LoadModule request and is then destroyed.                       */
   /*************************************************************************/

   if ((ulRC = AllocatePerm((*ppRmod)->MOD_usNumDSSegs *sizeof(BST),
                            (PVOID)&pbstDS)) != DSP_NOERROR)
      return (ulRC);

   if ((ulRC = AllocatePerm((*ppRmod)->MOD_usNumISSegs *sizeof(BST), (PVOID)
      &pbstIS)) != DSP_NOERROR)
      return (ulRC);

   /* initialize 1st element in each tree                                   */

   (*ppRmod)->MOD_pbstIS = pbstIS;     /* init MOD structure pointer        */
   (*ppRmod)->MOD_pbstDS = pbstDS;
   pbstDS->prSeg = (*ppRmod)->MOD_prtskTail->TSK_prsegDTail;/* form tree    */
                                       /* root                              */
   pbstDS->pbstLeftson = 0;
   pbstDS->pbstRightson = 0;
   pbstIS->prSeg = (*ppRmod)->MOD_prtskTail->TSK_prsegITail;/* form tree    */
                                       /* root                              */
   pbstIS->pbstLeftson = 0;
   pbstIS->pbstRightson = 0;

   prTskNode = (*ppRmod)->MOD_prtskTail;
   pbstISLast = pbstIS;                /* points to last PBST in IS buffer  */
   pbstDSLast = pbstDS;                /* points to last PBST in DS buffer  */
   bFirstPassIS = TRUE;                /* allows duplicate name when        */
                                       /* inserting IS root                 */
   bFirstPassDS = TRUE;                /* allows duplicate name when        */
                                       /* inserting DS root                 */
   do {                                /* for all tasks                     */

      prSegNode = prTskNode->TSK_prsegITail;
      do {                             /* for all IS segments               */
         pbstTemp = pbstIS;            /* start at root of tree             */
         bFound = FALSE;
         while (!bFound) {             /* search tree                       */
            cmp = mwstrcmp(prSegNode->SEG_pszRealSegName,
               pbstTemp->prSeg->SEG_pszRealSegName);
            if (cmp < 0) {
               if (pbstTemp->pbstLeftson) {
                  pbstTemp = pbstTemp->pbstLeftson;
               }
               else {                  /* Add new leftson node              */
                  pbstTemp = pbstTemp->pbstLeftson = ++pbstISLast;
                  pbstTemp->prSeg = prSegNode;
                  pbstTemp->pbstLeftson = 0;/* init new node                */
                  pbstTemp->pbstRightson = 0;
                  bFound = TRUE;
               }                       /* endif                             */
            }
            else
               if (cmp > 0) {
                  if (pbstTemp->pbstRightson) {
                     pbstTemp = pbstTemp->pbstRightson;
                  }
                  else {               /* Add new rightson node             */
                     pbstTemp = pbstTemp->pbstRightson = ++pbstISLast;
                     pbstTemp->prSeg = prSegNode;
                     pbstTemp->pbstLeftson = 0;/* init new node             */
                     pbstTemp->pbstRightson = 0;
                     bFound = TRUE;
                  }                    /* endif                             */
               }
               else
                  if (bFirstPassIS)
                     bFound = TRUE;    /* Root -> duplicate OK              */
                  else {
		    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspparse::ProcessModule error ulRC %x Duplicate Code Seg: %s\n",DSP_DUPLICATE_SEGMENT_NAME,
                               prSegNode->SEG_pszRealSegName);
                     return (DSP_DUPLICATE_SEGMENT_NAME);
                  }

            bFirstPassIS = FALSE;
         }                             /* endwhile                          */
         prSegNode = prSegNode->SEG_prsegNext;
      }  while (prSegNode != prTskNode->TSK_prsegITail);/* enddo            */

      /* process DS segments                                                */

      prSegNode = prTskNode->TSK_prsegDTail;
      do {                             /* for all DS segments               */
         pbstTemp = pbstDS;            /* start at root of tree             */
         bFound = FALSE;
         while (!bFound) {             /* search tree                       */
            cmp = mwstrcmp(prSegNode->SEG_pszRealSegName,
               pbstTemp->prSeg->SEG_pszRealSegName);
            if (cmp < 0) {
               if (pbstTemp->pbstLeftson) {
                  pbstTemp = pbstTemp->pbstLeftson;
               }
               else {                  /* Add new leftson node              */
                  pbstTemp = pbstTemp->pbstLeftson = ++pbstDSLast;
                  pbstTemp->prSeg = prSegNode;
                  pbstTemp->pbstLeftson = 0;/* init new node                */
                  pbstTemp->pbstRightson = 0;
                  bFound = TRUE;
               }                       /* endif                             */
            }
            else
               if (cmp > 0) {
                  if (pbstTemp->pbstRightson) {
                     pbstTemp = pbstTemp->pbstRightson;
                  }
                  else {               /* Add new rightson node             */
                     pbstTemp = pbstTemp->pbstRightson = ++pbstDSLast;
                     pbstTemp->prSeg = prSegNode;
                     pbstTemp->pbstLeftson = 0;/* init new node             */
                     pbstTemp->pbstRightson = 0;
                     bFound = TRUE;
                  }                    /* endif                             */
               }
               else
                  if (bFirstPassDS)
                     bFound = TRUE;    /* Root -> duplicate OK              */
                  else {
		    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspparse::ProcessModule error ulRC %x Duplicate Data Seg: %s\n",DSP_DUPLICATE_SEGMENT_NAME,
                              prSegNode->SEG_pszRealSegName);
                     return (DSP_DUPLICATE_SEGMENT_NAME);
                  }

            bFirstPassDS = FALSE;
         }                             /* endwhile                          */
         prSegNode = prSegNode->SEG_prsegNext;
      }  while (prSegNode != prTskNode->TSK_prsegDTail);/* enddo            */
      prTskNode = prTskNode->TSK_prtskNext;
   }  while (prTskNode != (*ppRmod)->MOD_prtskTail);

   /*************************************************************************/
   /* Begin reading in segments from file                                   */
   /*************************************************************************/

   while (*pulFileRead <= ulRiffSize) {
      while ((TAG_CODE == aTag) || (TAG_DATA == aTag)) {
         if (DSP_NOERROR != (ulRC = SEGSection(hfile, aTag, ulBodySize,
                                    pulFileRead, ppRmod, NULL, ENTIRE_FILE)))
            return (ulRC);

         *pulFileRead += ulTagSize;
         if (*pulFileRead < ulRiffSize) {
            if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                                  &ulTagSize)))
               return (ulRC);
         }
         else {
            *pulFileRead = ulRiffSize+1;
            break;
         }
      }                                /* endwhile                          */
   }                                   /* endwhile                          */

   /* Freeup binary segment structure                                       */

   if ((ulRC = FreePerm(pbstDS, (*ppRmod)->MOD_usNumDSSegs *sizeof(BST))) !=
      DSP_NOERROR)
      return (ulRC);

   if ((ulRC = FreePerm(pbstIS, (*ppRmod)->MOD_usNumISSegs *sizeof(BST))) !=
      DSP_NOERROR)
      return (ulRC);

   (*ppRmod)->MOD_pbstDS = 0;
   (*ppRmod)->MOD_pbstIS = 0;

   MW_SYSLOG_1(TRACE_MANAGER_CORE,"dspparse::ProcessModule exit\n");

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  GetStringElement                                       */
/*                                                                          */
/* DISCRIPTIVE NAME: Get String Elements from the .DSP file.                */
/*                                                                          */
/* FUNCTION:         This routine will get string elements from the .DSP    */
/*                   file.  The elements are in the form of a DWORD length  */
/*                   for the string and the String of that length.          */
/*                                                                          */
/*                   The elements LENGTH-STRING pairs reside in a buffer    */
/*                   that presumably contains the contents of the body      */
/*                   of a tag.  Each time a read is done, a pointer is      */
/*                   updated (ppulLocator) within the buffer.               */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   .DSP file must follow the LENGTH - STRING guideline.   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     GetStringElement                                       */
/*                                                                          */
/* INPUT:            pBody - pointer to buffer that contains the body info. */
/*                   ppLocator - pointer to the next element to be read     */
/*                               within the buffer.                         */
/*                                                                          */
/* EXIT-NORMAL:      ppsz          - address of the pointer to where string */
/*                                   should go.                             */
/*                   ppLocator     - pointer to next string to read.        */
/*                   pulStringSize - address of Length of string read.      */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC GetStringElement(PSZ FAR *ppsz,PSZ FAR *ppLocator,PULONG pulStringSize)
{
   RC         ulRC;
   PSZ        pBuffer;
   PSZ        pTempLocator;
   ULONG      ulSize;

   pTempLocator = *ppLocator;
   ulSize = (ULONG)*pTempLocator++;    /* read BSTR size                    */
   if (ulSize) {

      if (DSP_NOERROR != (ulRC = AllocatePerm((ULONG)ulSize+1,
                                              (PVOID *)&pBuffer)))
         return (ulRC);
      *ppsz = memcpy(pBuffer, pTempLocator, ulSize);
      pBuffer[ulSize] = '\0';          /* add terminating zero              */
      pTempLocator += ulSize;          /* move ptr to end of string         */
   }
   else
      *ppsz = '\0';

   *ppLocator = pTempLocator;
   *pulStringSize = ulSize;

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ReadStringElement                                      */
/*                                                                          */
/* DISCRIPTIVE NAME: Get String Elements from the .DSP file and place in    */
/*                   user's buffer.  Replaces GetStringElement for temp     */
/*                   strings read into a char buffer.                       */
/*                                                                          */
/* FUNCTION:         This routine will get string elements from the .DSP    */
/*                   file.  The elements are in the form of a DWORD length  */
/*                   for the string and the String of that length.          */
/*                                                                          */
/*                   The elements LENGTH-STRING pairs reside in a buffer    */
/*                   that presumably contains the contents of the body      */
/*                   of a tag.  Each time a read is done, a pointer is      */
/*                   updated (ppulLocator) within the buffer.               */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   .DSP file must follow the LENGTH - STRING guideline.   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     ReadStringElement                                      */
/*                                                                          */
/* INPUT:            pBody - pointer to buffer that contains the body info. */
/*                   ppLocator - pointer to the next element to be read     */
/*                               within the buffer.                         */
/*                                                                          */
/* EXIT-NORMAL:      ppsz          - address of the pointer to where string */
/*                                   should go.                             */
/*                   ppLocator     - pointer to next string to read.        */
/*                   pulStringSize - address of Length of string read.      */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ReadStringElement(PSZ pCharbuffer, PSZ FAR *ppLocator, PULONG pulStringSize)
{
   PSZ        pTempLocator;
   ULONG      ulSize;

   pTempLocator = *ppLocator;
   ulSize = (ULONG)*pTempLocator++;    /* read BSTR size                    */
   if (ulSize) {
      memcpy(pCharbuffer, pTempLocator, ulSize);
      pCharbuffer[ulSize] = '\0';      /* add terminating zero              */
      pTempLocator += ulSize;          /* move ptr to end of string         */
   }
   else
      *pCharbuffer = '\0';

   *ppLocator = pTempLocator;
   *pulStringSize = ulSize;

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  GetStr                                                 */
/*                                                                          */
/* DISCRIPTIVE NAME: Creates permanent copy of string in memory pool.       */
/*                                                                          */
/* FUNCTION:         This routine allocates permanent memory for the        */
/*                   string passed to it and copies the string.             */
/* NOTE:                                                                    */
/*   DEPENDENCIES:                                                          */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     GetStr                                                 */
/*                                                                          */
/* INPUT:            pszString - pointer to 0 terminated string.            */
/*                   ppsz      - pointer to a psz to return                 */
/*                                                                          */
/* EXIT-NORMAL:      ppsz          - address of the pointer to where string */
/*                                   should go.                             */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC GetStr(PSZ pszString,PSZ FAR *ppsz)
{
   RC         ulRC;
   PSZ        pBuffer;
   ULONG      ulSize;

   ulSize = strlen(pszString);
   if (DSP_NOERROR != (ulRC = AllocatePerm((ULONG)(ulSize+1),
                                           (PVOID *)&pBuffer)))
      return (ulRC);

   *ppsz = memcpy(pBuffer, pszString, ulSize);/* pass back pointer          */
   pBuffer[ulSize] = '\0';             /* terminate zero                    */
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  GetStringTable                                         */
/*                                                                          */
/* DISCRIPTIVE NAME: Get String Tables from the .DSP file.                  */
/*                                                                          */
/* FUNCTION:         This routine will get string tables, both the          */
/*                   Discardable and Nondiscardable, from the .DSP file.    */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   .DSP file                                              */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     GetStringTable                                         */
/*                                                                          */
/* INPUT:            pBody - pointer to buffer that contains the body info. */
/*                   ulBodySize - Size of buffer with strings.              */
/*                   aTag - NSTR or DSTR depending on the type of strings.  */
/*                                                                          */
/* EXIT-NORMAL:      pprsTable     - ponter to the table.                   */
/*                   puCount       - pointer number of entries in table.    */
/*                   pulSizen      - pointer to total size of table.        */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC GetStringTable(PVOID pBody,ULONG ulBodySize,ULONG aTag,PVOID *pprsTable,
                   PUSHORT puCount,PULONG pulSize)
{

   RC         ulRC;
   PBYTE      pBuffer,ptr;
   USHORT     usStrcnt = 0;

   if (ulBodySize != 0) {              /* if entries in table, process.     */
      if (TAG_NSTR == aTag) {
         if (DSP_NOERROR != (ulRC = AllocatePerm(ulBodySize+1,
                                                 (PVOID)&pBuffer)))
            return (ulRC);
      }
      else
         if (DSP_NOERROR != (ulRC = AllocateTemp(ulBodySize+1, BYTEALIGN,
                                                 (PVOID)&pBuffer)))

            return (ulRC);

      *pprsTable = pBuffer;            /* return pointer to caller          */
      memcpy(pBuffer, pBody, ulBodySize);/* copy buffer                     */

      /* count number of BSTR in buffer                                     */

      ptr = pBuffer;
      while (ptr < pBuffer+ulBodySize) {
         ptr = ptr+*ptr+2;             /* ptr = ptr+strleng+2               */
         usStrcnt++;
      }                                /* endwhile                          */
   }                                   /* endif                             */
   else
      *pprsTable = NULL;

   *pulSize = ulBodySize+1;
   *puCount = usStrcnt;                /* number of BSTR in Buffer          */
   return (DSP_NOERROR);

}                                      /* end GetStringTable                */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  GetTagHdr                                              */
/*                                                                          */
/* DISCRIPTIVE NAME: Read a Tag Header from the users buffer and updates    */
/*                   users buffer pointer.  Replaces ReadTagHdr for getting */
/*                   the tag and length from a buffer rather than file.     */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     GetTagHdr                                              */
/*                                                                          */
/* INPUT:            pulCounter                                             */
/*                                                                          */
/* EXIT-NORMAL:                                                             */
/*                   pTag        - pointer to the Tag String                */
/*                   pulBodySize - pointer to body size.                    */
/*                   pulTagSize  - pointer to tag size.                     */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/************************** END OF SPECIFICATIONS ***************************/

RC GetTagHdr( PBYTE *ppbBuffer, PULONG pTag, PULONG pulBodySize,
              PULONG pulTagSize )
{
   typedef    struct {                 /* structure of TAG/SIZE header      */
      ULONG      Tag;
      ULONG      Bodysize;
   }
   TAGHEAD;
   PBYTE      pbBuffer;                /* local pointer to buffer           */
   ULONG      ulBodySize;              /* temp bodysize var                 */

   pbBuffer = *ppbBuffer;              /* get local copy of pointer         */
   if (1&(ULONG)pbBuffer)
      pbBuffer++;                      /* move to even boundary             */
   *pTag = ((TAGHEAD *)pbBuffer)->Tag; /* return 4-Char tag                 */

   ulBodySize = (((TAGHEAD *)pbBuffer)->Bodysize+1)&~1;/* round up          */

   *pulTagSize = ulBodySize+TAGHEADERSIZE;/* Return entire section size     */
   *ppbBuffer = pbBuffer+TAGHEADERSIZE;/* Update users pointer              */
   *pulBodySize = ulBodySize;          /* Return rounded body size          */
   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  MODSection                                             */
/*                                                                          */
/* DISCRIPTIVE NAME: Fill in the information from the MOD section into RMOD */
/*                   shell, create the Task Shell and do general processing */
/*                   (not detailed).                                        */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     MODSection                                             */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   ulBodySize - Size of the Module.                       */
/*                                                                          */
/* EXIT-NORMAL:      Module filled out with Name info and Task Shells       */
/*                   in place.                                              */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC MODSection( HFILE hfile, ULONG ulModSize,
               PRMOD *ppRmod, PULONG pulFileRead)
{
   RC         ulRC;
   ULONG      aTag;
   ULONG      ulBodySize;
   ULONG      ulRead;
   ULONG      ulStringSize;
   ULONG      ulTagSize;

   PVOID      pBody;
   PSZ        pulCounter;

   PRMOD      pRmod;
   PRTSK      prTskNode;
   PRGPCCL    prgpcclNode;
   PRITCBCL   pritcbclNode;

   ulRead = 0;
   ulBodySize = 0;
   pRmod = *ppRmod;

   /*************************************************************************/
   /*   Create memory for the Mod Body info.                                */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocateTemp(ulModSize, BYTEALIGN, &pBody)))
      return (ulRC);

   /* read entire MODSECTION into buffer                                    */

   if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulModSize, pBody)))
      return (ulRC);
   pulCounter = pBody;                 /* init buffer pointer               */

   /*************************************************************************/
   /*   Read each element of the Tag and process it accordingly.            */
   /*************************************************************************/

   while (ulRead < ulModSize) {

      if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                           &ulTagSize)))
         return (ulRC);

      ulRead += ulTagSize;

      if (TAG_MHDR == aTag) {

         pRmod->MOD_rver.bVersion = *pulCounter++;
         pRmod->MOD_rver.bRelease = *pulCounter++;
         GetStringElement(&(pRmod->MOD_pszName), &pulCounter, &ulStringSize);
      }

      else if ((TAG_OSIN == aTag) || (TAG_TINS == aTag)) {

         if (DSP_NOERROR != (ulRC = AddRTaskTail(&(pRmod->MOD_prtskTail),
                                                 &prTskNode)))
            return (ulRC);

         prTskNode->TSK_prmod = pRmod;

         if (TAG_OSIN == aTag)
            prTskNode->TSK_usflgType = TSK_OS;

         GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_pszFileName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_pszRealName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prTskNode->TSK_pszVirtName), &pulCounter,
            &ulStringSize);
      }

      else if (TAG_GPCC == aTag) {

         if (DSP_NOERROR != (ulRC =
                AddRGPCConnListTail(&(pRmod->MOD_prgpcclTail), &prgpcclNode)))
            return (ulRC);

         GetStringElement(&(prgpcclNode->GPCCL_pszGpc1FileName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc1ModName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc1TaskName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc1SegName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc1GPCName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc2FileName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc2ModName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc2TaskName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc2SegName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(prgpcclNode->GPCCL_pszGpc2GPCName),
            &pulCounter, &ulStringSize);
         GETWORDELEMENT((PUSHORT *)&pulCounter,
            prgpcclNode->GPCCL_usProtocol);
      }

      else if (TAG_ITCC == aTag) {

         if (DSP_NOERROR != (ulRC =
             AddRITCBConnListTail(&(pRmod->MOD_pritcbclTail), &pritcbclNode)))
            return (ulRC);

         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBPFileName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBPModName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBPTaskName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBPSegName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBPName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBSFileName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBSModName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBSTaskName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBSSegName),
            &pulCounter, &ulStringSize);
         GetStringElement(&(pritcbclNode->ITCBCL_pszITCBSName),
            &pulCounter, &ulStringSize);
      }

      else {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::MODSection error ulRC %x A tag is missing OR wrong: 'itcc','gpcc','osin','tins',mhdr'\n",DSP_IO_INV_DSPFILE);
         return (DSP_IO_INV_DSPFILE);
      }

   }                                   /* endwhile                          */

   /*************************************************************************/
   /*   Free the memory for the Mod Body info.                              */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulModSize)))
      return (ulRC);

   pulFileRead = pulFileRead;          /* Warning error                     */
   return (ulRC);

}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  OSSection                                              */
/*                                                                          */
/* DISCRIPTIVE NAME: Fill in the information from the OS section.           */
/*                                                                          */
/* FUNCTION:         Same and create the data and code segments for the OS. */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     OSSection                                              */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   ulCodeSize - Size of the Code.                         */
/*                   ppRmod  - address of pointer to the module.            */
/*                                                                          */
/* EXIT-NORMAL:     TASK filled out with Name info and Code and Data Shells */
/*                  in place.                                               */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC OSSection( HFILE hfile, ULONG ulOSSize, PRMOD *ppRmod)
{
   RC         ulRC;
   ULONG      ulRead;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulStringSize;
   ULONG      aTag;

   PSZ        pulCounter;
   PVOID      pBody;

   PRTSK      prTskNode;
   PRSEG      prSegNode;
   BOOL       bFound;
   CHAR       cTempBuffer[MAXSTRINGSIZE];

   ulRead = 0;

   /*************************************************************************/
   /*   Create memory for the OS Task Body info.                            */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocateTemp(ulOSSize, BYTEALIGN, &pBody)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulOSSize, pBody)))
      return (ulRC);
   pulCounter = pBody;                 /* initialize buffer pointer         */

   /*************************************************************************/
   /*   Read each element of the Tag and process it accordingly.            */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the OSHD section.  This is the header information.             */
   /*   This section MUST be in the file.                                   */
   /*   Get the name of the task from the body and find the task            */
   /*   with the same real name.                                            */
   /*************************************************************************/

   if (TAG_OSHD == aTag) {

      prTskNode = (*ppRmod)->MOD_prtskTail;

      /* the version comes before the task name for OSHD                    */

      prTskNode->TSK_rver.bVersion = *pulCounter++;
      prTskNode->TSK_rver.bRelease = *pulCounter++;
      ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

      /**********************************************************************/
      /*  loop through the tasks until the names match                      */
      /**********************************************************************/

      bFound = FALSE;
      do {
         if (mwstrcmp(cTempBuffer, prTskNode->TSK_pszVirtName) == 0)
            bFound = TRUE;
         else
            prTskNode = prTskNode->TSK_prtskNext;

      }  while (!bFound && (prTskNode != (*ppRmod)->MOD_prtskTail));/* enddo*/
      if (!bFound)
         return (DSP_IO_INV_DSPFILE);

      GETDWORDELEMENT((PULONG *)&pulCounter, prTskNode->TSK_ulCPF);

   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::OSSection error ulRC %x 'oshd' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the DDS  section.  Default Data Segment.                       */
   /*   This section MUST be in the file.                                   */
   /*   ?? This creates a Data Segment Structure and add it to the          */
   /*   tail of the Segments hanging off the task.                          */
   /*   Fill the segment in with its name.                                  */
   /*************************************************************************/

   if (TAG_DDS == aTag) {

      if (DSP_NOERROR != (ulRC = AddRSegmentTail(&(prTskNode->TSK_prsegDTail),
                                                 &prSegNode)))
         return (ulRC);

      (*ppRmod)->MOD_usNumDSSegs++;

      GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
         &ulStringSize);

      /**********************************************************************/
      /*      Since realname always equals virtname for OS                  */
      /*      make pointers the same rather than a copy.                    */
      /**********************************************************************/

      prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

      prSegNode->SEG_usflg = SEG_DATA|SEG_DDS;
      prSegNode->SEG_prtsk = prTskNode;
      prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
      prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;
   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::OSSection error ulRC %x 'dds ' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the Data Segments (DS) and the Code Segments (CS) and          */
   /*   Add each to their respective linked lists.                          */
   /*   CS must occur at least once but can occur more than once.           */
   /*   DS can occur zero or more times.                                    */
   /*   Fill the segment in with its name.                                  */
   /*************************************************************************/

   while (TAG_DS == aTag) {

      if (DSP_NOERROR != (ulRC = AddRSegmentTail(&(prTskNode->TSK_prsegDTail),
                                                 &prSegNode)))
         return (ulRC);

      (*ppRmod)->MOD_usNumDSSegs++;

      GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
         &ulStringSize);

      /**********************************************************************/
      /*      Since realname always equals virtname for OS                  */
      /*      make pointers the same rather than a copy.                    */
      /**********************************************************************/

      prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

      prSegNode->SEG_usflg = SEG_DATA;
      prSegNode->SEG_prtsk = prTskNode;
      prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
      prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;
      if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                           &ulTagSize)))
         return (ulRC);

      ulRead += ulTagSize;

   }

   if (TAG_CS == aTag) {
      while (ulRead <= ulOSSize) {

         if (DSP_NOERROR != (ulRC =
                   AddRSegmentTail(&(prTskNode->TSK_prsegITail), &prSegNode)))
            return (ulRC);

         (*ppRmod)->MOD_usNumISSegs++;

         prSegNode->SEG_usflg = SEG_INST;
         prSegNode->SEG_prtsk = prTskNode;
         prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
         prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;
         GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
            &ulStringSize);

         /*******************************************************************/
         /*      Since realname always equals virtname for OS               */
         /*      make pointers the same rather than a copy.                 */
         /*******************************************************************/

         prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

         /*******************************************************************/
         /*   Do reads until the amount read is equal to the                */
         /*   total size If they match, break out of the loop.              */
         /*******************************************************************/

         if (ulRead < ulOSSize) {

            if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag,
                                                 &ulBodySize, &ulTagSize)))
               return (ulRC);

            ulRead += ulTagSize;

         }
         else
            break;                     /* break out of while loop.          */
      }                                /* end while                         */
   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::OSSection error ulRC %x 'cs ' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   /*************************************************************************/
   /*   Free the memory for the TASK Body info.                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulOSSize)))
      return (ulRC);

   return (DSP_NOERROR);

}                                      /* end OSSection                     */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ProcessMagic                                           */
/*                                                                          */
/* DISCRIPTIVE NAME: Process the RWMS Header in the .DSP file.              */
/*                                                                          */
/* FUNCTION:         Verifies that the file has a RWMS identifier.          */
/*                   At the moment this file just reads the body to         */
/*                   increment the Read/Write pointer.                      */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     ProcessMagic                                           */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                                                                          */
/* EXIT-NORMAL:      File Read/Write pointer set to Next   section.         */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessMagic(HFILE hfile, PULONG pulFileRead)
{
   PVOID      pBuffer;
   PSZ        pulCounter;
   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      aTag;

   /*************************************************************************/
   /*   Read tag in and check that its identifier is ISP.                   */
   /*************************************************************************/
   /*   If its ISP ,  point    the Read/Write file pointer to               */
   /*   the next section.                                                   */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_ISP != aTag) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ProcessMagic error ulRC %x 'isp ' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   /*************************************************************************/
   /*   If its ISP , read in the rest of the ISP  header.                   */
   /*   The pointer will get bumped to point to the next section.           */
   /*   Hopefully, it'll be the TOC    section.                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocateTemp(ulBodySize, BYTEALIGN, &pBuffer)))
      return (ulRC);

   if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulBodySize, pBuffer)))
      return (ulRC);

   *pulFileRead += ulTagSize;

   /*************************************************************************/
   /*   Get the DSP file version.                                           */
   /*************************************************************************/

   pulCounter = pBuffer;
   Dspfile.bVersion = *pulCounter++;
   Dspfile.bRelease = *pulCounter++;

   /*************************************************************************/
   /*   Put the processing of the ISP  info here.                           */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = FreeTemp(pBuffer, ulBodySize)))
      return (ulRC);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ProcessRIFF                                            */
/*                                                                          */
/* DISCRIPTIVE NAME: Process the RIFF Header in the .DSP file.              */
/*                                                                          */
/* FUNCTION:         Verifies that the file has a RIFF identifier and if so,*/
/*                   increments the pointer past this header.               */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     ProcessRIFF                                            */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                                                                          */
/* EXIT-NORMAL:      File Read/Write pointer set to "RWMS" section.         */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessRIFF( HFILE hfile, PULONG pulRiffSize, PULONG pulFileRead)
{

   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      aTag;

   /*************************************************************************/
   /*   Read tag in and check that its identifier is "RIFF".                */
   /*   If its RIFF, increment the Read/Write file pointer to               */
   /*   the next section.(RWMS)                                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, (PVOID)&aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_RIFF == aTag) {
      if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)RIFFHDRSIZE,
                                               FROMTOP)))
         return (ulRC);

      *pulRiffSize = ulBodySize;
      *pulFileRead += RIFFHDRSIZE;
   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ProcessRIFF error ulRC %x 'RIFF ' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  ProcessTOC                                             */
/*                                                                          */
/* DISCRIPTIVE NAME: Process the Table of Contents in the .DSP file.        */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     ProcessTOC                                             */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                                                                          */
/* EXIT-NORMAL:      File Read/Write pointer set to Next   section.         */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ProcessTOC( HFILE hfile, PULONG pulFileRead, PSZ pszRealName,
               PULONG pulFileOffset,USHORT usReadFlag)
{

   RC         ulRC;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      aTag;

   /*************************************************************************/
   /*   Read tag in and check that its identifier is TOC.                   */
   /*************************************************************************/
   /*   If its TOC read the body so the Read/Write file pointer             */
   /*   points to the next section.(RWMS)                                   */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_TOC != aTag) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::ProcessTOC error ulRC %x 'toc' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   /*************************************************************************/
   /*   If its TOC, and flag is ENTIRE_FILE read in the rest of the TOC     */
   /*   header. The pointer will get bumped to point to the next section.   */
   /*   Hopefully, it'll be the MODULE section.                             */
   /*   If flag is READ_TASK_OFFSET or READ_SEGMENT_OFFSET then call        */
   /*   SearchTOC to find offset.                                           */
   /*************************************************************************/

   if ((usReadFlag == FIND_TASK_OFFSET) ||
      (usReadFlag == FIND_SEGMENT_OFFSET)) {
      if (DSP_NOERROR != (ulRC = SearchTOC(hfile, pszRealName, pulFileOffset,
                                           usReadFlag)))
         return (ulRC);
   }

   else
      if (usReadFlag == ENTIRE_FILE) {

         DspChgFilePtr(hfile, ulBodySize, FROMCURRENT);
         *pulFileRead += ulTagSize;

      }
      else
         return (DSP_ERROR);

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS ***********************  */
/*                                                                          */
/* SUBROUTINE NAME:  RecordBios                                             */
/*                                                                          */
/* DISCRIPTIVE NAME: Record the BIOS Tasks in the File specified.           */
/*                                                                          */
/* FUNCTION:         Record the Bios Tasks in the file specified and        */
/*                   hang them off the MGR structure.                       */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     RecordBios                                             */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   hfile - Handle to the file.                            */
/*                                                                          */
/* EXIT-NORMAL:                                                             */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ************************  */

RC RecordBios( PRMGR prmgr, PSZ pszfilename)
{

   RC         ulRC;
   PSZ        pszTaskName=0;
   PSZ        pulCounter;
   PRBIOT     prbiot;
   PVOID      pBuffer = NULL;
   ULONG      aTag;
   ULONG      ulStringSize;
   ULONG      ulRiffSize;
   ULONG      ulFileRead;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulCount;
   USHORT     usTemp,i,j;
   USHORT     usTaskCount;
   USHORT     usSizeTemp;
   USHORT     usCount;
   HFILE      hfile;
   char       tempBuffer[MAXSTRINGSIZE];

   /* Verify that the parms are valid and Open the file.                    */

   if ((pszfilename == NULL) || (prmgr == NULL)) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::RecordBIOS error ulRC %x The filename is NULL OR prmgr is NULL\n",DSP_INV_PARAMETER);
      return (DSP_INV_PARAMETER);
   }

   if ((ulRC = DspOpenFile(pszfilename, &hfile, DSPOPENBINARY|DSPOPENRDONLY))
      != DSP_NOERROR)
      return (ulRC);

   /* Verify that we are dealing with a Correct .DSP File.                  */

   if (ulRC == DSP_NOERROR)
      ulRC = ProcessRIFF(hfile, &ulRiffSize, &ulFileRead);

   if (ulRC == DSP_NOERROR)
      ulRC = ProcessMagic(hfile, &ulFileRead);

   /*************************************************************
    *   Read tag in and check that its identifier is TOC.
    *   If its TOC read the body so the Read/Write file pointer
    *   points to the next section.(RWMS)
    *************************************************************
    *   Read tag in and check that its identifier is BIOT
    *   If its a BIOT search the BIOS tasks and Record them in
    *   the MGR.
    **************************************************************          */

   if (ulRC == DSP_NOERROR)
      ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize, &ulTagSize);

   if (ulRC == DSP_NOERROR) {
      if (TAG_TOC != aTag) {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::RecordBIOS error ulRC %x 'toc ' tag not found\n",DSP_IO_INV_DSPFILE);
         return (DSP_IO_INV_DSPFILE);
      }
   }

   if (ulRC == DSP_NOERROR)
      ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize, &ulTagSize);

   if (ulRC == DSP_NOERROR) {
      if (TAG_BIOT != aTag) {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::RecordBIOS error ulRC %x 'biot' tag not found\n",DSP_NO_BIOS_TASKS);
         return (DSP_NO_BIOS_TASKS);
      }
   }

   /*************************************************************
    *   For each BIOS task, allocate a control block for each
    *   Task and add the BIOS task "pool" for the MGR input.
    *   Read the Number of BIOS tasks and process each.
    *   Note that only certain information goes in the control
    *   Block.
    **************************************************************          */

   if (ulRC == DSP_NOERROR)
      ulRC = AllocateTemp(ulBodySize, BYTEALIGN, &pBuffer);

   if (ulRC == DSP_NOERROR)
      ulRC = DspReadFile(hfile, ulBodySize, pBuffer);

   if (ulRC == DSP_NOERROR) {
      pulCounter = pBuffer;
      GETWORDELEMENT((PUSHORT *)&pulCounter, usTaskCount);
       
      PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspparse::RecordBIOS entry");
      PRINT_DEBUG(EMBEDFILEPTR, "  usTaskCount %x\n", usTaskCount);
      PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);

      for (i = 0; i < usTaskCount; i++) {

         if (ulRC == DSP_NOERROR) {
            GetStringElement(&pszTaskName, &pulCounter, &ulStringSize);

            /* Offset in DSP file of Task Header, ignore for now.           */

            GETDWORDELEMENT((PULONG *)&pulCounter, ulCount);

            /* Read the HWID Count for the Task                             */

            GETWORDELEMENT((PUSHORT *)&pulCounter, usCount);

			if (pszTaskName) {
			PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspparse::RecordBIOS");
			PRINT_DEBUG(EMBEDFILEPTR, "  pszTaskName %s", pszTaskName);
			PRINT_DEBUG(EMBEDFILEPTR, "  ulCount %x", ulCount);
			PRINT_DEBUG(EMBEDFILEPTR, "  usCount %x\n", usCount);
			PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);
			}
            /* Read the HWIDs for the Task                                  */

            for (j = 0; j < usCount; j++) {
               if (Dspfile.bRelease < 8) {
                  GETWORDELEMENT((PUSHORT *)&pulCounter, usTemp);
               }
               else
                ReadStringElement(tempBuffer, &pulCounter, &ulStringSize);
            }                          /* endfor                            */

            /****************************************************************/
            /* Read the Connector Count for the Task,                       */
            /* For each connector in the Task, create a BIOT control        */
            /* Block and add it to the "pool" of BIOS Tasks for the MGR.    */
            /* Fill the Block in with Connector Name, Task Name, File Name. */
            /****************************************************************/

            GETWORDELEMENT((PUSHORT *)&pulCounter, usCount);
			PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspparse::RecordBIOS");
			PRINT_DEBUG(EMBEDFILEPTR, "  Connector Count %x\n", usCount);
			PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);

            for (j = 0; j < usCount; j++) {
               ulRC = AddRBiotTail(&(prmgr->MGR_prBiotTail), &prbiot);

               if (ulRC == DSP_NOERROR) {

                  GetStringElement(&(prbiot->BT_pszConnector), &pulCounter,
                     &ulStringSize);
				  if (prbiot->BT_pszConnector) {
					PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspparse::RecordBIOS");
					PRINT_DEBUG(EMBEDFILEPTR, "  Connector name %s\n", prbiot->BT_pszConnector);
					PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);
				  }
                  prbiot->BT_pszTaskname = pszTaskName;

                  usSizeTemp = (USHORT)(strlen(pszfilename)+1);
                  ulRC = AllocatePerm((ULONG)usSizeTemp, (PVOID *)
                     &(prbiot->BT_pszFilename));

                  if (ulRC == DSP_NOERROR) {
                     strcpy(prbiot->BT_pszFilename, pszfilename);
                  }
                  else
                     break;

               }
               else
                  break;
            }                          /* endfor                            */
         }
         else
            break;
      }                                /* endfor                            */
   }                                   /* endif                             */
   DspCloseFile(hfile);

   if (pBuffer != NULL)
      ulRC = FreeTemp(pBuffer, ulBodySize);

   PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a", "dspparse::RecordBIOS exit");
   PRINT_DEBUG(EMBEDFILEPTR, "  ulRC %x\n", ulRC);
   PRINT_DEBUG_FCLOSE(EMBEDFILEPTR);

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  SEGSection                                             */
/*                                                                          */
/* DISCRIPTIVE NAME: Fill in the information from the CODE or DATA section. */
/*                                                                          */
/* FUNCTION:  This function reads the entire segment section into a buffer  */
/*            and processes tags contained in the chunk.  A pointer         */
/*            (pulCounter) is moved through the buffer and acts similar to  */
/*            a file pointer.  Segments are processed differently depending */
/*            on whether they are loaded as part of a LoadModule, LoadTask, */
/*            or LoadSegment call.  For LoadModule, all segments that are   */
/*            linked to the module structure are searched to determine where*/
/*            the info is to go.  For LoadTask both a prtsk and prseg is    */
/*            passed to the function but the segment structure was created  */
/*            by the TaskSection function and still needs to be filled in.  */
/*            For LoadSegment the seg structure needs to be created and     */
/*            filled in.                                                    */
/*                                                                          */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/* ENTRY POINTS:     SEGSection                                             */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   ulCodeSize - Size of the Code.                         */
/*                   ppRmod  - address of pointer to the module.            */
/*                                                                          */
/* EXIT-NORMAL:     TASK filled out with Name info and Code and Data Shells */
/*                  in place.                                               */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC SEGSection( HFILE hfile, ULONG ulSegType, ULONG ulSegSize,
               PULONG pulFileRead, PRMOD *ppRmod, PRSEG *ppRseg,
               USHORT usReadFlag)
{
   RC         ulRC;
   PRGPC      prGPCNode;
   PRIMG      prImageNode;
   PRITCB     prITCBNode;
   PRLDMA     prldmaNode;
   PRSEG      prSegNode=0;
   PRTSK      prTskNode;
   PSZ        pulCounter;
   PSZ        pszTemp;
   PSZ        pBuffer;
   PVOID      pBody;
   PVOID      pTemp;
   PBST       pbstTemp;
   RPUB       *pRpub;

   ULONG      ulRead;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulStringSize;
   ULONG      ulTemp;
   ULONG      aTag;
   ULONG      ulstride;                /* Used for calculation of stride    */

   PARSTBL    parstbl;
   BOOL       bFound;
   INT        cmp;
   USHORT     usName,usTemp;
   CHAR       cTempBuffer[MAXSTRINGSIZE];  // buffer for tempstrings

   ulRead = 0;
   ulBodySize = 0;

   /*************************************************************************/
   /*   Create memory for the Code Body info.                               */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocateTemp(ulSegSize, BYTEALIGN, &pBody)))

      return (ulRC);

   /* read entire segment body into buffer                                  */

   if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulSegSize, pBody)))
      return (ulRC);
   pulCounter = pBody;                 /* set pointer to beginning of buffer*/

   /*************************************************************************/
   /*   Read each element of the Tag and process it accordingly.            */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the SHDR section.  This is the header information.             */
   /*   This section MUST be in the file.                                   */
   /*   If the segment is a code segment, traverse the Code segments        */
   /*   otherwise traverse the data segments.                               */
   /*   Get the name of the Segment from the body traverse the code         */
   /*   segments in each of the tasks and fill in the info.                 */
   /*************************************************************************/
   /*************************************************************************/
   /*   usReadFlag control                                                  */
   /*     ENTIRE_FILE          - LoadModule processing                      */
   /*     READ_SEGMENT_OFFSET  - LoadSegment processing                     */
   /*************************************************************************/

   if ((TAG_SHDR == aTag) && (usReadFlag == ENTIRE_FILE)) {

      prTskNode = (*ppRmod)->MOD_prtskTail->TSK_prtskNext;

      ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

       /*********************************************************************/
       /* Loop through the segments through the tasks until the segment's   */
       /* Virtual name matches the name in the .DSP file.                   */
       /*********************************************************************/

      bFound = FALSE;

       /*********************************************************************/
       /* This section searches a binary tree to determine which RSEG       */
       /* is to be filled in by the data read from the file.  When the      */
       /* section is exited prSegNode points to the correct RSEG iff        */
       /* bFound=TRUE.  If the segment name was not found bFound=False.     */
       /*********************************************************************/

      if (TAG_CODE == ulSegType) {
         pbstTemp = (*ppRmod)->MOD_pbstIS;
      }
      else {
         pbstTemp = (*ppRmod)->MOD_pbstDS;
      }
      do {
         cmp = mwstrcmp(cTempBuffer, pbstTemp->prSeg->SEG_pszRealSegName);/**/
         if (cmp < 0) {                /* search left nodes                 */
            pbstTemp = pbstTemp->pbstLeftson;
         }
         else
            if (cmp > 0) {             /* search right nodes                */
               pbstTemp = pbstTemp->pbstRightson;
            }
            else {                     /* found it!                         */
               prSegNode = pbstTemp->prSeg;
               prTskNode = prSegNode->SEG_prtsk;
               bFound = TRUE;
            }                          /* endif                             */
      }  while ((!bFound) && (pbstTemp));/* enddo                           */

      /**********************************************************************/
      /* If after the search if the segment is not found then it could      */
      /* be an 'EXTRA' segment so ignore the info and return                */
      /**********************************************************************/

      if (!bFound) {
         if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulSegSize)))
            return (ulRC);
         return (DSP_NOERROR);
      }

      prSegNode->SEG_rver.bVersion = *pulCounter++;
      prSegNode->SEG_rver.bRelease = *pulCounter++;
      prSegNode->SEG_rverDSP.bVersion = *pulCounter++;
      prSegNode->SEG_rverDSP.bRelease = *pulCounter++;
      *pulCounter++;
      *pulCounter++;
      GETWORDELEMENT((PUSHORT *)&pulCounter,
         prSegNode->SEG_prsegtmp->SEG_ulAlign);

      GETWORDELEMENT((PUSHORT *)&pulCounter, usTemp);

      /* Note:  SEG_usflg is set in Task section                            */

      prSegNode->SEG_usflg |= usTemp;
      GETDWORDELEMENT((PULONG *)&pulCounter,
         prSegNode->SEG_prsegtmp->SEG_ulMemSize);

   }
   else

     /***********************************************************************/
     /* Process Segsection for LoadSegment and LoadTask                     */
     /***********************************************************************/

      if ((TAG_SHDR == aTag) && (usReadFlag != ENTIRE_FILE)) {

         prSegNode = *ppRseg;

       /* only create a new string element for loadsegment                  */

         if (usReadFlag == FIND_SEGMENT_OFFSET) {/* create segname buffer   */
            GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
               &ulStringSize);
            prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;/*   */
         }
         else                          /* dont create perm element          */
                                       /* (loadtask)                        */
            ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

         prSegNode->SEG_rver.bVersion = *pulCounter++;
         prSegNode->SEG_rver.bRelease = *pulCounter++;
         prSegNode->SEG_rverDSP.bVersion = *pulCounter++;
         prSegNode->SEG_rverDSP.bRelease = *pulCounter++;
         *pulCounter++;
         *pulCounter++;
         GETWORDELEMENT((PUSHORT *)&pulCounter,
            prSegNode->SEG_prsegtmp->SEG_ulAlign);

         GETWORDELEMENT((PUSHORT *)&pulCounter, usTemp);

       /* SEG_usflg is set in Task section or ParseSeg                      */

         prSegNode->SEG_usflg |= usTemp;
         GETDWORDELEMENT((PULONG *)&pulCounter,
            prSegNode->SEG_prsegtmp->SEG_ulMemSize);

         prTskNode = prSegNode->SEG_prtsk;/* Always set prTskNode           */
      }
      else {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::SegSection error ulRC %x 'shdr' tag not found\n",DSP_IO_INV_DSPFILE);
         return (DSP_IO_INV_DSPFILE);
      }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the NSTR section.  Nondiscardable String Table.                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_NSTR == aTag) {

      if (DSP_NOERROR != (ulRC = GetStringTable(pulCounter, ulBodySize, aTag,
                                   &(pTemp), &(prSegNode->SEG_usStaticCount),
                                   &(prSegNode->SEG_ulStaticSize))))
         return (ulRC);
      pulCounter += ulBodySize;

      prSegNode->SEG_parstblStatic = (PARSTBL)pTemp;

     /***********************************************************************/
     /*   Read the next section if there is info left.                      */
     /***********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the DSTR section.  Discardable String Table.                   */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_DSTR == aTag) {

      if (DSP_NOERROR != (ulRC = GetStringTable(pulCounter, ulBodySize, aTag,
                    &(pTemp), &(prSegNode->SEG_prsegtmp->SEG_usDiscardCount),
                    &(prSegNode->SEG_prsegtmp->SEG_ulDiscardSize))))
         return (ulRC);
      pulCounter += ulBodySize;

      prSegNode->SEG_prsegtmp->SEG_parstblDiscard = (PARSTBL)pTemp;

     /***********************************************************************/
     /*   Read the next section if there is info left.                      */
     /***********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the XTRN section.  Will use discardable string table           */
   /*   data is fixup table.                                                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_XTRN == aTag) {

      prSegNode->SEG_prsegtmp->SEG_usFixupCount =
                            (USHORT)(ulBodySize/FIXUPENTRY);      /* CH02  */

      ulTemp = prSegNode->SEG_prsegtmp->SEG_usFixupCount *sizeof(RFIX);

      prSegNode->SEG_prsegtmp->SEG_ulFixupSize = ulTemp;

      if (ulTemp) {                    /* allocate & read if size>0         */
         if (DSP_NOERROR != (ulRC = AllocateTemp(ulTemp, BYTEALIGN,
                                                 (PVOID)&pBuffer)))
            return (ulRC);
         memcpy(pBuffer, pulCounter, ulTemp);    /* CH03 */
         pulCounter += ulTemp;                   /* CH03 */

         prSegNode->SEG_prsegtmp->SEG_parfix = (RFIX *)pBuffer;
      }
      else {
         prSegNode->SEG_prsegtmp->SEG_parfix = NULL;
      }                                /* endif                             */

     /***********************************************************************/
     /*   Read the next section if there is info left.                      */
     /***********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the PUBS section.                                              */
   /*   Data is fixup table.                                                */
   /*   This section has to occur once.                                     */
   /*************************************************************************/

   if (TAG_PUBS == aTag) {

      if (DSP_NOERROR != (ulRC = AllocatePerm(ulBodySize, (PVOID)&pBuffer)))
         return (ulRC);

      prSegNode->SEG_ulPubsSize = ulBodySize;
      pRpub = (RPUB *)pBuffer;
      ulTemp = 0;

      while (ulTemp < ulBodySize) {

         GETWORDELEMENT((PUSHORT *)&pulCounter, pRpub->PUB_usLabel);
         GETDWORDELEMENT((PULONG *)&pulCounter, pRpub->PUB_ulOffset);
         GETWORDELEMENT((PUSHORT *)&pulCounter, pRpub->PUB_usflg);
         prSegNode->SEG_usPubsCount += 1;
         ulTemp += sizeof(RPUB);
         pRpub++;

      }                                /* endwhile                          */
      prSegNode->SEG_parpub = (RPUB *)pBuffer;

      /**********************************************************************/
      /*   Read the next section if there is info left.                     */
      /**********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   }

   /*************************************************************************/
   /*     Read the GPC section.  Will use nondiscardable string table       */
   /*     This section can occur zero or more time.                         */
   /*     GPCs will only be declared in Data Segments.                      */
   /*************************************************************************/

   parstbl = prSegNode->SEG_parstblStatic;
   while (TAG_GPC == aTag) {

      if (DSP_NOERROR != (ulRC =
             AddRGPCTail(&(prSegNode->SEG_prtsk->TSK_prgpcTail), &prGPCNode)))
         return (ulRC);

     /***********************************************************************/
     /*   Fill in the GPC from the Buffer                                   */
     /*   The name is and index into Static table so                        */
     /*   turn it into a string.                                            */
     /***********************************************************************/

      GETWORDELEMENT((PUSHORT *)&pulCounter, usName);
      FindStr(parstbl, usName, &(prGPCNode->GPC_pszName));

      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usflgKind);
      GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_lBufferSize);
      GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_ulOffset);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usAddrMode);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usMode);
      GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_ulSamplerate);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usMinPtrSep);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usMaxWPF);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usIntegerStride);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usFractStride);
      GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_lMaxdelay);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prGPCNode->GPC_usProtocol);

      if (Dspfile.bRelease >= 6) {
         GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_ulMaxCycles);
         GETDWORDELEMENT((PULONG *)&pulCounter, prGPCNode->GPC_ulVAllocSize);
      }
      prGPCNode->GPC_prseg = prSegNode;
      prGPCNode->GPC_prseg->SEG_usflg |= SEG_GPC;   /* seg has GPC CH04 */

     /* Begin CH01 **********************************************************/
     /*   If Protocol is GPC_SYNC and stride is zero, calculate stride      */
     /***********************************************************************/
      if ((prGPCNode->GPC_usProtocol == GPC_SYNC) &&
          (prGPCNode->GPC_usIntegerStride == 0) &&
          (prGPCNode->GPC_usFractStride == 0)) {

          ulstride = (ULONG)(prGPCNode->GPC_usMaxWPF);
          ulstride = ulstride << 15;
          ulstride = ulstride/(prTskNode->TSK_prtsktmp->TSK_usSamplesPerFrame);
          prGPCNode->GPC_usIntegerStride = (USHORT)(ulstride>>15);
          prGPCNode->GPC_usFractStride = (USHORT)(ulstride & 0x7FFF);
      }
     /** End  CH01 **********************************************************/

     /***********************************************************************/
     /*   Read the next section if there is info left.                      */
     /***********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*     Read the ITCB section.  Will use nondiscardable string table      */
   /*     This section can occur zero or more time.                         */
   /*     ITCBs will only be declared in Data Segments.                     */
   /*************************************************************************/

   while (TAG_ITCB == aTag) {

      if (DSP_NOERROR != (ulRC =
          AddRITCBTail(&(prSegNode->SEG_prtsk->TSK_pritcbTail), &prITCBNode)))
         return (ulRC);

     /***********************************************************************/
     /*   Fill in the ITCB from the Buffer                                  */
     /*   The name is and index into Static table so                        */
     /*   turn it into a string.                                            */
     /***********************************************************************/

      ulTemp = (ULONG)pulCounter;      /* remember beginning of ITCB chunk  */
      GETWORDELEMENT((PUSHORT *)&pulCounter, usName);
      FindStr(parstbl, usName, &(prITCBNode->ITCB_pszName));

      GETWORDELEMENT((PUSHORT *)&pulCounter, prITCBNode->ITCB_usflgKind);
      GETDWORDELEMENT((PULONG *)&pulCounter, prITCBNode->ITCB_ulSize);
      GETDWORDELEMENT((PULONG *)&pulCounter, prITCBNode->ITCB_ulOffset);

//    GETWORDELEMENT((PUSHORT *)&pulCounter, prITCBNode->ITCB_usLabelCount);
     /***********************************************************************/
     /* The module compiler does not follow the file format spec for        */
     /* the remainder of the ITCB chunk.  It leaves out the Vsegname        */
     /* and DeltaCPF fields, the IMSI linker has them.  To stay comp-       */
     /* atible with both we will skip the remainder of the ITCB chunk       */
     /* because it is not used by the manager anyway                        */
     /***********************************************************************/

      pulCounter = (PCHAR)(ulTemp+ulBodySize);
      prITCBNode->ITCB_pritcblab = NULL; /* no ITCB label info saved        */
      prITCBNode->ITCB_usLabelCount = 0; /* no label info                   */

     /***********************************************************************/
     /*   Fill in the ITCB labels for the ITCB                              */
     /***********************************************************************/

//   if (prITCBNode->ITCB_usLabelCount > 0) {
//      ulTemp = prITCBNode->ITCB_usLabelCount*sizeof(RITCBLAB);
//      if (DSP_NOERROR != (ulRC = AllocatePerm(ulTemp, (PVOID) &pBuffer)))
//         return (ulRC);
//
//      memcpy(pBuffer, pulCounter, ulTemp);  /*CH18*/
//      pulCounter+=ulTemp;
//      pritcblab = (PRITCBLAB) pBuffer;
//      for (i=0;i<prITCBNode->ITCB_usLabelCount;i++) {
//          GETWORDELEMENT((PUSHORT *)&pulCounter, pritcblab->ITCBLAB_usName);
//          GETWORDELEMENT((PUSHORT *)&pulCounter, pritcblab->ITCBLAB_usFlags);
//          GETWORDELEMENT((PUSHORT *)&pulCounter, pritcblab->ITCBLAB_sMaxdel);
//          pritcblab++;
//      } /* endfor */
//      prITCBNode->ITCB_pritcblab = (PRITCBLAB) pBuffer;
//   }  else prITCBNode->ITCB_pritcblab = NULL;

      prITCBNode->ITCB_prseg = prSegNode;

     /***********************************************************************/
     /*   Read the next section if there is info left.                      */
     /***********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the IPC section.                                               */
   /*   Data is fixup table.                                                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_IPC == aTag) {

      if (prTskNode->TSK_prseg_IPC != NULL)/* if IPC already exists         */
         return (DSP_IPC_COMMITTED);

      GETDWORDELEMENT((PULONG *)&pulCounter, prTskNode->TSK_ulOffset_IPC);/**/
      prTskNode->TSK_prseg_IPC = prSegNode;

      /**********************************************************************/
      /*   Read the next section if there is info left.                     */
      /**********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the LDMA section. ldma changed to dma at rel 8.                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   while ( ((Dspfile.bRelease < 8) && (TAG_LDMA == aTag)) ||
           ((Dspfile.bRelease >= 8) && (TAG_DMA == aTag)) ) {
      if (DSP_NOERROR != (ulRC =
          AddRLDMATail(&(prSegNode->SEG_prtsk->TSK_prldmaTail), &prldmaNode)))
         return (ulRC);

      /**********************************************************************/
      /*   Fill in the DMA from the Buffer                                  */
      /**********************************************************************/

      prldmaNode->LDMA_pszDMAName = NULL;
      if (Dspfile.bRelease >= 8)
         {
         GETWORDELEMENT((PUSHORT *)&pulCounter, usName);
         FindStr(parstbl, usName, &(pszTemp));
         GetStr(pszTemp, &(prldmaNode->LDMA_pszDMAName));
      }
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_lPackets);
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_ulMaxPkSize);
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_lMaxPerDMA);
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_ulOffset);
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_ulStride);
      GETDWORDELEMENT((PULONG *)&pulCounter, prldmaNode->LDMA_ulHold);
      GETWORDELEMENT((PUSHORT *)&pulCounter, prldmaNode->LDMA_usflg);

      /*Add segment pointer to structure for Stride/hold                    */

      prldmaNode->LDMA_prseg = prSegNode;

      /**********************************************************************/
      /*   Read the next section if there is info left.                     */
      /**********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the HW section.  Will use nondiscardable string table          */
   /*   data is fixup table.                                                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_HW == aTag) {

      pulCounter += ulBodySize;        /* Skip info                         */

      /** process HW *                                                      */
      /**********************************************************************/
      /*   Read the next section if there is info left.                     */
      /**********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the OVLY section.                                              */
   /*   Data is fixup table.                                                */
   /*   This section can occur zero or one time.                            */
   /*************************************************************************/

   if (TAG_OVLY == aTag) {

      pulCounter += ulBodySize;        /* Skip overlay info if present      */

      /** process OVLY *                                                    */
      /**********************************************************************/
      /*   Read the next section if there is info left.                     */
      /**********************************************************************/

      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
   };

   /*************************************************************************/
   /*   Read the IMAG sections. Instruction Block.                          */
   /*   IMAG can occur zero or more times.                                  */
   /*   Fill the segment in with its name.                                  */
   /*************************************************************************/

   while (TAG_IMAG == aTag) {

      if (DSP_NOERROR != (ulRC =
                AddRImageTail(&(prSegNode->SEG_prsegtmp->SEG_primgTail),
                              &prImageNode)))
         return (ulRC);

      GETDWORDELEMENT((PULONG *)&pulCounter, prImageNode->IMG_ulOffset);

      GETDWORDELEMENT((PULONG *)&pulCounter, prImageNode->IMG_ulSize);

      prImageNode->IMG_ulFileOffset = *pulFileRead+ulRead-ulTagSize+
         TAGHEADERSIZE                 /* Code hdr                          */
         +TAGHEADERSIZE                /* Image hdr                         */
         +sizeof(prImageNode->IMG_ulOffset)+sizeof(prImageNode->IMG_ulSize);

      pulCounter += ulBodySize-2*sizeof(ULONG);/* skip image                */
      if (ulRead < ulSegSize) {
         if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                              &ulTagSize)))
            return (ulRC);
         ulRead += ulTagSize;
      }
      else
         break;

   };

   /*************************************************************************/
   /*   Free the memory for the TASK Body info.                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulSegSize)))
      return (ulRC);

   return (DSP_NOERROR);

}                                      /* end SegSection                    */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  TASKSection                                            */
/*                                                                          */
/* DISCRIPTIVE NAME: Fill in the information from the TASK section into TSK */
/*                   shell, create the CODE and DATA Segment Shells and fill*/
/*                   in with initial general info (not detailed info).      */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     TASKSection                                            */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                   ulTaskSize - Size of the Task.                         */
/*                   ppRmod  - address of pointer to the module.            */
/*                   ppRtsk  - address of pointer to the single task        */
/*                             that needs to be parsed.                     */
/*                                                                          */
/*                                                                          */
/*                                                                          */
/* EXIT-NORMAL:     TASK filled out with Name info and Code and Data Shells */
/*                  in place.                                               */
/*                                                                          */
/* EXTERNAL REFERENCES: System Calls                                        */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC TASKSection( HFILE hfile, ULONG ulTaskSize, PRMOD *ppRmod,
                PRTSK *ppRtsk, USHORT usReadFlag)
{

#define  NUMHWINT      4
#define  NUMHWID       7               /* HW id's                           */

   RC         ulRC;
   PSZ        pulCounter;
   PSZ        pszTemp;
   PVOID      pBody;

   PRTSK      prTskNode;
   PRSEG      prSegNode;
   PRHW       prhwNode;

   PSZ        pszDefHWInt[] =  {
      "CDDAC","AIC1","AIC2","ADC"      /* HW interrupt sources              */
   };

   PSZ        pszDefHWID[] =  {
      "CDDAC","AIC1","AIC2","ADC","ACI","UART","MIDI"
   };

   ULONG      ulRead;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulStringSize;
   ULONG      aTag;
   USHORT     usflg,usCount,i;
   USHORT     notfound,usLoop;

   USHORT     usHWID[] =  {
      0,1,2,3,4,DSPFILE_UART,DSPFILE_MIDI
   };

   CHAR       cTempBuffer[MAXSTRINGSIZE];
   BOOL       bFound;

   ulRead = 0;
   ulBodySize = 0;

   /*************************************************************************/
   /*   Create memory for the Task Body info.                               */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocateTemp(ulTaskSize, BYTEALIGN, &pBody)))
      return (ulRC);

   /* Read entire TaskSection into buffer                                   */

   if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulTaskSize, pBody)))
      return (ulRC);
   pulCounter = pBody;

   /*************************************************************************/
   /*   Read each element of the Tag and process it accordingly.            */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the THDR section.  This is the header information.             */
   /*   This section MUST be in the file.                                   */
   /*   If usReadFlag = ENTIRE_FILE, get the name of the task from          */
   /*   the body and find the task with the same real name.                 */
   /*   If usReadFlag = READ_TASK_OFFSET, use task provided.                */
   /*************************************************************************/

   if ((TAG_THDR == aTag) && (usReadFlag == ENTIRE_FILE)) {

      prTskNode = (*ppRmod)->MOD_prtskTail;

      ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

      /**********************************************************************/
      /*  loop through the tasks until the names match                      */
      /**********************************************************************/

      bFound = FALSE;
      do {
         if (mwstrcmp(cTempBuffer, prTskNode->TSK_pszVirtName) == 0)
            bFound = TRUE;
         else
            prTskNode = prTskNode->TSK_prtskNext;

      }  while (!bFound && (prTskNode != (*ppRmod)->MOD_prtskTail));/* enddo*/

      /**********************************************************************/
      /* skip any extra tasks                                               */
      /**********************************************************************/

      if (!bFound) {
         if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulTaskSize)))
            return (ulRC);
         return (DSP_NOERROR);
      }
   }
   else
      if ((TAG_THDR == aTag) &&
         (usReadFlag == FIND_TASK_OFFSET)) {

         prTskNode = *ppRtsk;

         GetStringElement(&(prTskNode->TSK_pszVirtName), &pulCounter,
            &ulStringSize);

      }
      else {
	MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'thdr' tag not found\n",DSP_IO_INV_DSPFILE);
         return (DSP_IO_INV_DSPFILE);
      }
   prTskNode->TSK_rver.bVersion = *pulCounter++;
   prTskNode->TSK_rver.bRelease = *pulCounter++;

      /**********************************************************************/
      /*  Read in HW Int Source. TSK_hwidHWIntSrc is PSZ starting with      */
      /*  file release 8. Release 7 and lower is USHORT. HWID is also       */
      /*  PSZ starting with file release 8.                                 */
      /**********************************************************************/

   if (Dspfile.bRelease < 8) {
      GETWORDELEMENT((PUSHORT *)&pulCounter,
         prTskNode->TSK_prtsktmp->TSK_hwidHWIntSrc);
   }
   else {
      notfound = 1;
      GetStringElement(&pszTemp, &pulCounter, &ulStringSize);

      if (pszTemp == NULL) {           /* NULL for NRT tasks                */
         prTskNode->TSK_prtsktmp->TSK_hwidHWIntSrc = 0xFFFF;
         notfound = 0;
      }
      else {                           /* if not NULL then HWIntSrc         */
                                       /* string read                       */
         for (i = 0; i < NUMHWINT && notfound; i++) {
            if (mwstrcmp(pszTemp, pszDefHWInt[i]) == 0)
               {                       /* get string and convert to ushort  */
               prTskNode->TSK_prtsktmp->TSK_hwidHWIntSrc = i;
               notfound = 0;
            }
         }                             /* endfor                            */
         FreePerm(pszTemp, strlen(pszTemp)+1);/* free buffer                */
      }                                /* endif                             */

      if (notfound)                    /* no match for int source found     */
         return (DSP_HWDEVICE_NOT_AVAILABLE);
   }                                   /* end else                          */
   GETWORDELEMENT((PUSHORT *)&pulCounter, prTskNode->TSK_usflgType);
   prTskNode->TSK_usflgType <<= 8;
   GETWORDELEMENT((PUSHORT *)&pulCounter, usflg);
   prTskNode->TSK_usflgType |= usflg;

   GETDWORDELEMENT((PULONG *)&pulCounter, prTskNode->TSK_ulCPF);
   GETDWORDELEMENT((PULONG *)&pulCounter,
      prTskNode->TSK_prtsktmp->TSK_ulTimeBase);

   GETWORDELEMENT((PUSHORT *)&pulCounter,
      prTskNode->TSK_prtsktmp->TSK_usSamplesPerFrame);

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the STND section.  Standby Entry point.                        */
   /*   This section MUST be in the file.                                   */
   /*************************************************************************/

   if (TAG_STND == aTag) {

      GetStringElement
         (&(prTskNode->TSK_prtsktmp->TSK_rslStandbyEntry.pszSegment),
         &pulCounter, &ulStringSize);
      GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_rslStandbyEntry.pszLabel
         ), &pulCounter, &ulStringSize);

   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'stnd' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the ACTV section.  Active Entry point.                         */
   /*   This section MUST be in the file.                                   */
   /*************************************************************************/

   if (TAG_ACTV == aTag) {

      GetStringElement
         (&(prTskNode->TSK_prtsktmp->TSK_rslActiveEntry.pszSegment),
         &pulCounter, &ulStringSize);
      GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_rslActiveEntry.pszLabel)
         , &pulCounter, &ulStringSize);

   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'actv' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the ALUO section.  ALU Overflow.                               */
   /*   This section MUST be in the file.                                   */
   /*************************************************************************/

   if (TAG_ALUO == aTag) {

      GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_rslALUEntry.pszSegment),
         &pulCounter, &ulStringSize);
      GetStringElement(&(prTskNode->TSK_prtsktmp->TSK_rslALUEntry.pszLabel),
         &pulCounter, &ulStringSize);

   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'aluo' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /************************************************************************
    *   Read the HWID Section for the Task.
    *   This section can occur zero or more times in a file and only applies to
    *   BIOS Tasks.
    *************************************************************************/

   if (TAG_HWID == aTag) {

      /* Read the HWID Count for the Task                                   */

      GETWORDELEMENT((PUSHORT *)&pulCounter, usCount);

      /************************************************/
      /*  Add RHW control Blocks for the BIOS task.   */
      /*  And set them to point to the BIOS Task.     */
      /************************************************/

      for (i = 0; i < usCount; i++) {
         AddRHwTail(&(prTskNode->TSK_prhwTail), &prhwNode);

         if (Dspfile.bRelease < 8) {
            GETWORDELEMENT((PUSHORT *)&pulCounter, prhwNode->HW_ulhwType);
         }

         else
            {
            notfound = 1;
            GetStringElement(&pszTemp, &pulCounter, &ulStringSize);
            for (usLoop = 0; usLoop < NUMHWID && notfound; usLoop++) {
               if (mwstrcmp(pszTemp, pszDefHWID[usLoop]) == 0)
                  {
                  notfound = 0;
                  prhwNode->HW_ulhwType = usHWID[usLoop];
               }

            }                          /* end for usLoop                    */
            FreePerm(pszTemp, strlen(pszTemp)+1);/* free buffer             */
            if (notfound)              /* no match for HWID found           */
               return (ulRC = DSP_HWDEVICE_NOT_AVAILABLE);

         }                             /* end else > 8                      */

          /******************************************************************/
          /* Convert the HW type to our internal representation.            */
          /******************************************************************/

         switch (prhwNode->HW_ulhwType) {
            case  DSPFILE_ACI :
               prhwNode->HW_ulhwType = HW_ACI;
               break;
            case  DSPFILE_ADC :
               prhwNode->HW_ulhwType = HW_ADC;
               break;
            case  DSPFILE_CDDAC :
               prhwNode->HW_ulhwType = HW_CDDAC;
               break;
            case  DSPFILE_AIC2 :
               prhwNode->HW_ulhwType = HW_AIC2;
               break;
            case  DSPFILE_AIC1 :
               prhwNode->HW_ulhwType = HW_AIC1;
               break;
            case  DSPFILE_UART :
               prhwNode->HW_ulhwType = HW_UART;
               break;
            case  DSPFILE_MIDI :
               prhwNode->HW_ulhwType = HW_MIDI;
               break;
            default  :
               break;
         }                             /* endswitch                         */
         prhwNode->HW_prtsk = prTskNode;
      }                                /* endfor                            */
      if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                           &ulTagSize)))
         return (ulRC);

      ulRead += ulTagSize;
   }

   /*************************************************************************/
   /*   Read the DDS  section.  Default Data Segment.                       */
   /*   This section MUST be in the file.                                   */
   /*   ?? This creates a Data Segment Structure and add it to the          */
   /*   tail of the Segments hanging off the task.                          */
   /*   Fill the segment in with its name.                                  */
   /*************************************************************************/

   if (TAG_DDS == aTag) {

      if (DSP_NOERROR != (ulRC = AddRSegmentTail(&(prTskNode->TSK_prsegDTail),
                                                 &prSegNode)))
         return (ulRC);

      (*ppRmod)->MOD_usNumDSSegs++;

      GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
         &ulStringSize);

      /* Make real/virt name point to same string                           */
      prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

      prSegNode->SEG_usflg = SEG_DATA|SEG_DDS;
      prSegNode->SEG_prtsk = prTskNode;
      prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
      prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;
   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'dds' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                        &ulTagSize)))
      return (ulRC);

   ulRead += ulTagSize;

   /*************************************************************************/
   /*   Read the Data Segments (DS) and the Code Segments (CS) and          */
   /*   Add each to their respective linked lists.                          */
   /*   CS must occur at least once but can occur more than once.           */
   /*   DS can occur zero or more times.                                    */
   /*   Fill the segment in with its name.                                  */
   /*************************************************************************/

   while (TAG_DS == aTag) {

      if (DSP_NOERROR != (ulRC = AddRSegmentTail(&(prTskNode->TSK_prsegDTail),
                                                 &prSegNode)))
         return (ulRC);

      (*ppRmod)->MOD_usNumDSSegs++;

      GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
         &ulStringSize);

      /* Make real/virt name point to same string                           */
      prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

      prSegNode->SEG_usflg = SEG_DATA;
      prSegNode->SEG_prtsk = prTskNode;
      prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
      prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;
      if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag, &ulBodySize,
                                           &ulTagSize)))
         return (ulRC);

      ulRead += ulTagSize;

   }

   if (TAG_CS == aTag) {
      while (ulRead <= ulTaskSize) {

         if (DSP_NOERROR != (ulRC =
                   AddRSegmentTail(&(prTskNode->TSK_prsegITail), &prSegNode)))
            return (ulRC);

         (*ppRmod)->MOD_usNumISSegs++;

         GetStringElement(&(prSegNode->SEG_pszVirtName), &pulCounter,
            &ulStringSize);

        /* Make real/virt name point to same string                        */

         prSegNode->SEG_pszRealSegName = prSegNode->SEG_pszVirtName;

         prSegNode->SEG_usflg = SEG_INST;
         prSegNode->SEG_prtsk = prTskNode;
         prSegNode->SEG_prsegtmp->SEG_hfile = hfile;
         prSegNode->SEG_prsegtmp->SEG_ulMemSize = 0;

   /*************************************************************************/
   /*   Do reads until the amount read is equal to the total size.          */
   /*   If they match, break out of the loop.                               */
   /*************************************************************************/

         if (ulRead < ulTaskSize) {

            if (DSP_NOERROR != (ulRC = GetTagHdr((PBYTE *)&pulCounter, &aTag,
                                                 &ulBodySize, &ulTagSize)))
               return (ulRC);

            ulRead += ulTagSize;

         }
         else
            break;

      }                                /* end while                         */
   }
   else {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::TASKSection error ulRC %x 'cs' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   /*************************************************************************/
   /*   Free the memory for the TASK Body info.                             */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = FreeTemp(pBody, ulTaskSize)))
      return (ulRC);

   return (DSP_NOERROR);

}                                      /* end TaskSection                   */

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  VerifyAllocMod                                         */
/*                                                                          */
/* DISCRIPTIVE NAME: Verify the MOD tag and create the memory for Module.   */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     VerifyAllocMod                                         */
/*                                                                          */
/* INPUT:            hfile - Handle to the file.                            */
/*                                                                          */
/* EXIT-NORMAL:      ppRmod points to the memory area created.              */
/*                   ulBodySize set to the tag body info.                   */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC VerifyAllocMod(HFILE hfile,PULONG pulBodySize,PRMOD *ppRmod,PULONG
                   pulFileRead)
{

   RC         ulRC;
   ULONG      aTag;
   ULONG      ulTagSize;

   /*************************************************************************/
   /*   Read in the module tag and verify that its correct.                 */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, pulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if (TAG_MOD != aTag) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::VerifyAllocMod error ulRC %x 'mod' tag not found\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }
   *pulFileRead += ulTagSize;

   /*************************************************************************/
   /*   Create memory for the module.                                       */
   /*************************************************************************/

   if (DSP_NOERROR != (ulRC = AllocRModule(ppRmod)))
      return (ulRC);

   pg->prmodCurrentMod = *ppRmod;

   return (DSP_NOERROR);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME:  SearchTOC                                              */
/*                                                                          */
/* DISCRIPTIVE NAME: Find offset in DSP file of a task or a segment         */
/*                                                                          */
/* FUNCTION:         Same                                                   */
/* NOTE:                                                                    */
/* NOTE:                                                                    */
/*   DEPENDENCIES:   NONE                                                   */
/*                                                                          */
/*   RESTRICTIONS:   NONE                                                   */
/*                                                                          */
/*                                                                          */
/* ENTRY POINTS:     SearchTOC                                              */
/*                                                                          */
/* INPUT:            hfile       - Handle to the file.                      */
/*                   pszRealName - Name of task or seg to be located.       */
/*                   usRead Flag   -  type of search to do on TOC.          */
/*                                                                          */
/* EXIT-NORMAL:      pulFileOffset points offset of a task or a segment.    */
/*                                                                          */
/* EXTERNAL REFERENCES:                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC SearchTOC( HFILE hfile, PSZ pszRealName, PULONG pulFileOffset,
              USHORT usReadFlag)
{
   RC         ulRC;
   PVOID      pBuffer;
   PSZ        pulCounter = NULL;
   ULONG      ulBodySize;
   ULONG      ulTagSize;
   ULONG      ulStringSize;
   ULONG      ulTemp;
   ULONG      aTag;
   USHORT     i,j;
   USHORT     usTaskCount,usSegmentCount,usCount;
   CHAR       cTempBuffer[MAXSTRINGSIZE]; /* buffer to hold temp strings  */

   if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                         &ulTagSize)))
      return (ulRC);

   if ((TAG_BIOT != aTag) &&
      (TAG_TSK != aTag)) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspparse::SearchTOC error ulRC %x 'biot' or 'tsk' tag expected\n",DSP_IO_INV_DSPFILE);
      return (DSP_IO_INV_DSPFILE);
   }

   if (usReadFlag == FIND_TASK_OFFSET) { // find task offset
      if (DSP_NOERROR != (ulRC = AllocateTemp(ulBodySize, BYTEALIGN, &pBuffer)))

         return (ulRC);

      if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulBodySize, pBuffer)))
         return (ulRC);

      pulCounter = pBuffer;

      GETWORDELEMENT((PUSHORT *)&pulCounter, usTaskCount); // Task count
      for (i = 0; i < usTaskCount; ++i) {
         ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

        /*
         *  The string element is followed by a DWORD entry that specifies
         *  the offset in dsp file of the task header.  This offset gets
         *  read in whether or not theres a match in order to bump up the
         *  buffer pointer.
                                                                            */

         if (mwstrcmp(pszRealName, cTempBuffer) == 0) {
            GETDWORDELEMENT((PULONG *)&pulCounter, *pulFileOffset);
            break;                     // Find task, break out of for loop
         }                             // endif

        /* Return the String Storage if nothing found                       */

         GETDWORDELEMENT((PULONG *)&pulCounter, ulTemp);

        /*
         *  BIOT entries in the TOC have additional information related to
         *  BIOS connector names.  This also has to be "bumped" around in a search.
         *  Use usSegmentCount as a Temporary variable.
                                                                            */

         if (TAG_BIOT == aTag) {

           /* Read in HW IDs to traverse buffer                             */

            GETWORDELEMENT((PUSHORT *)&pulCounter, usCount); // HWID #
            for (j = 0; j < usCount; j++) {
               if (Dspfile.bRelease < 8) {
                  GETWORDELEMENT((PUSHORT *)&pulCounter, usSegmentCount);
               }
               else
                  ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);
            }

           /* Read in Connector Names and Discard to traverse buffer.       */

            GETWORDELEMENT((PUSHORT *)&pulCounter, usCount);
            for (j = 0; j < usCount; j++) {
               ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);
            }
         }                             // end if "biot"
      }                                // end "for" loop

      if (DSP_NOERROR != (ulRC = FreeTemp(pBuffer, ulBodySize)))
         return (ulRC);

      if (*pulFileOffset == 0)         // Find task in DSP ?
      {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspparse::SearchTOC error ulRC %x Parse task: %s\n", DSP_NAME_NOT_FOUND,pszRealName);
         return (DSP_NAME_NOT_FOUND);
      }

   }                                   // endif FIND_TASK_OFFSET

   else if (usReadFlag == FIND_SEGMENT_OFFSET) { // find segment offset

       /******************************************************************/
       /* Move past tsk section in TOC                                   */
       /* or the biot section in TOC                                     */
       /******************************************************************/

      if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)ulBodySize,
                                               FROMCURRENT)))
         return (ulRC);

      if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                            &ulTagSize)))
         return (ulRC);

       /******************************************************************/
       /* If the Tag is tsk the first Task was biot                      */
       /* So move past the tsk Section in TOC                            */
       /******************************************************************/

      if (TAG_TSK == aTag) {
         if (DSP_NOERROR != (ulRC = DspChgFilePtr(hfile, (LONG)ulBodySize,
                                                  FROMCURRENT)))
            return (ulRC);

         if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                               &ulTagSize)))
            return (ulRC);
      }

       /******************************************************************/
       /* In TOC, after tsk section it has to be either                  */
       /* iseg or dseg section until end of TOC                          */
       /******************************************************************/

      while ((TAG_ISEG == aTag) || (TAG_DSEG == aTag)) {
         if (DSP_NOERROR != (ulRC = AllocateTemp(ulBodySize, BYTEALIGN,
                                                 &pBuffer)))
            return (ulRC);

         if (DSP_NOERROR != (ulRC = DspReadFile(hfile, ulBodySize, pBuffer)))
            return (ulRC);

         pulCounter = pBuffer;
         GETWORDELEMENT((PUSHORT *)&pulCounter, usSegmentCount);
                                            // get segment count
         for (i = 0; i < usSegmentCount; ++i) {
            ReadStringElement(cTempBuffer, &pulCounter, &ulStringSize);

            if (mwstrcmp(pszRealName, cTempBuffer) == 0) {
               GETDWORDELEMENT((PULONG *)&pulCounter, *pulFileOffset);
               break;               // find seg, break out of for loop
            }                       // endif

            GETDWORDELEMENT((PULONG *)&pulCounter, ulTemp);
         }                          // endfor
         if (*pulFileOffset == 0) { // Find segment in this seg section ?
            if (DSP_NOERROR != (ulRC = FreeTemp(pBuffer, ulBodySize)))
               return (ulRC);

            if (DSP_NOERROR != (ulRC = ReadTagHdr(hfile, &aTag, &ulBodySize,
                                                  &ulTagSize)))
               return (ulRC);

         }
         else
            break;                 // find seg, break out of while loop

      }                             // endwhile

      if (*pulFileOffset == 0)      // Find segment in this DSP file ?
      {
	MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspparse::SearchTOC error ulRC %x Parse Segment: %s\n",DSP_NAME_NOT_FOUND, pszRealName);
         return (DSP_NAME_NOT_FOUND);
      }
      else
         if (DSP_NOERROR != (ulRC = FreeTemp(pBuffer, ulBodySize)))
            return (ulRC);
   }
   else
      return (DSP_ERROR);

   return (DSP_NOERROR);
}
