#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* command.c
 *
 * Command handler.  All command flow through this handler.  This is done
 * in order to allow standard things to be done before and after each
 * command.
 *
 * All operations that the user can select (through menu selection or through
 * interacting with the tree) are implemented as commands that flow through
 * this handler.
 *
 * This is the process that occurs each time a command is called:  the error
 * message is reset, and the canvas is reset (so that it has no update area).
 * Then the command is called, which may place a message in the error message,
 * and/or may invalidate part of the canvas.  Then if there's an error message,
 * it is displayed, and if there's an invalid part of the canvas, that part
 * is redrawn
 *
 * Normally, you call do_command with the command name as the first
 * parameter.  However, if you want the command to be performed on every
 * selected item, you call do_command with the command name as the last
 * parameter, and the first parameter is one of EveryNode, EveryTree,
 * EveryBranch, or EveryNodeAndSubtree.
 *
 */

#include "interface.h"

#include "treedata.h"

int doingselection=0;

int do_command(c, p1, p2, d1, d2, i1, i2, i3, i4, c2)
/* returns 1 if no error, 0 if error */
command c;
void *p1, *p2;
double d1, d2;
int i1, i2, i3, i4;
command c2;
{
	int result;
	int wasmodified;

	if(c==NoCommand)
		return(1);

	result=1;
	if(!doingselection)
	{
		/* error_string is always set to a constant string.  If that becomes
			no longer true, must uncomment following two lines and make
			tterr() strdup the error string */
		/*if(globals->error_string!=NULL)*/
			/*free(globals->error_string);*/
		globals->error_string=NULL;
		globals->error=NoError;
		set_main_message("");
		canvas_reset();
		wasmodified=globals->modified;
	}
	/* call function */
	switch(c)
	{
		case Load:
			set_all_busy();
			uninstall(do_change_text);
			result=do_load(p1, i1);
			set_all_notbusy();
			break;
		case Save:
			set_all_busy();
			uninstall(do_change_text);
			result=do_save(p1, i1);
			set_all_notbusy();
			break;
		case Print:
			set_all_busy();
			uninstall(do_change_text);
			result=do_print(p1);
			set_all_notbusy();
			break;
		/*case Quit:*/
			/*exit(0);*/
			/*break;*/
		case AutoformatOn:
			globals->autoformat=1;
			modify();
			canvas_add_all();
			break;
		case AutoformatOff:
			globals->autoformat=0;
			canvas_add_all();
			break;
		case AutomergeOn:
			globals->automerge=1;
			break;
		case AutomergeOff:
			globals->automerge=0;
			break;
		case HideScaleBar:
			result=tree_hide_scale_bar(p1, NoTreeType);
			break;
		case ShowScaleBar:
			result=tree_show_scale_bar(p1, NoTreeType);
			break;
		case EvenSpacing:
			result=reset_even_spacing(p1, NoTreeType);
			break;
		case SetScale:
			result=tree_change_scale(p1, NoTreeType, d1);
			break;
		case SetScaleBarValue:
			result=tree_change_scale_value(p1, NoTreeType, d1);
			break;
		case SetScaleBarFont:
			result=set_font_of_scale_bar(p1, NoTreeType, p2);
			break;
		case SetNodeFont:
			result=set_font_of_node(p1, NoTreeType, p2);
			break;
		case SetScaleBarColor:
			result=set_color_of_scale_bar(p1, NoTreeType, p2);
			break;
		case SetNodeColor:
			result=set_color_of_node(p1, NoTreeType, p2);
			break;
		case SetThickness:
			result=branch_set_thickness(p1, NoTreeType, i1);
			break;
		case SetBranchColor:
			result=set_color_of_branch(p1, NoTreeType, p2);
			break;
		case FlipNode:
			result=flip_node(p1);
			break;
		case AddToSelection:
			result=add_to_selection(p1, i1, 1, d1);
			break;
		case RemoveFromSelection:
			result=remove_from_selection(p1, i1);
			break;
		case AddNodeListToSelection:
			result=add_node_list_to_selection(p1, p2, 1);
			break;
		case Select:
			result=add_to_selection(p1, i1, 0, d1);
			break;
		case Copy:
			result=copy_to_clipboard();
			break;
		case Paste:
			result=paste_selection();
			break;
		case PasteSubtree:
			result=paste_selection_subtree();
			break;
		case Cut:
			result=cut_selection();
			break;
		case Delete:
			result=delete_selection();
			break;
		case GoEditTree:
			result=change_editing_mode(MouseTree);
			break;
		case GoEditFlip:
			result=change_editing_mode(MouseFlip);
			break;
		case GoEditHide:
			result=change_editing_mode(MouseHide);
			break;
		case GoEditMove:
			result=change_editing_mode(MouseMove);
			break;
		case GoEditText:
			result=change_editing_mode(MouseText);
			break;
		case MoveTree:
			result=move_whole_tree(p1, NoTreeType, i1, i2);
			break;
		case ResizeTree:
			result=resize_whole_tree(p1, NoTreeType, i1, i2, i3, i4);
			break;
		case ShowInternalNames:
			result=tree_show_internal_names(p1, NoTreeType);
			break;
		case HideInternalNames:
			result=tree_hide_internal_names(p1, NoTreeType);
			break;
		case ShowTerminalNames:
			result=tree_show_terminal_names(p1, NoTreeType);
			break;
		case HideTerminalNames:
			result=tree_hide_terminal_names(p1, NoTreeType);
			break;
		case ChangeOutgroup:
			result=change_outgroup_at_selection();
			break;
		case SetTreeBranchFormat:
			result=tree_change_branch_type(p1, NoTreeType, i1);
			break;
		case SetTreeFormat:
			result=set_tree_type(p1, i1);
			modify();
			break;
		case ShowHandles:
			canvas_show_handles();
			break;
		case HideHandles:
			canvas_hide_handles();
			break;
		case MoveAllOnscreen:
			move_all_on_screen(p1, i1);
			break;
		case MoveOnscreen:
			move_on_screen(p1, i1);
			break;
		case SetNodeName:
			set_node_name(p1, NoTreeType, p2);
			break;
		case MoveNodeLabel:
			result=move_node_label(p1, NoTreeType, i1, i2);
			break;
		case FixLabel:
			result=fix_node_label(p1, NoTreeType);
			break;
		case MoveNode:
			result=shift_node_position(p1, NoTreeType, i1, i2);
			break;
		case RotateNode:
			result=rotate_node_position(p1, NoTreeType, d1);
			break;
		case ConnectSelections:
			result=connect_selections(p1);
			break;
		case SelectAncestor:
			result=select_common_ancestor(p1);
			break;
		case PruneToReps:
			result=prune_to_representatives(p1);
			break;
		case MergeMergeableNodes:
			result=merge_mergeable_node(p1, NoTreeType);
			break;
		case SetScaleBarPosition:
			result=tree_move_scale_bar(p1, NoTreeType, i1, i2);
			break;
		case SelectAll:
			result=select_all();
			break;
		case HideNode:
			result=hide_node(p1);
			break;
		case ShowNode:
			result=show_node(p1);
			break;
		case HideLevel:
			result=hide_level(p1);
			break;
			/* add more commands */
		case EveryNode:
		case EveryTree:
		case EveryBranch:
		case EveryNodeAndSubtree:
			doingselection=1;
			result=do_on_selection(c, c2, p1, p2, d1, d2, i1, i2, i3, i4);
			doingselection=0;
			break;
		default:
			result=1;
			break;
	}
	if(!doingselection)
	{
		if(globals->modified^wasmodified)
			show_filename();
		if(globals->autoformat)
			reset_invalid_trees();
		recalc_invalid_trees();
		move_invalid_labels();
		refit_invalid_trees();
		switch(globals->error)
		{
			case NoError:
				break;
			case Message:
				set_main_message(globals->error_string);
				break;
			case BeepMessage:
#ifdef DEBUG
				printf("%s\n", globals->error_string);
#endif
				set_main_message(globals->error_string);
				if(globals->beeping)
					tdisplay_beep(globals->c);
				break;
			case OutOfMemory:
				tnotice_no_choice(globals->ewin,
					globals->ex, globals->ey, "Out of Memory");
				break;
			case Notice:
				tnotice_no_choice(globals->ewin,
					globals->ex, globals->ey, globals->error_string);
				break;
			
		}
		tcanvas_paint_area(globals->c);
	}
    return(result);
}
