/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: marshal.cxx,v $
 *
 *  $Revision: 1.6 $
 *
 *  last change: $Author: kz $ $Date: 2005/10/05 14:59:08 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include <osl/diagnose.h>
#include <bridges/remote/marshal.hxx>
#include <bridges/remote/remote.hxx>
#include <bridges/remote/helper.hxx>

#include "conversion.h"

#ifdef major
#undef major
#undef minor
#endif


#include <com/sun/star/corba/TCKind.hpp>
#include <com/sun/star/corba/CorbaString8.hpp>
#include <com/sun/star/corba/CorbaUnion.hpp>
#include <com/sun/star/corba/ObjectKey.hpp>
#include <com/sun/star/corba/iop/IOR.hpp>
#include <com/sun/star/corba/iop/ProfileIdGroup.hpp>
#include <com/sun/star/corba/iiop/ProfileBody_1_1.hpp>

using namespace ::bridges_remote;
using namespace ::rtl;

using namespace ::com::sun::star::uno;
using namespace ::com::sun::star::corba;
using namespace ::com::sun::star::corba::iop;
using namespace ::com::sun::star::corba::iiop;


namespace bridges_remote {

static int g_nDetectLittleEndian = 1;	
static char g_bSystemIsLittleEndian = ((char*)&g_nDetectLittleEndian)[0];
	
Marshal::Marshal( sal_Bool bIsLittleEndian ,
				  uno_Environment *pEnvRemote,
				  extractOidCallback callback,
				  sal_Int32 nStartMarshalingAt) :
	m_seqData( 512 ),
	m_nStartMarshalingAt( nStartMarshalingAt ),
	m_bIsLittleEndian( bIsLittleEndian ),
	m_pos( (( sal_Int8 * ) m_seqData.getConstArray()) + nStartMarshalingAt ),
	m_base( m_pos ),
	m_callback( callback ),
	m_pEnvRemote( pEnvRemote )
{}

Marshal::Marshal(
	sal_Bool bIsLittleEndian,
	const Sequence< sal_Int8 > &seqFixedSized
	) :
	m_seqData( seqFixedSized ),
	m_nStartMarshalingAt( 0 ),
	m_bIsLittleEndian( bIsLittleEndian ),
	m_pos( ( sal_Int8 * ) m_seqData.getConstArray() ),
	m_base( ( sal_Int8 * ) m_seqData.getConstArray() ),
	m_callback( 0 ),
	m_pEnvRemote( 0 )
{}
	

inline void Marshal::align( sal_Int32 nToAlign )
{
// This alignment is an iiop-specification violation, but it was
// discovered too late.
	sal_Int32 n = ( sal_uIntPtr(m_pos) + m_nStartMarshalingAt ) % nToAlign;
// This is the correct alignment	
//	sal_Int32 n = ( sal_uIntPtr(m_pos) % nToAlign);
	if( n )
	{
		n = nToAlign -n;
	}
	m_pos += n;

}

inline void Marshal::ensureAdditionalMem( sal_Int32 nMemToAdd )
{
	sal_Int32 nDiff = m_pos - m_base + m_nStartMarshalingAt;
	if( nDiff + nMemToAdd > m_seqData.getLength() )
	{
		m_seqData.realloc(
			m_seqData.getLength() *2 > nDiff+nMemToAdd ?
			m_seqData.getLength() *2 :
			nDiff + nMemToAdd );
		m_base = ((sal_Int8*)m_seqData.getConstArray()) + m_nStartMarshalingAt;
		m_pos = ((sal_Int8*)m_seqData.getConstArray()) + nDiff;
	}
}

Sequence< sal_Int8 >  Marshal::remove(  )
{
	// the below hack works only, if no one else knows the sequence
	OSL_ASSERT( 1 == (*((uno_Sequence** ) &m_seqData))->nRefCount );
	// m_seqData.realloc( (sal_Int32) ( m_pos - m_base ) );
	// fast, but memory can't be reused till the sequence is destructed
	// Either dtor's don't need to be called
	(*((uno_Sequence** ) &m_seqData))->nElements =
		((sal_Int32) ( m_pos - m_base )) + m_nStartMarshalingAt;

	return m_seqData;
}

void Marshal::packByteSequence( sal_Int8 *pData , sal_Int32 nLength )
{
	pack( &nLength , getCppuType( &nLength ) );
	ensureAdditionalMem( nLength );
	memcpy( m_pos , pData , nLength );
	m_pos += nLength;
}

void Marshal::pack( void *pSource , const ::com::sun::star::uno::Type &rType )
{
	typelib_TypeDescription * pDataTD = 0;
	TYPELIB_DANGER_GET( &pDataTD, rType.getTypeLibType() );
	pack( pSource, pDataTD );
	TYPELIB_DANGER_RELEASE( pDataTD );	
}


void packString8( Marshal *pMarshal , void *pSource )
{
	rtl_uString *p = *(rtl_uString **) pSource;
	sal_Int32 nLenPlus = p->length+1;

	(*pMarshal) <<= nLenPlus;

	OString str = OUStringToOString( OUString(p) , RTL_TEXTENCODING_ASCII_US );
	for( sal_Int32 i = 0 ; i < nLenPlus ; i ++ )
	{
		sal_Int8 int8 = str.pData->buffer[i];
		(*pMarshal) <<= int8;
	}
}


/*****
 * packs a discrimnative union. (see corba specification)
 * The discriminative type and one value is packed.
 *****/
void packUnion( Marshal *pMarshal , void *pVoidSource , typelib_CompoundTypeDescription *pCompType )
{
	sal_Int8 *pSource = ( sal_Int8 * ) pVoidSource;

	sal_Int32 nPosToMarshal = 0;
	typelib_TypeDescription *pType = 0;
	TYPELIB_DANGER_GET( &pType , pCompType->ppTypeRefs[0] );

	if( typelib_TypeClass_BYTE == pType->eTypeClass ||
		typelib_TypeClass_BOOLEAN == pType->eTypeClass )
	{
		// retrieve the position to marshal
		nPosToMarshal =  (sal_Int32 ) *(sal_Int8 * ) (pSource + pCompType->pMemberOffsets[0]);
	}
	else if( typelib_TypeClass_SHORT   == pType->eTypeClass ||
			 typelib_TypeClass_UNSIGNED_SHORT   == pType->eTypeClass )
	{
		nPosToMarshal = ( sal_Int32 ) *(sal_Int16 *) (pSource + pCompType->pMemberOffsets[0]);
	}
	else if( typelib_TypeClass_LONG    == pType->eTypeClass ||
			 typelib_TypeClass_ENUM    == pType->eTypeClass )
	{
		// retrieve the position to marshal
		nPosToMarshal =  (sal_Int32 ) *(sal_Int32 * ) (pSource + pCompType->pMemberOffsets[0]);
	}
	else {
		OSL_ASSERT( 0 );
	}

	pMarshal->pack( pSource + pCompType->pMemberOffsets[0] , pType );
	TYPELIB_DANGER_RELEASE( pType );

	pType = 0;
	TYPELIB_DANGER_GET( &pType , pCompType->ppTypeRefs[1+nPosToMarshal] );
	pMarshal->pack( pSource + pCompType->pMemberOffsets[1+nPosToMarshal], pType );
	TYPELIB_DANGER_RELEASE( pType );
}

void packTypecode( Marshal *pMarshal , typelib_TypeDescription *pType )
{
	TCKind kind;
	switch( pType->eTypeClass )
	{
	case typelib_TypeClass_BYTE:
	{
		kind = TCKind_tk_octet;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_BOOLEAN:
	{
		kind = TCKind_tk_boolean;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_CHAR:
	{
		kind = TCKind_tk_wchar;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_SHORT:
	{
		kind = TCKind_tk_short;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_UNSIGNED_SHORT:
	{
		kind = TCKind_tk_ushort;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_FLOAT:
	{
		kind = TCKind_tk_float;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_LONG:
	{
		kind = TCKind_tk_long;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_UNSIGNED_LONG:
	{
		kind = TCKind_tk_ulong;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_HYPER:
	{
		kind = TCKind_tk_longlong;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_UNSIGNED_HYPER:
	{
		kind = TCKind_tk_ulonglong;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_DOUBLE:
	{
		kind = TCKind_tk_double;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_STRING:
	{
		kind = TCKind_tk_wstring;
		(*pMarshal) <<= kind;
		sal_uInt32 nBound = 0;
		(*pMarshal) <<= nBound;
		break;
	}
	case typelib_TypeClass_ANY:
	{
		kind = TCKind_tk_any;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_ENUM:
	{
		kind = TCKind_tk_enum;
		(*pMarshal) <<= kind;

		CorbaString8 string8;
		string8.theString = OUString( pType->pTypeName );
		(*pMarshal) <<= string8;

		CorbaString8 dummy;
		(*pMarshal) <<= dummy;

		// TODO : all enum names 
		sal_uInt32 nCount = 0;
		(*pMarshal) <<= nCount;		
		break;
	}
	case typelib_TypeClass_TYPEDEF:
	{
		OSL_ASSERT( 0 );
		break;
	}
	case typelib_TypeClass_EXCEPTION:
	{
		kind = TCKind_tk_except;
		(*pMarshal) <<= kind;

		CorbaString8 string8;
		string8.theString = OUString( pType->pTypeName );
		(*pMarshal) <<= string8;

		CorbaString8 dummy;
		(*pMarshal) <<= dummy;

		// TODO : all struct member names 
		sal_uInt32 nCount = 0;
		(*pMarshal) <<= nCount;
		break;
	}
	case typelib_TypeClass_STRUCT:
	{		
		typelib_CompoundTypeDescription * pCompType =
			(typelib_CompoundTypeDescription *)pType;
		if( my_str_equalsWithLength( CORBA_STRING8_NAME , CORBA_STRING8_NAME_LENGTH,
									 pType->pTypeName->buffer, pType->pTypeName->length ) )
		{
			kind = TCKind_tk_string;
			(*pMarshal) <<= kind;

			sal_uInt32 nBound = 0;
			(*pMarshal) <<= nBound;
		}
		else if( pCompType->pBaseTypeDescription &&
				 my_str_equalsWithLength(
					 CORBA_UNION_NAME ,
					 CORBA_UNION_NAME_LENGTH,
					 pCompType->pBaseTypeDescription->aBase.pTypeName->buffer,
					 pCompType->pBaseTypeDescription->aBase.pTypeName->length ) )
		{
			// CorbaUnion 
			kind = TCKind_tk_union;
			(*pMarshal) <<= kind;

			CorbaString8 string8;
			string8.theString = OUString( pType->pTypeName );
			(*pMarshal) <<= string8;
			
			typelib_TypeDescription *pDiscriminativeType = 0;
			TYPELIB_DANGER_GET( & pDiscriminativeType , pCompType->ppTypeRefs[0] );
			packTypecode( pMarshal , pDiscriminativeType );
			TYPELIB_DANGER_RELEASE( pDiscriminativeType );
			
			sal_Int32 nDefault = 0;
			(*pMarshal) <<= nDefault;
			
			sal_uInt32 nCount = 0;
			(*pMarshal) <<= nCount;
			// TODO : add members of the unions
		}
		else
		{
			kind = TCKind_tk_struct;
			(*pMarshal) <<= kind;
			
			CorbaString8 string8;
			string8.theString = OUString( pType->pTypeName );
			(*pMarshal) <<= string8;
			
			CorbaString8 dummy;
			(*pMarshal) <<= dummy;
			
			// TODO : all struct member names 
			sal_uInt32 nCount = 0;
			(*pMarshal) <<= nCount;
		}
		break;		
	}
	case typelib_TypeClass_SEQUENCE:
	{
		kind = TCKind_tk_sequence;
		(*pMarshal) <<= kind;

		typelib_TypeDescription *pElementType = 0;
		typelib_typedescriptionreference_getDescription(
			&pElementType , ((typelib_IndirectTypeDescription *)pType)->pType );
		packTypecode( pMarshal , pElementType );
		typelib_typedescription_release( pElementType );

		sal_uInt32 nBound = 0;
		(*pMarshal) <<= nBound;
		break;
	}
	case typelib_TypeClass_INTERFACE:
	{
		kind = TCKind_tk_objref;
		(*pMarshal) <<= kind;
		
		CorbaString8 string8;
		string8.theString = OUString( pType->pTypeName );
		(*pMarshal) <<= string8;
		
		CorbaString8 dummy;
		(*pMarshal) <<= dummy;
		break;
	}
	case typelib_TypeClass_VOID:
	{
		kind = TCKind_tk_void;
		(*pMarshal) <<= kind;		
		break;
	}
	case typelib_TypeClass_TYPE:
	{
		kind = TCKind_tk_TypeCode;
		(*pMarshal) <<= kind;
		break;
	}
	case typelib_TypeClass_UNION:
	case typelib_TypeClass_ARRAY:
	case typelib_TypeClass_SERVICE:
	case typelib_TypeClass_MODULE:
	case typelib_TypeClass_INTERFACE_METHOD:
	case typelib_TypeClass_INTERFACE_ATTRIBUTE:
	case typelib_TypeClass_UNKNOWN:
	default:
		OSL_ASSERT( 0 );
	}
}

void Marshal::pack( void *pSource , typelib_TypeDescription *pType )
{
	switch( pType->eTypeClass )
	{
	case typelib_TypeClass_BYTE:
	{
		ensureAdditionalMem( 1 );
		*m_pos = *((sal_Int8*) pSource );
		m_pos++;
		break;
	}
	case typelib_TypeClass_BOOLEAN:
	{
		ensureAdditionalMem( 1 );
		*m_pos = ( *((sal_Bool*) pSource ) ) ? 1 : 0;
		m_pos++;
		break;
	}

	case typelib_TypeClass_CHAR:
	case typelib_TypeClass_SHORT:
	case typelib_TypeClass_UNSIGNED_SHORT:
	{
		align( 2 );
		ensureAdditionalMem( 2 );
		sal_uInt16 *p = ( sal_uInt16 *  ) pSource;
		if( m_bIsLittleEndian && g_bSystemIsLittleEndian ||
			! m_bIsLittleEndian && ! g_bSystemIsLittleEndian ) {
			*((sal_uInt16*) m_pos ) = *p;
		}
		else {
			m_pos[0] = ((unsigned char *)pSource)[1];
			m_pos[1] = ((unsigned char *)pSource)[0];
		}
		m_pos +=2;
		break;
	}
	case typelib_TypeClass_FLOAT:
	case typelib_TypeClass_LONG:
	case typelib_TypeClass_UNSIGNED_LONG:
	{
		align( 4 );
		ensureAdditionalMem( 4 );
		sal_uInt32 *p = ( sal_uInt32 * ) pSource;
		if( m_bIsLittleEndian && g_bSystemIsLittleEndian ||
			! m_bIsLittleEndian && ! g_bSystemIsLittleEndian ) {
			*((sal_uInt32*) m_pos ) = *p;
		}
		else {
			m_pos[0] = ((unsigned char *)pSource)[3];
			m_pos[1] = ((unsigned char *)pSource)[2];
			m_pos[2] = ((unsigned char *)pSource)[1];
			m_pos[3] = ((unsigned char *)pSource)[0];
		}
		m_pos +=4;
		break;
	}
	case typelib_TypeClass_HYPER:
	case typelib_TypeClass_UNSIGNED_HYPER:
	case typelib_TypeClass_DOUBLE:
	{
		align( 8 );
		ensureAdditionalMem( 8 );
		sal_uInt64 *p = ( sal_uInt64 * ) pSource;
		if( m_bIsLittleEndian && g_bSystemIsLittleEndian ||
			! m_bIsLittleEndian && ! g_bSystemIsLittleEndian ) {
			*((sal_uInt64*) m_pos ) = *p;
		}
		else {
			m_pos[0] = ((unsigned char *)pSource)[7];
			m_pos[1] = ((unsigned char *)pSource)[6];
			m_pos[2] = ((unsigned char *)pSource)[5];
			m_pos[3] = ((unsigned char *)pSource)[4];
			m_pos[4] = ((unsigned char *)pSource)[3];
			m_pos[5] = ((unsigned char *)pSource)[2];
			m_pos[6] = ((unsigned char *)pSource)[1];
			m_pos[7] = ((unsigned char *)pSource)[0];
		}
		m_pos += 8;
		break;
	}
	case typelib_TypeClass_STRING:
	{
		rtl_uString *p = *( rtl_uString ** ) pSource;
		sal_Int32 nLenPlus = p->length +1;
		(*this) <<= nLenPlus;

		ensureAdditionalMem( nLenPlus * sizeof( sal_Unicode ) );
		
		if( m_bIsLittleEndian && g_bSystemIsLittleEndian ||
			! m_bIsLittleEndian && ! g_bSystemIsLittleEndian ) {
			memcpy( m_pos , p->buffer , sizeof( sal_Unicode ) * nLenPlus );
			m_pos += ( sizeof( sal_Unicode ) * nLenPlus );
		}
		else
		{
			sal_Unicode *pUnicode = p->buffer;
			for( sal_Int32 i = 0; i < nLenPlus ; i ++ )
			{
				m_pos[0] = ((unsigned char *)pUnicode)[1];
				m_pos[1] = ((unsigned char *)pUnicode)[0];
				m_pos += 2;
				pUnicode ++;
			}
		}
		break;
	}
	case typelib_TypeClass_ANY:
	{
		uno_Any *pAny = (uno_Any * ) pSource;

		typelib_TypeDescription *pValueType = 0;
		typelib_typedescriptionreference_getDescription( &pValueType , pAny->pType );
		packTypecode( this, pValueType );
		pack( pAny->pData , pValueType );
		typelib_typedescription_release( pValueType );
		break;
	}
	case typelib_TypeClass_ENUM:
	{
		sal_Int32 nValue = *(sal_Int32*)pSource;
		typelib_EnumTypeDescription *pEnumType = ( typelib_EnumTypeDescription * ) pType;
		
		sal_Int32 i;
		for( i = 0; i < pEnumType->nEnumValues ; i ++ )
		{
			if( pEnumType->pEnumValues[i] == nValue )
			{
				// this is against iiop-specification !!!
				// must be (*this) <<= i 
				(*this) <<= nValue;
				break;
			}
		}
//		OSL_ASSERT( i < pEnumType->nEnumValues );
		if( i == pEnumType->nEnumValues )
		{
			(*this) <<= pEnumType->nDefaultEnumValue;
		}
		break;
	}
	case typelib_TypeClass_TYPEDEF:
	{
		pack( pSource , ((typelib_IndirectTypeDescription *) pType )->pType );
		break;
	}
	case typelib_TypeClass_EXCEPTION:
	case typelib_TypeClass_STRUCT:
	{		
		typelib_CompoundTypeDescription * pCompType =
			(typelib_CompoundTypeDescription *)pType;

		if( my_str_equalsWithLength(
			CORBA_STRING8_NAME , CORBA_STRING8_NAME_LENGTH,
			pType->pTypeName->buffer, pType->pTypeName->length ) )
		{
			// pack string8 !
			rtl_uString *p = *(rtl_uString **) pSource;
			sal_Int32 nLenPlus = p->length+1;

			(*this) <<= nLenPlus;

			ensureAdditionalMem( nLenPlus );
			for( sal_Int32 i = 0 ; i < nLenPlus ; i ++ )
			{
				OSL_ASSERT( p->buffer[i] <= 127 );
				m_pos[i] = p->buffer[i];
			}
			m_pos += nLenPlus;

		}
		else if( pCompType->pBaseTypeDescription &&
				 my_str_equalsWithLength(
					CORBA_UNION_NAME ,
					CORBA_UNION_NAME_LENGTH,
					pCompType->pBaseTypeDescription->aBase.pTypeName->buffer,
					pCompType->pBaseTypeDescription->aBase.pTypeName->length ) )
		{
			packUnion( this , pSource , pCompType );
		}
		else
		{
			if (pCompType->pBaseTypeDescription)
			{
				pack( pSource , (typelib_TypeDescription * )pCompType->pBaseTypeDescription );
			}
			
			// then construct members
			typelib_TypeDescriptionReference ** ppTypeRefs = pCompType->ppTypeRefs;
			sal_Int32 * pMemberOffsets = pCompType->pMemberOffsets;
			sal_Int32 nDescr = pCompType->nMembers;
			
			for ( sal_Int32 nPos = 0; nPos < nDescr; ++nPos )
			{
				typelib_TypeDescription * pMemberType = 0;
				TYPELIB_DANGER_GET( &pMemberType, ppTypeRefs[nPos] );
				pack( (char*)pSource + pMemberOffsets[nPos] , pMemberType );
				TYPELIB_DANGER_RELEASE( pMemberType );
			}
		}
		break;		
	}
	case typelib_TypeClass_SEQUENCE:
	{
		typelib_TypeDescription * pElementType = 0;
		typelib_typedescriptionreference_getDescription( 
			&pElementType , ((typelib_IndirectTypeDescription *)pType)->pType );

		if( pElementType )
		{
			sal_Int32 nElements		 = (*(uno_Sequence **)pSource)->nElements;		
			sal_Int32 nElementSize	 = pElementType->nSize;
			
			char * pSourceElements	 = (char *)(*(uno_Sequence **)pSource)->elements;
			
			if( typelib_TypeClass_BYTE == pElementType->eTypeClass )
			{
				// Byte sequences are optimized
				packByteSequence( (sal_Int8*)pSourceElements , nElements );
			}
			else
			{
				(*this) <<= nElements;
				for ( sal_Int32 i = 0 ; i < nElements; i++ )
				{
					pack( pSourceElements + (nElementSize*i) , pElementType );
				}
			}
			typelib_typedescription_release( pElementType );
		}
		else
		{
			OSL_ENSURE(0 , "urp-bridge : unknown type" );
		}
		break;
	}
	case typelib_TypeClass_INTERFACE:
	{
		iop::IOR ior;
		ObjectKey key;

		remote_Interface *pRemoteI = *( remote_Interface ** )pSource;
		if( pRemoteI )
		{

			remote_retrieveOidFromProxy( pRemoteI, &(key.sOid.theString.pData));
			key.sType.theString = pType->pTypeName;
//  			m_callback( pRemoteI ,
//  						&(key.sOid.theString.pData),
//  						&(key.sType.theString.pData) );

			ior.type_id.theString  = key.sType.theString;
			
			TaggedProfile profile;
			profile.tag = ProfileIdGroup::TAG_INTERNET_IOP;
			{
				ProfileBody_1_1 body;
				body.port = 0;
				body.iiop_version.major = 1;
				body.iiop_version.minor = 2;
				{
					Marshal marsh( m_bIsLittleEndian );
					marsh <<= key;
					body.object_key =  marsh.remove();
				}
				{
					Marshal marsh( m_bIsLittleEndian );
					marsh <<= body;
					profile.profile_data = marsh.remove();
				}
			}
			ior.profiles = Sequence < TaggedProfile > ( &profile , 1 );
		}
		else
		{
			// empty ior structure is null reference
		}
		(*this ) <<= ior;

		break;		
	}
	case typelib_TypeClass_VOID:
		// do nothing 
		break;

	case typelib_TypeClass_TYPE:
	{
	    typelib_TypeDescriptionReference *pRef =
		  *(typelib_TypeDescriptionReference ** ) pSource;
		typelib_TypeDescription *pType = 0;
		typelib_typedescriptionreference_getDescription(
			   &pType ,
			   pRef );

		if( ! pType )
		{
			// Unknown type !
			// Marshal to void
			Type type;
			type.getDescription( &pType );
		}
		
		packTypecode( this , pType );

		typelib_typedescription_release( pType );
		
	    break;
	}
	case typelib_TypeClass_UNION:
	case typelib_TypeClass_ARRAY:
	case typelib_TypeClass_SERVICE:
	case typelib_TypeClass_MODULE:
	case typelib_TypeClass_INTERFACE_METHOD:
	case typelib_TypeClass_INTERFACE_ATTRIBUTE:
	case typelib_TypeClass_UNKNOWN:
	default:
		OSL_ASSERT( 0 );
	}	
}

} // end namespace bridges

