/*************************************************************************
 *
 *  OpenOffice.org - a multi-platform office productivity suite
 *
 *  $RCSfile: salsound2.cxx,v $
 *
 *  $Revision: 1.8 $
 *
 *  last change: $Author: rt $ $Date: 2005/09/09 13:01:54 $
 *
 *  The Contents of this file are made available subject to
 *  the terms of GNU Lesser General Public License Version 2.1.
 *
 *
 *    GNU Lesser General Public License Version 2.1
 *    =============================================
 *    Copyright 2005 by Sun Microsystems, Inc.
 *    901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *    This library is free software; you can redistribute it and/or
 *    modify it under the terms of the GNU Lesser General Public
 *    License version 2.1, as published by the Free Software Foundation.
 *
 *    This library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *    Lesser General Public License for more details.
 *
 *    You should have received a copy of the GNU Lesser General Public
 *    License along with this library; if not, write to the Free Software
 *    Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *    MA  02111-1307  USA
 *
 ************************************************************************/
#include <sys/stat.h>
#include <sys/mman.h>
#include <fcntl.h>
#include <unistd.h>

#include <salunx.h>
#include <saldata.hxx>
#include <saldisp.hxx>

#include <salsound.h>
#include <salstype.hxx>
#include <salimpsound.hxx>

using namespace vcl_sal;

ULONG X11SalSound::s_nSoundState = SOUND_STATE_UNLOADED;

X11SalSound::X11SalSound() :
		m_pVSound( NULL ),
		m_bLoop( false ),
		m_bPlaying( false ),
		m_bPaused( false ),
		m_nStartTime( 0 ),
		m_nPlayTime( SOUND_PLAYALL )
{
	SalDbgAssert( "X11SalSound::X11SalSound()\n" );
    #ifdef USE_PASF
    // initialize the portaudio library (typically just once)
    if (s_nSoundState == SOUND_STATE_UNLOADED) 
    {
        PaError err = Pa_Initialize();  
        if( err == paNoError )
            s_nSoundState = SOUND_STATE_VALID;
    }
    if (s_nSoundState != SOUND_STATE_VALID )
        s_nSoundState = SOUND_STATE_INVALID;
    #else
    s_nSoundState = SOUND_STATE_VALID;
    #endif
}

X11SalSound::~X11SalSound()
{
	SalDbgAssert( "X11SalSound::~X11SalSound()\n" );
	if( m_pVSound )
		delete m_pVSound;
}

void X11SalSound::Release()
{
#ifdef USE_PASF
        // terminate the portaudio library
        if ( s_nSoundState == SOUND_STATE_VALID ) Pa_Terminate();
#endif
        s_nSoundState = SOUND_STATE_UNLOADED;
}

bool X11SalSound::IsValid()
{
  return (X11SalSound::s_nSoundState == SOUND_STATE_VALID);
}


bool X11SalSound::Init( const String& rSoundName, ULONG& rSoundLen )
{
    
    // FIXME: NULL file names are passed in
    // This can happen before play of previous sound has completed and this disrupts things
    // simpy ignore this for now
    if (! (rSoundName.Len()))
        return false;
    
	if( m_pVSound )
		delete m_pVSound;

    #ifdef MACOSX
	m_aSoundFile = ByteString( rSoundName, RTL_TEXTENCODING_UTF8 );
    #else
	m_aSoundFile = ByteString( rSoundName, osl_getThreadTextEncoding() );
    #endif
	SalDbgAssert( "X11SalSound::Init( \"%s\", %d )\n", m_aSoundFile.GetBuffer(), rSoundLen );
    
	m_pVSound = ( !m_aSoundFile.Len() || access( m_aSoundFile.GetBuffer(), R_OK ) ) ? NULL : VSound::createVSound( this );
    
	return m_pVSound ? true : false;
}

void X11SalSound::Play( ULONG nStartTime, ULONG nPlayTime, bool bLoop )
{
	SalDbgAssert( "SalSound::Play( %d, %d, %s )\n", nStartTime, nPlayTime, bLoop ? "TRUE" : "FALSE" );
	if( m_bPlaying && m_pVSound )
		m_pVSound->stop();
    
	m_bLoop			= bLoop;
	m_nStartTime	= nStartTime;
	m_nPlayTime		= nPlayTime;
    
	if( m_pVSound )
		m_pVSound->play();
}

void X11SalSound::Stop()
{
	SalDbgAssert( "X11SalSound::Stop()\n" );
	if( m_bPlaying && m_pVSound )
		m_pVSound->stop();
}

void X11SalSound::Pause()
{
	if( m_bPlaying && m_pVSound )
		m_pVSound->pause();
}

void X11SalSound::Continue()
{
	if( m_bPaused && m_pVSound )
		m_pVSound->cont();
}

bool X11SalSound::IsLoopMode() const
{
    return m_bLoop;
}

bool X11SalSound::IsPlaying() const
{
    return m_bPlaying;
}

bool X11SalSound::IsPaused() const
{
    return m_bPaused;
}

// --------------------------------------------------

void VSound::initBuffer()
{
	if( stat( m_pSalSound->m_aSoundFile.GetBuffer(), &m_aStat ) )
	{
		SalDbgAssert( "could not stat \"%s\"\n", m_pSalSound->m_aSoundFile.GetBuffer() );
		return;
	}
	int nFile;
	if( ( nFile = ::open( m_pSalSound->m_aSoundFile.GetBuffer(), O_RDONLY ) ) == -1 )
	{
		SalDbgAssert( "could not open %s\n", m_pSalSound->m_aSoundFile.GetBuffer() );
		return;
	}

	m_pBuffer = (char *)mmap( NULL, m_aStat.st_size, PROT_READ, MAP_SHARED, nFile, 0 );
	::close( nFile );
	if( m_pBuffer == MAP_FAILED )
	{
		SalDbgAssert( "could not mmap %s\n", m_pSalSound->m_aSoundFile.GetBuffer() );
		m_pBuffer = NULL;
		return;
	}
}

void VSound::releaseBuffer()
{
	if( m_pBuffer )
	{
		munmap( m_pBuffer, m_aStat.st_size );
		m_pBuffer = NULL;
	}
}


VSound* VSound::createVSound( X11SalSound* pX11SalSound )
{
	struct stat aStat;
	if( stat( pX11SalSound->m_aSoundFile.GetBuffer(), & aStat ) )
		return NULL;

	VSound* pRet = NULL;

	if( GetSalData()->GetDisplay()->IsLocal() )
    {
        #ifdef USE_PASF
        {
            pRet = new PASFSound( pX11SalSound );
            if( pRet && ! pRet->isValid() )
            {
                delete pRet;
                pRet = NULL;
            }
            else
                SalDbgAssert( "got an PASFSound\n" );
        }
        #endif

        #ifdef USE_OSS
        if( ! pRet )
        {
            pRet = new OSSSound( pX11SalSound );
            if( pRet && ! pRet->isValid() )
            {
                delete pRet;
                pRet = NULL;
            }
            else
                SalDbgAssert( "got an OSSSound\n" );
        }
        #endif

        #ifdef USE_DEVAUDIO
        if( ! pRet )
        {
            pRet = new DevAudioSound( pX11SalSound );
            if( pRet && ! pRet->isValid() )
            {
                delete pRet;
                pRet = NULL;
            }
            else
                SalDbgAssert( "got an DevAudioSound\n" );
        }
        #endif
    }
    
#ifdef USE_NAS
	if( ! pRet )
	{
		pRet = new NASSound( pX11SalSound );
		if( pRet && ! pRet->isValid() )
		{
			delete pRet;
			pRet = NULL;
		}
		else
			SalDbgAssert( "got an AUSound\n" );
	}
#endif

	if( ! pRet )
	{
		pRet = new RPTPSound( pX11SalSound );
		if( pRet && ! pRet->isValid() )
		{
			delete pRet;
			pRet = NULL;
		}
		else
			SalDbgAssert( "got an RPTPSound\n" );
	}
	return pRet;
}


void X11SalSound::changeStatePlay()
{
	SalDbgAssert( "SalSound(%p)::changeStatePlay()\n", this );

	m_bPlaying	= TRUE;
	m_bPaused 	= FALSE;

    CallNotifyProc( SOUND_NOTIFY_SUCCESS, SOUNDERR_SUCCESS );
}

void X11SalSound::changeStateStop()
{
	SalDbgAssert( "X11SalSound(%p)::changeStateStop()\n", this );

	m_bPlaying	= FALSE;
	m_bPaused 	= FALSE;

    CallNotifyProc( SOUND_NOTIFY_SUCCESS, SOUNDERR_SUCCESS );
}

void X11SalSound::changeStatePause()
{
	SalDbgAssert( "X11SalSound(%p)::changeStatePause()\n", this );

	m_bPlaying	= TRUE;
	m_bPaused	= TRUE;

    CallNotifyProc( SOUND_NOTIFY_SUCCESS, SOUNDERR_SUCCESS );
}

void X11SalSound::changeStateCont()
{
	SalDbgAssert( "X11SalSound(%p)::changeStateCont()\n", this );

	m_bPlaying	= TRUE;
	m_bPaused	= FALSE;

    CallNotifyProc( SOUND_NOTIFY_SUCCESS, SOUNDERR_SUCCESS );
}

void X11SalSound::setError( ULONG nError )
{
    CallNotifyProc( SOUND_NOTIFY_ERROR, nError );
}

