/*
 * ratDisFolder.c --
 *
 *      This file contains code which implements disconnected folders.
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

/*
 *  * One directory per folder
 *  * In that directory we find the following files:
 *    master	- File containing information about master folder.
 *		  Contains the following entries one on each line:
 *			name
 *			folder_spec
 *			user
 *			protocol
 *    state	- State against master.
 *		  Contains the following entries one on each line:
 *			uidvalidity
 *			last known UID in master
 *    mappings	- File with mappings msgid <> uid in master
 *    folder	- The local copy of the folder
 *    changes   - Changes which should be applied to the master once
 *		  we synchronize. This file should contain:
 *			delete uid		- message to delete
 *			flag UID flag value	- set flag to value
 *
 * TODO,
 *    messages (add and translate)
 *    deletion of folder
 */

#include "ratFolder.h"
#include "ratStdFolder.h"
#include "mbx.h"

/*
 * This is the private part of a disconnected folder info structure.
 */

typedef struct DisFolderInfo {
    char *dir;			/* Directory where local data is stored */
    Tcl_HashTable map;		/* Mappings local_uid > remote_uid */
    int mapChanged;		/* non null if mappings needs to be rewritten */

    /* Original procs for local folder */
    RatInitProc *initProc;
    RatCloseProc *closeProc;
    RatUpdateProc *updateProc;
    RatInsertProc *insertProc;
    RatSetFlagProc *setFlagProc;
    RatGetFlagProc *getFlagProc;
    RatInfoProc *infoProc;
    RatSetInfoProc *setInfoProc;
    RatCreateProc *createProc;
} DisFolderInfo;

/*
 * Hashtable containing open disfolders
 * The dirname is the key and the infoPtr is the value
 */
Tcl_HashTable openFolders;

/*
 * The uid map
 */
static struct {
    unsigned long *map;
    unsigned long alloc;
    unsigned long size;
} uidMap;

/*
 * Procedures private to this module.
 */
static RatInitProc Dis_InitProc;
static RatCloseProc Dis_CloseProc;
static RatUpdateProc Dis_UpdateProc;
static RatInsertProc Dis_InsertProc;
static RatInfoProc Dis_InfoProc;
static RatSetFlagProc Dis_SetFlagProc;
static RatGetFlagProc Dis_GetFlagProc;
static RatCreateProc Dis_CreateProc;
static RatSetInfoProc Dis_SetInfoProc;
static RatSyncProc Dis_SyncProc;
static int CreateDir(char *dir);
static Tcl_CmdProc RatSyncDisconnected;
static void Dis_FindFolders(Tcl_Interp *interp, char *dir);
static void Dis_SyncFolder(Tcl_Interp *interp, char *dir, off_t size,int force);
static unsigned long GetMasterUID(MAILSTREAM *s, Tcl_HashTable *mapPtr,
	int index);
static void UpdateFolderFlag(Tcl_Interp *interp, RatFolderInfoPtr infoPtr,
	int index, RatFlag flag, int value);
static void ReadMappings(MAILSTREAM *s, const char *dir, Tcl_HashTable *mapPtr);
static void ReadOldMappings(MAILSTREAM *s, Tcl_HashTable *mapPtr, char *buf,
	int buflen, FILE *fp);
static void InitUidMap(MAILSTREAM *s);
static unsigned long MsgNo(unsigned long uid);
static void CheckDeletion(RatFolderInfoPtr infoPtr, Tcl_Interp *interp);


/*
 *----------------------------------------------------------------------
 *
 * RatDisFolderInit --
 *
 *      Initializes the disconnected folder command.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	the result area.
 *
 * Side effects:
 *	None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatDisFolderInit(Tcl_Interp *interp)
{
    Tcl_InitHashTable(&openFolders, TCL_STRING_KEYS);
    Tcl_CreateCommand(interp, "RatSyncDisconnected", RatSyncDisconnected,
	    NULL, NULL);
    uidMap.map = NULL;
    uidMap.alloc = 0;
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDisFolderCreate --
 *
 *      Creates a disconnected folder entity.
 *
 * Results:
 *      The return value is normally TCL_OK; if something goes wrong
 *	TCL_ERROR is returned and an error message will be left in
 *	the result area.
 *
 * Side effects:
 *	A disconnected folder is created.
 *
 *
 *----------------------------------------------------------------------
 */

RatFolderInfo*
RatDisFolderCreate(Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[])
{
    const char *dir;
    Tcl_Obj *lobjv[4];
    RatFolderInfo *infoPtr;
    DisFolderInfo *disPtr;
    Tcl_HashEntry *entryPtr;
    Tcl_DString ds;
    int unused;

    if (objc != 7) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		Tcl_GetString(objv[0]), " ", Tcl_GetString(objv[1]),
		" flags name folder_spec user prot\"", (char *) NULL);
	return (RatFolderInfo *) NULL;
    }

    /*
     * Prepare directory
     */
    dir = RatDisPrepareDir(interp, Tcl_GetString(objv[3]),
			   Tcl_GetString(objv[4]), Tcl_GetString(objv[5]),
			   Tcl_GetString(objv[6]));
    if (!dir) {
	return NULL;
    }
    disPtr = (DisFolderInfo *) ckalloc(sizeof(*disPtr));
    disPtr->dir = cpystr(dir);

    /*
     * Open filefolder
     */
    Tcl_DStringInit(&ds);
    Tcl_DStringAppend(&ds, disPtr->dir, -1);
    Tcl_DStringAppend(&ds, "/folder", 7);
    lobjv[0] = objv[0];
    lobjv[1] = Tcl_NewStringObj("std", -1);
    lobjv[2] = objv[2];
    lobjv[3] = Tcl_NewStringObj(Tcl_DStringValue(&ds), Tcl_DStringLength(&ds));
    infoPtr = RatStdFolderCreate(interp, 4, lobjv);
    Tcl_DecrRefCount(lobjv[1]);
    Tcl_DecrRefCount(lobjv[3]);
    if (NULL == infoPtr) {
	Tcl_DStringFree(&ds);
	goto error;
    }

    /*
     * Read mappings
     */
    Tcl_InitHashTable(&disPtr->map, TCL_ONE_WORD_KEYS);
    ReadMappings(((StdFolderInfo*)infoPtr->private)->stream, 
	    disPtr->dir, &disPtr->map);

    infoPtr->name = Tcl_GetString(objv[3]);
    if (!*infoPtr->name) {
	infoPtr->name = "INBOX";
    }
    infoPtr->name = cpystr(infoPtr->name);
    infoPtr->type = "dis";
    infoPtr->private2 = (ClientData) disPtr;

    disPtr->initProc = infoPtr->initProc;
    disPtr->closeProc = infoPtr->closeProc;
    disPtr->updateProc = infoPtr->updateProc;
    disPtr->insertProc = infoPtr->insertProc;
    disPtr->setFlagProc = infoPtr->setFlagProc;
    disPtr->getFlagProc = infoPtr->getFlagProc;
    disPtr->infoProc = infoPtr->infoProc;
    disPtr->setInfoProc = infoPtr->setInfoProc;
    disPtr->createProc = infoPtr->createProc;

    infoPtr->initProc = Dis_InitProc;
    infoPtr->closeProc = Dis_CloseProc;
    infoPtr->updateProc = Dis_UpdateProc;
    infoPtr->insertProc = Dis_InsertProc;
    infoPtr->setFlagProc = Dis_SetFlagProc;
    infoPtr->getFlagProc = Dis_GetFlagProc;
    infoPtr->infoProc = Dis_InfoProc;
    infoPtr->setInfoProc = Dis_SetInfoProc;
    infoPtr->createProc = Dis_CreateProc;
    infoPtr->syncProc = Dis_SyncProc;

    /*
     * Add to hash table
     */
    entryPtr = Tcl_CreateHashEntry(&openFolders, disPtr->dir, &unused);
    Tcl_SetHashValue(entryPtr, (ClientData)infoPtr);

    Tcl_DStringFree(&ds);
    return infoPtr;

error:
    ckfree(disPtr);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDisPrepareDir --
 *
 *      Prepares the directory for a disconnected folder
 *
 * Results:
 *	A pointer to a stattic are holde the name of the directory or
 *	NULL on errors;
 *
 * Side effects:
 *	Updates the master-file
 *
 *----------------------------------------------------------------------
 */
const char*
RatDisPrepareDir(Tcl_Interp *interp, const char *name, const char *folder_spec,
	   const char *user, const char *prot)
{
    const char *dir;
    struct stat sbuf;
    Tcl_DString ds;
    FILE *fp;

    /*
     * Find directory and make sure it exists
     */
    if (!(dir = RatDisFolderDir(interp, folder_spec, user, prot))) {
	return NULL;
    }
    if (!*name) {
	name = "INBOX";
    }

    /*
     * Initialize state-file and create folder-file if it does not exist
     */
    Tcl_DStringInit(&ds);
    Tcl_DStringAppend(&ds, dir, -1);
    Tcl_DStringAppend(&ds, "/state", 7);
    if (0 != stat(Tcl_DStringValue(&ds), &sbuf)) {
	fp = fopen(Tcl_DStringValue(&ds), "w");
	if (NULL == fp) {
	    Tcl_DStringFree(&ds);
	    return NULL;
	}
	fprintf(fp, "0\n0\n");
	fclose(fp);

	Tcl_DStringSetLength(&ds, strlen(dir));
	Tcl_DStringAppend(&ds, "/folder", 7);
	mbx_create(NIL, Tcl_DStringValue(&ds));
    }

    /*
     * Always update the master-file (the user may have changed some setting)
     */
    Tcl_DStringSetLength(&ds, strlen(dir));
    Tcl_DStringAppend(&ds, "/master", 7);
    fp = fopen(Tcl_DStringValue(&ds), "w");
    if (NULL == fp) {
	Tcl_DStringFree(&ds);
	return NULL;
    }
    fprintf(fp, "%s\n%s\n%s\n%s\n", name, folder_spec, user, prot);
    fclose(fp);

    Tcl_DStringFree(&ds);
    return dir;
}

/*
 *----------------------------------------------------------------------
 *
 * RatDisFolderOpenStream --
 *
 *      Opens the local part of a disconnected folder.
 *
 * Results:
 *      A pointer to a MAILSTREAM or NULL on failures.
 *
 * Side effects:
 *	A disconnected folder is created.
 *
 *----------------------------------------------------------------------
 */

MAILSTREAM*
RatDisFolderOpenStream(Tcl_Interp *interp, int objc, Tcl_Obj *CONST objv[],
	char **mailbox)
{
    static Tcl_DString ds;
    static int initialized = 0;
    const char *dir;
    MAILSTREAM *stream;

    if (objc != 5) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		"name ", Tcl_GetString(objv[1]),
		" flags folder_spec user\"", (char *) NULL);
	return (MAILSTREAM*)NULL;
    }

    if (initialized) {
	Tcl_DStringSetLength(&ds, 0);
    } else {
	Tcl_DStringInit(&ds);
    }

    /*
     * Prepare directory
     */
    dir = RatDisPrepareDir(interp, Tcl_GetString(objv[0]),
			   Tcl_GetString(objv[3]), Tcl_GetString(objv[4]),
			   "imap");
    if (!dir) {
	return NULL;
    }

    /*
     * Open filefolder
     */
    Tcl_DStringAppend(&ds, dir, -1);
    Tcl_DStringAppend(&ds, "/folder", 7);
    stream = OpenStdFolder(interp, Tcl_DStringValue(&ds), "std",
			   Tcl_GetString(objv[2]), NULL);
    if (mailbox) {
	*mailbox = Tcl_DStringValue(&ds);
    }
    return stream;
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_InitProc --
 *
 *      See the documentation for initProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for initProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static void
Dis_InitProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    /* Do nothing */
    (*disPtr->initProc)(infoPtr, interp, index);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_CloseProc --
 *
 *      See the documentation for closeProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for closeProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_CloseProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int expunge)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    char buf[1024];
    unsigned long *lPtr;
    int result;
    FILE *fp;

    if (expunge) {
	CheckDeletion(infoPtr, interp);
    }
    result = (*disPtr->closeProc)(infoPtr, interp, expunge);
    entryPtr = Tcl_FindHashEntry(&openFolders, disPtr->dir);
    Tcl_DeleteHashEntry(entryPtr);
    if (disPtr->mapChanged) {
	snprintf(buf, sizeof(buf), "%s/mappings", disPtr->dir);
	fp = fopen(buf, "w");
    } else {
	fp = NULL;
    }
    for (entryPtr = Tcl_FirstHashEntry(&disPtr->map, &search); entryPtr;
	    entryPtr = Tcl_NextHashEntry(&search)) {
	lPtr = (unsigned long*)Tcl_GetHashValue(entryPtr);
	if (fp) {
	    fprintf(fp, "%ld %ld\n", *lPtr,
		    (unsigned long)Tcl_GetHashKey(&disPtr->map, entryPtr));
	}
	ckfree(lPtr);
    }
    if (fp) {
	fclose(fp);
    }
    Tcl_DeleteHashTable(&disPtr->map);
    ckfree(disPtr->dir);
    ckfree(disPtr);
    return result;
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_UpdateProc --
 *
 *      See the documentation for updateProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for updateProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_UpdateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, RatUpdateType mode)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;

    if (RAT_SYNC == mode) {
	CheckDeletion(infoPtr, interp);
    }
    return (*disPtr->updateProc)(infoPtr, interp, mode);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_InsertProc --
 *
 *      See the documentation for insertProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for insertProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_InsertProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int argc,
	char *argv[])
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    /* Do nothing */
    return (*disPtr->insertProc)(infoPtr, interp, argc, argv);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_SetFlagProc --
 *
 *      See the documentation for setFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for setFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_SetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag, int value)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    FILE *fp = NULL;
    char buf[1024];
    unsigned long uid;

    uid = GetMasterUID(((StdFolderInfo*)infoPtr->private)->stream,
	    &disPtr->map, index);
    if (uid) {
	snprintf(buf, sizeof(buf), "%s/changes", disPtr->dir);
	if (NULL != (fp = fopen(buf, "a"))) {
	    fprintf(fp, "flag %ld %d %d\n", uid, flag, value);
	    fclose(fp);
	}
    }
    return (*disPtr->setFlagProc)(infoPtr, interp, index, flag, value);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_GetFlagProc --
 *
 *      See the documentation for getFlagProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for getFlagProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_GetFlagProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index,
	RatFlag flag)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    /* Do Nothing */
    return (*disPtr->getFlagProc)(infoPtr, interp, index,flag);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_InfoProc --
 *
 *      See the documentation for infoProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for infoProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */

Tcl_Obj*
Dis_InfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int index)
{
    /* Do Nothing */
    return Std_InfoProc(interp, clientData, type, index);
}

/*
 *----------------------------------------------------------------------
 *
 * Dis_CreateProc --
 *
 *      See the documentation for createProc in ratFolder.h
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the documentation for createProc in ratFolder.h
 *
 *
 *----------------------------------------------------------------------
 */
static char*
Dis_CreateProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp, int index)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    /* Do Nothing */
    return (*disPtr->createProc)(infoPtr, interp, index);
}

/*
 *----------------------------------------------------------------------
 *
 * Dis_SetInfoProc --
 *
 *      Sets information about a message
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static void
Dis_SetInfoProc(Tcl_Interp *interp, ClientData clientData,
	RatFolderInfoType type, int index, Tcl_Obj *oPtr)
{
    /* Do Nothing */
    Std_SetInfoProc(interp, clientData, type,index,oPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * CreateDir --
 *
 *      Checks that a given directory exists and creates it and any
 *	parent directories if they do not exist. This routine expects
 *	a complete path starting with '/'.
 *
 * Results:
 *	Non zero if failed and sets errno.
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static int
CreateDir(char *dir)
{
    struct stat sbuf;
    char *cPtr;

    /*
     * First we check if it already exists.
     */
    if (0 == stat(dir, &sbuf) && S_ISDIR(sbuf.st_mode)) {
	return 0;
    }

    /*
     * Go through all directories from the top and down and create those
     * which do not exist.
     */
    for (cPtr = strchr(dir+1, '/'); cPtr; cPtr = strchr(cPtr+1, '/')) {
	*cPtr = '\0';
	if (0 != stat(dir, &sbuf)) {
	    if (mkdir(dir, 0700)) {
		return 1;
	    }
	} else if (!S_ISDIR(sbuf.st_mode)) {
	    errno = ENOTDIR;
	    return 1;
	}
	*cPtr = '/';
    }
    if (0 != stat(dir, &sbuf)) {
	if (mkdir(dir, 0700)) {
	    return 1;
	}
    } else if (!S_ISDIR(sbuf.st_mode)) {
	errno = ENOTDIR;
	return 1;
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * RatSyncDisconnected --
 *
 *	Synchronizes all disconnected folders
 *
 * Results:
 *	None
 *
 * Side effects:
 *	All disconnected folders are updated
 *
 *
 *----------------------------------------------------------------------
 */
static int
RatSyncDisconnected(ClientData op, Tcl_Interp *interp, int argc, char *argv[])
{
    Tcl_DString tmpDs;
    char *value, *dirname;

    value = Tcl_GetVar2(interp, "option", "disconnected_dir", TCL_GLOBAL_ONLY);
    dirname = Tcl_TranslateFileName(interp, value, &tmpDs);
    Dis_FindFolders(interp, dirname);
    Tcl_DStringFree(&tmpDs);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Dis_FindFolders --
 *
 *	Recurses over a directory tree and calls Dis_SyncFolder for each
 *	found folder.
 *
 * Results:
 *	None
 *
 * Side effects:
 *	All disconnected folders are updated
 *
 *
 *----------------------------------------------------------------------
 */
static void
Dis_FindFolders(Tcl_Interp *interp, char *dir)
{
    struct stat sbuf;
    char buf[1024];
    DIR *dirPtr;
    struct dirent *direntPtr;

    /*
     * Check if this is a folder directory (contains a master-file)
     */
    RatStrNCpy(buf, dir, sizeof(buf)-7);
    strcat(buf, "/master");
    if (0 == stat(buf, &sbuf) && S_ISREG(sbuf.st_mode)) {
	Dis_SyncFolder(interp, dir, sbuf.st_size, 0);
	return;
    }

    /*
     * Otherwise check all entries and call Dis_FindFolders for all
     * descendants
     */
    if (NULL == (dirPtr = opendir(dir))) {
	return;
    }
    while (dirPtr && 0 != (direntPtr = readdir(dirPtr))) {
	snprintf(buf, sizeof(buf), "%s/%s", dir, direntPtr->d_name);
	if (0 != stat(buf, &sbuf)
		|| !S_ISDIR(sbuf.st_mode)
		|| !strcmp(".", direntPtr->d_name)
		|| !strcmp("..", direntPtr->d_name)) {
	    continue;
	}
	Dis_FindFolders(interp, buf);
    }
    closedir(dirPtr);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_SyncFolder --
 *
 *	Synchronizes a specified folder
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static void
Dis_SyncFolder(Tcl_Interp *interp, char *dir, off_t size, int force)
{
    char buf[1024], *name, *spec, *specMutf7, *user, *prot, *data,
	    *header, *body, localMailbox[1024], datebuf[128], *cPtr;
    MESSAGECACHE *elt;
    unsigned long uid, lastUid, folderNextUid, msgno, uidvalidity, len, nmsgs;
    MAILSTREAM *masterStream, *localStream;
    Tcl_HashTable *mapPtr;
    RatFolderInfoPtr infoPtr = NULL;
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    Tcl_CmdInfo cmdInfo;
    STRING string;  
    ENVELOPE *envPtr;
    int fd, i, masterError = 0;
    SEARCHPGM *pgm;
    FILE *fp = NULL;
    Tcl_DString ds;

    Tcl_DStringInit(&ds);

    /*
     * Read and parse masterfile & statefile
     */
    snprintf(buf, sizeof(buf), "%s/master", dir);
    if (-1 == (fd = open(buf, O_RDONLY))
	    || NULL == (data = (char*)ckalloc(size+1))
	    || size != read(fd, data, size)
	    || 0 != close(fd)) {
	RatLogF(interp, RAT_ERROR, "Failed to read masterfile", RATLOG_TIME);
	return;
    }
    name = data;
    if (NULL == (spec = strchr(data, '\n'))) return;
    *spec++ = '\0';
    if (NULL == (user = strchr(spec, '\n'))) return;
    *user++ = '\0';
    if (NULL == (prot = strchr(user, '\n'))) return;
    *prot++ = '\0';
    if (NULL == (cPtr = strchr(prot, '\n'))) return;
    *cPtr = '\0';
    snprintf(buf, sizeof(buf), "%s/state", dir);
    if (NULL == (fp = fopen(buf, "r"))
	    || 2 != fscanf(fp, "%ld\n%ld", &uidvalidity, &lastUid)
	    || 0 != fclose(fp)) {
	RatLog(interp, RAT_ERROR, "Failed to read statefile", RATLOG_TIME);
	return;
    }

    if (!force && Tcl_GetCommandInfo(interp, "RatUP_NetsyncFolder", &cmdInfo)) {
	Tcl_Obj *oPtr = Tcl_NewObj();
	int doSync = 0;

	Tcl_ListObjAppendElement(interp, oPtr,
		Tcl_NewStringObj("RatUP_NetsyncFolder", -1));
	Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewStringObj(spec, -1));
	Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewStringObj(user, -1));
	Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewStringObj(prot, -1));
	Tcl_IncrRefCount(oPtr);
	if (TCL_OK != Tcl_EvalObjEx(interp, oPtr, TCL_EVAL_GLOBAL)
		|| TCL_OK != Tcl_GetBooleanFromObj(interp,
			Tcl_GetObjResult(interp),&doSync)
		|| 0 == doSync) {
	    Tcl_DecrRefCount(oPtr);
	    ckfree(data);
	    return;
	}
	Tcl_DecrRefCount(oPtr);
    }

    RatLogF(interp, RAT_INFO, "synchronizing", RATLOG_EXPLICIT, name);

    /*
     * Open connection
     * Check uidvalidity
     * Apply deletion commands and expunge
     * Apply flag commands
     * loop over messages in local folder
     *   Check if has master uid
     *   if YES check if still exists in master
     *     if NO delete
     *     if YES check flags and update from master
     *   if NO then append to master, should update with masters uid
     * Loop over new messages in master and append them to local
     */

    /*
     * Open connections
     */
    if (NULL == (masterStream = Std_StreamOpen(interp, spec, user, NIL,
	    &masterError))){
	RatLog(interp, RAT_INFO, "", RATLOG_EXPLICIT);
	return;
    }
    specMutf7 = RatUtf8toMutf7(spec);
    if (!(0 == uidvalidity && 0 == lastUid)
	    && uidvalidity != masterStream->uid_validity) {
	RatLogF(interp, RAT_ERROR, "uidvalidity_changed", RATLOG_TIME);
	mail_close(masterStream);
	return;
    }
    if (0 == masterStream->uid_last) {
	mail_status(masterStream, specMutf7, SA_UIDNEXT);
	folderNextUid = stdStatus.uidnext;
    } else {
	folderNextUid = masterStream->uid_last+1;
    }
    if (NULL != (entryPtr = Tcl_FindHashEntry(&openFolders, dir))) {
	infoPtr = Tcl_GetHashValue(entryPtr);
	localStream = ((StdFolderInfo*)infoPtr->private)->stream;
	RatStrNCpy(localMailbox, ((StdFolderInfo*)infoPtr->private)->origName,
		sizeof(localMailbox));
	mapPtr = &(((DisFolderInfo*)infoPtr->private2)->map);
    } else {
	snprintf(localMailbox, sizeof(localMailbox), "%s/folder", dir);
	localStream = mail_open(NIL, localMailbox, NIL);
	mapPtr = (Tcl_HashTable*)ckalloc(sizeof(*mapPtr));
	Tcl_InitHashTable(mapPtr, TCL_ONE_WORD_KEYS);
	ReadMappings(localStream, dir, mapPtr);
    }
    if (NULL == localStream) {
	Std_StreamClose(interp, masterStream);
	return;
    }

    /*
     * Apply deletion commands and expunge
     */
    RatLogF(interp, RAT_INFO, "uploading", RATLOG_EXPLICIT);
    snprintf(buf, sizeof(buf), "%s/changes", dir);
    if (NULL != (fp = fopen(buf, "r"))) {
	buf[sizeof(buf)-1] = '\0';
	while (fgets(buf, sizeof(buf)-1, fp), !feof(fp)) {
	    if (!strncmp(buf, "delete", 6)) {
		if (Tcl_DStringLength(&ds)) {
		    Tcl_DStringAppend(&ds, ",", 1);
		}
		sprintf(buf, "%ld", atol(buf+7));
		Tcl_DStringAppend(&ds, buf, -1);
	    }
	}
	if (Tcl_DStringLength(&ds)) {
	    mail_setflag_full(masterStream, Tcl_DStringValue(&ds),
		    RAT_DELETED_STR, ST_UID);
	    mail_expunge(masterStream);
	}
	if (masterError) goto error;
    }

    /*
     * Build list of uids
     */
    InitUidMap(masterStream);
    if (masterError) goto error;

    /*
     * Apply flag commands (and remove changes file)
     */
    if (NULL != fp) {
	RatFlag flag;
	char *flagStr;
	int value;

	fseek(fp, 0, SEEK_SET);
	while (fgets(buf, sizeof(buf)-1, fp), !feof(fp)) {
	    if (!strncmp(buf, "flag", 4)) {
		sscanf(buf+5, "%ld %d %d", &uid, (int*)&flag, &value);
		sprintf(buf, "%ld", uid);
		if (0 == (msgno = MsgNo(uid))) {
		    continue;
		}
		mail_fetchenvelope(masterStream, msgno);
		elt = mail_elt(masterStream, msgno);
		switch(flag) {
		    case RAT_SEEN:
			flagStr = RAT_SEEN_STR;
			elt->seen = value;
			break;
		    case RAT_FLAGGED:
			flagStr = RAT_FLAGGED_STR;
			elt->flagged = value;
			break;
		    case RAT_DELETED:
			flagStr = RAT_DELETED_STR;
			elt->deleted = value;
			break;
		    case RAT_ANSWERED:
			flagStr = RAT_ANSWERED_STR;
			elt->answered = value;
			break;
		    case RAT_DRAFT:
			flagStr = RAT_DRAFT_STR;
			elt->draft = value;
			break;
		    default:
			flagStr = NULL;
			break;
		}
		if (value) {
		    mail_setflag_full(masterStream, buf, flagStr, ST_UID);
		} else {
		    mail_clearflag_full(masterStream, buf, flagStr, ST_UID);
		}
	    }
	}
	fclose(fp);
	if (masterError) goto error;
	snprintf(buf, sizeof(buf), "%s/changes", dir);
	unlink(buf);
    }

    /*
     * Download new messages
     */
    nmsgs = localStream->nmsgs;
    snprintf(buf, sizeof(buf), "%s/mappings", dir);
    fp = fopen(buf, "a");
    pgm = mail_newsearchpgm();
    pgm->uid = mail_newsearchset();
    pgm->uid->first = lastUid+1;
    pgm->uid->last = folderNextUid;
    pgm->uid->next = NULL;
    searchResultNum = 0;
    mail_search_full(masterStream, NULL, pgm, SE_FREE);
    lastUid = localStream->uid_last;
    for (i = 0; i < searchResultNum; i++) {
	RatLogF(interp, RAT_INFO, "downloading", RATLOG_EXPLICIT, i+1,
		searchResultNum);
	envPtr = mail_fetchenvelope(masterStream, searchResultPtr[i]);
	elt = mail_elt(masterStream, searchResultPtr[i]);
	body =mail_fetchtext_full(masterStream,searchResultPtr[i],&len,FT_PEEK);
	header = mail_fetchheader(masterStream, searchResultPtr[i]);
	if (masterError) goto error;
	Tcl_DStringSetLength(&ds, 0);
	Tcl_DStringAppend(&ds, header, -1);
	Tcl_DStringAppend(&ds, body, len);
	INIT(&string, mail_string,Tcl_DStringValue(&ds),Tcl_DStringLength(&ds));
	mail_date(datebuf, elt);
	mail_append_full(localStream, localMailbox, MsgFlags(elt), datebuf,
		&string);
	fprintf(fp, "%ld %ld\n", mail_uid(masterStream, searchResultPtr[i]),
		++lastUid);
	if (infoPtr) {
	    unsigned long *lPtr = (unsigned long*)ckalloc(sizeof(*lPtr));
	    int unused;

	    *lPtr = mail_uid(masterStream, searchResultPtr[i]);
	    entryPtr = Tcl_CreateHashEntry(mapPtr, (char*)lastUid,
		    &unused);
	    Tcl_SetHashValue(entryPtr, lPtr);
	}
    }

    /*
     * Loop over messages and update
     */
    RatLogF(interp, RAT_INFO, "downloading_flags", RATLOG_EXPLICIT);
    for (i = 1; i <= nmsgs; i++) {
	if ((uid = GetMasterUID(localStream, mapPtr, i-1))) {
	    msgno = MsgNo(uid);
	    if (0 == msgno) {
		if (infoPtr) {
		    UpdateFolderFlag(interp, infoPtr, i, RAT_DELETED, 1);
		} else {
		    sprintf(buf, "%d", i);
		    mail_setflag(localStream, buf, RAT_DELETED_STR);
		}
		continue;
	    }
	    /*
	     * Update flags from master
	     */
	    envPtr = mail_fetchenvelope(masterStream,
		    MsgNo(uid));
	    elt = mail_elt(masterStream, MsgNo(uid));
	    if (infoPtr) {
		UpdateFolderFlag(interp,infoPtr,i,RAT_SEEN,elt->seen);
		UpdateFolderFlag(interp,infoPtr,i,RAT_DELETED,elt->deleted);
		UpdateFolderFlag(interp,infoPtr,i,RAT_FLAGGED,elt->flagged);
		UpdateFolderFlag(interp,infoPtr,i,RAT_ANSWERED,
			elt->answered);
		UpdateFolderFlag(interp,infoPtr,i,RAT_DRAFT,elt->draft);
		UpdateFolderFlag(interp,infoPtr,i,RAT_RECENT,elt->recent);
	    } else {
		MESSAGECACHE *lelt;

		lelt = mail_elt(localStream, i);
		sprintf(buf, "%d", i);
		if (elt->seen != lelt->seen) {
		    if (elt->seen) {
			mail_setflag(localStream, buf, RAT_SEEN_STR);
		    } else {
			mail_clearflag(localStream, buf, RAT_SEEN_STR);
		    }
		}
		if (elt->deleted != lelt->deleted) {
		    if (elt->deleted) {
			mail_setflag(localStream, buf, RAT_DELETED_STR);
		    } else {
			mail_clearflag(localStream, buf, RAT_DELETED_STR);
		    }
		}
		if (elt->flagged != lelt->flagged) {
		    if (elt->flagged) {
			mail_setflag(localStream, buf, RAT_FLAGGED_STR);
		    } else {
			mail_clearflag(localStream, buf, RAT_FLAGGED_STR);
		    }
		}
		if (elt->answered != lelt->answered) {
		    if (elt->answered) {
			mail_setflag(localStream, buf, RAT_ANSWERED_STR);
		    } else {
			mail_clearflag(localStream, buf, RAT_ANSWERED_STR);
		    }
		}
		if (elt->draft != lelt->draft) {
		    if (elt->draft) {
			mail_setflag(localStream, buf, RAT_DRAFT_STR);
		    } else {
			mail_clearflag(localStream, buf, RAT_DRAFT_STR);
		    }
		}
	    }
	} else {
	    /*
	     * Append the message to the master stream
	     */
	    Tcl_DStringSetLength(&ds, 0);
	    header = mail_fetchheader(localStream, i);
	    Tcl_DStringAppend(&ds, header, -1);
	    body = mail_fetchtext_full(localStream, i, &len, FT_PEEK);
	    Tcl_DStringAppend(&ds, body, len);
	    INIT(&string, mail_string, Tcl_DStringValue(&ds),
		    Tcl_DStringLength(&ds));
	    elt = mail_elt(localStream, i);
	    mail_date(datebuf, elt);
	    mail_append_full(masterStream, specMutf7, MsgFlags(elt), datebuf,
		    &string);

	    envPtr = mail_fetchenvelope(localStream, i);
	    pgm = mail_newsearchpgm();
	    if (envPtr->subject) {
		pgm->subject = mail_newstringlist();
		pgm->subject->text.data = envPtr->subject;
		pgm->subject->text.size = strlen(envPtr->subject);
	    }
	    if (envPtr->in_reply_to) {
		pgm->in_reply_to = mail_newstringlist();
		pgm->in_reply_to->text.data = envPtr->in_reply_to;
		pgm->in_reply_to->text.size = strlen(envPtr->in_reply_to);
	    }
	    if (envPtr->message_id) {
		pgm->message_id = mail_newstringlist();
		pgm->message_id->text.data = envPtr->message_id;
		pgm->message_id->text.size = strlen(envPtr->message_id);
	    }
	    if (envPtr->date) {
		pgm->header = mail_newsearchheader("date", envPtr->date);
	    }
	    searchResultNum = 0;
	    mail_search_full(masterStream, NULL, pgm, SE_FREE|SE_UID);
	    if (searchResultNum == 1) {
		fprintf(fp, "%ld %ld\n", searchResultPtr[0],
			mail_uid(localStream, i));
		if (infoPtr) {
		    unsigned long *lPtr =(unsigned long*)ckalloc(sizeof(*lPtr));
		    int unused;
  
		    *lPtr = searchResultPtr[0];
		    entryPtr = Tcl_CreateHashEntry(mapPtr,
			    (char*)mail_uid(localStream, i), &unused);
		    Tcl_SetHashValue(entryPtr, lPtr);
		}
		if (folderNextUid == searchResultPtr[0]) {
		    folderNextUid = searchResultPtr[0]+1;
		}
	    }
	}
    }
    fclose(fp);

    /*
     * Update state file
     */
    snprintf(buf, sizeof(buf), "%s/state", dir);
    fp = fopen(buf, "w");
    fprintf(fp, "%ld\n%ld\n", masterStream->uid_validity, folderNextUid-1);
    fclose(fp);

    /*
     * Cleanup
     */
    if (!infoPtr) {
	mail_close(localStream);
	for (entryPtr = Tcl_FirstHashEntry(mapPtr, &search); entryPtr;
		entryPtr = Tcl_NextHashEntry(&search)) {
	    ckfree(Tcl_GetHashValue(entryPtr));
	}
	Tcl_DeleteHashTable(mapPtr);
	ckfree(mapPtr);
    }
    Std_StreamClose(interp, masterStream);
    RatLog(interp, RAT_INFO, "", RATLOG_EXPLICIT);
    ckfree(data);
    Tcl_DStringFree(&ds);
    return;

error:
    Std_StreamClose(interp, masterStream);
    return;
}


/*
 *----------------------------------------------------------------------
 *
 * GetMasterUID
 *
 *	Returns the UID in the master folder of the specified message
 *
 * Results:
 *	The UID is returned or zero if the message is not present
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static unsigned long
GetMasterUID(MAILSTREAM *s, Tcl_HashTable *mapPtr, int index)
{
    Tcl_HashEntry *entryPtr;

    if ((entryPtr = Tcl_FindHashEntry(mapPtr, (char*)mail_uid(s, index+1)))) {
	return *((unsigned long*)Tcl_GetHashValue(entryPtr));
    } else {
	return 0;
    }
}


/*
 *----------------------------------------------------------------------
 *
 * UpdateFolderFlag --
 *
 *	Synchronizes a flag with master
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static void
UpdateFolderFlag(Tcl_Interp *interp, RatFolderInfoPtr infoPtr,
	int index, RatFlag flag, int value)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    int local;

    local = (*disPtr->getFlagProc)(infoPtr, interp, index-1, flag);
    if (value == local) {
	return;
    }
    (*disPtr->setFlagProc)(infoPtr, interp, index-1, flag, value);
}


/*
 *----------------------------------------------------------------------
 *
 * ReadMappings --
 *
 *	Reads the mappings-file into the given hash-table
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static void
ReadMappings(MAILSTREAM *s, const char *dir, Tcl_HashTable *mapPtr)
{
    Tcl_HashEntry *entryPtr;
    char buf[1024];
    int unused;
    unsigned long *lPtr, uid;
    FILE *fp;

    snprintf(buf, sizeof(buf), "%s/mappings", dir);
    if (NULL != (fp = fopen(buf, "r"))) {
	buf[sizeof(buf)-1] = '\0';
	while(fgets(buf, sizeof(buf)-1, fp), !feof(fp)) {
	    if (strchr(buf, '<')) {
		ReadOldMappings(s, mapPtr, buf, sizeof(buf)-1, fp);
		break;
	    }
	    buf[strlen(buf)-1] = '\0';
	    uid = atol(strchr(buf, ' '));
	    entryPtr = Tcl_CreateHashEntry(mapPtr, (char*)uid, &unused);
	    lPtr = (unsigned long*)ckalloc(sizeof(unsigned long));
	    *lPtr = atol(buf);
	    Tcl_SetHashValue(entryPtr, lPtr);
	}
	fclose(fp);
    }
}


/*
 *----------------------------------------------------------------------
 *
 * ReadOldMappings --
 *
 *	Reads the old style mappings-file into the given hash-table
 *
 * Results:
 *	None
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
static void
ReadOldMappings(MAILSTREAM *s, Tcl_HashTable *mapPtr, char *buf, int buflen,
	FILE *fp)
{
    Tcl_HashTable tmap;
    Tcl_HashEntry *entryPtr;
    Tcl_HashSearch search;
    unsigned long *lPtr, l, uid;
    ENVELOPE *envPtr;
    int unused;

    /*
     * Read file into local hash-table tmap
     */
    Tcl_InitHashTable(&tmap, TCL_STRING_KEYS);
    do {
	buf[strlen(buf)-1] = '\0';
	entryPtr = Tcl_CreateHashEntry(&tmap, strchr(buf, '<'), &unused);
	lPtr = (unsigned long*)ckalloc(sizeof(unsigned long));
	*lPtr = atol(buf);
	Tcl_SetHashValue(entryPtr, lPtr);
    } while (fgets(buf, buflen, fp), !feof(fp));

    /*
     * Loop through folder and add the new mappings to the real map
     */
    for (l=1; l <= s->nmsgs; l++) {
	envPtr = mail_fetch_structure(s, l, NIL, 0);
	entryPtr = Tcl_FindHashEntry(&tmap, envPtr->message_id);
	if (entryPtr) {
	    uid = *(unsigned long*)Tcl_GetHashValue(entryPtr);
	    entryPtr = Tcl_CreateHashEntry(mapPtr, (char*)mail_uid(s, l),
		    &unused);
	    lPtr = (unsigned long*)ckalloc(sizeof(unsigned long));
	    *lPtr = uid;
	    Tcl_SetHashValue(entryPtr, lPtr);
	}
    }

    /*
     * Free the temporary hashtable from memory
     */
    for (entryPtr = Tcl_FirstHashEntry(&tmap, &search); entryPtr;
	    entryPtr = Tcl_NextHashEntry(&search)) {
	ckfree(Tcl_GetHashValue(entryPtr));
    }
    Tcl_DeleteHashTable(&tmap);
}


/*
 *----------------------------------------------------------------------
 *
 * RatDisFolderDir --
 *
 *	Calculates the directory name of a disconnected folder and
 *	makes sure the directory exists.
 *
 * Results:
 *	A pointer to a static area containign the name
 *
 * Side effects:
 *	None
 *
 *
 *----------------------------------------------------------------------
 */
char*
RatDisFolderDir(Tcl_Interp *interp, const char *spec, const char *user,
	const char *prot)
{
    static Tcl_DString ds;
    static int initialized = 0;
    char *value, *dir, *host, *folderName;
    Tcl_DString tmpDs;
    int hostlen;

    if (!initialized) {
	Tcl_DStringInit(&ds);
    } else {
	Tcl_DStringSetLength(&ds, 0);
    }

    value = Tcl_GetVar2(interp, "option", "disconnected_dir", TCL_GLOBAL_ONLY);
    dir = Tcl_TranslateFileName(interp, value, &tmpDs);
    host = strchr(spec, '{')+1;
    folderName = strchr(spec, '}');
    hostlen = folderName-host;
    if (!folderName || folderName[1] == '\0') {
	folderName = "INBOX";
    } else {
	folderName++;
    }
    Tcl_DStringInit(&ds);
    Tcl_DStringAppend(&ds, dir, -1);
    Tcl_DStringAppend(&ds, "/", 1);
    Tcl_DStringAppend(&ds, host, hostlen);
    Tcl_DStringAppend(&ds, "/", 1);
    Tcl_DStringAppend(&ds, folderName, -1);
    Tcl_DStringAppend(&ds, "+", 1);
    Tcl_DStringAppend(&ds, user, -1);
    Tcl_DStringAppend(&ds, "+", 1);
    Tcl_DStringAppend(&ds, prot, -1);
    Tcl_DStringFree(&tmpDs);
    if (CreateDir(Tcl_DStringValue(&ds))) {
	return NULL;
    }
    return Tcl_DStringValue(&ds);
}


/*
 *----------------------------------------------------------------------
 *
 * Dis_SyncProc --
 *
 *	Synchronizes the specified folder
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The folder may be modified
 *
 *
 *----------------------------------------------------------------------
 */
static int
Dis_SyncProc(RatFolderInfoPtr infoPtr, Tcl_Interp *interp)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    struct stat sbuf;
    char buf[1024];

    snprintf(buf, sizeof(buf), "%s/master", disPtr->dir);
    stat(buf, &sbuf);
    Dis_SyncFolder(interp, disPtr->dir, sbuf.st_size, 1);
    return TCL_OK;
}


/*
 *----------------------------------------------------------------------
 *
 * InitUidMap --
 *
 *	initializes the uid map
 *
 * Results:
 *	None
 *
 * Side effects:
 *	The uidMap structure is initialized
 *
 *----------------------------------------------------------------------
 */
static void
InitUidMap(MAILSTREAM *s)
{
    unsigned long i, j, uid;

    if (uidMap.alloc < s->nmsgs) {
	uidMap.alloc = s->nmsgs;
	uidMap.map = (unsigned long*)ckrealloc(uidMap.map,
		uidMap.alloc*sizeof(unsigned long));
    }
    uidMap.size = s->nmsgs;
    for (i=0; i<s->nmsgs; i++) {
	uid = mail_uid(s, i+1);
	if (i == 0) {
	    uidMap.map[i] = uid;
	} else if (uidMap.map[i-1] < uid) {
	    uidMap.map[i] = uid;
	} else {
	    for (j=0; uidMap.map[j] < uid; j++);
	    memmove(&uidMap.map[j+1], &uidMap.map[j],
		    (i-j)*sizeof(unsigned long));
	    uidMap.map[j] = uid;
	}
    }
}


/*
 *----------------------------------------------------------------------
 *
 * MsgNo --
 *
 *	Lookup a message uid in the map and return teh msgno
 *
 * Results:
 *	The msgno for the given uid or 0 if there is no such message
 *
 * Side effects:
 *	None
 *
 *----------------------------------------------------------------------
 */
static unsigned long
MsgNo(unsigned long uid)
{
    unsigned long i, s, e, m;

    s = 0;
    e = uidMap.size;
    while (e-s > 10) {
	m = (e-s)/2+s;
	if (uid == uidMap.map[m]) {
	    return m+1;
	} else if (uid > uidMap.map[m]) {
	    s = m;
	} else {
	    e = m;
	}
    }
    for (i=s; i < e; i++) {
	if (uidMap.map[i] == uid) {
	    return i+1;
	}
    }
    return 0;
}

/*
 *----------------------------------------------------------------------
 *
 * CheckDeletion --
 *
 *      Checks which messages are going to be deleted before an expunge
 *
 * Results:
 *	None
 *
 * Side effects:
 *	Appends things to the changes file
 *
 *----------------------------------------------------------------------
 */
static void
CheckDeletion(RatFolderInfoPtr infoPtr, Tcl_Interp *interp)
{
    DisFolderInfo *disPtr = (DisFolderInfo*)infoPtr->private2;
    Tcl_HashEntry *entryPtr;
    FILE *fp = NULL;
    char buf[1024];
    unsigned long uid;
    int i;

    for (i = 0; i < infoPtr->number; i++) {
	if (0 != (*disPtr->getFlagProc)(infoPtr, interp, i, RAT_DELETED)) {
	    if (NULL == fp) {
		snprintf(buf, sizeof(buf), "%s/changes", disPtr->dir);
		fp = fopen(buf, "a");
	    }
	    uid = GetMasterUID(((StdFolderInfo*)infoPtr->private)->stream,
		    &disPtr->map, i);
	    if (uid && NULL != fp) {
		fprintf(fp, "delete %ld\n", uid);
	    }
	    entryPtr = Tcl_FindHashEntry(&disPtr->map, (char*)
		    mail_uid(((StdFolderInfo*)infoPtr->private)->stream,
			     i+1));
	    if (entryPtr) {
		disPtr->mapChanged = 1;
		ckfree(Tcl_GetHashValue(entryPtr));
		Tcl_DeleteHashEntry(entryPtr);
	    }
	}
    }
    if (NULL != fp) {
	fclose(fp);
    }
}
