package de.lmu.ifi.dbs.elki.math.statistics.distribution;

/*
 This file is part of ELKI:
 Environment for Developing KDD-Applications Supported by Index-Structures

 Copyright (C) 2015
 Ludwig-Maximilians-Universität München
 Lehr- und Forschungseinheit für Datenbanksysteme
 ELKI Development Team

 This program is free software: you can redistribute it and/or modify
 it under the terms of the GNU Affero General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU Affero General Public License for more details.

 You should have received a copy of the GNU Affero General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */
import org.junit.Test;

import de.lmu.ifi.dbs.elki.JUnit4Test;

/**
 * Unit test for the Weibull distribution in ELKI.
 * 
 * The reference values were computed using GNU R and SciPy.
 * 
 * @author Erich Schubert
 * @since 0.6.0
 */
public class WeibullDistributionTest extends AbstractDistributionTest implements JUnit4Test {
  public static final double[] P_CDFPDF = { //
  1e-10, 1e-05, 0.1, 0.1234567, 0.2, 0.271828182846, 0.3, 0.314159265359, 0.4, 0.5, 0.6, 0.7, 0.8, 0.9, 1.0, 1.1, 1.2, 1.3, 1.4, 1.5, 1.6, 1.7, 1.8, 1.9, 2.0, 2.71828182846, 3.14159265359 //
  };

  public static final double[] SCIPY_WEIBULL_CDF_1_1 = { //
  9.99999999950000070330663866405545691512513073462287e-11, // 0.000000
  9.99995000016666629922851977640974041605659294873476e-06, // 0.000010
  9.51625819640404269073030718573136255145072937011719e-02, // 0.100000
  1.16140088450309583723019102308171568438410758972168e-01, // 0.123457
  1.81269246922018151257915974383649881929159164428711e-01, // 0.200000
  2.38014828033141334628908225568011403083801269531250e-01, // 0.271828
  2.59181779318282123902861258102348074316978454589844e-01, // 0.300000
  2.69597308951354364126018481329083442687988281250000e-01, // 0.314159
  3.29679953964360727969307163220946677029132843017578e-01, // 0.400000
  3.93469340287366575736882623459678143262863159179688e-01, // 0.500000
  4.51188363905973555123551932410919107496738433837891e-01, // 0.600000
  5.03414696208590473069932613725541159510612487792969e-01, // 0.700000
  5.50671035882778436842954761232249438762664794921875e-01, // 0.800000
  5.93430340259400890268182138242991641163825988769531e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  6.67128916301920504849931603530421853065490722656250e-01, // 1.100000
  6.98805788087797918883836700842948630452156066894531e-01, // 1.200000
  7.27468206965987462808698182925581932067871093750000e-01, // 1.300000
  7.53403036058393538176858328370144590735435485839844e-01, // 1.400000
  7.76869839851570209710018843907164409756660461425781e-01, // 1.500000
  7.98103482005344644356625849468400701880455017089844e-01, // 1.600000
  8.17316475947265308477085454796906560659408569335938e-01, // 1.700000
  8.34701111778413440411839019361650571227073669433594e-01, // 1.800000
  8.50431380777364909206994525447953492403030395507812e-01, // 1.900000
  8.64664716763387297682186272140825167298316955566406e-01, // 2.000000
  9.34011964154687457373427150741918012499809265136719e-01, // 2.718282
  9.56786081736227700389463279861956834793090820312500e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_1_1 = { //
  9.99999999899999991725962900090962648391723632812500e-01, // 0.000000
  9.99990000049999827602675850357627496123313903808594e-01, // 0.000010
  9.04837418035959517581545696884859353303909301757812e-01, // 0.100000
  8.83859911549690457910344321135198697447776794433594e-01, // 0.123457
  8.18730753077981820986508409987436607480049133300781e-01, // 0.200000
  7.61985171966858665371091774431988596916198730468750e-01, // 0.271828
  7.40818220681717876097138741897651925683021545410156e-01, // 0.300000
  7.30402691048645635873981518670916557312011718750000e-01, // 0.314159
  6.70320046035639327541844068036880344152450561523438e-01, // 0.400000
  6.06530659712633424263117376540321856737136840820312e-01, // 0.500000
  5.48811636094026389365296836331253871321678161621094e-01, // 0.600000
  4.96585303791409526930067386274458840489387512207031e-01, // 0.700000
  4.49328964117221563157045238767750561237335205078125e-01, // 0.800000
  4.06569659740599109731817861757008358836174011230469e-01, // 0.900000
  3.67879441171442334024277442949824035167694091796875e-01, // 1.000000
  3.32871083698079550661219627727405168116092681884766e-01, // 1.100000
  3.01194211912202136627314530414878390729427337646484e-01, // 1.200000
  2.72531793034012592702453048332245089113712310791016e-01, // 1.300000
  2.46596963941606489578717287258768919855356216430664e-01, // 1.400000
  2.23130160148429818045556771721749100834131240844727e-01, // 1.500000
  2.01896517994655383398949766160512808710336685180664e-01, // 1.600000
  1.82683524052734663767338929574179928749799728393555e-01, // 1.700000
  1.65298888221586531832585365009435918182134628295898e-01, // 1.800000
  1.49568619222635063037429858923132997006177902221680e-01, // 1.900000
  1.35335283236612702317813727859174832701683044433594e-01, // 2.000000
  6.59880358453125426265728492580819875001907348632812e-02, // 2.718282
  4.32139182637722579771732966946728993207216262817383e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_1_1 = { //
  9.99999999950000070330663866405545691512513073462287e-11, // 0.000000
  9.99995000016666629922851977640974041605659294873476e-06, // 0.000010
  9.51625819640404269073030718573136255145072937011719e-02, // 0.100000
  1.16140088450309583723019102308171568438410758972168e-01, // 0.123457
  1.81269246922018151257915974383649881929159164428711e-01, // 0.200000
  2.38014828033214137503748020208149682730436325073242e-01, // 0.271828
  2.59181779318282123902861258102348074316978454589844e-01, // 0.300000
  2.69597308951369518670304614715860225260257720947266e-01, // 0.314159
  3.29679953964360727969307163220946677029132843017578e-01, // 0.400000
  3.93469340287366575736882623459678143262863159179688e-01, // 0.500000
  4.51188363905973555123551932410919107496738433837891e-01, // 0.600000
  5.03414696208590473069932613725541159510612487792969e-01, // 0.700000
  5.50671035882778436842954761232249438762664794921875e-01, // 0.800000
  5.93430340259400890268182138242991641163825988769531e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  6.67128916301920504849931603530421853065490722656250e-01, // 1.100000
  6.98805788087797918883836700842948630452156066894531e-01, // 1.200000
  7.27468206965987462808698182925581932067871093750000e-01, // 1.300000
  7.53403036058393538176858328370144590735435485839844e-01, // 1.400000
  7.76869839851570209710018843907164409756660461425781e-01, // 1.500000
  7.98103482005344644356625849468400701880455017089844e-01, // 1.600000
  8.17316475947265308477085454796906560659408569335938e-01, // 1.700000
  8.34701111778413440411839019361650571227073669433594e-01, // 1.800000
  8.50431380777364909206994525447953492403030395507812e-01, // 1.900000
  8.64664716763387297682186272140825167298316955566406e-01, // 2.000000
  9.34011964154750407018923397117760032415390014648438e-01, // 2.718282
  9.56786081736236693195962743629934266209602355957031e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_1_1 = { //
  9.99999999899999991725962900090962648391723632812500e-01, // 0.000000
  9.99990000049999827602675850357627496123313903808594e-01, // 0.000010
  9.04837418035959517581545696884859353303909301757812e-01, // 0.100000
  8.83859911549690457910344321135198697447776794433594e-01, // 0.123457
  8.18730753077981820986508409987436607480049133300781e-01, // 0.200000
  7.61985171966785834740676364162936806678771972656250e-01, // 0.271828
  7.40818220681717876097138741897651925683021545410156e-01, // 0.300000
  7.30402691048630536840846616541966795921325683593750e-01, // 0.314159
  6.70320046035639327541844068036880344152450561523438e-01, // 0.400000
  6.06530659712633424263117376540321856737136840820312e-01, // 0.500000
  5.48811636094026389365296836331253871321678161621094e-01, // 0.600000
  4.96585303791409526930067386274458840489387512207031e-01, // 0.700000
  4.49328964117221563157045238767750561237335205078125e-01, // 0.800000
  4.06569659740599109731817861757008358836174011230469e-01, // 0.900000
  3.67879441171442334024277442949824035167694091796875e-01, // 1.000000
  3.32871083698079550661219627727405168116092681884766e-01, // 1.100000
  3.01194211912202136627314530414878390729427337646484e-01, // 1.200000
  2.72531793034012592702453048332245089113712310791016e-01, // 1.300000
  2.46596963941606489578717287258768919855356216430664e-01, // 1.400000
  2.23130160148429818045556771721749100834131240844727e-01, // 1.500000
  2.01896517994655383398949766160512808710336685180664e-01, // 1.600000
  1.82683524052734663767338929574179928749799728393555e-01, // 1.700000
  1.65298888221586531832585365009435918182134628295898e-01, // 1.800000
  1.49568619222635063037429858923132997006177902221680e-01, // 1.900000
  1.35335283236612702317813727859174832701683044433594e-01, // 2.000000
  6.59880358452495374699253716244129464030265808105469e-02, // 2.718282
  4.32139182637633137429311602772941114380955696105957e-02, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_2_1 = { //
  1.00000000000000009561654835946237267172778046723484e-20, // 0.000000
  9.99999999950000199577634577816119890170121209393983e-11, // 0.000010
  9.95016625083194883316650702909100800752639770507812e-03, // 0.100000
  1.51259921218065002124220086443528998643159866333008e-02, // 0.123457
  3.92105608476767952819130869102082215249538421630859e-02, // 0.200000
  7.12266676857742225825731452459876891225576400756836e-02, // 0.271828
  8.60688147287718141598134025116451084613800048828125e-02, // 0.300000
  9.39819442110770147946752217649191152304410934448242e-02, // 0.314159
  1.47856211033788681463718717168376315385103225708008e-01, // 0.400000
  2.21199216928595121522960198490181937813758850097656e-01, // 0.500000
  3.02323673928968916779780329306959174573421478271484e-01, // 0.600000
  3.87373605815583887945763308380264788866043090820312e-01, // 0.700000
  4.72707575956951508988623800178174860775470733642578e-01, // 0.800000
  5.55141933777058893539901873737107962369918823242188e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  7.01802720570112725795297592412680387496948242187500e-01, // 1.100000
  7.63072241317878208199942946521332487463951110839844e-01, // 1.200000
  8.15480476007010790517881559935631230473518371582031e-01, // 1.300000
  8.59141579078954942261248106660787016153335571289062e-01, // 1.400000
  8.94600775438135653594429186341585591435432434082031e-01, // 1.500000
  9.22695259556700286829311608016723766922950744628906e-01, // 1.600000
  9.44423787388516955410011632920941337943077087402344e-01, // 1.700000
  9.60836104901012899581758119893493130803108215332031e-01, // 1.800000
  9.72948153133649573653940478834556415677070617675781e-01, // 1.900000
  9.81684361111265779697987454710528254508972167968750e-01, // 2.000000
  9.99382021010668863958414931403240188956260681152344e-01, // 2.718282
  9.99948276813796166351266947458498179912567138671875e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_2_1 = { //
  2.00000000000000007286439463099548315833109413119928e-10, // 0.000000
  1.99999999980000009935313803888234929218015167862177e-05, // 0.000010
  1.98009966749833621335596944845747202634811401367188e-01, // 0.100000
  2.43178589856831550708449185549397952854633331298828e-01, // 0.123457
  3.84315775660929281887234765235916711390018463134766e-01, // 0.200000
  5.04933534397422723039028369385050609707832336425781e-01, // 0.271828
  5.48358711162736933708572450996143743395805358886719e-01, // 0.300000
  5.69267933617237598120652819488896057009696960449219e-01, // 0.314159
  6.81715031172969165851327488780952990055084228515625e-01, // 0.400000
  7.78800783071404878477039801509818062186241149902344e-01, // 0.500000
  8.37211591285237255455342619825387373566627502441406e-01, // 0.600000
  8.57676951858182490262549890758236870169639587402344e-01, // 0.700000
  8.43667878468877696640504382230574265122413635253906e-01, // 0.800000
  8.00744519201293969423716134770074859261512756347656e-01, // 0.900000
  7.35758882342884668048554885899648070335388183593750e-01, // 1.000000
  6.56034014745752180886029236717149615287780761718750e-01, // 1.100000
  5.68626620837092255911215943342540413141250610351562e-01, // 1.200000
  4.79750762381772055675810406683012843132019042968750e-01, // 1.300000
  3.94403578578926050646202838834142312407493591308594e-01, // 1.400000
  3.16197673685592983705561209717416204512119293212891e-01, // 1.500000
  2.47375169418559082146202854346483945846557617187500e-01, // 1.600000
  1.88959122879042484832723403087584301829338073730469e-01, // 1.700000
  1.40990022356353439381138059616205282509326934814453e-01, // 1.800000
  1.02797018092131570154990072296641301363706588745117e-01, // 1.900000
  7.32625555549367146745964873844059184193611145019531e-02, // 2.000000
  3.35968211413639750162474229000508785247802734375000e-03, // 2.718282
  3.24986363596307557474740512759581179125234484672546e-04, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_2_1 = { //
  1.00000000000000009561654835946237267172778046723484e-20, // 0.000000
  9.99999999950000199577634577816119890170121209393983e-11, // 0.000010
  9.95016625083194883316650702909100800752639770507812e-03, // 0.100000
  1.51259921218065002124220086443528998643159866333008e-02, // 0.123457
  3.92105608476767952819130869102082215249538421630859e-02, // 0.200000
  7.12266676858224756507809161121258512139320373535156e-02, // 0.271828
  8.60688147287718141598134025116451084613800048828125e-02, // 0.300000
  9.39819442110888109143118640531611163169145584106445e-02, // 0.314159
  1.47856211033788681463718717168376315385103225708008e-01, // 0.400000
  2.21199216928595121522960198490181937813758850097656e-01, // 0.500000
  3.02323673928968916779780329306959174573421478271484e-01, // 0.600000
  3.87373605815583887945763308380264788866043090820312e-01, // 0.700000
  4.72707575956951508988623800178174860775470733642578e-01, // 0.800000
  5.55141933777058893539901873737107962369918823242188e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  7.01802720570112725795297592412680387496948242187500e-01, // 1.100000
  7.63072241317878208199942946521332487463951110839844e-01, // 1.200000
  8.15480476007010790517881559935631230473518371582031e-01, // 1.300000
  8.59141579078954942261248106660787016153335571289062e-01, // 1.400000
  8.94600775438135653594429186341585591435432434082031e-01, // 1.500000
  9.22695259556700286829311608016723766922950744628906e-01, // 1.600000
  9.44423787388516955410011632920941337943077087402344e-01, // 1.700000
  9.60836104901012899581758119893493130803108215332031e-01, // 1.800000
  9.72948153133649573653940478834556415677070617675781e-01, // 1.900000
  9.81684361111265779697987454710528254508972167968750e-01, // 2.000000
  9.99382021010672083605186344357207417488098144531250e-01, // 2.718282
  9.99948276813796277373569409974152222275733947753906e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_2_1 = { //
  2.00000000000000007286439463099548315833109413119928e-10, // 0.000000
  1.99999999980000009935313803888234929218015167862177e-05, // 0.000010
  1.98009966749833621335596944845747202634811401367188e-01, // 0.100000
  2.43178589856831550708449185549397952854633331298828e-01, // 0.123457
  3.84315775660929281887234765235916711390018463134766e-01, // 0.200000
  5.04933534397574046437284778221510350704193115234375e-01, // 0.271828
  5.48358711162736933708572450996143743395805358886719e-01, // 0.300000
  5.69267933617267685164620161231141537427902221679688e-01, // 0.314159
  6.81715031172969165851327488780952990055084228515625e-01, // 0.400000
  7.78800783071404878477039801509818062186241149902344e-01, // 0.500000
  8.37211591285237255455342619825387373566627502441406e-01, // 0.600000
  8.57676951858182490262549890758236870169639587402344e-01, // 0.700000
  8.43667878468877696640504382230574265122413635253906e-01, // 0.800000
  8.00744519201293969423716134770074859261512756347656e-01, // 0.900000
  7.35758882342884668048554885899648070335388183593750e-01, // 1.000000
  6.56034014745752180886029236717149615287780761718750e-01, // 1.100000
  5.68626620837092255911215943342540413141250610351562e-01, // 1.200000
  4.79750762381772055675810406683012843132019042968750e-01, // 1.300000
  3.94403578578926050646202838834142312407493591308594e-01, // 1.400000
  3.16197673685592983705561209717416204512119293212891e-01, // 1.500000
  2.47375169418559082146202854346483945846557617187500e-01, // 1.600000
  1.88959122879042484832723403087584301829338073730469e-01, // 1.700000
  1.40990022356353439381138059616205282509326934814453e-01, // 1.800000
  1.02797018092131570154990072296641301363706588745117e-01, // 1.900000
  7.32625555549367146745964873844059184193611145019531e-02, // 2.000000
  3.35968211412013880584614966551271209027618169784546e-03, // 2.718282
  3.24986363595906402670920876118998421588912606239319e-04, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_4_1 = { //
  1.00000000000000013320864402483403646202733603332908e-40, // 0.000000
  1.00000000000000039654310216996797471172433399612973e-20, // 0.000010
  9.99950001666625217910250311703634906734805554151535e-05, // 0.100000
  2.32278072192682402295413335302498580858809873461723e-04, // 0.123457
  1.59872068239368765249497350566798559157177805900574e-03, // 0.200000
  5.44493730219271481612919316717125184368342161178589e-03, // 0.271828
  8.06728339442883482512947068698849761858582496643066e-03, // 0.300000
  9.69362011877886869948284243037051055580377578735352e-03, // 0.314159
  2.52750983982060611898745605685689952224493026733398e-02, // 0.400000
  6.05869371865242131725004526288103079423308372497559e-02, // 0.500000
  1.21553260650068689630032281456806231290102005004883e-01, // 0.600000
  2.13450797786544993472901410314079839736223220825195e-01, // 0.700000
  3.36084236664526470050873285799752920866012573242188e-01, // 0.800000
  4.81129009534547624760136841359781101346015930175781e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  7.68713944715671448904004137148149311542510986328125e-01, // 1.100000
  8.74267670405572072667155225644819438457489013671875e-01, // 1.200000
  9.42507455473835520365355478134006261825561523437500e-01, // 1.300000
  9.78540760919919594407190288620768114924430847167969e-01, // 1.400000
  9.93670284572514272980470195761881768703460693359375e-01, // 1.500000
  9.98575023561807495475761697889538481831550598144531e-01, // 1.600000
  9.99764099393370408996872811258072033524513244628906e-01, // 1.700000
  9.99972397383803390802370358869666233658790588378906e-01, // 1.800000
  9.99997811074995079927418828447116538882255554199219e-01, // 1.900000
  9.99999887464825265581680469040293246507644653320312e-01, // 2.000000
  1.00000000000000000000000000000000000000000000000000e+00, // 2.718282
  1.00000000000000000000000000000000000000000000000000e+00, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_4_1 = { //
  4.00000000000000033334568243034394140372534410747462e-30, // 0.000000
  4.00000000000000109968250472765612236495730768756629e-15, // 0.000010
  3.99960001999933451033442111111071426421403884887695e-03, // 0.100000
  7.52494092430275145211293619240677799098193645477295e-03, // 0.123457
  3.19488409381634141936956439167261123657226562500000e-02, // 0.200000
  7.99046897358400981525861084264761302620172500610352e-02, // 0.271828
  1.07128733393401662232768956073414301499724388122559e-01, // 0.300000
  1.22822854451452953550294466822379035875201225280762e-01, // 0.314159
  2.49529574810059301404052689576928969472646713256836e-01, // 0.400000
  4.69706531406737903822090629546437412500381469726562e-01, // 0.500000
  7.58977982798340611303444802615558728575706481933594e-01, // 0.600000
  1.07914550543685994199449851294048130512237548828125e+00, // 0.700000
  1.35969948331105006111840793892042711377143859863281e+00, // 0.800000
  1.51302780819725923855401106266072019934654235839844e+00, // 0.900000
  1.47151776468576933609710977179929614067077636718750e+00, // 1.000000
  1.23136695833376563768979394808411598205566406250000e+00, // 1.100000
  8.69061862156685571711989268806064501404762268066406e-01, // 1.200000
  5.05244481295933911546569561323849484324455261230469e-01, // 1.300000
  2.35536608142962594625302585882309358566999435424805e-01, // 1.400000
  8.54511582710575784416207056892744731158018112182617e-02, // 1.500000
  2.33468139633453281023278691463929135352373123168945e-02, // 1.600000
  4.63591872148445566931274086641678877640515565872192e-03, // 1.700000
  6.43913830635281774633638107729893818031996488571167e-04, // 1.800000
  6.00553464339657769527694441968890259886393323540688e-05, // 1.900000
  3.60112559101629172664759369337783567743827006779611e-06, // 2.000000
  1.56051574726592369321459464025011473518080818529307e-22, // 2.718282
  6.15572040972424438967743933644082832917854496729035e-41, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_4_1 = { //
  1.00000000000000013320864402483403646202733603332908e-40, // 0.000000
  1.00000000000000039654310216996797471172433399612973e-20, // 0.000010
  9.99950001666625217910250311703634906734805554151535e-05, // 0.100000
  2.32278072192682402295413335302498580858809873461723e-04, // 0.123457
  1.59872068239368765249497350566798559157177805900574e-03, // 0.200000
  5.44493730220034933414696709519375872332602739334106e-03, // 0.271828
  8.06728339442883482512947068698849761858582496643066e-03, // 0.300000
  9.69362011878141006937514845276382402516901493072510e-03, // 0.314159
  2.52750983982060611898745605685689952224493026733398e-02, // 0.400000
  6.05869371865242131725004526288103079423308372497559e-02, // 0.500000
  1.21553260650068689630032281456806231290102005004883e-01, // 0.600000
  2.13450797786544993472901410314079839736223220825195e-01, // 0.700000
  3.36084236664526470050873285799752920866012573242188e-01, // 0.800000
  4.81129009534547624760136841359781101346015930175781e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  7.68713944715671448904004137148149311542510986328125e-01, // 1.100000
  8.74267670405572072667155225644819438457489013671875e-01, // 1.200000
  9.42507455473835520365355478134006261825561523437500e-01, // 1.300000
  9.78540760919919594407190288620768114924430847167969e-01, // 1.400000
  9.93670284572514272980470195761881768703460693359375e-01, // 1.500000
  9.98575023561807495475761697889538481831550598144531e-01, // 1.600000
  9.99764099393370408996872811258072033524513244628906e-01, // 1.700000
  9.99972397383803390802370358869666233658790588378906e-01, // 1.800000
  9.99997811074995079927418828447116538882255554199219e-01, // 1.900000
  9.99999887464825265581680469040293246507644653320312e-01, // 2.000000
  1.00000000000000000000000000000000000000000000000000e+00, // 2.718282
  1.00000000000000000000000000000000000000000000000000e+00, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_4_1 = { //
  4.00000000000000033334568243034394140372534410747462e-30, // 0.000000
  4.00000000000000109968250472765612236495730768756629e-15, // 0.000010
  3.99960001999933451033442111111071426421403884887695e-03, // 0.100000
  7.52494092430275145211293619240677799098193645477295e-03, // 0.123457
  3.19488409381634141936956439167261123657226562500000e-02, // 0.200000
  7.99046897359237395797038061573402956128120422363281e-02, // 0.271828
  1.07128733393401662232768956073414301499724388122559e-01, // 0.300000
  1.22822854451476920489838562389195431023836135864258e-01, // 0.314159
  2.49529574810059301404052689576928969472646713256836e-01, // 0.400000
  4.69706531406737903822090629546437412500381469726562e-01, // 0.500000
  7.58977982798340611303444802615558728575706481933594e-01, // 0.600000
  1.07914550543685994199449851294048130512237548828125e+00, // 0.700000
  1.35969948331104983907380301388911902904510498046875e+00, // 0.800000
  1.51302780819725901650940613762941211462020874023438e+00, // 0.900000
  1.47151776468576933609710977179929614067077636718750e+00, // 1.000000
  1.23136695833376519360058409802149981260299682617188e+00, // 1.100000
  8.69061862156685904778896656353026628494262695312500e-01, // 1.200000
  5.05244481295934022568872023839503526687622070312500e-01, // 1.300000
  2.35536608142962705647605048397963400930166244506836e-01, // 1.400000
  8.54511582710575784416207056892744731158018112182617e-02, // 1.500000
  2.33468139633453072856461574247077805921435356140137e-02, // 1.600000
  4.63591872148445566931274086641678877640515565872192e-03, // 1.700000
  6.43913830635280690431465622225459810579195618629456e-04, // 1.800000
  6.00553464339657769527694441968890259886393323540688e-05, // 1.900000
  3.60112559101629172664759369337783567743827006779611e-06, // 2.000000
  1.56051574714787161674795427931483364971526738525937e-22, // 2.718282
  6.15572040956738830521263230405220044059347242007796e-41, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_4_10 = { //
  1.00000000000000045083134380686670919466036533826663e-44, // 0.000000
  1.00000000000000065838446881563251711176039766238496e-24, // 0.000010
  9.99999995000000102046765708798184424388466595701175e-09, // 0.100000
  2.32305050223913381984605750103789478799853895907290e-08, // 0.123457
  1.59999987200000687481607052758958431581959303002805e-07, // 0.200000
  5.45981351283569954638893052117909476805834856349975e-07, // 0.271828
  8.09999671950088426490164863341680856478888017591089e-07, // 0.300000
  9.74090435913627788344247224183725109014631016179919e-07, // 0.314159
  2.55999672320279633532034806175925467641718569211662e-06, // 0.400000
  6.24998046879069103125628470429830940702231600880623e-06, // 0.500000
  1.29599160195627932342429117218429723834560718387365e-05, // 0.600000
  2.40097117622568567696633018426766170705377589911222e-05, // 0.700000
  4.09591611506531300059170963567822809636709280312061e-05, // 0.800000
  6.56078477110208019762102593652741688856622204184532e-05, // 0.900000
  9.99950001666625217910250311703634906734805554151535e-05, // 1.000000
  1.46399282579002344489502163149552416143706068396568e-04, // 1.100000
  2.07338502401139678377878228232589208346325904130936e-04, // 1.200000
  2.85569217346686983569764572976623639988247305154800e-04, // 1.300000
  3.84086219995277808499090088645289142732508480548859e-04, // 1.400000
  5.06121877090403020534037370481428297352977097034454e-04, // 1.500000
  6.55145298540010952680068267994784037000499665737152e-04, // 1.600000
  8.34861309211383712837462045541769839474000036716461e-04, // 1.700000
  1.04920919472584086955257998141632924671284854412079e-03, // 1.800000
  1.30236119061361726480241518544289647252298891544342e-03, // 1.900000
  1.59872068239368765249497350566798559157177805900574e-03, // 2.000000
  5.44493730219271481612919316717125184368342161178589e-03, // 2.718282
  9.69362011877886869948284243037051055580377578735352e-03, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_4_10 = { //
  4.00000000000000056598312279676865044379763820833960e-34, // 0.000000
  4.00000000000000182691365172827638047889457880763485e-19, // 0.000010
  3.99999996000000073268388234987158469380119640845805e-07, // 0.100000
  7.52668903431345566337837965642520998699183110147715e-07, // 0.123457
  3.19999948800004119659748033632773456247377907857299e-06, // 0.200000
  8.03421038274362870484652382474521914446086157113314e-06, // 0.271828
  1.07999912520035405629360153456630655455228406935930e-05, // 0.300000
  1.24024985909529048835279324558555913426971528679132e-05, // 0.314159
  2.55999344640838884224441629466895165023743174970150e-05, // 0.400000
  4.99996875009765654775117926789107514196075499057770e-05, // 0.500000
  8.63988802632558908862689395213863008393673226237297e-05, // 0.600000
  1.37196705867546154882993802992530163464834913611412e-04, // 0.700000
  2.04791611563796387964972445239197895716642960906029e-04, // 0.800000
  2.91580868751607432971112432440463635430205613374710e-04, // 0.900000
  3.99960001999933461875463835966115766495931893587112e-04, // 1.000000
  5.32322057021955194998363847247446756227873265743256e-04, // 1.100000
  6.91056687627140129107083943438283313298597931861877e-04, // 1.200000
  8.78549041771795719522886169983166837482713162899017e-04, // 1.300000
  1.09717842696493267894142231710929991095326840877533e-03, // 1.400000
  1.34931673546592788628939452166832779766991734504700e-03, // 1.500000
  1.63732660994287239700217817528482555644586682319641e-03, // 1.600000
  1.96355933055513721952056727104718447662889957427979e-03, // 1.700000
  2.33035240479054304302275468785410339478403329849243e-03, // 1.800000
  2.74002684183743266954635942056484054774045944213867e-03, // 1.900000
  3.19488409381634124589721679399190179537981748580933e-03, // 2.000000
  7.99046897358400912136922045192477526143193244934082e-03, // 2.718282
  1.22822854451452960489188370729607413522899150848389e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_4_10 = { //
  1.00000000000000045083134380686670919466036533826663e-44, // 0.000000
  1.00000000000000065838446881563251711176039766238496e-24, // 0.000010
  9.99999995000000102046765708798184424388466595701175e-09, // 0.100000
  2.32305050223913381984605750103789478799853895907290e-08, // 0.123457
  1.59999987200000687481607052758958431581959303002805e-07, // 0.200000
  5.45981351284337366489105448225105376991450611967593e-07, // 0.271828
  8.09999671950088426490164863341680856478888017591089e-07, // 0.300000
  9.74090435913884227569028463611378043651711777783930e-07, // 0.314159
  2.55999672320279633532034806175925467641718569211662e-06, // 0.400000
  6.24998046879069103125628470429830940702231600880623e-06, // 0.500000
  1.29599160195627932342429117218429723834560718387365e-05, // 0.600000
  2.40097117622568567696633018426766170705377589911222e-05, // 0.700000
  4.09591611506531300059170963567822809636709280312061e-05, // 0.800000
  6.56078477110208019762102593652741688856622204184532e-05, // 0.900000
  9.99950001666625217910250311703634906734805554151535e-05, // 1.000000
  1.46399282579002344489502163149552416143706068396568e-04, // 1.100000
  2.07338502401139678377878228232589208346325904130936e-04, // 1.200000
  2.85569217346686983569764572976623639988247305154800e-04, // 1.300000
  3.84086219995277808499090088645289142732508480548859e-04, // 1.400000
  5.06121877090403020534037370481428297352977097034454e-04, // 1.500000
  6.55145298540010952680068267994784037000499665737152e-04, // 1.600000
  8.34861309211383712837462045541769839474000036716461e-04, // 1.700000
  1.04920919472584086955257998141632924671284854412079e-03, // 1.800000
  1.30236119061361726480241518544289647252298891544342e-03, // 1.900000
  1.59872068239368765249497350566798559157177805900574e-03, // 2.000000
  5.44493730220034412997653916477247548755258321762085e-03, // 2.718282
  9.69362011878141006937514845276382402516901493072510e-03, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_4_10 = { //
  4.00000000000000056598312279676865044379763820833960e-34, // 0.000000
  4.00000000000000182691365172827638047889457880763485e-19, // 0.000010
  3.99999996000000073268388234987158469380119640845805e-07, // 0.100000
  7.52668903431345566337837965642520998699183110147715e-07, // 0.123457
  3.19999948800004119659748033632773456247377907857299e-06, // 0.200000
  8.03421038275209734025317231953522423282265663146973e-06, // 0.271828
  1.07999912520035405629360153456630655455228406935930e-05, // 0.300000
  1.24024985909553511146796028752348206580791156738997e-05, // 0.314159
  2.55999344640838884224441629466895165023743174970150e-05, // 0.400000
  4.99996875009765654775117926789107514196075499057770e-05, // 0.500000
  8.63988802632558908862689395213863008393673226237297e-05, // 0.600000
  1.37196705867546154882993802992530163464834913611412e-04, // 0.700000
  2.04791611563796387964972445239197895716642960906029e-04, // 0.800000
  2.91580868751607432971112432440463635430205613374710e-04, // 0.900000
  3.99960001999933461875463835966115766495931893587112e-04, // 1.000000
  5.32322057021955194998363847247446756227873265743256e-04, // 1.100000
  6.91056687627140129107083943438283313298597931861877e-04, // 1.200000
  8.78549041771795719522886169983166837482713162899017e-04, // 1.300000
  1.09717842696493267894142231710929991095326840877533e-03, // 1.400000
  1.34931673546592788628939452166832779766991734504700e-03, // 1.500000
  1.63732660994287239700217817528482555644586682319641e-03, // 1.600000
  1.96355933055513721952056727104718447662889957427979e-03, // 1.700000
  2.33035240479054304302275468785410339478403329849243e-03, // 1.800000
  2.74002684183743266954635942056484054774045944213867e-03, // 1.900000
  3.19488409381634124589721679399190179537981748580933e-03, // 2.000000
  7.99046897359236875379995268531274632550776004791260e-03, // 2.718282
  1.22822854451476917020391610435581242199987173080444e-02, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_01_10 = { //
  7.63599351831150269243053685386257711797952651977539e-02, // 0.000000
  2.22124383190155039180524454422993585467338562011719e-01, // 0.000010
  4.67917828829414239599771008215611800551414489746094e-01, // 0.100000
  4.75019399203212733251433519399142824113368988037109e-01, // 0.123457
  4.91476239432907469062428162942524068057537078857422e-01, // 0.200000
  5.02079925428299933543030419968999922275543212890625e-01, // 0.271828
  5.05508894037498435203303870366653427481651306152344e-01, // 0.300000
  5.07115955277961494296334876707987859845161437988281e-01, // 0.314159
  5.15568704992931170494330217479728162288665771484375e-01, // 0.400000
  5.23427039033618868302255577873438596725463867187500e-01, // 0.500000
  5.29881674984790551619084908452350646257400512695312e-01, // 0.600000
  5.35361615308103777621795416052918881177902221679688e-01, // 0.700000
  5.40124560432196254566861171042546629905700683593750e-01, // 0.800000
  5.44337595924840766947738757153274491429328918457031e-01, // 0.900000
  5.48115306569588400797954363952158018946647644042969e-01, // 1.000000
  5.51539729658985344862287547584855929017066955566406e-01, // 1.100000
  5.54671638850380399610173753899289295077323913574219e-01, // 1.200000
  5.57557321771014247069331304373918101191520690917969e-01, // 1.300000
  5.60232853193344815245779955148464068770408630371094e-01, // 1.400000
  5.62726898877477954563630646589444950222969055175781e-01, // 1.500000
  5.65062617878835049545216406841063871979713439941406e-01, // 1.600000
  5.67258990450220990098273432522546499967575073242188e-01, // 1.700000
  5.69331767989112891292791118758032098412513732910156e-01, // 1.800000
  5.71294167272651032263297565805260092020034790039062e-01, // 1.900000
  5.73157387426872722002713089750614017248153686523438e-01, // 2.000000
  5.84332020844404609327682464936515316367149353027344e-01, // 2.718282
  5.89617748244351203013025042309891432523727416992188e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_01_10 = { //
  7.33673382206617593765258789062500000000000000000000e+07, // 0.000000
  1.95393520726679525978397578001022338867187500000000e+03, // 0.000010
  3.35721153767048474669820734561653807759284973144531e-01, // 0.100000
  2.74018609307649319006827681732829660177230834960938e-01, // 0.123457
  1.71942902599836300314706249992013908922672271728516e-01, // 0.200000
  1.27730497083068800234784134772780817002058029174805e-01, // 0.271828
  1.16077849986715483932009362888493342325091361999512e-01, // 0.300000
  1.10996650714204211918278986104269279167056083679199e-01, // 0.314159
  8.77764877675434873616211461921920999884605407714844e-02, // 0.400000
  7.06409277770171650789876593989902175962924957275391e-02, // 0.500000
  5.91386021268343400891787098316854098811745643615723e-02, // 0.600000
  5.08776272124714656897026543447282165288925170898438e-02, // 0.700000
  4.46538827467651763813982768169807968661189079284668e-02, // 0.800000
  3.97946728437166968861404825474892277270555496215820e-02, // 0.900000
  3.58944770831502030228321586946549359709024429321289e-02, // 1.000000
  3.26941881616923946118902222224278375506401062011719e-02, // 1.100000
  3.00204524249116540268733643870291416533291339874268e-02, // 1.200000
  2.77528760154124205794534674396345508284866809844971e-02, // 1.300000
  2.58052190382604060281401814336277311667799949645996e-02, // 1.400000
  2.41140762238259073746426963680278277024626731872559e-02, // 1.500000
  2.26317820314968741468586443943422636948525905609131e-02, // 1.600000
  2.13218078236213533227516592205574852414429187774658e-02, // 1.700000
  2.01556859486728690011858589059556834399700164794922e-02, // 1.800000
  1.91109005040240170769649807880341541022062301635742e-02, // 1.900000
  1.81694078358287762486877880974134313873946666717529e-02, // 2.000000
  1.34239874155454609605531146598877967335283756256104e-02, // 2.718282
  1.16346596858789251560661881512714899145066738128662e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_01_10 = { //
  7.63599351831150269243053685386257711797952651977539e-02, // 0.000000
  2.22124383190155039180524454422993585467338562011719e-01, // 0.000010
  4.67917828829414239599771008215611800551414489746094e-01, // 0.100000
  4.75019399203212733251433519399142824113368988037109e-01, // 0.123457
  4.91476239432907469062428162942524068057537078857422e-01, // 0.200000
  5.02079925428312145996301296690944582223892211914062e-01, // 0.271828
  5.05508894037498435203303870366653427481651306152344e-01, // 0.300000
  5.07115955277963825764686589536722749471664428710938e-01, // 0.314159
  5.15568704992931170494330217479728162288665771484375e-01, // 0.400000
  5.23427039033618868302255577873438596725463867187500e-01, // 0.500000
  5.29881674984790551619084908452350646257400512695312e-01, // 0.600000
  5.35361615308103777621795416052918881177902221679688e-01, // 0.700000
  5.40124560432196254566861171042546629905700683593750e-01, // 0.800000
  5.44337595924840766947738757153274491429328918457031e-01, // 0.900000
  5.48115306569588400797954363952158018946647644042969e-01, // 1.000000
  5.51539729658985344862287547584855929017066955566406e-01, // 1.100000
  5.54671638850380399610173753899289295077323913574219e-01, // 1.200000
  5.57557321771014247069331304373918101191520690917969e-01, // 1.300000
  5.60232853193344815245779955148464068770408630371094e-01, // 1.400000
  5.62726898877477954563630646589444950222969055175781e-01, // 1.500000
  5.65062617878835049545216406841063871979713439941406e-01, // 1.600000
  5.67258990450220990098273432522546499967575073242188e-01, // 1.700000
  5.69331767989112891292791118758032098412513732910156e-01, // 1.800000
  5.71294167272651032263297565805260092020034790039062e-01, // 1.900000
  5.73157387426872722002713089750614017248153686523438e-01, // 2.000000
  5.84332020844417376892465654236730188131332397460938e-01, // 2.718282
  5.89617748244353534481376755138626322150230407714844e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_01_10 = { //
  7.33673382206617444753646850585937500000000000000000e+07, // 0.000000
  1.95393520726679525978397578001022338867187500000000e+03, // 0.000010
  3.35721153767048474669820734561653807759284973144531e-01, // 0.100000
  2.74018609307649374517978912990656681358814239501953e-01, // 0.123457
  1.71942902599836300314706249992013908922672271728516e-01, // 0.200000
  1.27730497083025279492218828636396210640668869018555e-01, // 0.271828
  1.16077849986715470054221555074036587029695510864258e-01, // 0.300000
  1.10996650714197092613133577287953812628984451293945e-01, // 0.314159
  8.77764877675434734838333383777353446930646896362305e-02, // 0.400000
  7.06409277770171650789876593989902175962924957275391e-02, // 0.500000
  5.91386021268343470280726137389137875288724899291992e-02, // 0.600000
  5.08776272124714656897026543447282165288925170898438e-02, // 0.700000
  4.46538827467651833202921807242091745138168334960938e-02, // 0.800000
  3.97946728437166968861404825474892277270555496215820e-02, // 0.900000
  3.58944770831502030228321586946549359709024429321289e-02, // 1.000000
  3.26941881616924015507841261296562151983380317687988e-02, // 1.100000
  3.00204524249116540268733643870291416533291339874268e-02, // 1.200000
  2.77528760154124205794534674396345508284866809844971e-02, // 1.300000
  2.58052190382604025586932294800135423429310321807861e-02, // 1.400000
  2.41140762238259073746426963680278277024626731872559e-02, // 1.500000
  2.26317820314968706774116924407280748710036277770996e-02, // 1.600000
  2.13218078236213498533047072669432964175939559936523e-02, // 1.700000
  2.01556859486728690011858589059556834399700164794922e-02, // 1.800000
  1.91109005040240136075180288344199652783572673797607e-02, // 1.900000
  1.81694078358287727792408361437992425635457038879395e-02, // 2.000000
  1.34239874155408032280201169328393007162958383560181e-02, // 2.718282
  1.16346596858781688166306622633783263154327869415283e-02, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_01_20 = { //
  7.14336532604261248424748487195756752043962478637695e-02, // 0.000000
  2.08928790931204783065666674701787997037172317504883e-01, // 0.000010
  4.44953851727799498672055733550223521888256072998047e-01, // 0.100000
  4.51868924426242546399379307331400923430919647216797e-01, // 0.123457
  4.67917828829414239599771008215611800551414489746094e-01, // 0.200000
  4.78277040424782917682478000642731785774230957031250e-01, // 0.271828
  4.81630099462710015156829967963858507573604583740234e-01, // 0.300000
  4.83202119329382395740424271934898570179939270019531e-01, // 0.314159
  4.91476239432907469062428162942524068057537078857422e-01, // 0.400000
  4.99177179518956792847461656492669135332107543945312e-01, // 0.500000
  5.05508894037498435203303870366653427481651306152344e-01, // 0.600000
  5.10889043159064359933552168513415381312370300292969e-01, // 0.700000
  5.15568704992931170494330217479728162288665771484375e-01, // 0.800000
  5.19710777811082103028184064896777272224426269531250e-01, // 0.900000
  5.23427039033618868302255577873438596725463867187500e-01, // 1.000000
  5.26797557160076923565839024377055466175079345703125e-01, // 1.100000
  5.29881674984790551619084908452350646257400512695312e-01, // 1.200000
  5.32724609600441545431692702550208196043968200683594e-01, // 1.300000
  5.35361615308103777621795416052918881177902221679688e-01, // 1.400000
  5.37820716140765320290029194438830018043518066406250e-01, // 1.500000
  5.40124560432196254566861171042546629905700683593750e-01, // 1.600000
  5.42291715817192598869667108374414965510368347167969e-01, // 1.700000
  5.44337595924840766947738757153274491429328918457031e-01, // 1.800000
  5.46275137813967970323858480696799233555793762207031e-01, // 1.900000
  5.48115306569588400797954363952158018946647644042969e-01, // 2.000000
  5.59163067520781487118597397056873887777328491210938e-01, // 2.718282
  5.64395683719495666608167994127143174409866333007812e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_01_20 = { //
  6.88192507850085347890853881835937500000000000000000e+07, // 0.000000
  1.85401216383494602268910966813564300537109375000000e+03, // 0.000010
  3.26757898025421766874387685675173997879028320312500e-01, // 0.100000
  2.66942809682530424186808204467524774372577667236328e-01, // 0.123457
  1.67860576883524237334910367280826903879642486572266e-01, // 0.200000
  1.24873969109886198691761194368154974654316902160645e-01, // 0.271828
  1.13534446544550443602972222834068816155195236206055e-01, // 0.300000
  1.08588251220167866795129896217986242845654487609863e-01, // 0.314159
  8.59714512999181501573531249960069544613361358642578e-02, // 0.400000
  6.92640857666013964877294029065524227917194366455078e-02, // 0.500000
  5.80389249933577419660046814442466711625456809997559e-02, // 0.600000
  4.99707831949509073687920590600697323679924011230469e-02, // 0.700000
  4.38882438837717436808105730960960499942302703857422e-02, // 0.800000
  3.91364638781799556443985466103185899555683135986328e-02, // 0.900000
  3.53204638885085825394938296994951087981462478637695e-02, // 1.000000
  3.21877457163609673185256099259277107194066047668457e-02, // 1.100000
  2.95693010634171700445893549158427049405872821807861e-02, // 1.200000
  2.73477053682244612697882502061474951915442943572998e-02, // 1.300000
  2.54388136062357328448513271723641082644462585449219e-02, // 1.400000
  2.37807387894387115001570975891809212043881416320801e-02, // 1.500000
  2.23269413733825881906991384084903984330594539642334e-02, // 1.600000
  2.10417445599888272567135061308363219723105430603027e-02, // 1.700000
  1.98973364218583484430702412737446138635277748107910e-02, // 1.800000
  1.88717135986907202926943227794254198670387268066406e-02, // 1.900000
  1.79472385415751015114160793473274679854512214660645e-02, // 2.000000
  1.32834210371973356068497906790071283467113971710205e-02, // 2.718282
  1.15227009149735971915706755908104241825640201568604e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_01_20 = { //
  7.14336532604261248424748487195756752043962478637695e-02, // 0.000000
  2.08928790931204783065666674701787997037172317504883e-01, // 0.000010
  4.44953851727799498672055733550223521888256072998047e-01, // 0.100000
  4.51868924426242546399379307331400923430919647216797e-01, // 0.123457
  4.67917828829414239599771008215611800551414489746094e-01, // 0.200000
  4.78277040424794852579992721075541339814662933349609e-01, // 0.271828
  4.81630099462710015156829967963858507573604583740234e-01, // 0.300000
  4.83202119329384616186473522247979417443275451660156e-01, // 0.314159
  4.91476239432907469062428162942524068057537078857422e-01, // 0.400000
  4.99177179518956792847461656492669135332107543945312e-01, // 0.500000
  5.05508894037498435203303870366653427481651306152344e-01, // 0.600000
  5.10889043159064359933552168513415381312370300292969e-01, // 0.700000
  5.15568704992931170494330217479728162288665771484375e-01, // 0.800000
  5.19710777811082103028184064896777272224426269531250e-01, // 0.900000
  5.23427039033618868302255577873438596725463867187500e-01, // 1.000000
  5.26797557160076923565839024377055466175079345703125e-01, // 1.100000
  5.29881674984790551619084908452350646257400512695312e-01, // 1.200000
  5.32724609600441545431692702550208196043968200683594e-01, // 1.300000
  5.35361615308103777621795416052918881177902221679688e-01, // 1.400000
  5.37820716140765320290029194438830018043518066406250e-01, // 1.500000
  5.40124560432196254566861171042546629905700683593750e-01, // 1.600000
  5.42291715817192598869667108374414965510368347167969e-01, // 1.700000
  5.44337595924840766947738757153274491429328918457031e-01, // 1.800000
  5.46275137813967970323858480696799233555793762207031e-01, // 1.900000
  5.48115306569588400797954363952158018946647644042969e-01, // 2.000000
  5.59163067520794143661078123841434717178344726562500e-01, // 2.718282
  5.64395683719498109098822169471532106399536132812500e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_01_20 = { //
  6.88192507850085496902465820312500000000000000000000e+07, // 0.000000
  1.85401216383494602268910966813564300537109375000000e+03, // 0.000010
  3.26757898025421711363236454417346976697444915771484e-01, // 0.100000
  2.66942809682530313164505741951870732009410858154297e-01, // 0.123457
  1.67860576883524237334910367280826903879642486572266e-01, // 0.200000
  1.24873969109843857561159552460594568401575088500977e-01, // 0.271828
  1.13534446544550429725184415019612060859799385070801e-01, // 0.300000
  1.08588251220160983412377220247435616329312324523926e-01, // 0.314159
  8.59714512999181501573531249960069544613361358642578e-02, // 0.400000
  6.92640857666013964877294029065524227917194366455078e-02, // 0.500000
  5.80389249933577350271107775370182935148477554321289e-02, // 0.600000
  4.99707831949509143076859629672981100156903266906738e-02, // 0.700000
  4.38882438837717367419166691888676723465323448181152e-02, // 0.800000
  3.91364638781799487055046427030902123078703880310059e-02, // 0.900000
  3.53204638885085825394938296994951087981462478637695e-02, // 1.000000
  3.21877457163609673185256099259277107194066047668457e-02, // 1.100000
  2.95693010634171735140363068694568937644362449645996e-02, // 1.200000
  2.73477053682244578003412982525333063676953315734863e-02, // 1.300000
  2.54388136062357328448513271723641082644462585449219e-02, // 1.400000
  2.37807387894387115001570975891809212043881416320801e-02, // 1.500000
  2.23269413733825916601460903621045872569084167480469e-02, // 1.600000
  2.10417445599888272567135061308363219723105430603027e-02, // 1.700000
  1.98973364218583484430702412737446138635277748107910e-02, // 1.800000
  1.88717135986907202926943227794254198670387268066406e-02, // 1.900000
  1.79472385415751015114160793473274679854512214660645e-02, // 2.000000
  1.32834210371927524674262599546636920422315597534180e-02, // 2.718282
  1.15227009149728529951994815405669214669615030288696e-02, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_01_4 = { //
  8.33733718538829660094080509225022979080677032470703e-02, // 0.000000
  2.40649836762494045983018509105022531002759933471680e-01, // 0.000010
  4.99177179518956792847461656492669135332107543945312e-01, // 0.100000
  5.06498248113537830050745469634421169757843017578125e-01, // 0.123457
  5.23427039033618868302255577873438596725463867187500e-01, // 0.200000
  5.34307105228540524244351672678021714091300964355469e-01, // 0.271828
  5.37820716140765320290029194438830018043518066406250e-01, // 0.300000
  5.39466643360843156607131732016569003462791442871094e-01, // 0.314159
  5.48115306569588400797954363952158018946647644042969e-01, // 0.400000
  5.56142802864377228111436579638393595814704895019531e-01, // 0.500000
  5.62726898877477954563630646589444950222969055175781e-01, // 0.600000
  5.68309949920494883457422474748454988002777099609375e-01, // 0.700000
  5.73157387426872722002713089750614017248153686523438e-01, // 0.800000
  5.77441148174444141005778874387033283710479736328125e-01, // 0.900000
  5.81279046613597527581873691815417259931564331054688e-01, // 1.000000
  5.84755370118739237206284542480716481804847717285156e-01, // 1.100000
  5.87932517498272577682882911176420748233795166015625e-01, // 1.200000
  5.90857985095490256810535356635227799415588378906250e-01, // 1.300000
  5.93568768820695535914921947551192715764045715332031e-01, // 1.400000
  5.96094250977751372033708321396261453628540039062500e-01, // 1.500000
  5.98458157890379149890236476494465023279190063476562e-01, // 1.600000
  6.00679925779863399526448120013810694217681884765625e-01, // 1.700000
  6.02775677450659741474225938873132690787315368652344e-01, // 1.800000
  6.04758935771455119478900996909942477941513061523438e-01, // 1.900000
  6.06641154768875257019544733338989317417144775390625e-01, // 2.000000
  6.17913083162751508581322923419065773487091064453125e-01, // 2.718282
  6.23234757420644713832302841183263808488845825195312e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_01_4 = { //
  7.97969827464829683303833007812500000000000000000000e+07, // 0.000000
  2.09043222444361890666186809539794921875000000000000e+03, // 0.000010
  3.46320428833006954683071398903848603367805480957031e-01, // 0.100000
  2.82305605668253523088395695594954304397106170654297e-01, // 0.123457
  1.76602319442542926575256956311932299286127090454102e-01, // 0.200000
  1.30926807720308208127235616302641574293375015258789e-01, // 0.271828
  1.18903693947193564439679391853132983669638633728027e-01, // 0.300000
  1.13663285983750791952040515297994716092944145202637e-01, // 0.314159
  8.97361927078755006181864928294089622795581817626953e-02, // 0.400000
  7.21048144026743415002655979151313658803701400756836e-02, // 0.500000
  6.02851905595647649671597889664553804323077201843262e-02, // 0.600000
  5.18057307044127585138326708147360477596521377563477e-02, // 0.700000
  4.54235195895719423564429462203406728804111480712891e-02, // 0.800000
  4.04448236551064599364480045551317743957042694091797e-02, // 0.900000
  3.64517761834422873712391321987524861469864845275879e-02, // 1.000000
  3.31775726553523511719312466539122397080063819885254e-02, // 1.100000
  3.04438254891350006114958404168646666221320629119873e-02, // 1.200000
  2.81267182882828474521641481942424434237182140350342e-02, // 1.300000
  2.61376080752830329545499665755414753220975399017334e-02, // 1.400000
  2.44113559017523976124586226887913653627038002014160e-02, // 1.500000
  2.28990161555940596738167869261815212666988372802734e-02, // 1.600000
  2.15630959198551165612034452578882337547838687896729e-02, // 1.700000
  2.03743883245797278991329193331694114021956920623779e-02, // 1.800000
  1.93098019737789824978868580274138366803526878356934e-02, // 1.900000
  1.83508390056730183448330251394509105011820793151855e-02, // 2.000000
  1.35235631786254032610239406153596064541488885879517e-02, // 2.718282
  1.17065775592872373112030004449479747563600540161133e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_01_4 = { //
  8.33733718538829660094080509225022979080677032470703e-02, // 0.000000
  2.40649836762494045983018509105022531002759933471680e-01, // 0.000010
  4.99177179518956792847461656492669135332107543945312e-01, // 0.100000
  5.06498248113537830050745469634421169757843017578125e-01, // 0.123457
  5.23427039033618868302255577873438596725463867187500e-01, // 0.200000
  5.34307105228552958742227474431274458765983581542969e-01, // 0.271828
  5.37820716140765320290029194438830018043518066406250e-01, // 0.300000
  5.39466643360845488075483444845303893089294433593750e-01, // 0.314159
  5.48115306569588400797954363952158018946647644042969e-01, // 0.400000
  5.56142802864377228111436579638393595814704895019531e-01, // 0.500000
  5.62726898877477954563630646589444950222969055175781e-01, // 0.600000
  5.68309949920494883457422474748454988002777099609375e-01, // 0.700000
  5.73157387426872722002713089750614017248153686523438e-01, // 0.800000
  5.77441148174444141005778874387033283710479736328125e-01, // 0.900000
  5.81279046613597527581873691815417259931564331054688e-01, // 1.000000
  5.84755370118739237206284542480716481804847717285156e-01, // 1.100000
  5.87932517498272577682882911176420748233795166015625e-01, // 1.200000
  5.90857985095490256810535356635227799415588378906250e-01, // 1.300000
  5.93568768820695535914921947551192715764045715332031e-01, // 1.400000
  5.96094250977751372033708321396261453628540039062500e-01, // 1.500000
  5.98458157890379149890236476494465023279190063476562e-01, // 1.600000
  6.00679925779863399526448120013810694217681884765625e-01, // 1.700000
  6.02775677450659741474225938873132690787315368652344e-01, // 1.800000
  6.04758935771455119478900996909942477941513061523438e-01, // 1.900000
  6.06641154768875257019544733338989317417144775390625e-01, // 2.000000
  6.17913083162764387168408575234934687614440917968750e-01, // 2.718282
  6.23234757420647156322957016527652740478515625000000e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_01_4 = { //
  7.97969827464829534292221069335937500000000000000000e+07, // 0.000000
  2.09043222444361890666186809539794921875000000000000e+03, // 0.000010
  3.46320428833006954683071398903848603367805480957031e-01, // 0.100000
  2.82305605668253523088395695594954304397106170654297e-01, // 0.123457
  1.76602319442542898819681340683018788695335388183594e-01, // 0.200000
  1.30926807720263271850313913091667927801609039306641e-01, // 0.271828
  1.18903693947193564439679391853132983669638633728027e-01, // 0.300000
  1.13663285983743478357865797079284675419330596923828e-01, // 0.314159
  8.97361927078755006181864928294089622795581817626953e-02, // 0.400000
  7.21048144026743276224777901006746105849742889404297e-02, // 0.500000
  6.02851905595647719060536928736837580800056457519531e-02, // 0.600000
  5.18057307044127585138326708147360477596521377563477e-02, // 0.700000
  4.54235195895719354175490423131122952327132225036621e-02, // 0.800000
  4.04448236551064599364480045551317743957042694091797e-02, // 0.900000
  3.64517761834422804323452282915241084992885589599609e-02, // 1.000000
  3.31775726553523511719312466539122397080063819885254e-02, // 1.100000
  3.04438254891349936726019365096362889744341373443604e-02, // 1.200000
  2.81267182882828474521641481942424434237182140350342e-02, // 1.300000
  2.61376080752830329545499665755414753220975399017334e-02, // 1.400000
  2.44113559017523976124586226887913653627038002014160e-02, // 1.500000
  2.28990161555940596738167869261815212666988372802734e-02, // 1.600000
  2.15630959198551200306503972115024225786328315734863e-02, // 1.700000
  2.03743883245797348380268232403977890498936176300049e-02, // 1.800000
  1.93098019737789824978868580274138366803526878356934e-02, // 1.900000
  1.83508390056730183448330251394509105011820793151855e-02, // 2.000000
  1.35235631786206692006579999087989563122391700744629e-02, // 2.718282
  1.17065775592864670939796667425980558618903160095215e-02, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_CDF_01_1 = { //
  9.51625819640404130295152640428568702191114425659180e-02, // 0.000000
  2.71106585889975382208660903415875509381294250488281e-01, // 0.000010
  5.48115306569588400797954363952158018946647644042969e-01, // 0.100000
  5.55694967119628291385424745385535061359405517578125e-01, // 0.123457
  5.73157387426872722002713089750614017248153686523438e-01, // 0.200000
  5.84332020844404609327682464936515316367149353027344e-01, // 0.271828
  5.87932517498272577682882911176420748233795166015625e-01, // 0.300000
  5.89617748244351203013025042309891432523727416992188e-01, // 0.314159
  5.98458157890379149890236476494465023279190063476562e-01, // 0.400000
  6.06641154768875257019544733338989317417144775390625e-01, // 0.500000
  6.13336400730758013466470401908736675977706909179688e-01, // 0.600000
  6.19001977866595609611977124586701393127441406250000e-01, // 0.700000
  6.23912243750147643694958787818904966115951538085938e-01, // 0.800000
  6.28244633588060930051710784027818590402603149414062e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  6.35626771185953010068203639093553647398948669433594e-01, // 1.100000
  6.38827420796338696362681730533950030803680419921875e-01, // 1.200000
  6.41771286235222282456902576086577028036117553710938e-01, // 1.300000
  6.44496325671848335758795656147412955760955810546875e-01, // 1.400000
  6.47032658735232191205000162881333380937576293945312e-01, // 1.500000
  6.49404586860663601832754920906154438853263854980469e-01, // 1.600000
  6.51632001805449334597142296843230724334716796875000e-01, // 1.700000
  6.53731392071210781757883978571044281125068664550781e-01, // 1.800000
  6.55716577591103022548679746250854805111885070800781e-01, // 1.900000
  6.57599256245357088879188722785329446196556091308594e-01, // 2.000000
  6.68845722847091206375580441090278327465057373046875e-01, // 2.718282
  6.74138343168612985678578297665808349847793579101562e-01, // 3.141593
  };

  public static final double[] SCIPY_WEIBULL_PDF_01_1 = { //
  9.04837418035960197448730468750000000000000000000000e+07, // 0.000000
  2.30496336008399066486163064837455749511718750000000e+03, // 0.000010
  3.58944770831502002472745971317635849118232727050781e-01, // 0.100000
  2.91956423650103102929165288514923304319381713867188e-01, // 0.123457
  1.81694078358287769425771784881362691521644592285156e-01, // 0.200000
  1.34239874155454602666637242691649589687585830688477e-01, // 0.271828
  1.21775301956540002445983361667458666488528251647949e-01, // 0.300000
  1.16346596858789258499555785419943276792764663696289e-01, // 0.314159
  9.15960646223762386952671477047260850667953491210938e-02, // 0.400000
  7.34033560226920733793321005578036420047283172607422e-02, // 0.500000
  6.12346392900829789018857240989746060222387313842773e-02, // 0.600000
  5.25211812394763555289500800427049398422241210937500e-02, // 0.700000
  4.59735675855645273268912376352091087028384208679199e-02, // 0.800000
  4.08732327117912602743032834951009135693311691284180e-02, // 0.900000
  3.67879441171442347902065250764280790463089942932129e-02, // 1.000000
  3.34420617455878435575478135888261022046208381652832e-02, // 1.100000
  3.06514941118683631460228866671968717128038406372070e-02, // 1.200000
  2.82885949087761413844877012024880968965590000152588e-02, // 1.300000
  2.62620643803951336892055223870556801557540893554688e-02, // 1.400000
  2.45048692989323843582916140348970657214522361755371e-02, // 1.500000
  2.29666813847786074886947460527153452858328819274902e-02, // 1.600000
  2.16089778205777807706944315668806666508316993713379e-02, // 1.700000
  2.04017709011452205714043373063759645447134971618652e-02, // 1.800000
  1.93213679270299865109450365707743912935256958007812e-02, // 1.900000
  1.83488015354457897898576845818752190098166465759277e-02, // 2.000000
  1.34637281783702208309838610489350685384124517440796e-02, // 2.718282
  1.16304997397552699334166703692972077988088130950928e-02, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_CDF_01_1 = { //
  9.51625819640404130295152640428568702191114425659180e-02, // 0.000000
  2.71106585889975382208660903415875509381294250488281e-01, // 0.000010
  5.48115306569588400797954363952158018946647644042969e-01, // 0.100000
  5.55694967119628291385424745385535061359405517578125e-01, // 0.123457
  5.73157387426872722002713089750614017248153686523438e-01, // 0.200000
  5.84332020844417376892465654236730188131332397460938e-01, // 0.271828
  5.87932517498272577682882911176420748233795166015625e-01, // 0.300000
  5.89617748244353534481376755138626322150230407714844e-01, // 0.314159
  5.98458157890379149890236476494465023279190063476562e-01, // 0.400000
  6.06641154768875257019544733338989317417144775390625e-01, // 0.500000
  6.13336400730758013466470401908736675977706909179688e-01, // 0.600000
  6.19001977866595609611977124586701393127441406250000e-01, // 0.700000
  6.23912243750147643694958787818904966115951538085938e-01, // 0.800000
  6.28244633588060930051710784027818590402603149414062e-01, // 0.900000
  6.32120558828557665975722557050175964832305908203125e-01, // 1.000000
  6.35626771185953010068203639093553647398948669433594e-01, // 1.100000
  6.38827420796338696362681730533950030803680419921875e-01, // 1.200000
  6.41771286235222282456902576086577028036117553710938e-01, // 1.300000
  6.44496325671848335758795656147412955760955810546875e-01, // 1.400000
  6.47032658735232191205000162881333380937576293945312e-01, // 1.500000
  6.49404586860663601832754920906154438853263854980469e-01, // 1.600000
  6.51632001805449334597142296843230724334716796875000e-01, // 1.700000
  6.53731392071210781757883978571044281125068664550781e-01, // 1.800000
  6.55716577591103022548679746250854805111885070800781e-01, // 1.900000
  6.57599256245357088879188722785329446196556091308594e-01, // 2.000000
  6.68845722847103862918061167874839156866073608398438e-01, // 2.718282
  6.74138343168615428169232473010197281837463378906250e-01, // 3.141593
  };

  public static final double[] GNUR_WEIBULL_PDF_01_1 = { //
  9.04837418035960048437118530273437500000000000000000e+07, // 0.000000
  2.30496336008399021011427976191043853759765625000000e+03, // 0.000010
  3.58944770831502002472745971317635849118232727050781e-01, // 0.100000
  2.91956423650103102929165288514923304319381713867188e-01, // 0.123457
  1.81694078358287741670196169252449180930852890014648e-01, // 0.200000
  1.34239874155408001055178601745865307748317718505859e-01, // 0.271828
  1.21775301956539974690407746038545155897736549377441e-01, // 0.300000
  1.16346596858781695105200526541011640802025794982910e-01, // 0.314159
  9.15960646223762386952671477047260850667953491210938e-02, // 0.400000
  7.34033560226920733793321005578036420047283172607422e-02, // 0.500000
  6.12346392900829858407796280062029836699366569519043e-02, // 0.600000
  5.25211812394763624678439839499333174899220466613770e-02, // 0.700000
  4.59735675855645273268912376352091087028384208679199e-02, // 0.800000
  4.08732327117912672131971874023292912170290946960449e-02, // 0.900000
  3.67879441171442347902065250764280790463089942932129e-02, // 1.000000
  3.34420617455878435575478135888261022046208381652832e-02, // 1.100000
  3.06514941118683666154698386208110605366528034210205e-02, // 1.200000
  2.82885949087761448539346531561022857204079627990723e-02, // 1.300000
  2.62620643803951267503116184798273025080561637878418e-02, // 1.400000
  2.45048692989323808888446620812828768976032733917236e-02, // 1.500000
  2.29666813847786040192477940991011564619839191436768e-02, // 1.600000
  2.16089778205777807706944315668806666508316993713379e-02, // 1.700000
  2.04017709011452171019573853527617757208645343780518e-02, // 1.800000
  1.93213679270299865109450365707743912935256958007812e-02, // 1.900000
  1.83488015354457863204107326282610301859676837921143e-02, // 2.000000
  1.34637281783654468719779728758112469222396612167358e-02, // 2.718282
  1.16304997397544927772994327597189112566411495208740e-02, // 3.141593
  };

  public static final double[] P_QUANT = { //
  0.0001, 0.001, 0.01, 0.1, 0.25, 0.5, 0.75, 0.9, 0.99, 0.999, 0.9999 //
  };

  public static final double[] SCIPY_WEIBULL_QUANT_1_1 = { //
  1.00005000333358343134713963618054322068928740918636e-04, // 0.000100
  1.00050033358353349620395444929954464896582067012787e-03, // 0.001000
  1.00503358535014419156139453548348683398216962814331e-02, // 0.010000
  1.05360515657826309121603003404743503779172897338867e-01, // 0.100000
  2.87682072451780901367612841568188741803169250488281e-01, // 0.250000
  6.93147180559945286226763982995180413126945495605469e-01, // 0.500000
  1.38629436111989057245352796599036082625389099121094e+00, // 0.750000
  2.30258509299404590109361379290930926799774169921875e+00, // 0.900000
  4.60517018598809002583038818556815385818481445312500e+00, // 0.990000
  6.90775527898213592692400197847746312618255615234375e+00, // 0.999000
  9.21034037197629373849849798716604709625244140625000e+00, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_1_1 = { //
  1.00005000333358343134713963618054322068928740918636e-04, // 0.000100
  1.00050033358353349620395444929954464896582067012787e-03, // 0.001000
  1.00503358535014419156139453548348683398216962814331e-02, // 0.010000
  1.05360515657826309121603003404743503779172897338867e-01, // 0.100000
  2.87682072451780901367612841568188741803169250488281e-01, // 0.250000
  6.93147180559945286226763982995180413126945495605469e-01, // 0.500000
  1.38629436111989057245352796599036082625389099121094e+00, // 0.750000
  2.30258509299404590109361379290930926799774169921875e+00, // 0.900000
  4.60517018598809002583038818556815385818481445312500e+00, // 0.990000
  6.90775527898213592692400197847746312618255615234375e+00, // 0.999000
  9.21034037197629373849849798716604709625244140625000e+00, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_2_1 = { //
  1.00002500135425789751231206992088118568062782287598e-02, // 0.000100
  3.16306865809696524660665772898937575519084930419922e-02, // 0.001000
  1.00251363349839001104513158679765183478593826293945e-01, // 0.010000
  3.24592845974501276185009146502125076949596405029297e-01, // 0.100000
  5.36360021302651590602295073040295392274856567382812e-01, // 0.250000
  8.32554611157697688206269504007650539278984069824219e-01, // 0.500000
  1.17741002251547466350700688053620979189872741699219e+00, // 0.750000
  1.51742712938514645593102159182308241724967956542969e+00, // 0.900000
  2.14596602628934673973049029882531613111495971679688e+00, // 0.990000
  2.62826088487846565300287693389691412448883056640625e+00, // 0.999000
  3.03485425877031111951964703621342778205871582031250e+00, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_2_1 = { //
  1.00002500135425789751231206992088118568062782287598e-02, // 0.000100
  3.16306865809696524660665772898937575519084930419922e-02, // 0.001000
  1.00251363349839001104513158679765183478593826293945e-01, // 0.010000
  3.24592845974501276185009146502125076949596405029297e-01, // 0.100000
  5.36360021302651590602295073040295392274856567382812e-01, // 0.250000
  8.32554611157697688206269504007650539278984069824219e-01, // 0.500000
  1.17741002251547466350700688053620979189872741699219e+00, // 0.750000
  1.51742712938514645593102159182308241724967956542969e+00, // 0.900000
  2.14596602628934673973049029882531613111495971679688e+00, // 0.990000
  2.62826088487846565300287693389691412448883056640625e+00, // 0.999000
  3.03485425877031111951964703621342778205871582031250e+00, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_4_1 = { //
  1.00001250059899640665328490740648703649640083312988e-01, // 0.000100
  1.77850180154448117564314202354580629616975784301758e-01, // 0.001000
  3.16624956928286072077582957717822864651679992675781e-01, // 0.010000
  5.69730502934941429593607153947232291102409362792969e-01, // 0.100000
  7.32366043247945519567565497709438204765319824218750e-01, // 0.250000
  9.12444305784028575345701028709299862384796142578125e-01, // 0.500000
  1.08508526048208508996140153612941503524780273437500e+00, // 0.750000
  1.23183892185023369236773760349024087190628051757812e+00, // 0.900000
  1.46491161040157868988842437829589471220970153808594e+00, // 0.990000
  1.62119119319050875027699021302396431565284729003906e+00, // 0.999000
  1.74208330993965687660818275617202743887901306152344e+00, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_4_1 = { //
  1.00001250059899640665328490740648703649640083312988e-01, // 0.000100
  1.77850180154448117564314202354580629616975784301758e-01, // 0.001000
  3.16624956928286072077582957717822864651679992675781e-01, // 0.010000
  5.69730502934941429593607153947232291102409362792969e-01, // 0.100000
  7.32366043247945519567565497709438204765319824218750e-01, // 0.250000
  9.12444305784028575345701028709299862384796142578125e-01, // 0.500000
  1.08508526048208508996140153612941503524780273437500e+00, // 0.750000
  1.23183892185023369236773760349024087190628051757812e+00, // 0.900000
  1.46491161040157868988842437829589471220970153808594e+00, // 0.990000
  1.62119119319050875027699021302396431565284729003906e+00, // 0.999000
  1.74208330993965687660818275617202743887901306152344e+00, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_4_10 = { //
  1.00001250059899637889770929177757352590560913085938e+00, // 0.000100
  1.77850180154448112013199079228797927498817443847656e+00, // 0.001000
  3.16624956928286049873122465214692056179046630859375e+00, // 0.010000
  5.69730502934941451798067646450363099575042724609375e+00, // 0.100000
  7.32366043247945519567565497709438204765319824218750e+00, // 0.250000
  9.12444305784028486527859058696776628494262695312500e+00, // 0.500000
  1.08508526048208508996140153612941503524780273437500e+01, // 0.750000
  1.23183892185023360354989563347771763801574707031250e+01, // 0.900000
  1.46491161040157873429734536330215632915496826171875e+01, // 0.990000
  1.62119119319050888350375316804274916648864746093750e+01, // 0.999000
  1.74208330993965674338141980115324258804321289062500e+01, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_4_10 = { //
  1.00001250059899637889770929177757352590560913085938e+00, // 0.000100
  1.77850180154448112013199079228797927498817443847656e+00, // 0.001000
  3.16624956928286049873122465214692056179046630859375e+00, // 0.010000
  5.69730502934941451798067646450363099575042724609375e+00, // 0.100000
  7.32366043247945519567565497709438204765319824218750e+00, // 0.250000
  9.12444305784028486527859058696776628494262695312500e+00, // 0.500000
  1.08508526048208508996140153612941503524780273437500e+01, // 0.750000
  1.23183892185023360354989563347771763801574707031250e+01, // 0.900000
  1.46491161040157873429734536330215632915496826171875e+01, // 0.990000
  1.62119119319050888350375316804274916648864746093750e+01, // 0.999000
  1.74208330993965674338141980115324258804321289062500e+01, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_01_10 = { //
  1.00050014586584021700243769737484031845767568464672e-39, // 0.000100
  1.00501461589481204170765583646174510341434104691366e-29, // 0.001000
  1.05149145760085306110809830513112699455704405376921e-19, // 0.010000
  1.68569451981823333734044073758343712521323709552235e-09, // 0.100000
  3.88264771194455187157575093426231660487246699631214e-05, // 0.250000
  2.56008632895631083048471055008121766149997711181641e-01, // 0.500000
  2.62152840085126229041634360328316688537597656250000e+02, // 0.750000
  4.18944879802952418685890734195709228515625000000000e+04, // 0.900000
  4.28999556918221637606620788574218750000000000000000e+07, // 0.990000
  2.47382762074844741821289062500000000000000000000000e+09, // 0.999000
  4.39295546284313201904296875000000000000000000000000e+10, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_01_10 = { //
  1.00050014586584021700243769737484031845767568464672e-39, // 0.000100
  1.00501461589481204170765583646174510341434104691366e-29, // 0.001000
  1.05149145760085306110809830513112699455704405376921e-19, // 0.010000
  1.68569451981823333734044073758343712521323709552235e-09, // 0.100000
  3.88264771194455187157575093426231660487246699631214e-05, // 0.250000
  2.56008632895631083048471055008121766149997711181641e-01, // 0.500000
  2.62152840085126229041634360328316688537597656250000e+02, // 0.750000
  4.18944879802952418685890734195709228515625000000000e+04, // 0.900000
  4.28999556918221637606620788574218750000000000000000e+07, // 0.990000
  2.47382762074844741821289062500000000000000000000000e+09, // 0.999000
  4.39295546284313201904296875000000000000000000000000e+10, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_01_20 = { //
  2.00100029173168043400487539474968063691535136929345e-39, // 0.000100
  2.01002923178962408341531167292349020682868209382732e-29, // 0.001000
  2.10298291520170612221619661026225398911408810753842e-19, // 0.010000
  3.37138903963646667468088147516687425042647419104469e-09, // 0.100000
  7.76529542388910374315150186852463320974493399262428e-05, // 0.250000
  5.12017265791262166096942110016243532299995422363281e-01, // 0.500000
  5.24305680170252458083268720656633377075195312500000e+02, // 0.750000
  8.37889759605904837371781468391418457031250000000000e+04, // 0.900000
  8.57999113836443275213241577148437500000000000000000e+07, // 0.990000
  4.94765524149689483642578125000000000000000000000000e+09, // 0.999000
  8.78591092568626403808593750000000000000000000000000e+10, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_01_20 = { //
  2.00100029173168043400487539474968063691535136929345e-39, // 0.000100
  2.01002923178962408341531167292349020682868209382732e-29, // 0.001000
  2.10298291520170612221619661026225398911408810753842e-19, // 0.010000
  3.37138903963646667468088147516687425042647419104469e-09, // 0.100000
  7.76529542388910374315150186852463320974493399262428e-05, // 0.250000
  5.12017265791262166096942110016243532299995422363281e-01, // 0.500000
  5.24305680170252458083268720656633377075195312500000e+02, // 0.750000
  8.37889759605904837371781468391418457031250000000000e+04, // 0.900000
  8.57999113836443275213241577148437500000000000000000e+07, // 0.990000
  4.94765524149689483642578125000000000000000000000000e+09, // 0.999000
  8.78591092568626403808593750000000000000000000000000e+10, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_01_4 = { //
  4.00200058346336103114236248946247444158805547000059e-40, // 0.000100
  4.02005846357924816683062334584698041365736418765464e-30, // 0.001000
  4.20596583040341236480301474472674879422679762663480e-20, // 0.010000
  6.74277807927293355615691608859066721870512139958009e-10, // 0.100000
  1.55305908477782068086766459336089951648318674415350e-05, // 0.250000
  1.02403453158252430443830860440357355400919914245605e-01, // 0.500000
  1.04861136034050488774482801090925931930541992187500e+02, // 0.750000
  1.67577951921180974750313907861709594726562500000000e+04, // 0.900000
  1.71599822767288647592067718505859375000000000000000e+07, // 0.990000
  9.89531048299378991127014160156250000000000000000000e+08, // 0.999000
  1.75718218513725280761718750000000000000000000000000e+10, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_01_4 = { //
  4.00200058346336103114236248946247444158805547000059e-40, // 0.000100
  4.02005846357924816683062334584698041365736418765464e-30, // 0.001000
  4.20596583040341236480301474472674879422679762663480e-20, // 0.010000
  6.74277807927293355615691608859066721870512139958009e-10, // 0.100000
  1.55305908477782068086766459336089951648318674415350e-05, // 0.250000
  1.02403453158252430443830860440357355400919914245605e-01, // 0.500000
  1.04861136034050488774482801090925931930541992187500e+02, // 0.750000
  1.67577951921180974750313907861709594726562500000000e+04, // 0.900000
  1.71599822767288647592067718505859375000000000000000e+07, // 0.990000
  9.89531048299378991127014160156250000000000000000000e+08, // 0.999000
  1.75718218513725280761718750000000000000000000000000e+10, // 0.999900
  };

  public static final double[] SCIPY_WEIBULL_QUANT_01_1 = { //
  1.00050014586584025778559062236561861039701386750015e-40, // 0.000100
  1.00501461589481204170765583646174510341434104691366e-30, // 0.001000
  1.05149145760085309120075368618168719855669940665870e-20, // 0.010000
  1.68569451981823338903922902214766680467628034989502e-10, // 0.100000
  3.88264771194455170216916148340224879120796686038375e-06, // 0.250000
  2.56008632895631076109577151100893388502299785614014e-02, // 0.500000
  2.62152840085126221936207002727314829826354980468750e+01, // 0.750000
  4.18944879802952436875784769654273986816406250000000e+03, // 0.900000
  4.28999556918221618980169296264648437500000000000000e+06, // 0.990000
  2.47382762074844747781753540039062500000000000000000e+08, // 0.999000
  4.39295546284313201904296875000000000000000000000000e+09, // 0.999900
  };

  public static final double[] GNUR_WEIBULL_QUANT_01_1 = { //
  1.00050014586584025778559062236561861039701386750015e-40, // 0.000100
  1.00501461589481204170765583646174510341434104691366e-30, // 0.001000
  1.05149145760085309120075368618168719855669940665870e-20, // 0.010000
  1.68569451981823338903922902214766680467628034989502e-10, // 0.100000
  3.88264771194455170216916148340224879120796686038375e-06, // 0.250000
  2.56008632895631076109577151100893388502299785614014e-02, // 0.500000
  2.62152840085126221936207002727314829826354980468750e+01, // 0.750000
  4.18944879802952436875784769654273986816406250000000e+03, // 0.900000
  4.28999556918221618980169296264648437500000000000000e+06, // 0.990000
  2.47382762074844747781753540039062500000000000000000e+08, // 0.999000
  4.39295546284313201904296875000000000000000000000000e+09, // 0.999900
  };

  @Test
  public void testPDF() {
    checkPDF(new WeibullDistribution(1., 1.), P_CDFPDF, SCIPY_WEIBULL_PDF_1_1, 1e-13);
    checkPDF(new WeibullDistribution(2., 1.), P_CDFPDF, SCIPY_WEIBULL_PDF_2_1, 1e-12);
    checkPDF(new WeibullDistribution(4., 1.), P_CDFPDF, SCIPY_WEIBULL_PDF_4_1, 1e-11);
    checkPDF(new WeibullDistribution(4., 10.), P_CDFPDF, SCIPY_WEIBULL_PDF_4_10, 1e-11);
    checkPDF(new WeibullDistribution(.1, 1.), P_CDFPDF, SCIPY_WEIBULL_PDF_01_1, 1e-12);
    checkPDF(new WeibullDistribution(.1, 4.), P_CDFPDF, SCIPY_WEIBULL_PDF_01_4, 1e-12);
    checkPDF(new WeibullDistribution(.1, 10.), P_CDFPDF, SCIPY_WEIBULL_PDF_01_10, 1e-12);
    checkPDF(new WeibullDistribution(.1, 20.), P_CDFPDF, SCIPY_WEIBULL_PDF_01_20, 1e-12);

    checkPDF(new WeibullDistribution(1., 1.), P_CDFPDF, GNUR_WEIBULL_PDF_1_1, 1e-15);
    checkPDF(new WeibullDistribution(2., 1.), P_CDFPDF, GNUR_WEIBULL_PDF_2_1, 1e-15);
    checkPDF(new WeibullDistribution(4., 1.), P_CDFPDF, GNUR_WEIBULL_PDF_4_1, 1e-14);
    checkPDF(new WeibullDistribution(4., 10.), P_CDFPDF, GNUR_WEIBULL_PDF_4_10, 1e-15);
    checkPDF(new WeibullDistribution(.1, 1.), P_CDFPDF, GNUR_WEIBULL_PDF_01_1, 1e-15);
    checkPDF(new WeibullDistribution(.1, 4.), P_CDFPDF, GNUR_WEIBULL_PDF_01_4, 1e-15);
    checkPDF(new WeibullDistribution(.1, 10.), P_CDFPDF, GNUR_WEIBULL_PDF_01_10, 1e-15);
    checkPDF(new WeibullDistribution(.1, 20.), P_CDFPDF, GNUR_WEIBULL_PDF_01_20, 1e-15);
  }

  @Test
  public void testCDF() {
    checkCDF(new WeibullDistribution(1., 1.), P_CDFPDF, SCIPY_WEIBULL_CDF_1_1, 1e-12);
    checkCDF(new WeibullDistribution(2., 1.), P_CDFPDF, SCIPY_WEIBULL_CDF_2_1, 1e-12);
    checkCDF(new WeibullDistribution(4., 1.), P_CDFPDF, SCIPY_WEIBULL_CDF_4_1, 1e-11);
    checkCDF(new WeibullDistribution(4., 10.), P_CDFPDF, SCIPY_WEIBULL_CDF_4_10, 1e-11);
    checkCDF(new WeibullDistribution(.1, 1.), P_CDFPDF, SCIPY_WEIBULL_CDF_01_1, 1e-13);
    checkCDF(new WeibullDistribution(.1, 4.), P_CDFPDF, SCIPY_WEIBULL_CDF_01_4, 1e-13);
    checkCDF(new WeibullDistribution(.1, 10.), P_CDFPDF, SCIPY_WEIBULL_CDF_01_10, 1e-13);
    checkCDF(new WeibullDistribution(.1, 20.), P_CDFPDF, SCIPY_WEIBULL_CDF_01_20, 1e-13);

    checkCDF(new WeibullDistribution(1., 1.), P_CDFPDF, GNUR_WEIBULL_CDF_1_1, 1e-14);
    checkCDF(new WeibullDistribution(2., 1.), P_CDFPDF, GNUR_WEIBULL_CDF_2_1, 1e-14);
    checkCDF(new WeibullDistribution(4., 1.), P_CDFPDF, GNUR_WEIBULL_CDF_4_1, 1e-13);
    checkCDF(new WeibullDistribution(4., 10.), P_CDFPDF, GNUR_WEIBULL_CDF_4_10, 1e-13);
    checkCDF(new WeibullDistribution(.1, 1.), P_CDFPDF, GNUR_WEIBULL_CDF_01_1, 1e-15);
    checkCDF(new WeibullDistribution(.1, 4.), P_CDFPDF, GNUR_WEIBULL_CDF_01_4, 1e-15);
    checkCDF(new WeibullDistribution(.1, 10.), P_CDFPDF, GNUR_WEIBULL_CDF_01_10, 1e-15);
    checkCDF(new WeibullDistribution(.1, 20.), P_CDFPDF, GNUR_WEIBULL_CDF_01_20, 1e-15);
  }

  @Test
  public void testQuantile() {
    checkQuantile(new WeibullDistribution(1., 1.), P_QUANT, SCIPY_WEIBULL_QUANT_1_1, 1e-15);
    checkQuantile(new WeibullDistribution(2., 1.), P_QUANT, SCIPY_WEIBULL_QUANT_2_1, 1e-15);
    checkQuantile(new WeibullDistribution(4., 1.), P_QUANT, SCIPY_WEIBULL_QUANT_4_1, 1e-13);
    checkQuantile(new WeibullDistribution(4., 10.), P_QUANT, SCIPY_WEIBULL_QUANT_4_10, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 1.), P_QUANT, SCIPY_WEIBULL_QUANT_01_1, 1e-14);
    checkQuantile(new WeibullDistribution(.1, 4.), P_QUANT, SCIPY_WEIBULL_QUANT_01_4, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 10.), P_QUANT, SCIPY_WEIBULL_QUANT_01_10, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 20.), P_QUANT, SCIPY_WEIBULL_QUANT_01_20, 1e-13);

    checkQuantile(new WeibullDistribution(1., 1.), P_QUANT, GNUR_WEIBULL_QUANT_1_1, 1e-15);
    checkQuantile(new WeibullDistribution(2., 1.), P_QUANT, GNUR_WEIBULL_QUANT_2_1, 1e-15);
    checkQuantile(new WeibullDistribution(4., 1.), P_QUANT, GNUR_WEIBULL_QUANT_4_1, 1e-13);
    checkQuantile(new WeibullDistribution(4., 10.), P_QUANT, GNUR_WEIBULL_QUANT_4_10, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 1.), P_QUANT, GNUR_WEIBULL_QUANT_01_1, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 4.), P_QUANT, GNUR_WEIBULL_QUANT_01_4, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 10.), P_QUANT, GNUR_WEIBULL_QUANT_01_10, 1e-13);
    checkQuantile(new WeibullDistribution(.1, 20.), P_QUANT, GNUR_WEIBULL_QUANT_01_20, 1e-13);
  }
}
