package cabf_br

/*
 * ZLint Copyright 2024 Regents of the University of Michigan
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy
 * of the License at http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License.
 */

import (
	"github.com/zmap/zcrypto/x509"
	"github.com/zmap/zlint/v3/lint"
	"github.com/zmap/zlint/v3/util"
)

type rootCAContainsEKU struct{}

/************************************************
BRs: 7.1.2.1d extendedKeyUsage
This extension MUST NOT be present.
************************************************/

func init() {
	lint.RegisterCertificateLint(&lint.CertificateLint{
		LintMetadata: lint.LintMetadata{
			Name:          "e_root_ca_extended_key_usage_present",
			Description:   "Root CA Certificate: extendedKeyUsage MUST NOT be present.t",
			Citation:      "BRs: 7.1.2.1",
			Source:        lint.CABFBaselineRequirements,
			EffectiveDate: util.CABEffectiveDate,
		},
		Lint: NewRootCAContainsEKU,
	})
}

func NewRootCAContainsEKU() lint.LintInterface {
	return &rootCAContainsEKU{}
}

func (l *rootCAContainsEKU) CheckApplies(c *x509.Certificate) bool {
	return util.IsRootCA(c)
}

func (l *rootCAContainsEKU) Execute(c *x509.Certificate) *lint.LintResult {
	if util.IsExtInCert(c, util.EkuSynOid) {
		return &lint.LintResult{Status: lint.Error}
	} else {
		return &lint.LintResult{Status: lint.Pass}
	}
}
