/* 
 * File: wejpscroller.c
 *
 * A gtk2 text scroll widget
 * Copyright (c) 2005 Johannes Heimansberg
 *
 * Released under the GNU General Public License v2
 */

#include "wejpscroller.h"
#include <string.h>
#define MAX_HEIGHT 30
#define MAX_WIDTH  2000

void wejpscroller_scroll_one_pixel(WejpScroller *ws)
{
	if (ws->extents.width <= ws->drawing_area->allocation.width && 
	    ws->smart_scrolling)
		ws->pos = ws->width;
	else if (ws->scroll == TRUE)
		ws->pos++;
	if (ws->pos > ws->extents.width + ws->drawing_area->allocation.width)
		ws->pos = 0;
}

static gboolean cb_timer(gpointer userdata)
{
	gboolean result = TRUE;

	if (GTK_IS_WIDGET(userdata)) {
		WejpScroller *ws = WEJPSCROLLER(userdata);

		if (ws->shortmessage_timeout > 0) {
			ws->shortmessage_timeout--;
		} else {
			if (ws->shortmessage_pos_backup > 0) {
				ws->pos = ws->shortmessage_pos_backup;
				ws->shortmessage_pos_backup = -1;
				if (strlen(ws->shortmessage_text_backup) > 0) {
					wejpscroller_set_text(ws, ws->shortmessage_text_backup, FALSE);
				}
			}
			if (WEJPSCROLLER(ws)->scroll)
				wejpscroller_scroll_one_pixel(ws);
		}
		wejpscroller_draw(ws);
	} else {
		result = FALSE;
	}
	return result;
}

GtkWidget *wejpscroller_new(guint scroll_delay, gboolean smart)
{
	GtkWidget *ws = GTK_WIDGET(g_object_new(WEJPSCROLLER_TYPE, NULL));
	WEJPSCROLLER(ws)->scroll_delay = scroll_delay;
	WEJPSCROLLER(ws)->smart_scrolling = smart;
	if (scroll_delay != 0) {
		WEJPSCROLLER(ws)->scroll = TRUE;
	} else {
		WEJPSCROLLER(ws)->scroll = FALSE;
	}
	WEJPSCROLLER(ws)->shortmessage_timeout = 0;
	WEJPSCROLLER(ws)->shortmessage_text_backup[0] = '\0';
	if (WEJPSCROLLER(ws)->scroll)
		g_timeout_add(WEJPSCROLLER(ws)->scroll_delay, &cb_timer, ws);
	else
		g_timeout_add(100, &cb_timer, ws);
	return ws;
}

static void wejpscroller_set_text_without_backup(WejpScroller *ws,
                                                 gchar        *new_text,
                                                 gboolean      reset_pos)
{
	GdkColor *text_color, *bg_color;

	gdk_window_show(ws->drawing_area->window);

	bg_color = &ws->drawing_area->style->bg[GTK_WIDGET_STATE(ws->drawing_area)];

	text_color = &ws->drawing_area->style->text[GTK_WIDGET_STATE(ws->drawing_area)];

	gdk_gc_set_foreground(ws->drawing_area->
	                      	style->fg_gc[GTK_WIDGET_STATE(ws->drawing_area)], 
	                      bg_color);

	if (ws->pixmap == NULL)
		ws->pixmap = gdk_pixmap_new(GDK_DRAWABLE(ws->drawing_area->window), 
		                            MAX_WIDTH, MAX_HEIGHT, -1);

	gdk_draw_rectangle(GDK_DRAWABLE(ws->pixmap),
	                   ws->drawing_area->
	                   	style->fg_gc[GTK_WIDGET_STATE(ws->drawing_area)],
	                   TRUE,
	                   0, 0, MAX_WIDTH, MAX_HEIGHT);

	gdk_gc_set_foreground(ws->drawing_area->style->
	                      	fg_gc[GTK_WIDGET_STATE(ws->drawing_area)],
	                      text_color);

	pango_layout_set_text(ws->pgl, new_text, strlen(new_text));

	pango_layout_get_pixel_extents(ws->pgl, NULL, &ws->extents);

	gdk_draw_layout(GDK_DRAWABLE(ws->pixmap),
	                ws->drawing_area->
	                	style->fg_gc[GTK_WIDGET_STATE(ws->drawing_area)], 
	                ws->drawing_area->allocation.width, 0, ws->pgl);
	if (reset_pos) {
		if (ws->scroll_delay > 0)
			ws->pos = 0;
		else
			ws->pos = ws->width;
	}
}

void wejpscroller_set_text(WejpScroller *ws,
                           gchar        *new_text,
                           gboolean      reset_pos)
{
	g_strlcpy(ws->shortmessage_text_backup, new_text, 
	          sizeof(ws->shortmessage_text_backup));
	wejpscroller_set_text_without_backup(ws, new_text, reset_pos);
}

const gchar *wejpscroller_get_text(WejpScroller *ws)
{
	return pango_layout_get_text(ws->pgl);
}

void wejpscroller_set_dimensions(WejpScroller *ws, gint w, gint h)
{
	gint height;

	ws->width  = (w > 0) ? w : -1;
	ws->height = (h > 0) ? h : -1;
	height     = (ws->height > MAX_HEIGHT) ? MAX_HEIGHT : ws->height;

	gtk_widget_set_size_request(ws->drawing_area, 
	                            ws->width,
	                            ws->height);
	gtk_widget_set_size_request(GTK_WIDGET(ws), 
	                            ws->width,
	                            ws->height);
	if (height > 2)
		pango_font_description_set_absolute_size(ws->pfd, (height-2)*PANGO_SCALE);
	pango_font_description_set_family_static(ws->pfd, ws->font);
	pango_layout_set_font_description(ws->pgl, ws->pfd);
}

void wejpscroller_set_font(WejpScroller *ws, gchar *font, gint fontsize)
{
	strncpy(ws->font, font, FONT_STR_LENGTH);
	pango_font_description_set_family_static(ws->pfd, ws->font);
	if (fontsize > 0)
		pango_font_description_set_absolute_size(ws->pfd, fontsize*PANGO_SCALE);
	pango_layout_set_font_description(ws->pgl, ws->pfd);
}

void wejpscroller_shortmessage_show(WejpScroller *ws,
                                    gchar        *message,
                                    guint         timeout)
{
	ws->shortmessage_timeout = timeout;
	ws->shortmessage_pos_backup = ws->pos;
	ws->pos = ws->width;
	if (strlen(ws->shortmessage_text_backup) > 0) {
		wejpscroller_set_text(ws, ws->shortmessage_text_backup, FALSE);
	}
	g_strlcpy(ws->shortmessage_text_backup, wejpscroller_get_text(ws), 
	          sizeof(ws->shortmessage_text_backup));
	wejpscroller_set_text_without_backup(ws, message, FALSE);
}

static void wejpscroller_class_init(WejpScrollerClass *class)
{
}

void wejpscroller_draw(WejpScroller *ws)
{
	GdkRectangle rect;

	rect.x = 0;
	rect.y = 0;
	rect.width  = ws->drawing_area->allocation.width;
	rect.height = ws->drawing_area->allocation.height;
	gtk_widget_draw(ws->drawing_area, &rect);
}

static gboolean expose_event_callback(GtkWidget      *widget,
                                      GdkEventExpose *event,
                                      gpointer        data)
{
	WejpScroller *ws = WEJPSCROLLER(widget->parent);

	if (ws->pixmap != NULL)
		gdk_draw_drawable(widget->window,
		                  ws->drawing_area->style->
		                  	fg_gc[GTK_WIDGET_STATE(ws->drawing_area)],
		                  GDK_DRAWABLE(ws->pixmap),
		                  ws->pos, 0, 0, 0,
		                  ws->drawing_area->allocation.width,
		                  ws->drawing_area->allocation.height);
	return TRUE;
}

static gboolean cb_button_press(GtkWidget      *widget,
                                GdkEventButton *event,
                                gpointer        data)
{
	WEJPSCROLLER(data)->scroll = FALSE;
	WEJPSCROLLER(data)->mouse_x = event->x;
	return TRUE;
}

static gboolean cb_button_release(GtkWidget      *widget,
                                  GdkEventButton *event,
                                  gpointer        data)
{
	if (WEJPSCROLLER(data)->scroll_delay > 0)
		WEJPSCROLLER(data)->scroll = TRUE;
	return TRUE;
}

static gboolean cb_motion_notify_event(GtkWidget      *widget,
                                       GdkEventMotion *event,
                                       gpointer        data)
{
	gint             x, y, tmp;
	GdkModifierType  state;
	WejpScroller    *ws = WEJPSCROLLER(data);

	if (event->is_hint) {
		gdk_window_get_pointer(event->window, &x, &y, &state);
	} else {
		x = event->x;
		y = event->y;
		state = event->state;
	}

	if (state & GDK_BUTTON1_MASK && ws->scroll == FALSE) {
		tmp = ws->mouse_x - x;
		if ((gint)ws->pos + tmp > 0 && 
		    (gint)ws->pos + tmp < 
		    ws->extents.width + ws->drawing_area->allocation.width) {
			ws->pos = ws->pos + tmp;
		} else if ((gint)ws->pos + tmp > 
		           ws->extents.width + ws->drawing_area->allocation.width) {
			ws->pos = 0;
		} else {
			ws->pos = ws->extents.width + ws->drawing_area->allocation.width;
		}
		ws->mouse_x = x;
	}
	return TRUE;
}

static void wejpscroller_init(WejpScroller *ws)
{
	ws->pos = 0;
	ws->pixmap = NULL;
	ws->drawing_area = gtk_drawing_area_new();
	gtk_container_add(GTK_CONTAINER(ws), ws->drawing_area);

	gtk_widget_show_all(GTK_WIDGET(ws));
	ws->colormap = gdk_colormap_new(gdk_visual_get_best(), FALSE);

	ws->pgl = pango_layout_new(gdk_pango_context_get_for_screen(
	                           gdk_screen_get_default()));

	ws->pfd = 
		pango_context_get_font_description(pango_layout_get_context(ws->pgl));

	/*pango_font_description_set_absolute_size(ws->pfd, 12 * PANGO_SCALE);*/
	pango_font_description_set_family_static(ws->pfd, "Sans Serif");
	pango_layout_set_font_description(ws->pgl, ws->pfd);

	g_signal_connect(G_OBJECT(ws->drawing_area), "expose_event",  
	                 G_CALLBACK(expose_event_callback), NULL);
	g_signal_connect(G_OBJECT(ws), "button_press_event",
	                 G_CALLBACK(cb_button_press), ws);
	g_signal_connect(G_OBJECT(ws), "button_release_event",
	                 G_CALLBACK(cb_button_release), ws);
	g_signal_connect(G_OBJECT(ws), "motion_notify_event",
	                 G_CALLBACK(cb_motion_notify_event), ws);
}

GType wejpscroller_get_type(void)
{
	static GType ws_type = 0;

	if (!ws_type)
	{
		static const GTypeInfo ws_info =
		{
			sizeof(WejpScrollerClass),
			NULL, /* base_init */
			NULL, /* base_finalize */
			(GClassInitFunc) wejpscroller_class_init,
			NULL, /* class_finalize */
			NULL, /* class_data */
			sizeof(WejpScroller),
			0,    /* n_preallocs */
			(GInstanceInitFunc) wejpscroller_init,
		};

		ws_type = g_type_register_static(GTK_TYPE_EVENT_BOX,
		                                 "WejpScroller",
		                                 &ws_info,
		                                 0);
	}
	return ws_type;
}
