/* -*- c++ -*- ----------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   https://www.lammps.org/, Sandia National Laboratories
   LAMMPS development team: developers@lammps.org

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#ifdef PAIR_CLASS
// clang-format off
PairStyle(dpd/ext/tstat/kk,PairDPDExtTstatKokkos<LMPDeviceType>);
PairStyle(dpd/ext/tstat/kk/device,PairDPDExtTstatKokkos<LMPDeviceType>);
PairStyle(dpd/ext/tstat/kk/host,PairDPDExtTstatKokkos<LMPHostType>);
// clang-format off
#else

#ifndef LMP_PAIR_DPD_EXT_TSTAT_KOKKOS_H
#define LMP_PAIR_DPD_EXT_TSTAT_KOKKOS_H

#include "pair_dpd_ext_tstat.h"
#include "pair_kokkos.h"
#include "kokkos_type.h"

#if !defined(DPD_USE_RAN_MARS) && !defined(DPD_USE_Random_XorShift64) && !defined(Random_XorShift1024)
#define DPD_USE_Random_XorShift64
#endif

#ifdef DPD_USE_RAN_MARS
#include "rand_pool_wrap_kokkos.h"
#else
#include "Kokkos_Random.hpp"
#endif

namespace LAMMPS_NS {

template<class DeviceType>
class PairDPDExtTstatKokkos : public PairDPDExtTstat {
 public:
  typedef DeviceType device_type;
  typedef ArrayTypes<DeviceType> AT;
  typedef EV_FLOAT value_type;

  PairDPDExtTstatKokkos(class LAMMPS*);
  ~PairDPDExtTstatKokkos() override;

  void allocate() override;

  void init_style() override;
  double init_one(int i, int j) override;
  void compute(int, int) override;

  struct params_dpd {
    KOKKOS_INLINE_FUNCTION
    params_dpd() {cut=ws=wsT=gamma=sigma=gammaT=sigmaT=0;}
    KOKKOS_INLINE_FUNCTION
    params_dpd(int /*i*/) {cut=ws=wsT=gamma=sigma=gammaT=sigmaT=0;}
    KK_FLOAT cut,ws,wsT,gamma,sigma,gammaT,sigmaT;
  };

  template<int NEIGHFLAG, int VFLAG>
  struct TagDPDExtTstatKokkos{};

  template<int NEIGHFLAG, int VFLAG>
  KOKKOS_INLINE_FUNCTION
  void operator () (TagDPDExtTstatKokkos<NEIGHFLAG,VFLAG>, const int &i) const;

  template<int NEIGHFLAG, int VFLAG>
  KOKKOS_INLINE_FUNCTION
  void operator () (TagDPDExtTstatKokkos<NEIGHFLAG,VFLAG>, const int &i, EV_FLOAT&) const;

  template<int NEIGHFLAG>
  KOKKOS_INLINE_FUNCTION
  void v_tally_xyz(EV_FLOAT &ev, const int &i, const int &j,
      const KK_FLOAT &fx,const KK_FLOAT &fy, const KK_FLOAT &fz,
      const KK_FLOAT &delx,const KK_FLOAT &dely, const KK_FLOAT &delz) const;
 private:
  KK_FLOAT special_lj[4], special_rf[4];
  int eflag,vflag;
  int neighflag,nlocal;
  KK_FLOAT dtinvsqrt;

  int need_dup;

  using KKDeviceType = typename KKDevice<DeviceType>::value;

  template<typename DataType, typename Layout>
  using DupScatterView = KKScatterView<DataType, Layout, KKDeviceType, KKScatterSum, KKScatterDuplicated>;

  template<typename DataType, typename Layout>
  using NonDupScatterView = KKScatterView<DataType, Layout, KKDeviceType, KKScatterSum, KKScatterNonDuplicated>;

  DupScatterView<KK_ACC_FLOAT*[3], typename DAT::t_kkacc_1d_3::array_layout> dup_f;
  DupScatterView<KK_ACC_FLOAT*[6], typename DAT::t_kkacc_1d_6::array_layout> dup_vatom;
  NonDupScatterView<KK_ACC_FLOAT*[3], typename DAT::t_kkacc_1d_3::array_layout> ndup_f;
  NonDupScatterView<KK_ACC_FLOAT*[6], typename DAT::t_kkacc_1d_6::array_layout> ndup_vatom;

#ifdef DPD_USE_RAN_MARS
  RandPoolWrap rand_pool;
  typedef RandWrap rand_type;
#elif defined(DPD_USE_Random_XorShift64)
  Kokkos::Random_XorShift64_Pool<DeviceType> rand_pool;
  typedef typename Kokkos::Random_XorShift64_Pool<DeviceType>::generator_type rand_type;
#elif defined(DPD_USE_Random_XorShift1024)
  Kokkos::Random_XorShift1024_Pool<DeviceType> rand_pool;
  typedef typename Kokkos::Random_XorShift1024_Pool<DeviceType>::generator_type rand_type;
#endif
  typename AT::t_kkfloat_1d_3_lr_randomread x;
  typename AT::t_kkfloat_1d_3_randomread v;
  typename AT::t_kkacc_1d_3 f;
  typename AT::t_int_1d_randomread type;

  typename AT::t_neighbors_2d d_neighbors;
  typename AT::t_int_1d_randomread d_ilist;
  typename AT::t_int_1d_randomread d_numneigh;

  DAT::ttransform_kkfloat_2d k_cutsq;
  typename AT::t_kkfloat_2d d_cutsq;

  Kokkos::DualView<params_dpd**,Kokkos::LayoutRight,DeviceType> k_params;
  typename Kokkos::DualView<params_dpd**,
    Kokkos::LayoutRight,DeviceType>::t_dev_const_um params;

  DAT::ttransform_kkacc_1d_6 k_vatom;
  typename AT::t_kkacc_1d_6 d_vatom;

  KOKKOS_INLINE_FUNCTION
  int sbmask(const int& j) const;
  friend void pair_virial_fdotr_compute<PairDPDExtTstatKokkos>(PairDPDExtTstatKokkos*);

};
}
#endif
#endif
