/*****************************************************************************
   xdwapi.h - header file of the xdesktopwaves API

   Copyright (C) 2004 Oliver Hamann (olha@users.sourceforge.net)

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*****************************************************************************/

#ifndef XDWAPI_H
#define XDWAPI_H

#ifdef __cplusplus
extern "C" {
#endif

#ifndef _XLIB_H_
#include <X11/Xlib.h>
#endif


typedef void * xdwApiContext;


xdwApiContext xdwApiOpen(Display * display);
	/* Open the API for a given display.
	 * Parameters:
	 *   display    - The X display.
	 * Return value:
	 *   A context for the API. It is required for the other functions.
	 */

void xdwApiClose(xdwApiContext context);
	/* Close the API.
	 * Parameters:
	 *   context  - The API context.
	 */

Bool xdwApiPing(xdwApiContext context);
	/* This sends a command which performs nothing, just for testing the
	 * command sending.
	 * Parameters:
	 *   context  - The API context.
	 * Return value:
	 *   True on success, False on error (xdesktopwaves probably not
	 *   running).
	 */

Bool xdwApiClear(xdwApiContext context);
	/* Clear the water from all waves.
	 * Parameters:
	 *   context  - The API context.
	 * Return value:
	 *   True on success, False on error (xdesktopwaves probably not
	 *   running).
	 */

Bool xdwApiSetRain(xdwApiContext context, int intensity);
	/* Set intensity of the rain feature.
	 * Parameters:
	 *   context   - The API context.
	 *   intensity - Intensity of the rain. Valid range: 0 to 10
	 * Return value:
	 *   True on success, False on error (xdesktopwaves probably not
	 *   running).
	 */

Bool xdwApiSetStorm(xdwApiContext context, int intensity);
	/* Set intensity of the storm feature.
	 * Parameters:
	 *   context   - The API context.
	 *   intensity - Intensity of the storm. Valid range: 0 to 10
	 * Return value:
	 *   True on success, False on error (xdesktopwaves probably not
	 *   running).
	 */

Bool xdwApiPutRaindrop(xdwApiContext context, int x, int y, int volume);
	/* Put one raindrop.
	 * Parameters:
	 *   context   - The API context.
	 *   x,y       - X- and Y-coordinates of the raindrop, in pixel units
	 *               relative to the root window origin.
	 *   volume    - Volume of the raindrop. Valid range: 0 to 100
	 * Return value:
	 *   True on success, False on error (xdesktopwaves probably not
	 *   running).
	 */

int xdwApiPutGhostRect(xdwApiContext context, int milliSecs, int removeId,
                       int x, int y, int width, int height);
int xdwApiPutGhostCircle(xdwApiContext context, int milliSecs, int removeId,
                        int x, int y, int radius);
	/* Put a temporary custom rectangular or circular "ghost" onto the
	 * water. It generates waves like a window (ship or shore), but it is
	 * not drawn. The ghost is automatically removed after a given lifetime.
	 * Additionally, putting a ghost can remove another ghost. This way a
	 * ghost can be animated by substituting it again and again.
	 * Parameters:
	 *   context   - The API context.
	 *   milliSecs - Lifetime of the ghost in milliseconds. The ghost is
	 *               removed automatically after this time has passed.
	 *               Valid range: 0 to 32767
	 *   removeId  - Identification of another ghost to be removed, or 0 for
	 *               not removing any ghost.
	 *   x,y       - Coordinates in pixel units relative to the root window
	 *               origin. It is the upper-left corner of the rectangle
	 *               respectively the center of the circle.
	 *   width     - Width of the rectangle in pixel units.
	 *   height    - Height of the rectangle in pixel units.
	 *   radius    - Radius of the circle in pixel units.
	 * Return value:
	 *   The identification for the new ghost, or 0 on error. The
	 *   identification comes from a pseudo random number generator. Most
	 *   likely it is unique, but there is no guarantee.
	 */

void xdwApiSetWavesByWindow(xdwApiContext context, Window window,
                            Bool wavesByWindow);
	/* Override the waves-by-windows option for a given window.
	 * Parameters:
	 *   context        - The API context.
	 *   window         - The window.
	 *   wavesByWindow  - True if waves are to be generated by the window,
	 *                    False if not.
	 */

void xdwApiDefaultWavesByWindow(xdwApiContext context, Window window);
	/* Remove the overriding of the waves-by-windows option for a given
	 * window.
	 * Parameters:
	 *   context        - The API context.
	 *   window         - The window.
	 */


#ifdef __cplusplus
}
#endif

#endif
