#include <chrono>
#include "reportevent.h"
#include "logger.h"
#include "common.h"

ReportEvent::ReportEvent()
    : m_busproxy(nullptr)
    , m_mainloop(nullptr)
    , m_pthrd(0)
{
}

ReportEvent::~ReportEvent()
{
    Stop();
    FreeBusProxy();
}

bool ReportEvent::Init()
{
    if (m_busproxy == nullptr) {
        if (!GetBusProxy()) {
            return false;
        }
    }

    // 创建线程，循环检测
    int ret = pthread_create(&m_pthrd, nullptr, &ReportEvent::LoopWatch, this);
    if (ret != 0) {
        Logger::get_mutable_instance().Print("create thread failed");
        return false;
    }

    return true;
}

//void ReportEvent::InitAsync(const std::function<void(bool)> &cb)
//{
//    m_callback = cb;
//    int ret;

//    // 创建线程，循环检测
//    ret = pthread_create(&m_pthrd, nullptr, &ReportEvent::LoopWatch, this);

//    if (ret != 0) {
//        Logger::get_mutable_instance().Print("create thread failed");
//        if (m_callback)
//            m_callback(false);
//    } else {
//        if (m_busproxy == nullptr) {
//            GetBusProxyAsync(this);
//        } else {
//            if (m_callback)
//                m_callback(true);
//        }
//    }
//}

void ReportEvent::Stop()
{
    if (m_mainloop)
        g_main_loop_quit(m_mainloop);

    if (m_pthrd != 0)
        pthread_join(m_pthrd, nullptr);
}

// 异步发送
void ReportEvent::PostEvent(const string &packagename, const string &evdata)
{
    if (m_busproxy)
        event_log1_call_report_event_log(m_busproxy, packagename.c_str(), evdata.c_str(), nullptr, nullptr, nullptr);
}

// 同步发送
void ReportEvent::SendEvent(const string &packagename, const string &evdata)
{
    if (m_busproxy)
        event_log1_call_report_event_log_sync(m_busproxy, packagename.c_str(), evdata.c_str(), nullptr, nullptr);
}

//void ReportEvent::GetBusProxyAsync(void *p)
//{
//    event_log1_proxy_new_for_bus(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
//                                 SERVICE_NAME, OBJECT_PATH, nullptr, &ReportEvent::OnAsyncInitableReady, p);
//}

//void ReportEvent::OnAsyncInitableReady(GObject *source_object, GAsyncResult *res, gpointer user_data)
//{
//    bool result = false;
//    GError *error = nullptr;

//    ReportEvent *preport = reinterpret_cast<ReportEvent*>(user_data);
//    if (preport) {
//        preport->m_busproxy = event_log1_proxy_new_for_bus_finish(res, &error);

//        if (error) {
//            Logger::get_mutable_instance().Print("get dbus object async error:%s", error->message);
//            g_error_free(error);
//        } else {
//            result = true;
//        }

//        if (preport->m_callback)
//            preport->m_callback(result);
//    }
//}

bool ReportEvent::GetBusProxy()
{
    GError *error = nullptr;
    m_busproxy = event_log1_proxy_new_for_bus_sync(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
                                                   SERVICE_NAME, OBJECT_PATH, nullptr, &error);

    if (error) {
        Logger::get_mutable_instance().Print("get dbus error:%s", error->message);
        g_error_free(error);
        return false;
    }

    return true;
}

void ReportEvent::FreeBusProxy()
{
    if (m_busproxy) {
        g_object_unref(m_busproxy);
        m_busproxy = nullptr;
    }
}

// 服务出现
void ReportEvent::OnNameAppeared(GDBusConnection *connection, const gchar *name, const gchar *name_owner, gpointer user_data)
{
    Logger::get_mutable_instance().Print("service:%s appear, owner:%s", name, name_owner);

    ReportEvent *preport = reinterpret_cast<ReportEvent*>(user_data);
    if (preport) {
        if (preport->m_busproxy == nullptr)
            preport->GetBusProxy();
    }
}

// 服务消失
void ReportEvent::OnNameVanished(GDBusConnection *connection, const gchar *name, gpointer user_data)
{
    Logger::get_mutable_instance().Print("service:%s disappear", name);

    ReportEvent *preport = reinterpret_cast<ReportEvent*>(user_data);
    if (preport)
        preport->FreeBusProxy();
}

void* ReportEvent::LoopWatch(void *p)
{
    ReportEvent *preport = reinterpret_cast<ReportEvent*>(p);
    if (preport) {
        // 对目标 D-BUS 进行监控
        guint watchid = g_bus_watch_name(G_BUS_TYPE_SYSTEM, SERVICE_NAME, G_BUS_NAME_WATCHER_FLAGS_AUTO_START,
                                         &ReportEvent::OnNameAppeared, &ReportEvent::OnNameVanished, preport, nullptr);

        preport->m_mainloop = g_main_loop_new(nullptr, FALSE);
        g_main_loop_run(preport->m_mainloop);
        g_bus_unwatch_name(watchid);
        g_main_loop_unref(preport->m_mainloop);
    }

    return nullptr;
}
