// SPDX-FileCopyrightText: 2024 UnionTech Software Technology Co., Ltd.
//
// SPDX-License-Identifier: GPL-3.0-or-later

#include "systemenv.h"

#include <QFile>
#include <QDebug>
#include <QLibrary>
#include <QFileInfo>

#include <iostream>

GLOBAL_USE_NAMESPACE

QStringList SystemEnv::cpuInstructions()
{
    QStringList rets;
    QFile file("/proc/cpuinfo");
    if (!file.open(QFile::ReadOnly))
        return rets;

    QTextStream stream(&file);
    stream.skipWhiteSpace();

    while (!stream.atEnd()) {
        QString line = stream.readLine();
        if (line.startsWith("flags")) {
            QStringList all = line.split(" ");

            // remove label "flags\t\t:"
            if (all.size() > 0)
                all.removeAt(0);

            rets = all;
            break;
        }
    }

    file.close();
    return rets;
}

QString SystemEnv::cpuModelName()
{
    QString ret;
    QFile file("/proc/cpuinfo");
    if (!file.open(QFile::ReadOnly))
        return ret;

    QTextStream stream(&file);
    stream.skipWhiteSpace();

    while (!stream.atEnd()) {
        QString line = stream.readLine();
        if (line.startsWith("model name")) {
            QStringList all = line.split(":");
            if (all.size() > 1)
                ret = all.at(1);
            break;
        }
    }

    file.close();
    return ret;
}

QString SystemEnv::vga()
{
    char cmd[512] = {0};
    snprintf(cmd, sizeof(cmd), "lspci|grep VGA");
    FILE* pipe = popen(cmd, "r");
    if (!pipe) {
        std::cerr << "ERROR: lspci|grep VGA " << std::endl;
        return "";
    }

    char buffer[512];
    while (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
        if (strstr(buffer, "VGA")) {
            QString str(buffer);
            pclose(pipe);
            return str;
        }
    }

    pclose(pipe);
    return QString();
}

bool SystemEnv::checkLibrary(const QString &lib)
{
    char cmd[512] = {0};
    snprintf(cmd, sizeof(cmd), "ldd %s", lib.toStdString().c_str());
    FILE* pipe = popen(cmd, "r");
    if (!pipe) {
        std::cerr << "ERROR: ldd " << lib.toStdString() << std::endl;
        return false;
    }

    char buffer[512];
    while (fgets(buffer, sizeof(buffer), pipe) != nullptr) {
        if (strstr(buffer, "not found")) {
            std::cerr << QFileInfo(lib).fileName().toStdString() << ":" << buffer << std::endl;
            pclose(pipe);
            return false;
        }
    }

    pclose(pipe);
    return true;
}
