#!/bin/sh

die () {
	echo "$@" >&2
	exit 1
}

command_list () {
	while read cmd rest
	do
		case "$cmd" in
		"#"* | '')
			# Ignore comments and allow empty lines
			continue
			;;
		*)
			case "$exclude_programs" in
			*":$cmd:"*)
				;;
			*)
				echo "$cmd $rest"
				;;
			esac
		esac
	done <"$1"
}

category_list () {
	echo "$1" |
	cut -d' ' -f2- |
	tr ' ' '\012' |
	grep -v '^$' |
	LC_ALL=C sort -u
}

define_categories () {
	echo
	echo "/* Command categories */"
	bit=0
	echo "$1" |
	while read cat
	do
		echo "#define CAT_$cat (1UL << $bit)"
		bit=$(($bit+1))
	done
	test "$bit" -gt 32 && die "Urgh.. too many categories?"
}

define_category_names () {
	echo
	echo "/* Category names */"
	echo "static const char *category_names[] = {"
	bit=0
	echo "$1" |
	while read cat
	do
		echo "	\"$cat\", /* (1UL << $bit) */"
		bit=$(($bit+1))
	done
	echo "	NULL"
	echo "};"
}

print_command_list () {
	echo "static struct cmdname_help command_list[] = {"

	echo "$2" |
	while read cmd rest
	do
		synopsis=
		while read line
		do
			case "$line" in
			"$cmd - "*)
				synopsis=${line#$cmd - }
				break
				;;
			esac
		done <"$1/Documentation/$cmd.adoc"

		printf '\t{ "%s", N_("%s"), 0' "$cmd" "$synopsis"
		printf " | CAT_%s" $rest
		echo " },"
	done
	echo "};"
}

exclude_programs=:
while test "--exclude-program" = "$1"
do
	shift
	exclude_programs="$exclude_programs$1:"
	shift
done

if test "$#" -ne 2
then
	die "USAGE: $0 <SOURCE_DIR> <OUTPUT>"
fi

SOURCE_DIR="$1"
OUTPUT="$2"

{
	commands="$(command_list "$SOURCE_DIR"/command-list.txt)"
	categories="$(category_list "$commands")"

	echo "/* Automatically generated by generate-cmdlist.sh */
	struct cmdname_help {
		const char *name;
		const char *help;
		uint32_t category;
	};
	"
	define_categories "$categories"
	echo
	define_category_names "$categories"
	echo
	print_command_list "$SOURCE_DIR" "$commands"
} >"$OUTPUT"
