/*-
 *   BSD LICENSE
 *
 *   Copyright(c) 2015 Intel Corporation. All rights reserved.
 *
 *   Redistribution and use in source and binary forms, with or without
 *   modification, are permitted provided that the following conditions
 *   are met:
 *
 *	 * Redistributions of source code must retain the above copyright
 *	   notice, this list of conditions and the following disclaimer.
 *	 * Redistributions in binary form must reproduce the above copyright
 *	   notice, this list of conditions and the following disclaimer in
 *	   the documentation and/or other materials provided with the
 *	   distribution.
 *	 * Neither the name of Intel Corporation nor the names of its
 *	   contributors may be used to endorse or promote products derived
 *	   from this software without specific prior written permission.
 *
 *   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *   "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *   LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *   A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *   OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *   SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *   LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *   DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *   THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *   (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *   OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#ifndef TEST_CRYPTODEV_GCM_TEST_VECTORS_H_
#define TEST_CRYPTODEV_GCM_TEST_VECTORS_H_

#define GMAC_LARGE_PLAINTEXT_LENGTH		65376

struct gcm_test_data {
	struct {
		uint8_t data[64];
		unsigned len;
	} key;

	struct {
		uint8_t data[64] __rte_aligned(16);
		unsigned len;
	} iv;

	struct {
		uint8_t data[64];
		unsigned len;
	} aad;

	struct {
		uint8_t data[1024];
		unsigned len;
	} plaintext;

	struct {
		uint8_t data[1024];
		unsigned len;
	} ciphertext;

	struct {
		uint8_t data[16];
		unsigned len;
	} auth_tag;

};

struct gmac_test_data {
	struct {
		uint8_t data[64];
		unsigned len;
	} key;

	struct {
		uint8_t data[64] __rte_aligned(16);
		unsigned len;
	} iv;

	struct {
		uint8_t *data;
		unsigned len;
	} aad;

	struct {
		uint8_t *data;
		unsigned len;
	} plaintext;

	struct {
		uint8_t data[16];
		unsigned len;
	} gmac_tag;

};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_1 = {
	.key = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00 },
		.len = 12
	},
	.aad = {
		.data = { 0 },
		.len = 0
	},
	.plaintext = {
		.data = {
			0x00 },
		.len = 0
	},
	.ciphertext = {
		.data = {
			0x00
		},
		.len = 0
	},
	.auth_tag = {
		.data = {
			0x58, 0xe2, 0xfc, 0xce, 0xfa, 0x7e, 0x30, 0x61,
			0x36, 0x7f, 0x1d, 0x57, 0xa4, 0xe7, 0x45, 0x5a },
		.len = 16
	}
};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_2 = {
	.key = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.len = 16
	},
	.iv = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00 },
		.len = 12
	},
	.aad = {
		.data = { 0 },
		.len = 0
	},
	.plaintext = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.len = 16
	},
	.ciphertext = {
		.data = {
			0x03, 0x88, 0xda, 0xce, 0x60, 0xb6, 0xa3, 0x92,
			0xf3, 0x28, 0xc2, 0xb9, 0x71, 0xb2, 0xfe, 0x78 },
		.len = 16
	},
	.auth_tag = {
		.data = {
			0xab, 0x6e, 0x47, 0xd4, 0x2c, 0xec, 0x13, 0xbd,
			0xf5, 0x3a, 0x67, 0xb2, 0x12, 0x57, 0xbd, 0xdf },
		.len = 16
	}
};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_3 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88 },
		.len = 12
	},
	.aad = {
		.data = { 0 },
		.len = 0
	},
	.plaintext = {
		.data = {
			0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55 },
		.len = 64
	},
	.ciphertext = {
		.data = {
			0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			0x3d, 0x58, 0xe0, 0x91, 0x47, 0x3f, 0x59, 0x85
		},
		.len = 64
	},
	.auth_tag = {
		.data = {
			0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
			0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4 },
		.len = 16
	}
};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_4 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88 },
		.len = 12
	},
	.aad = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 },
		.len = 8
	},
	.plaintext = {
		.data = {
			0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			0xba, 0x63, 0x7b, 0x39
		},
		.len = 60
	},
	.ciphertext = {
		.data = {
			0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			0x3d, 0x58, 0xe0, 0x91
		},
		.len = 60
	},
	.auth_tag = {
		.data = {
			0xA2, 0xA4, 0x35, 0x75, 0xDC, 0xB0, 0x57, 0x74,
			0x07, 0x02, 0x30, 0xC2, 0xE7, 0x52, 0x02, 0x00
		},
		.len = 16
	}

};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_5 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88 },
		.len = 12
	},
	.aad = {
		.data = {
			0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef },
		.len = 8
	},
	.plaintext = {
		.data = {
			0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			0xba, 0x63, 0x7b, 0x39
		},
		.len = 60
	},
	.ciphertext = {
		.data = {
			0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			0x3d, 0x58, 0xe0, 0x91
		},
		.len = 60
	},
	.auth_tag = {
		.data = {
			0xC5, 0x2D, 0xFB, 0x54, 0xAF, 0xBB, 0x07, 0xA1,
			0x9A, 0xFF, 0xBE, 0xE0, 0x61, 0x4C, 0xE7, 0xA5
		},
		.len = 16
	}

};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_6 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88
		},
		.len = 12
	},
	.aad = {
		.data = {
			0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
			0x00, 0x00, 0x00, 0x00
		},
		.len = 12
	},
	.plaintext = {
		.data = {
			0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			0xba, 0x63, 0x7b, 0x39
		},
		.len = 60
	},
	.ciphertext = {
		.data = {
			0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			0x3d, 0x58, 0xe0, 0x91
		},
		.len = 60
	},
	.auth_tag = {
		.data = {
			0x74, 0xFC, 0xFA, 0x29, 0x3E, 0x60, 0xCC, 0x66,
			0x09, 0xD6, 0xFD, 0x00, 0xC8, 0x86, 0xD5, 0x42
		},
		.len = 16
	}
};

/** AES-128 Test Vectors */
static const struct gcm_test_data gcm_test_case_7 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88
		},
		.len = 12
	},
	.aad = {
		.data = {
			0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
			0xfe, 0xed, 0xfa, 0xce
		},
		.len = 12
	},
	.plaintext = {
		.data = {
			0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
			0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
			0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
			0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
			0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
			0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
			0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
			0xba, 0x63, 0x7b, 0x39
		},
		.len = 60
	},
	.ciphertext = {
		.data = {
			0x42, 0x83, 0x1e, 0xc2, 0x21, 0x77, 0x74, 0x24,
			0x4b, 0x72, 0x21, 0xb7, 0x84, 0xd0, 0xd4, 0x9c,
			0xe3, 0xaa, 0x21, 0x2f, 0x2c, 0x02, 0xa4, 0xe0,
			0x35, 0xc1, 0x7e, 0x23, 0x29, 0xac, 0xa1, 0x2e,
			0x21, 0xd5, 0x14, 0xb2, 0x54, 0x66, 0x93, 0x1c,
			0x7d, 0x8f, 0x6a, 0x5a, 0xac, 0x84, 0xaa, 0x05,
			0x1b, 0xa3, 0x0b, 0x39, 0x6a, 0x0a, 0xac, 0x97,
			0x3d, 0x58, 0xe0, 0x91
		},
		.len = 60
	},
	.auth_tag = {
		.data = {
			0xE9, 0xE4, 0xAB, 0x76, 0xB7, 0xFF, 0xEA, 0xDC,
			0x69, 0x79, 0x38, 0xA2, 0x0D, 0xCA, 0xF5, 0x92
		},
		.len = 16
	}
};

/** GMAC Test Vectors */
static uint8_t gmac_plaintext[GMAC_LARGE_PLAINTEXT_LENGTH] = {
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10,
			0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
			0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10
};

static const struct gmac_test_data gmac_test_case_1 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88 },
		.len = 12
	},
	.aad = {
		.data = gmac_plaintext,
		.len = 160
	},
	.plaintext = {
		.data = NULL,
		.len = 0
	},
	.gmac_tag = {
		.data = {
			0x4C, 0x0C, 0x4F, 0x47, 0x2D, 0x78, 0xF6, 0xD8,
			0x03, 0x53, 0x20, 0x2F, 0x1A, 0xDF, 0x90, 0xD0
		},
		.len = 16
	},
};

static const struct gmac_test_data gmac_test_case_2 = {
	.key = {
		.data = {
		    0xaa, 0x74, 0x0a, 0xbf, 0xad, 0xcd, 0xa7, 0x79,
		    0x22, 0x0d, 0x3b, 0x40, 0x6c, 0x5d, 0x7e, 0xc0,
		    0x9a, 0x77, 0xfe, 0x9d, 0x94, 0x10, 0x45, 0x39,
		},
		.len = 24
	},
	.iv = {
		.data = {
		    0xab, 0x22, 0x65, 0xb4, 0xc1, 0x68, 0x95,
		    0x55, 0x61, 0xf0, 0x43, 0x15, },
		.len = 12
	},
	.aad = {
		.data = gmac_plaintext,
		.len = 80
	},
	.plaintext = {
		.data = NULL,
		.len = 0
	},
	.gmac_tag = {
		.data = {
		    0xCF, 0x82, 0x80, 0x64, 0x02, 0x46, 0xF4, 0xFB,
		    0x33, 0xAE, 0x1D, 0x90, 0xEA, 0x48, 0x83, 0xDB
		},
		.len = 16
	},
};

static const struct gmac_test_data gmac_test_case_3 = {
	.key = {
		.data = {
		    0xb5, 0x48, 0xe4, 0x93, 0x4f, 0x5c, 0x64, 0xd3,
		    0xc0, 0xf0, 0xb7, 0x8f, 0x7b, 0x4d, 0x88, 0x24,
		    0xaa, 0xc4, 0x6b, 0x3c, 0x8d, 0x2c, 0xc3, 0x5e,
		    0xe4, 0xbf, 0xb2, 0x54, 0xe4, 0xfc, 0xba, 0xf7,
		},
		.len = 32
	},
	.iv = {
		.data = {
		    0x2e, 0xed, 0xe1, 0xdc, 0x64, 0x47, 0xc7,
		    0xaf, 0xc4, 0x41, 0x53, 0x58,
		},
		.len = 12
	},
	.aad = {
		.data = gmac_plaintext,
		.len = 65
	},
	.plaintext = {
		.data = NULL,
		.len = 0
	},
	.gmac_tag = {
		.data = {
			0x77, 0x46, 0x0D, 0x6F, 0xB1, 0x87, 0xDB, 0xA9,
			0x46, 0xAD, 0xCD, 0xFB, 0xB7, 0xF9, 0x13, 0xA1
		},
		.len = 16
	},
};

/******* GCM PERF VECTORS ***********/

struct cryptodev_perf_test_data {
	struct {
		uint8_t data[64];
		unsigned len;
	} key;

	struct {
		uint8_t data[64] __rte_aligned(16);
		unsigned len;
	} iv;

	struct {
		uint8_t data[64];
		unsigned len;
	} aad;

	struct {
		uint8_t data[2048];
		unsigned len;
	} plaintext;

	struct {
		uint8_t data[2048];
		unsigned len;
	} ciphertext;

	struct {
		uint8_t data[16];
		unsigned len;
	} auth_tag;

	struct {
		uint32_t size;
		uint8_t data[16];
		unsigned len;
	} auth_tags[7];

};

/* 2048B */
static const struct cryptodev_perf_test_data AES_GCM_128_12IV_0AAD = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08 },
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88 },
		.len = 12
	},
	.aad = {
		.data = { 0 },
		.len = 0
	},
	.plaintext = {
		.data = {
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55,
		    0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
		    0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
		    0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
		    0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
		    0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
		    0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
		    0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
		    0xba, 0x63, 0x7b, 0x39, 0x1a, 0xaf, 0xd2, 0x55
		},
		.len = 2048
	},
	.ciphertext = {
		.data = {
		    0x42, 0x83, 0x1E, 0xC2, 0x21, 0x77, 0x74, 0x24,
		    0x4B, 0x72, 0x21, 0xB7, 0x84, 0xD0, 0xD4, 0x9C,
		    0xE3, 0xAA, 0x21, 0x2F, 0x2C, 0x02, 0xA4, 0xE0,
		    0x35, 0xC1, 0x7E, 0x23, 0x29, 0xAC, 0xA1, 0x2E,
		    0x21, 0xD5, 0x14, 0xB2, 0x54, 0x66, 0x93, 0x1C,
		    0x7D, 0x8F, 0x6A, 0x5A, 0xAC, 0x84, 0xAA, 0x05,
		    0x1B, 0xA3, 0x0B, 0x39, 0x6A, 0x0A, 0xAC, 0x97,
		    0x3D, 0x58, 0xE0, 0x91, 0x47, 0x3F, 0x59, 0x85,
		    0x04, 0x99, 0x55, 0xE1, 0x36, 0x76, 0xB7, 0x14,
		    0x1D, 0xF0, 0xF6, 0x8C, 0x65, 0xD5, 0xAD, 0xFB,
		    0x90, 0x7F, 0x5D, 0xA2, 0xD6, 0xFD, 0xD0, 0xE5,
		    0x0D, 0x9B, 0x68, 0x21, 0x49, 0x42, 0x6E, 0x13,
		    0xEC, 0x22, 0x50, 0x2A, 0x30, 0x47, 0x49, 0xA1,
		    0x7F, 0xC3, 0x09, 0xE0, 0x56, 0x91, 0xC4, 0x54,
		    0x70, 0xD7, 0x19, 0x40, 0xCA, 0x6B, 0x65, 0x27,
		    0x3E, 0xE9, 0xD1, 0x0F, 0x1C, 0xB5, 0x45, 0x0D,
		    0x27, 0xE7, 0xCF, 0x94, 0x10, 0xBF, 0xA2, 0xFA,
		    0x86, 0x20, 0x3F, 0x6E, 0xE9, 0x95, 0x03, 0x5A,
		    0x46, 0x11, 0x75, 0xD5, 0x37, 0x71, 0x7F, 0xE0,
		    0xBC, 0x9F, 0xC8, 0xE9, 0xB1, 0x08, 0x2C, 0x59,
		    0x6E, 0x51, 0x4A, 0x83, 0x38, 0xC1, 0xED, 0xE2,
		    0x2E, 0x88, 0x90, 0xA5, 0x7D, 0xA4, 0x93, 0x9A,
		    0x30, 0xD6, 0x96, 0x34, 0x0F, 0xC4, 0xD1, 0x7E,
		    0xC9, 0x8F, 0xC5, 0xBB, 0x80, 0x50, 0x85, 0x75,
		    0x7D, 0x82, 0x36, 0xDB, 0x62, 0x15, 0xAF, 0x4B,
		    0x0A, 0x9D, 0xCD, 0x64, 0x00, 0xAB, 0x88, 0x28,
		    0xA8, 0x35, 0x17, 0x70, 0x6F, 0x47, 0x44, 0xCD,
		    0x65, 0xAE, 0xD5, 0x05, 0x0A, 0xA8, 0x2F, 0x48,
		    0xAC, 0xA1, 0x72, 0x64, 0x1C, 0x7E, 0xD3, 0xF5,
		    0xD8, 0x4E, 0x73, 0x17, 0x0C, 0xE5, 0x9F, 0xB6,
		    0x00, 0xFA, 0xD7, 0x2C, 0x3D, 0x6A, 0x10, 0x47,
		    0x7C, 0xF2, 0x6B, 0x13, 0x10, 0x8A, 0x76, 0x39,
		    0xF8, 0x50, 0x33, 0xAC, 0x08, 0x1D, 0xA3, 0x48,
		    0xE1, 0xD0, 0x05, 0x49, 0xB7, 0x76, 0x03, 0x72,
		    0x07, 0xC5, 0xD3, 0x08, 0x79, 0x38, 0x66, 0xC1,
		    0x52, 0xAF, 0x83, 0xCD, 0xF3, 0x86, 0x62, 0xBF,
		    0x92, 0x24, 0x97, 0xBD, 0x5D, 0x7D, 0x81, 0x56,
		    0x4C, 0xF3, 0xD2, 0x60, 0xC2, 0xDE, 0x61, 0xC1,
		    0x39, 0x61, 0xDA, 0x07, 0x50, 0xC7, 0x98, 0x63,
		    0x7E, 0xDD, 0x54, 0xCA, 0xDE, 0x12, 0xD2, 0xA8,
		    0x19, 0x08, 0x6E, 0xF9, 0xFA, 0x6F, 0x58, 0x97,
		    0xD4, 0x0B, 0x5C, 0x5B, 0xE5, 0x30, 0xE5, 0x4C,
		    0x0E, 0x16, 0x87, 0xF0, 0x2C, 0xCB, 0x53, 0xB8,
		    0x0C, 0xE5, 0xDF, 0x16, 0x7B, 0xE8, 0xC2, 0xCF,
		    0xCC, 0xFF, 0x51, 0x24, 0xC1, 0xDD, 0x59, 0x9C,
		    0xA7, 0x56, 0x03, 0xB9, 0x0A, 0x37, 0xA2, 0xAC,
		    0x28, 0x8B, 0xEB, 0x51, 0x4E, 0xF1, 0xAE, 0xB5,
		    0xC8, 0xB5, 0xCB, 0x8D, 0x23, 0xF6, 0x24, 0x2D,
		    0xF6, 0x59, 0x62, 0xC0, 0xCB, 0xD3, 0x18, 0xE4,
		    0xB7, 0x73, 0xEF, 0xDB, 0x13, 0x9A, 0xF5, 0xD3,
		    0xD5, 0x61, 0x01, 0x14, 0xA5, 0xE5, 0x0D, 0x27,
		    0xC9, 0xA5, 0x08, 0x1C, 0x60, 0xBA, 0x73, 0xFF,
		    0xA9, 0xE0, 0x27, 0x86, 0x3F, 0xF7, 0x15, 0x03,
		    0x69, 0xA7, 0x2B, 0x57, 0xAC, 0xA6, 0x70, 0x55,
		    0xE9, 0xB5, 0x3F, 0xEB, 0x6F, 0xCE, 0x8A, 0xA1,
		    0x9D, 0x8B, 0x84, 0xF1, 0x7C, 0xD0, 0x35, 0x21,
		    0x91, 0x3D, 0x3D, 0x6E, 0x83, 0xFC, 0x45, 0x36,
		    0x93, 0xDA, 0x66, 0xDF, 0x1A, 0x59, 0x22, 0xA5,
		    0xC4, 0x99, 0x9B, 0xF8, 0x48, 0x9A, 0x50, 0x09,
		    0xAB, 0xAE, 0x56, 0xB6, 0x49, 0x02, 0x3E, 0x90,
		    0xB6, 0x07, 0x7E, 0xA7, 0x6A, 0x0A, 0xB5, 0x85,
		    0x31, 0x0D, 0x84, 0xD4, 0x01, 0xE4, 0x48, 0x63,
		    0xF3, 0xC1, 0x54, 0x65, 0xA6, 0x4C, 0x8B, 0x33,
		    0xF9, 0x70, 0x59, 0x3B, 0xA6, 0xF6, 0x2B, 0x66,
		    0xC5, 0xD2, 0xEB, 0xAB, 0x67, 0xD2, 0xE3, 0x78,
		    0xA9, 0x1A, 0x4C, 0x99, 0xA9, 0xA6, 0xCA, 0xF7,
		    0x65, 0xF0, 0x48, 0xF8, 0x2A, 0xEA, 0x96, 0x9F,
		    0xC4, 0x50, 0x9A, 0x0C, 0xB6, 0x0D, 0x8A, 0x2F,
		    0xC3, 0x99, 0x4E, 0xA0, 0x06, 0x4D, 0xAB, 0x25,
		    0x2E, 0x44, 0x47, 0xB6, 0x98, 0xF1, 0x2C, 0x96,
		    0x54, 0x51, 0x12, 0x41, 0x0D, 0xEF, 0x32, 0x9A,
		    0x4A, 0xBD, 0xA2, 0x26, 0x53, 0xA8, 0xFD, 0x8B,
		    0x6C, 0x95, 0x0A, 0x1A, 0x96, 0xEF, 0x3C, 0x85,
		    0x34, 0x4E, 0x25, 0x9E, 0x1C, 0x67, 0x33, 0x8A,
		    0xFF, 0x6D, 0x98, 0x93, 0x3D, 0x3F, 0x49, 0x6B,
		    0xBF, 0x7C, 0x4F, 0x63, 0x5D, 0x62, 0x64, 0x67,
		    0x0D, 0x07, 0x7F, 0x24, 0x4A, 0x23, 0xBC, 0x35,
		    0xE0, 0x92, 0x6F, 0x51, 0xE7, 0x25, 0x97, 0xB9,
		    0x14, 0x35, 0x2B, 0x48, 0xAC, 0x6F, 0x54, 0xDF,
		    0xF2, 0xB4, 0xB0, 0xE0, 0xD3, 0x28, 0x0D, 0x66,
		    0x46, 0x28, 0x0A, 0x16, 0x9C, 0x87, 0x73, 0xB7,
		    0x9C, 0x2B, 0xB5, 0x43, 0xC9, 0x46, 0xB9, 0x1F,
		    0x5F, 0x3C, 0x45, 0x03, 0x4B, 0xBF, 0x44, 0x4D,
		    0xE1, 0x44, 0xDA, 0x54, 0xC5, 0x32, 0x3A, 0xFA,
		    0x21, 0x5C, 0xAD, 0xD5, 0x1E, 0x1B, 0x54, 0x7C,
		    0x9F, 0xEA, 0x92, 0x8C, 0xEA, 0x69, 0xC0, 0xCE,
		    0xDA, 0x09, 0xAD, 0x95, 0xA0, 0x8E, 0x0B, 0x8E,
		    0x10, 0x4F, 0x5B, 0x8F, 0xB8, 0x2D, 0xAC, 0x4C,
		    0x94, 0x4B, 0x7C, 0x1E, 0xF1, 0x53, 0x20, 0x9B,
		    0xD6, 0xC4, 0x92, 0x4C, 0x7F, 0xFB, 0x8B, 0x8E,
		    0x40, 0x2F, 0x24, 0xA3, 0x4E, 0x46, 0x64, 0xF4,
		    0xC6, 0x35, 0x0F, 0xC7, 0x40, 0x55, 0x43, 0xAF,
		    0x7E, 0x91, 0x76, 0x48, 0x6F, 0x97, 0x7A, 0xF8,
		    0x32, 0x1E, 0xD3, 0x5B, 0xBC, 0x19, 0xB5, 0x48,
		    0xFA, 0x4F, 0x52, 0x77, 0x5B, 0x9E, 0xA2, 0xC8,
		    0x9A, 0x83, 0x30, 0x8D, 0x9F, 0x0B, 0x6F, 0xA8,
		    0x2E, 0x84, 0xCC, 0xC1, 0x50, 0x96, 0x46, 0xAE,
		    0x73, 0x91, 0x7D, 0xCD, 0x88, 0xAB, 0x67, 0x3F,
		    0x66, 0x3A, 0x8D, 0xB1, 0x89, 0x07, 0x93, 0xDB,
		    0x42, 0x22, 0xDC, 0x13, 0xBD, 0xCD, 0xBB, 0x12,
		    0x8D, 0x88, 0x44, 0x13, 0x22, 0x52, 0x81, 0xDC,
		    0xEF, 0xA1, 0xE4, 0xA3, 0xA7, 0xBA, 0xEE, 0x98,
		    0x79, 0x45, 0x29, 0x05, 0x65, 0x3D, 0xDC, 0xAF,
		    0xA1, 0x37, 0x29, 0xFD, 0x05, 0xD1, 0x3A, 0xF7,
		    0x32, 0x1D, 0x02, 0xEC, 0x28, 0x1E, 0x0F, 0x96,
		    0xF3, 0x21, 0x19, 0x5F, 0x49, 0xB9, 0xEA, 0x9A,
		    0xAD, 0x34, 0x58, 0xD1, 0xD9, 0xB1, 0x7D, 0xD2,
		    0xEA, 0xED, 0x74, 0xE8, 0x25, 0x9A, 0x7B, 0xC5,
		    0xC8, 0xD8, 0x76, 0xB6, 0xBC, 0x0B, 0x78, 0xCE,
		    0xD9, 0xA6, 0xBB, 0x2F, 0x79, 0xA4, 0x45, 0x05,
		    0x55, 0x6E, 0x20, 0x84, 0xEB, 0xC8, 0x70, 0xB0,
		    0x3A, 0x2D, 0x06, 0x98, 0x29, 0x10, 0xB8, 0xC5,
		    0xE9, 0xE4, 0xB6, 0xDE, 0x97, 0x9A, 0x0D, 0x8C,
		    0xB6, 0x22, 0x16, 0x59, 0xAB, 0xB5, 0xD7, 0x14,
		    0xAB, 0x08, 0x02, 0x27, 0x7B, 0xF7, 0x0E, 0xAC,
		    0xC5, 0xAC, 0x4D, 0x7F, 0xE5, 0x65, 0x51, 0x40,
		    0x44, 0x92, 0xB1, 0x6A, 0xB7, 0x00, 0x76, 0x89,
		    0x6E, 0x08, 0x5F, 0x45, 0x2B, 0x53, 0x86, 0x86,
		    0xA7, 0x85, 0xBC, 0x62, 0xAC, 0xAA, 0x82, 0x73,
		    0x0A, 0xEB, 0x35, 0x16, 0x95, 0x26, 0xAB, 0x9E,
		    0xE9, 0x64, 0x53, 0x99, 0x08, 0x31, 0xF5, 0x6B,
		    0x1F, 0xFE, 0x47, 0x4B, 0x09, 0x33, 0x4F, 0xBF,
		    0x1F, 0x0B, 0x4C, 0xB2, 0xB4, 0xA4, 0x17, 0xA9,
		    0xAD, 0xC5, 0x62, 0x7C, 0xF1, 0x1B, 0xAE, 0x46,
		    0xD3, 0xAC, 0xFD, 0x43, 0xFE, 0x79, 0xD0, 0x58,
		    0x2F, 0x6C, 0x9F, 0xD0, 0x65, 0xA4, 0x64, 0x03,
		    0xAF, 0x73, 0x46, 0x75, 0x7D, 0x49, 0x1B, 0x4C,
		    0xFA, 0x49, 0xD8, 0x9A, 0xCC, 0x59, 0xC6, 0xC7,
		    0xA1, 0x05, 0xC2, 0x32, 0xC8, 0x6C, 0x50, 0xA8,
		    0x06, 0x58, 0xBE, 0x6C, 0x7D, 0x22, 0xD6, 0x0D,
		    0x74, 0x40, 0xCE, 0xD6, 0x64, 0xD6, 0x47, 0xD0,
		    0xBF, 0xF1, 0x5C, 0x54, 0xF9, 0x06, 0x3F, 0x3D,
		    0x86, 0xBA, 0xF2, 0x0F, 0x5E, 0x2C, 0x01, 0xCC,
		    0xD9, 0xC7, 0xB1, 0x4A, 0xB3, 0xD7, 0x26, 0xCC,
		    0xC3, 0x7A, 0x74, 0x2C, 0xE1, 0x22, 0x65, 0xA0,
		    0x5B, 0xCA, 0xF4, 0xE1, 0x7D, 0xE1, 0x56, 0xFD,
		    0x94, 0x10, 0xC6, 0xA1, 0x4A, 0xE8, 0x6B, 0x34,
		    0x4E, 0x71, 0x60, 0x77, 0x0F, 0x03, 0xDD, 0xFF,
		    0xC8, 0x59, 0x54, 0x6C, 0xD4, 0x4A, 0x55, 0x24,
		    0x35, 0x21, 0x60, 0x73, 0xDF, 0x6F, 0xE7, 0x3C,
		    0xC2, 0xF0, 0xDA, 0xA9, 0xE5, 0x8C, 0xAC, 0xB6,
		    0xFD, 0x2E, 0xF7, 0xA0, 0x18, 0xA7, 0x55, 0x47,
		    0xD1, 0xCB, 0x9E, 0xAA, 0x58, 0x54, 0x3B, 0x37,
		    0x18, 0xB5, 0xC1, 0xBB, 0x41, 0x59, 0xE4, 0x28,
		    0x4A, 0x13, 0x90, 0x6A, 0xF7, 0xD1, 0xB3, 0x71,
		    0xB6, 0x6E, 0xF6, 0x5D, 0x2E, 0x0E, 0x6C, 0x4A,
		    0x7B, 0xF7, 0xB6, 0x21, 0xD4, 0xFC, 0x47, 0x8C,
		    0x9B, 0x0A, 0x90, 0xAC, 0x11, 0x52, 0x86, 0x07,
		    0x24, 0xDA, 0xA9, 0x49, 0x50, 0xD9, 0xDC, 0xE2,
		    0x19, 0x87, 0x73, 0x88, 0xC3, 0xE4, 0xED, 0xC9,
		    0x1C, 0xA8, 0x7E, 0x39, 0x48, 0x91, 0x10, 0xAB,
		    0xFC, 0x3C, 0x1E, 0xEE, 0x08, 0xA1, 0xB9, 0xB2,
		    0x02, 0x57, 0xB1, 0xD1, 0x35, 0x5E, 0x3D, 0x94,
		    0xFB, 0x36, 0x27, 0x1A, 0x0E, 0x75, 0xFC, 0xBC,
		    0xDB, 0xF3, 0xF5, 0x7C, 0x08, 0x39, 0xAA, 0xF4,
		    0x2E, 0xEE, 0xCF, 0xCD, 0x2D, 0x70, 0xB8, 0x84,
		    0xE6, 0x22, 0x5C, 0xC0, 0xB9, 0x33, 0xCB, 0x97,
		    0xA1, 0xA3, 0xEE, 0x93, 0x71, 0xCF, 0xC9, 0x21,
		    0x31, 0x7A, 0xEC, 0xE7, 0x70, 0xF2, 0xAA, 0x91,
		    0xAA, 0x48, 0xAD, 0xAC, 0x03, 0xB1, 0x26, 0x52,
		    0xBC, 0x65, 0x22, 0xA1, 0x09, 0x3D, 0xAB, 0x16,
		    0x08, 0xBF, 0xCF, 0x3F, 0x59, 0x08, 0x6F, 0x68,
		    0xEB, 0x8A, 0xB3, 0xCF, 0x77, 0x82, 0xFB, 0x25,
		    0x78, 0x16, 0x4C, 0xDB, 0x72, 0xF5, 0xCF, 0x79,
		    0x71, 0xE4, 0x4E, 0x23, 0x15, 0x7F, 0x1E, 0xA8,
		    0x3E, 0xC0, 0x59, 0x91, 0x20, 0xAE, 0x2C, 0x1D,
		    0x90, 0xC8, 0x49, 0x42, 0x48, 0x29, 0x82, 0x66,
		    0x68, 0x49, 0x73, 0xDA, 0xE4, 0x28, 0xCD, 0x7B,
		    0x4D, 0xE4, 0x23, 0x34, 0xB9, 0xE1, 0xB4, 0x42,
		    0x67, 0x22, 0x5B, 0xEE, 0xE6, 0x74, 0x32, 0x6F,
		    0x21, 0x9F, 0x97, 0x46, 0x03, 0xE1, 0xC9, 0x7A,
		    0x14, 0x27, 0x30, 0xE1, 0xB2, 0x34, 0xE6, 0xAF,
		    0x7B, 0xAA, 0xDD, 0x89, 0x04, 0x30, 0xD6, 0x78,
		    0x0B, 0x3D, 0xC3, 0x69, 0xB0, 0x67, 0x4F, 0x4E,
		    0x12, 0x21, 0x93, 0x2D, 0x79, 0xDD, 0x8B, 0xDB,
		    0xEA, 0x90, 0x66, 0x54, 0xA8, 0x05, 0xF2, 0xE4,
		    0x59, 0x8A, 0x96, 0x52, 0x30, 0xF0, 0x4E, 0x9A,
		    0xE5, 0xD8, 0x72, 0x1C, 0x3B, 0x63, 0x02, 0xB9,
		    0xC7, 0xA1, 0xDA, 0xC8, 0x6C, 0x48, 0xE0, 0xDE,
		    0x59, 0x64, 0x89, 0x2C, 0xF9, 0xC8, 0x3B, 0x00,
		    0xEC, 0xF2, 0x68, 0x51, 0x67, 0x05, 0x85, 0xAF,
		    0xB8, 0xD5, 0x65, 0xEE, 0x73, 0x26, 0x88, 0xFB,
		    0xA9, 0xD6, 0x6C, 0x68, 0x9D, 0x9F, 0x23, 0x6A,
		    0x10, 0x24, 0x82, 0xB2, 0xB7, 0x40, 0x19, 0x3E,
		    0x6F, 0xA2, 0xD5, 0x2C, 0x6E, 0x8D, 0xE9, 0x33,
		    0x6E, 0x24, 0x94, 0x05, 0xE9, 0x2D, 0xD9, 0x3A,
		    0x8C, 0xE5, 0xCC, 0x1D, 0x3F, 0xB8, 0x71, 0xA8,
		    0x98, 0x33, 0xBB, 0x1A, 0xAC, 0x41, 0x0A, 0x04,
		    0xFE, 0x4D, 0x46, 0x17, 0x8A, 0xCB, 0xF3, 0x4B,
		    0x97, 0x02, 0xCC, 0x9D, 0x11, 0xF1, 0xBC, 0xA9,
		    0xC1, 0xD1, 0xB6, 0xD6, 0x7B, 0x5F, 0x9D, 0x22,
		    0x86, 0x71, 0xEC, 0x42, 0x53, 0xB7, 0x85, 0x30,
		    0xAF, 0x1D, 0x01, 0xA7, 0xBF, 0x72, 0xC2, 0xC6,
		    0xC9, 0xB8, 0xD8, 0xC7, 0xE9, 0xC4, 0xBA, 0xC5,
		    0xB1, 0x8A, 0xB8, 0x62, 0xBF, 0x75, 0x75, 0x69,
		    0xF8, 0x8D, 0x7E, 0xD9, 0xD2, 0x28, 0xB5, 0x40,
		    0xCE, 0xCB, 0xB8, 0x74, 0x31, 0x40, 0x7B, 0x0D,
		    0x73, 0x98, 0x99, 0x12, 0xB7, 0x75, 0x3E, 0xBC,
		    0xAE, 0x48, 0xCA, 0xA9, 0x1E, 0xA7, 0x95, 0x31,
		    0x87, 0x0F, 0x14, 0x52, 0xB6, 0x8E, 0x42, 0x50,
		    0xB2, 0x76, 0x75, 0xD8, 0x7E, 0x66, 0x23, 0x13,
		    0x8B, 0x29, 0xAA, 0x13, 0xCA, 0x8A, 0xD8, 0x9B,
		    0x7B, 0x38, 0xD2, 0xE8, 0x67, 0xD1, 0x89, 0x25,
		    0x9C, 0x63, 0x2F, 0xC3, 0x26, 0xC7, 0x74, 0x83,
		    0x05, 0xED, 0x67, 0x02, 0x85, 0xAD, 0x1D, 0x0E,
		    0xA9, 0xD6, 0xE1, 0xC7, 0x39, 0xA0, 0x6E, 0x72,
		    0xCE, 0x56, 0x6C, 0xB8, 0x4A, 0xDE, 0x11, 0xA2,
		    0xBF, 0xC1, 0x84, 0x98, 0x8F, 0xCA, 0x79, 0x74,
		    0xCA, 0x9F, 0x45, 0x16, 0xBC, 0xB1, 0xF4, 0x03,
		    0x76, 0x6E, 0xD5, 0x46, 0x60, 0xD7, 0x1D, 0xF0,
		    0x87, 0x29, 0x63, 0x07, 0x06, 0xB9, 0xC2, 0x69,
		    0x6D, 0xF9, 0x4B, 0x30, 0x96, 0x83, 0xB8, 0xC5,
		    0xBE, 0x3A, 0xBA, 0xD0, 0x3E, 0x2B, 0x04, 0x16,
		    0x6A, 0x00, 0x3B, 0x1A, 0x8E, 0xF8, 0xF6, 0x21,
		    0x01, 0xD6, 0x08, 0x41, 0x74, 0xA2, 0xFC, 0x36,
		    0xED, 0x11, 0x51, 0x5A, 0x4A, 0x21, 0x1A, 0x03,
		    0x11, 0x95, 0x11, 0xF6, 0x73, 0x38, 0x67, 0xFC,
		    0xF1, 0x2B, 0x22, 0x54, 0x65, 0x40, 0x7D, 0x8C,
		    0x13, 0xC4, 0x46, 0x87, 0x09, 0x2B, 0xB5, 0xA1,
		    0x82, 0x49, 0x46, 0x56, 0xF5, 0x5F, 0xF1, 0x04,
		    0xD8, 0x6F, 0xDB, 0x38, 0xAD, 0xF4, 0x1A, 0xA3,
		    0xFF, 0x7C, 0xC7, 0xA6, 0xAF, 0x87, 0x5C, 0x8C,
		    0xEA, 0x3C, 0x9D, 0x7A, 0x4A, 0xD8, 0xA8, 0x66,
		    0xDB, 0xBF, 0x12, 0x58, 0x98, 0x8E, 0xBA, 0x6F,
		    0xAF, 0x20, 0xDA, 0xEE, 0x82, 0x34, 0x2F, 0x33,
		    0x88, 0x98, 0xBA, 0xB2, 0x54, 0x7F, 0x9E, 0x63,
		    0x19, 0x6C, 0x7D, 0xCE, 0x85, 0xF8, 0xB6, 0x77,
		    0xCB, 0x38, 0x1F, 0xB1, 0x79, 0xBD, 0xED, 0x32,
		    0xE3, 0xB9, 0x40, 0xEF, 0x3E, 0x6C, 0x29, 0x88,
		    0x70, 0x99, 0x47, 0xA6, 0x4A, 0x1C, 0xCC, 0x0B,
		    0x9B, 0x72, 0xA9, 0x29, 0x83, 0x4C, 0xDE, 0x4F,
		    0x65, 0x4E, 0xCE, 0xBD, 0xFA, 0x76, 0x8D, 0xA6,
		    0x1A, 0xD8, 0x66, 0xFE, 0xA4, 0x2A, 0x61, 0x50,
		    0xEE, 0x15, 0xF1, 0xF0, 0x9D, 0xFF, 0xEC, 0xEE,
		    0x00, 0x03, 0xFE, 0xAC, 0x53, 0x02, 0xCC, 0x87,
		    0xB1, 0xA2, 0xD8, 0x34, 0x2C, 0xEC, 0xA6, 0x4C,
		    0x02, 0xC0, 0xC1, 0x72, 0xD6, 0x54, 0x35, 0x24,
		    0x25, 0x8B, 0xEC, 0xDA, 0x47, 0x5F, 0x5D, 0x7E,
		    0xD8, 0x01, 0x51, 0xDD, 0x8F, 0xB4, 0x48, 0xDD,
		    0x94, 0x99, 0x95, 0x77, 0xB3, 0x42, 0x14, 0xEB,
		    0x26, 0x61, 0xE9, 0x22, 0xE3, 0x07, 0x73, 0xFB,
		    0xEF, 0x38, 0x55, 0x35, 0x8F, 0xCC, 0x30, 0x1E,
		    0x38, 0xE0, 0x35, 0xF4, 0x9A, 0x7C, 0xCF, 0x38,
		    0x0B, 0x9E, 0xF4, 0x88, 0x4A, 0xEA, 0xF2, 0x67,
		    0x9F, 0x61, 0x40, 0x34, 0x09, 0xDC, 0xBF, 0xFB,
		    0x22, 0x27, 0x04, 0x8B, 0x8D, 0x85, 0x7F, 0xB2,
		    0x29, 0x62, 0x25, 0x73, 0x7F, 0x46, 0x2E, 0xA3,
		    0x8E, 0xAF, 0xEC, 0x55, 0x98, 0x1A, 0xEE, 0x29,
		    0xA0, 0x1A, 0x5F, 0xFE, 0x5D, 0xA5, 0x76, 0x93,
		    0xAB, 0x57, 0x56, 0xEA, 0xDB, 0x39, 0xAC, 0x48,
		    0xBE, 0x95, 0x92, 0x2B, 0xC6, 0xE1, 0x2F, 0x36,
		    0x4B, 0x08, 0x01, 0x90, 0x50, 0xD8, 0xFA, 0xF9,
		    0x94, 0x4E, 0x76, 0x9B, 0x72, 0x59, 0xC2, 0x2F,
		    0x61, 0x04, 0x0A, 0x9E, 0x28, 0xE5, 0x24, 0x1E,
		    0x79, 0xCF, 0x8D, 0xB6, 0x52, 0xA7, 0x79, 0x5F,
		    0x44, 0x98, 0xD5, 0x0E, 0x6E, 0x4B, 0x64, 0x9B,
		},
		.len = 2048
	},
	.auth_tags[0] = {
		.size = 64,
		.data = { 0x4d, 0x5c, 0x2a, 0xf3, 0x27, 0xcd, 0x64, 0xa6,
			0x2c, 0xf3, 0x5a, 0xbd, 0x2b, 0xa6, 0xfa, 0xb4 },
		.len = 16
	},
	.auth_tags[1] = {
		.size = 128,
		.data = { 0xE9, 0xA9, 0x75, 0xB6, 0xEF, 0x6F, 0x8C, 0xF1,
		    0xB3, 0xA9, 0x19, 0xA4, 0xAE, 0x66, 0xBD, 0x9E },
		.len = 16
	},
	.auth_tags[2] = {
		.size = 256,
		.data = { 0x29, 0xC3, 0x18, 0x96, 0x54, 0xCB, 0xF5, 0xAA,
		    0x4E, 0x62, 0xB6, 0xFF, 0x45, 0xA6, 0x18, 0x0C },
		.len = 16
	},
	.auth_tags[3] = {
		.size = 512,
		.data = { 0x3B, 0xD7, 0xC3, 0x5F, 0xE4, 0x1B, 0xC2, 0xBC,
		    0xE9, 0xAC, 0xF2, 0xCE, 0xA7, 0x7B, 0x1D, 0x70 },
		.len = 16
	},
	.auth_tags[4] = {
		.size = 1024,
		.data = { 0xCC, 0xBB, 0xBC, 0xCF, 0x86, 0x01, 0x4D, 0x93,
		    0x4B, 0x68, 0x55, 0x19, 0xA1, 0x40, 0xCD, 0xEA },
		.len = 16
	},
	.auth_tags[5] = {
		.size = 1536,
		.data = { 0x67, 0x31, 0x11, 0xA2, 0x58, 0xB5, 0x1C, 0x23,
		    0xC0, 0x41, 0x05, 0x30, 0xC6, 0xBA, 0xFA, 0x88 },
		.len = 16
	},
	.auth_tags[6] = {
		.size = 2048,
		.data = { 0x03, 0x9C, 0x6B, 0xB9, 0x57, 0xBF, 0x6E, 0x86,
			0x3A, 0x09, 0x5F, 0x08, 0xA9, 0xE4, 0xF2, 0x1F },
		.len = 16
	},
	.auth_tag = {
		.data = {
		    0x03, 0x9C, 0x6B, 0xB9, 0x57, 0xBF, 0x6E, 0x86,
		    0x3A, 0x09, 0x5F, 0x08, 0xA9, 0xE4, 0xF2, 0x1F
		},
		.len = 16
	},
};

static const struct gmac_test_data gmac_test_case_4 = {
	.key = {
		.data = {
			0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
			0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
		},
		.len = 16
	},
	.iv = {
		.data = {
			0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
			0xde, 0xca, 0xf8, 0x88
		},
		.len = 12
	},
	.aad = {
		.data = gmac_plaintext,
		.len = GMAC_LARGE_PLAINTEXT_LENGTH
	},
	.plaintext = {
		.data = NULL,
		.len = 0
	},
	.gmac_tag = {
		.data = {
			0x88, 0x82, 0xb4, 0x93, 0x8f, 0x04, 0xcd, 0x06,
			0xfd, 0xac, 0x6d, 0x8b, 0x9c, 0x9e, 0x8f, 0xec
		},
		.len = 16
	}
};

#endif /* TEST_CRYPTODEV_GCM_TEST_VECTORS_H_ */
