/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

/* (C) Marcin Kwadrans <quarter@users.sourceforge.net> */

#include "include/support.h"
#include "include/symbol.h"
#include "include/environment.h"

GData *LWSymbol::datalist_symbols = NULL;
GData *LWSymbol::datalist_symbols_with_id = NULL;

typedef LWSymbol * (*SymbolConstructor) ();

static void delete_cb (LWSymbol *symbol)
{
	g_return_if_fail (FALSE == symbol->canClone());
 	delete symbol;
}

/*! \brief Constructor

	Create the symbol
*/
LWSymbol::LWSymbol (): id(0) {};

/*! \brief Destructor

	Destroy the symbol
*/
LWSymbol::~LWSymbol ()
{
	if (id > 0)
		g_datalist_id_remove_data (&datalist_symbols, quark);
}

/*! \brief Clone the symbol

	Return the coloned symbol. 
	If symbol is a command, command is unintialized
*/
LWSymbol *LWSymbol::clone()
{
	g_return_val_if_fail (TRUE == canClone(), NULL);
	
	GData *symbols = LWEnvironment::getSymbols ();
	
	SymbolConstructor sc = (SymbolConstructor) g_datalist_id_get_data (&symbols, quark);
	
	if (sc == NULL) g_print ("%s", getName());
	
	g_return_val_if_fail (sc != NULL, NULL);

	LWSymbol *symbol = sc();
	symbol->quark = quark;
		
	return symbol;
}

/*! \brief Pobieranie identyfikatora

	Pobiera identyfikator symbolu
	\return Pobrany identyfikator
*/
guint LWSymbol::getId ()
{
	return id;
}

/*! \brief Czy symbol jest poleceniem

	Zwraca FAŁSZ. Klasa która implementuje polecenie
	przeciąża tę metodę.
	\return PRAWDA jeśli symbol stanowi polecenie
	w przeciwnym razie FAŁSZ
*/
gboolean LWSymbol::isCommand ()
{
	return FALSE;
}

/*! \brief Czy symbol jest poleceniem

	Zwraca FAŁSZ. Klasa która implementuje wartość
	przeciąża tę metodę.
	\return PRAWDA jeśli symbol stanowi wartość
	w przeciwnym razie FAŁSZ
*/
gboolean LWSymbol::isValue ()
{
	return FALSE;
}

void LWSymbol::onDndCopy (LWPiece *sourcepiece, LWPiece *destpiece)
{
	g_return_if_fail (sourcepiece->getSymbol() == this);
	g_return_if_fail (LW_TYPE_PROGRAM == destpiece->getRow()->getBoard()->getType());
	
	LWPiece *newpiece = new LWPiece (sourcepiece, destpiece->getRow());
	destpiece->getRow()->insertPieceBefore (newpiece, destpiece);	
}

void LWSymbol::onDndMove (LWPiece *sourcepiece, LWPiece *destpiece)
{
	g_return_if_fail (sourcepiece->getSymbol() == this);
	g_return_if_fail (LW_TYPE_PROGRAM == destpiece->getRow()->getBoard()->getType());
	
	LWPiece *newpiece = new LWPiece (sourcepiece, destpiece->row);
	destpiece->getRow()->insertPieceBefore (newpiece, destpiece);
	sourcepiece->getRow()->removePiece (sourcepiece);
}

void LWSymbol::onAttach (LWPiece *piece)
{
 	(void) piece;
}

/*! \brief Metoda fabryczna

	Tworzy polecenie na podstawie jego nazwy 
	\param name Nazwa polecenia
*/
LWSymbol *LWSymbol::factory (const gchar *name)
{
	g_return_val_if_fail (name != NULL, NULL);
	
	GQuark quark = g_quark_try_string (name);
	
	g_return_val_if_fail (quark != 0, NULL);
	
	if (datalist_symbols == NULL)
		g_datalist_init (&datalist_symbols);
	else {
		LWSymbol *symbol = (LWSymbol *) g_datalist_id_get_data (&datalist_symbols, quark);
	
		if (symbol != NULL)
			if (FALSE == symbol->canClone())
				return symbol;
	}
	
	GData *symbols = LWEnvironment::getSymbols ();
	
	SymbolConstructor sc = (SymbolConstructor) g_datalist_id_get_data (&symbols, quark);
	
	if (sc == NULL) g_print ("%s", name);
	
	g_return_val_if_fail (sc != NULL, NULL);

	LWSymbol *symbol = sc();

	symbol->quark = quark;
	
	if (FALSE == symbol->canClone())
		g_datalist_id_set_data_full (&datalist_symbols, quark, 
									 (gpointer) symbol, (GDestroyNotify) delete_cb);
	
	return symbol;
}

/*! \brief Metoda fabryczna uwzględniająca identyfikator

	Tworzy polecenie na podstawie jego nazwy 
	Dodatkowo stworzene polecenie będzie miało przyporządkowany
	identyfikator.
	\param name Nazwa polecenia
	\param id Identyfikator
*/
LWSymbol *LWSymbol::factoryId (const gchar *name, guint id)
{
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (id > 0, NULL);
	
	gchar *idString = g_strdup_printf ("%s@%u", name, id);
	
	if (datalist_symbols == NULL)
		g_datalist_init (&datalist_symbols_with_id);
	
	GQuark quark = g_quark_from_string (idString);
	
	LWSymbol *symbol = (LWSymbol *) g_datalist_id_get_data (&datalist_symbols_with_id, quark);
	
	if (symbol == NULL) {
		GData *symbols = LWEnvironment::getSymbols ();
		
		SymbolConstructor sc = (SymbolConstructor) g_datalist_get_data (&symbols, name);
		
		if (sc == NULL) g_print ("%s", name);
		
		g_return_val_if_fail (sc != NULL, NULL);

		symbol = sc();
		symbol->id = id;
		symbol->quark = quark;
		g_datalist_id_set_data (&datalist_symbols_with_id, quark, (gpointer) symbol);
	}
	
	g_free (idString);
	return symbol;
}

/*! \brief Destroy all symbols

	It destroys all 'nonclonable' symbols
*/
void LWSymbol::destroyAll ()
{
	g_datalist_clear (&datalist_symbols);
}
