package Sumibi;

our $VERSION = '0.09';

use strict;

use Term::ReadLine ();
use Data::Dumper ();
use SOAP::Lite ();
use Encode ();

use constant SUMIBI_WSDL_STABLE  => 'http://sumibi.org/sumibi/Sumibi_stable.wsdl';
use constant SUMIBI_WSDL_TESTING => 'http://sumibi.org/sumibi/Sumibi_testing.wsdl';

sub new{
  my $class = shift;
  my %hash = @_;
  my $self =
    {
     soap    => undef,
     count   => 0,
     encode  => 'euc-jp',
     history => $ENV{HOME} . '/.sumibi_pm_history',
     ca_file => undef,
     wsdl    => SUMIBI_WSDL_STABLE,
     can_choose => undef,
    };
  foreach my $key (qw/encode history ca_file wsdl/){
    $self->{$key} = $hash{$key} if exists $hash{$key};
  }
  if(my $server = $hash{wsdl}){
    if($server eq 'tesing'){
      $self->{server} = SUMIBI_WSDL_TESTING
    }elsif($server eq 'stable'){
      $self->{server} = SUMIBI_WSDL_STABLE;
    }
  }
  return bless $self => $class;
}

sub ca_file{
  my $self = shift;
  $ENV{HTTPS_CA_FILE} = shift if @_;
}

sub soap{
  my $self = shift;
  return $self->{soap} ||= SOAP::Lite->service($self->wsdl) or die;
}

sub wsdl{
  my $self = shift;
  return $self->{wsdl};
}

sub test_wsdl{
  my $self = shift;
  return $self->{wsdl} = SUMIBI_WSDL_TESTING;
}

sub encode{
  my $self = shift;
  $self->{encode} = shift if @_;
  return $self->{encode};
}

sub convert{
  my $self = shift;
  $self->count_plus();
  my $str = shift;
  my $r = $self->soap->doSumibiConvert($str, 'sumibi_current', '', '');
  if($r){
    return $self->parse($r->{resultElements});
  }else{
    $self->current_error({content => $r});
  }
}

sub _parse{
  my($self, $result, $ret, $choosen, $candidate) = @_;
  my $i = 0;
 LOOP: while($_ = shift @$result){
    my @candidate_list = @$_;
    foreach my $string (@candidate_list){
      if(!$self->can_choose){
        push @$ret, shift @candidate_list;
      }elsif(!$candidate){
        push @$ret, $candidate_list[$choosen->[$i]];
        $i++;
      }else{
        push @$candidate, \@candidate_list;
      }
      next LOOP;
    }
  }
}

sub parse{
  my $self = shift;
  my $contents = shift;
  my @result;
  my $encode = $self->encode;

  foreach (@$contents){
    $_->{word} = Encode::encode("utf-8", $_->{word});
    Encode::from_to($_->{word}, "utf-8", $encode);
    push @{$result[$_->{no}] ||= []}, $_->{word};
  }
  $self->result(@result);
  my @ret;
  my $candidate = [];
  $self->_parse(\@result, \@ret, undef, $candidate);
  if(@$candidate){
    $self->candidate($candidate);
    return '';
  }else{
    return join "", @ret;
  }
}

sub can_choose{
  my $self = shift;
  $self->{can_choose} = shift  if @_;
  return defined $self->{can_choose} ? $self->{can_choose} : $self->shell_mode;
}

sub get_string{
  my $self = shift;
  my $choosen = shift;
  my @ret;
  $self->_parse($self->result, \@ret, $choosen);
  return join "", @ret;
}

sub clear_candidate{
  my $self = shift;
  $self->{result} = $self->{candidate} = undef;
}

sub result{
  my $self = shift;
  $self->{result} = [ @_ ] if @_;
  return $self->{result};
}

sub candidate{
  my $self = shift;
  $self->{candidate} = shift if @_;
  return $self->{candidate};
}

sub count{
  my $self = shift;
  $self->{count};
}

sub count_plus{
  my $self = shift;
  $self->{count}++;
}

sub current_error{
  my $self = shift;
  my $count = $self->count;
  $self->{error}->{$count} = shift if @_;
  return $self->{error}->{$count};
}

sub dump_error{
  my $self = shift;
  print Data::Dumper::Dumper($self->error);
}

sub error{
  my $self = shift;
  $self->{error};
}

sub shell{
  my $self = $_[0];
  $self = shift->new(@_) if (ref $self and ref $self ne __PACKAGE__) or $self eq __PACKAGE__;
  $self = __PACKAGE__->new(@_) unless $self eq __PACKAGE__;
  $self->shell_mode(1);
  my $term = new Term::ReadLine 'sumibi shell';
  $self->{term} = $term;
  my $can_history = $self->history ? 1 : 0;
  $self->history_open() if $can_history;
  my($prompt, $prompt2) = ("Sumibi> ", "Sumibi-candidate> ");
  while(defined (my $str = $term->readline($prompt))){
    next if $str =~/^\s*$/;
    my $res = $self->convert($str);
    print $res, "\n" unless !$res or $self->current_error;
    $term->addhistory($str) if $can_history;
    $self->addhistory($str);
    if(my @candidate = @{$self->candidate || []}){
      print "[Sumibi candidate]\n";
      foreach my $candidate (@candidate){
        print "=> ", join("  ",map{ $_ . ':' . $candidate->[$_]}(0.. $#{$candidate})), "\n";
      }
      while(defined(my $str = $term->readline($prompt2))){
        my @define = $str =~/^\s*$/ ? ('0') x @candidate : $str =~/(\d+)/g;
        if(@define == @candidate){
          print $self->get_string(\@define);
          $self->clear_candidate();
          last;
        }else{
          print "nubmer of your choice is more or lesser. ( you must write ",scalar(@candidate)," nubmer.)\n";
        }
      }
      print "\n";
    }
  }
  print "\n";
  $self->history_close() if $can_history;
}

sub history{
  my $self = shift;
  $self->{history} = shift if @_;
  return $self->{history};
}

sub addhistory{
  my $self = shift;
  my $fh = $self->{histfh};
  print $fh shift, "\n";
}

sub term{
  my $self = shift;
  return $self->{term};
}

sub history_open{
  my $self = shift;
  $self->create_history unless -e $self->history;
  open my $fh, '+<', $self->history;
  seek $fh, 0, 0;
  unless(-z $self->history){
    while(<$fh>){
      chomp;
      next unless $_;
      $self->term->addhistory($_);
    }
  }
  return $self->{histfh} = $fh;
}

sub create_history{
  my $self = shift;
  open OUT, '>', $self->history;
  close OUT;
}

sub history_close{
  my $self = shift;
  close $self->{histfh};
}

sub shell_mode{
  my $self = shift;
  $self->{shell_mode} = shift if @_;
  return $self->{shell_mode};
}

1;

=pod

=head1 ̾

Sumibi -- Sumibi(http://sumibi.org/) Perl⥸塼

=head1 

Sumibi(http://sumibi.org/)  kiyoka󤬺޻Ѵ󥸥Ǥ
sourceforge.jp ˥ץȤޤ(http://sourceforge.jp/projects/sumibi/)
ϡSumibiڤिPerl⥸塼Ǥ

=head1 Ȥ

 use Sumibi;

 print Sumibi->new->convert("Perl no module wo tukutte mimashita.h .");
 # "PerlmoduleäƤߤޤ"ȽϤޤ

Ϥʸ Sumibi Ѵϡ

 perl -MSumibi -e 'Sumibi::shell();'

Sumibi shell Ω夬ꡢΤ褦ϤǤޤ

 Sumibi> konnnichiha .

=head1 ᥽å

=over 4

=item new

 my $sumibi = Sumibi->new(encode => 'euc-jp', history => '.sumibi_pm_history');

󥹥ȥ饯ϲ̤ꡣ

 encode .... Ѥ륨󥳡(ǥեȤeuc-jp)
 history ... shell ξ˻Ѥե(ǥեȤ $ENV{HOME}/.sumibi_pm_history)
 ca_file ... $ENV{HTTPS_CA_FILE} ޤ
 wsdl ...... SumibiWSDLեξꤷޤ(ǥեȤϡhttp://sumibi.org/sumibi/Sumibi_stable.wsdl)

=item shell

 $sumibi->shell;

饹᥽åɡؿȤƤƤ٤ޤ

 Sumibi->shell;
 Sumibi::shell()

˰Ϳ new ͿƱˤʤޤ

=item ca_file

 $sumibi->ca_file
 $sumibi->ca_file($ca_file);

$ENV{HTTPS_CA_FILE} ͤФ줷ޤ饹᥽åɤȤƤѤǤޤ

=item wsdl

 $sumibi->wsdl($url);

WSDLեξѹޤ

=item test_wsdl

 $sumibi->test_server;

WSDLƥѤˤޤhttp://sumibi.org/sumibi/Sumibi_testing.wsdl

=item encode

 $sumibi->encode;
 $sumibi->encode('sjis');

Sumibi ֤󥳡ɤꤷޤǥեȤ euc-jp Ǥ
ǻꤹΤϡEncode⥸塼ǻѤʸ̾ˤƤ

=item soap

 $sumibi->soap;

SOAP::Lite->service($wsdl_url)η̤֤äƤޤ
http://sumibi.org/sumibi/sumibi_api_stable.html Ƥ᥽åɤ
ƤӽФȤޤ

=item convert

 $sumibi->convert('konnnichiha .');

ʸѴޤ

=item can_choose

 $sumibi->can_choose(1);

ѴǤ뤫ɤǥեȤ0Ǥ
shell ⡼ɤǤϡ꤬ʤ¤1֤ޤ

=item history

¸ե̾ꤷޤ
ǥեȤϡ $ENV{HOME}/.sumibi_pm_history Ǥ

=item current_error

 $sumibi->current_error;

ߤΥ顼֤ޤ(ޤưޤ)

=item dump_error

 $sumibi->dump_error;

ΤΥ顼 Data::Dumper η֤ޤ(ޤưޤ)

=item error

ΤΥ顼äƤ롢$sumibi->{error} ֤ޤ(ޤưޤ)

=back

=head1 Sumibi shell

ʸϤԤ Sumibi ѴԤޤshell ȴϡctrl + d ȴ뤳ȤǤޤ

 Sumibi> konnnichiha .

Sumibi ʣѴ֤硢Τ褦ɽޤ

 [Sumibi-candidate]
 => 0:  1:  2:ˤ
 => 0:  1:  2:  3:

shell ΥץץȤΤ褦Ѥޤ

 sumibi-candidate>

ξ֤ǡ򤹤򥹥ڡڤϤޤ
 0 ɤϲԤǤޤϤϡΤ褦ˤʤޤ

 Sumibi-candidate> 2 0

Ѵ줿ΤǤޤΤήϲΤ褦ˤʤޤ

 Sumibi> konnnichiha .
 [Sumibi candidate]
 => 0:  1:  2:ˤ
 => 0:  1:  2:  3:
 Sumibi-candidate> 2 0
 ˤϡ
 sumibi>

=head1 Х

顼طEmacs LispѤCGIѤ˺ä᥽åɤĤäƤʤΤǡޤäȤޤ
¾ƥȤƤޤ󡣤äȤɤХäƤ뤳ȤǤ礦

=head1 ռ

򤤥޻Ѵ󥸥äƤä kiyoka˴ա
http://www.netfort.gr.jp/~kiyoka/
http://sumibi.org/
http://sourceforge.jp/projects/sumibi/


=head1 

 Ktat <atusi@pure.ne.jp>

=head1 

 Copyright 2005-2006 by Ktat <atusi@pure.ne.jp>.

 This program is free software; you can redistribute it
 and/or modify it under the same terms as Perl itself.

 See http://www.perl.com/perl/misc/Artistic.html

=cut
