
/*

    allin1.c - All in one dock applet: mem, swap, battery, cpu, net & fs
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include <unistd.h>
#include <limits.h>
#include <X11/Xlib.h>

#include "dockhelper.h"
#include "memory.h"
#include "battery.h"
#include "cpu.h"
#include "network.h"
#include "filesys.h"
#include "seti.h"
#include "confparse.h"


#define VERSION "0.5.0"
#define RELDATE "24 aug 2004"



/* colors for bars and graphs */
char background[30];


/* system memory struct */
struct memory_monitor mmon;

/* cpu monitor struct */
struct cpu_monitor cpum;

/* battery struct */
struct battery_monitor bmon;

/* net interfaces structs */
struct net_monitor eth,ppp;

/* filesys monitor */
struct fsys_monitor fmon;

/* seti histogram */
struct seti_monitor setimon;

/* update time for graphs */
int update_time = 200000;
unsigned int updates = 0;

/* border size and separator between areas */
int border = 2;


/* conf variables from .conf file */
struct confvars cv[] = {
  {"border",VAR_INT,0,&border},
  {"bgcolor",VAR_STR,30,background},
  {"ram",VAR_INT,0,&mmon.enable},
  {"ram_used",VAR_STR,30,mmon.used_color},
  {"ram_buff",VAR_STR,30,mmon.buffers_color},
  {"ram_share",VAR_STR,30,mmon.shared_color},
  {"ram_cache",VAR_STR,30,mmon.cache_color},
  {"swap_used",VAR_STR,30,mmon.paged_color},
  {"battery",VAR_INT,0,&bmon.enable},
  {"batt_acpi_info",VAR_STR,PATH_MAX,bmon.acpi_info},
  {"batt_acpi_status",VAR_STR,PATH_MAX,bmon.acpi_status},
  {"battnorm",VAR_STR,30,bmon.norm_color},
  {"battlow",VAR_STR,30,bmon.low_color},
  {"percent_low",VAR_INT,0,&bmon.percent},
  {"cpu",VAR_INT,0,&cpum.enable},
  {"cpu_mode",VAR_INT,0,&cpum.mode},
  {"cpu_size",VAR_INT,0,&cpum.vsize},
  {"cpu_user",VAR_STR,30,cpum.user_color},
  {"cpu_nice",VAR_STR,30,cpum.nice_color},
  {"cpu_sys",VAR_STR,30,cpum.sys_color},
  {"ppp",VAR_INT,0,&ppp.enable},
  {"eth",VAR_INT,0,&eth.enable},
  {"pppicon",VAR_INT,0,&ppp.icon_type},
  {"ethicon",VAR_INT,0,&eth.icon_type},
  {"pppname",VAR_STR,20,ppp.intf_name},
  {"ethname",VAR_STR,20,eth.intf_name},
  {"pppspeed",VAR_INT,0,&ppp.speedkb},
  {"ethspeed",VAR_INT,0,&eth.speedkb},
  {"ppp_log",VAR_INT,0,&ppp.log_scale},
  {"eth_log",VAR_INT,0,&eth.log_scale},
  {"ppp_act",VAR_INT,0,&ppp.activity_led},
  {"eth_act",VAR_INT,0,&eth.activity_led},
  {"eth_in_color",VAR_STR,30,eth.in_color},
  {"eth_out_color",VAR_STR,30,eth.out_color},
  {"ppp_in_color",VAR_STR,30,ppp.in_color},
  {"ppp_out_color",VAR_STR,30,ppp.out_color},
  {"path1",VAR_STR,PATH_MAX,fmon.fsys1},
  {"path2",VAR_STR,PATH_MAX,fmon.fsys2},
  {"path3",VAR_STR,PATH_MAX,fmon.fsys3},
  {"fslow1",VAR_INT,0,&fmon.perc1},
  {"fslow2",VAR_INT,0,&fmon.perc2},
  {"fslow3",VAR_INT,0,&fmon.perc3},
  {"fsnorm",VAR_STR,30,fmon.used_color},
  {"fscrit",VAR_STR,30,fmon.crit_color},
  {"seti",VAR_INT,0,&setimon.enable},
  {"seticolor",VAR_STR,30,setimon.seti_color},
  {"setipath",VAR_STR,PATH_MAX,setimon.setiwd},
  {NULL,0,0,NULL}
};


/* path where to search for config file */
char *confpaths[] = {
  ".allin1.conf",			/* config file in home dir */
  "/usr/share/allin1/allin1.conf",	/* config in package dir */
  "/usr/local/share/allin1/allin1.conf",  /* building by tarball */
  ""
};


/* command line line argument list */
static struct option long_options[] = {
  {"help",	no_argument,	    0, 'h'},
  {"version",	no_argument,	    0, 'v'},
  {"config",	required_argument,  0, 'c'},
  {0, 0, 0, 0}
};


/* Help and version displaying. Added on 11th september 2002 by Pawel J. Maczewski */


void GetVersion(void)
{
  fprintf(stdout,"This is Allin1 version: %s",VERSION);
  fprintf(stdout,", relased on %s\n",RELDATE);
}


void GetHelp(void)
{
  fprintf(stdout,"\nThis is a Allin1 dockapp written by Mario Pascucci (ilpettegolo (a) yahoo it).\n");
  fprintf(stdout,"To read more about it, see the README file in your\n\t/usr/local/share/doc/allin1,\n\t/usr/share/doc/allin1 or\n\t/usr/doc/allin1 directory (depending on type of installation you had)\n");
  fprintf(stdout,"or visit the author's homepage http://ilpettegolo.altervista.org/\n(mirror http://www.geocities.com/ilpettegolo/)\n\n");
  fprintf(stdout,"This program currently accepts such command-line parameters:\n");
  fprintf(stdout," --help, -h\t\t- Shows this help message and exits.\n");
  fprintf(stdout," --version, -v\t\t- Displays the program version and exits.\n");
  fprintf(stdout," --config, -c FILE\t- Uses the config FILE instead of default one.\n");
}


int main (int argc, char *argv[]) {
  
  XEvent event;
  FILE *fp;
  int lasty = 0;	/* last y position of area */
  int sizex,sizey;	/* size of the applet */
  int optindex;		/* used for getopt */
  int i,fileconfig,userconfig;

  /* no default error messages printed by getopt */
  opterr = 0;
  userconfig = 0;
  /* setting default config */
  strcpy(background,"gray10");
  strcpy(mmon.used_color,"red");
  strcpy(mmon.buffers_color,"yellow");
  strcpy(mmon.shared_color,"lightblue");
  strcpy(mmon.cache_color,"orange");
  strcpy(mmon.paged_color,"yellow");
  mmon.enable = 1;
  mmon.update_dly = 2;
  strcpy(bmon.norm_color,"green");
  strcpy(bmon.low_color,"red");
  bmon.percent = 20;
  bmon.enable = 0;
  bmon.update_dly = 10;
  bmon.anim_dly = 2;
  strcpy(bmon.acpi_info,"/proc/acpi/battery/BAT0/info");
  strcpy(bmon.acpi_status,"/proc/acpi/battery/BAT0/state");
  strcpy(cpum.user_color,"green");
  strcpy(cpum.nice_color,"yellow");
  strcpy(cpum.sys_color,"orange");
  cpum.enable = 1;
  cpum.vsize = 18;
  cpum.update_dly = 1;
  cpum.cpu_num = -1;
  cpum.mode = 1;
  strcpy(eth.in_color,"red3");
  strcpy(eth.out_color,"green3");
  strcpy(ppp.in_color,"red3");
  strcpy(ppp.out_color,"green3");
  strcpy(eth.intf_name,"eth");
  strcpy(ppp.intf_name,"ppp");
  eth.enable = 1;
  eth.update_dly = 1;
  eth.speedkb = 100000;
  eth.log_scale = 1;
  eth.activity_led = 1;
  eth.icon_type = NIC_ICO; 
  ppp.enable = 1;
  ppp.update_dly = 1;
  ppp.speedkb = 115;
  ppp.log_scale = 0;
  ppp.activity_led = 1;
  ppp.icon_type = MDM_ICO;
  fmon.update_dly = 5;
  fmon.fsys1[0] = 0;
  fmon.fsys2[0] = 0;
  fmon.fsys3[0] = 0;
  strcpy(fmon.used_color,"green2");
  strcpy(fmon.crit_color,"red2");
  setimon.update_dly=100;
  strcpy(setimon.seti_color,"gold");
  strcpy(setimon.setiwd,"seti/state.sah");
  setimon.enable=0;
  do
  {
    optindex = 0;
    i = getopt_long (argc, argv, "hvc:",long_options, &optindex);
    switch (i)
    {
      case 'v':
	GetVersion();
	exit(0);
      case 'h':
	GetHelp();
	exit(0);
      case '?':
	fprintf(stderr,"Unknown or illegal option :%c\n",(char)optopt);
	GetHelp();
	exit(1);
      case 'c':
        if ((fp = fopen(optarg,"r")) != NULL)
        {
          fprintf(stdout,"Processing config file: %s\n",optarg);
          confparse(cv,fp);
          fclose(fp);
          userconfig = -1;
        }
        else 
        {
          fprintf(stderr,"Unable to read '%s' config file, exiting...\n", optarg);
          exit(1);
        }
        break;
    }
  }
  while (i != -1);
  chdir(getenv("HOME"));
  i = fileconfig = 0;
  if (userconfig == 0)	  /* no --config option given */
  {  
    while ((strlen(confpaths[i]) > 0) && !fileconfig)
    {
      if ((fp = fopen(confpaths[i],"r")) != NULL)
      {
        confparse(cv,fp);
        fclose(fp);
        fileconfig = -1;
      }
      i++;
    }
  }
  if ( (!fileconfig) && (!userconfig) )   /* no conf file found */
    fprintf(stderr,"No .conf file found. Uses default settings.\n");
  sizey = border;	    /* minimum size: only border... */
  if (mmon.enable) sizey += 10 + border;
  if (cpum.enable)
  {
    if (cpum.mode == CPUMON_GRAPH)
      sizey += cpum.vsize + border;
    else
      sizey += 6 + border;
  }
  if (bmon.enable) sizey += 8 + border;
  if (ppp.enable) sizey += 8 + border;
  if (eth.enable) sizey += 8 + border;
  filesys_setpath(&fmon);
  if (!mmon.enable && !cpum.enable && !bmon.enable &&
      !ppp.enable && !eth.enable && !fmon.fstotals)
  {
    fprintf(stderr, "No monitors enabled, nothing to display... exit.\n");
    exit(0);
  }
  switch (fmon.fstotals)
  {
    case 1:
      sizey += 6 + border;
      break;
    case 2:
      sizey += 10 + border;
      break;
    case 3:
      sizey += 15 + border;
      break;
  }
  if (setimon.enable) sizey += 6 + border;
  /* applet size */
  sizex = 64;
  dh_setsize(sizex,sizey,border);
  lasty = border;
  /* individual sections */
  /* first define area used for each gauge and new y position from top */
  if (mmon.enable)
  {
    mmon.handler = dh_addarea(border,lasty,sizex-2*border,10,TRUE,NULL);	/* mem/swap area */
    lasty += 10 + border;
  }
  if (bmon.enable) {
    bmon.icon_handler = dh_addarea(border,lasty,24,8,TRUE,NULL);	/* icon batt area */
    bmon.handler = dh_addarea(border+26,lasty,sizex-26-2*border,8,TRUE,NULL);	/* batt histogram */
    lasty += 8 + border;
  }
  if (cpum.enable)
  {
    if (cpum.mode == 1)
    {
      cpum.handler = dh_addarea(border,lasty,sizex-2*border,cpum.vsize,TRUE,NULL);	/* cpu graph */
      lasty += cpum.vsize + border;
    }
    else
    {
      cpum.handler = dh_addarea(border,lasty,sizex-2*border,6,TRUE,NULL);	/* cpu histogram */
      lasty += 6 + border;
    }
  }
  /* filesystem load histograms */
  switch (fmon.fstotals)
  {
    case 1:
      fmon.handler = dh_addarea(border,lasty,sizex-2*border,6,TRUE,NULL);
      lasty += 6 + border;
      break;
    case 2:
      fmon.handler = dh_addarea(border,lasty,sizex-2*border,10,TRUE,NULL);
      lasty += 10 + border;
      break;
    case 3:
      fmon.handler = dh_addarea(border,lasty,sizex-2*border,15,TRUE,NULL);
      lasty += 15 + border;
      break;
  }
  if (ppp.enable) {
    ppp.icon_handler = dh_addarea(border,lasty,12,8,TRUE,NULL);
    ppp.handler = dh_addarea(border+14,lasty,sizex-14-2*border,8,TRUE,NULL);
    lasty += 8 + border;
  }
  if (eth.enable) {
    eth.icon_handler = dh_addarea(border,lasty,12,8,TRUE,NULL);
    eth.handler = dh_addarea(border+14,lasty,sizex-14-2*border,8,TRUE,NULL);
    lasty += 8 + border;
  }
  /* Seti@home histogram */
  if (setimon.enable == 1)
  {
    setimon.handler = dh_addarea(border,lasty,sizex-2*border,6,TRUE,NULL);
    lasty += 6 + border;
  }
  dh_mainwin("allin1", background);
  dh_bevel();
  /* next, define size of usable area, colors and icon sizes */
  if (mmon.enable) 
  {
    /* size of usable area */
    mem_size(&mmon);
  }
  if (bmon.enable) {
    batt_size(&bmon);
    batt_icon_size(&bmon);
  }
  if (ppp.enable) {
    net_size(&ppp);
    net_icon_size(&ppp);
  }
  if (eth.enable) {
    net_size(&eth);
    net_icon_size(&eth);
  }
  if (fmon.fstotals > 0) 
  {
    filesys_size(&fmon);
  }
  if (cpum.enable)
  {
    cpu_size(&cpum);
  }
  if (setimon.enable)
  {
    seti_size(&setimon);
  }
  /* main loop */
  while (1) {
    while (XPending(dh_display())) {
      XNextEvent(dh_display(), &event);
      switch (event.type) {
      case Expose:
	while (XCheckTypedEvent(dh_display(), Expose, &event));
	dh_update();
	break;
      case DestroyNotify:
	XCloseDisplay(dh_display());
	exit(0);
	break;
      }
    }
    /* draws individual gauge, updating values first */
    if (mmon.enable && ((updates % mmon.update_dly) == 0))
    {
      mem_update(&mmon);
      mem_draw(&mmon);
    }
    if (cpum.enable && ((updates % cpum.update_dly) == 0))
    {
      cpu_update(&cpum);
      cpu_draw(&cpum);
    }
    if (ppp.enable && ((updates % ppp.update_dly) == 0))
    {
      net_update(&ppp);
      net_draw(&ppp);
      net_icon_draw(&ppp);
    }
    if (eth.enable && ((updates % eth.update_dly) == 0))
    {
      net_update(&eth);
      net_draw(&eth);
      net_icon_draw(&eth);
    }
    if ((fmon.fstotals) && ((updates % fmon.update_dly) == 0))
    {
      filesys_update(&fmon);
      filesys_draw(&fmon);
    }
    if ((bmon.enable) && ((updates % bmon.update_dly) == 0))
    {
      batt_update(&bmon);
      batt_draw(&bmon);
    }
    if ((bmon.enable) && ((updates % bmon.anim_dly) == 0))
    {
      batt_icon_draw(&bmon);
    }
    if ((setimon.enable) && ((updates % setimon.update_dly) == 0))
    {
      seti_update(&setimon);
      seti_draw(&setimon);
    }
    dh_update();
    usleep(update_time);
    updates++;
  }
  return 0;
}


