
/*

    battery.c - battery histograms and status for dock applets
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "dockhelper.h"
#include "battery.h"

#include "batt.xpm"



/* pixmap for icons */
static Pixmap batt_icons = (Pixmap) NULL;



/* reads apm status from /proc/apm */
void batt_update(struct battery_monitor *bm) 
{
  
  int i;
  FILE *f;
  char s[255],*p;

  bm->ac_line_status = 1;
  bm->battery_status = 0xff;
  bm->battery_percentage = 0;
  if (!(f = fopen ("/proc/apm", "r"))) 
  {
    batt_update_acpi(bm);
    return;
  }
  while (!feof (f)) 
  {
    fgets (s, 254, f);
    p = strchr(s,' ');
    sscanf (p, "%d.%d %x %x %x %x %d",&i,&i,&i,
	&(bm->ac_line_status),&(bm->battery_status),&i,&(bm->battery_percentage));
  }
  fclose(f);
}


/* reads (badly) the acpi status for the battery */
void batt_update_acpi(struct battery_monitor *bm)
{

  FILE *acpi;
  char buf[512];
  char stat;
  char *ptr;
  int last_full_capacity = -1;
  int rcapacity;

  bm->ac_line_status = 1;
  bm->battery_status = 0xff;
  bm->battery_percentage = 0;

  /*read the last full capacity (to compute the percentage) */
  if (!(acpi = fopen (bm->acpi_info, "r"))) return;
  fread (buf, 512, 1, acpi);
  fclose (acpi);

  if ((ptr = strstr (buf, "present:")) || (ptr = strstr (buf, "Present:")))
  {
    stat = *(ptr + 25);
    if (stat == 'y')
    {
      if ((ptr = strstr (buf, "last full capacity:"))
      || (ptr = strstr (buf, "Last Full Capacity:")))
      {
        ptr += 25;
        sscanf (ptr, "%d", &last_full_capacity);
      }
    }
  }

  if (last_full_capacity == -1) return;

  if (!(acpi = fopen (bm->acpi_status, "r"))) return;
  fread (buf, 512, 1, acpi);
  fclose (acpi);

  if ((ptr = strstr (buf, "present:")) || (ptr = strstr (buf, "Present:")))
  {
    stat = *(ptr + 25);
    if (stat == 'y')
    {
      if ((ptr = strstr (buf, "charging state:"))
	  || (ptr = strstr (buf, "State:")))
      {
	stat = *(ptr + 25);
	switch (stat)
	{
	  case 'd':
	    /* discharging */
	    bm->battery_status = 0;
	    bm->ac_line_status = 0;
	    break;
	  case 'c':
	    bm->ac_line_status = 1;
	    if (*(ptr + 33) == '/')
	      /* full */
	      bm->battery_status = 0;
	    else
	      /* charging */
	      bm->battery_status = 3;
	    break;
	  case 'u':
	    /* unknown */
	    bm->battery_status = 0xfe;
	    break;
	}
      }

      /* This section of the code will calculate "percentage remaining"
      * using battery capacity, and the following formula 
      * (acpi spec 3.9.2):
      * 
      * percentage = (current_capacity / last_full_capacity) * 100;
      *
      */

      if ((last_full_capacity > 0)
	  && ((ptr = strstr (buf, "remaining capacity:"))
	    || (ptr = strstr (buf, "Remaining Capacity:"))))
      {
	ptr += 25;
	sscanf (ptr, "%d", &rcapacity);

	bm->battery_percentage = rcapacity * 100 / last_full_capacity;
      }
    }
  }
}



/* set size for battery histogram */
int batt_size(struct battery_monitor *bm)
{

  dh_getsize(bm->handler, &(bm->xb), &(bm->yb), &(bm->wb), &(bm->hb));
  /* if size is less than 2x10 pixel is unusable */
  if ((bm->hb < 2) || (bm->wb < 10)) return -1;
  return 0;
}



/* set size for status icons */
int batt_icon_size(struct battery_monitor *bm)
{

  dh_getsize(bm->icon_handler, &(bm->xi), &(bm->yi), &(bm->wi), &(bm->hi));
  /* if size is less than 8x24 pixel is unusable */
  if ((bm->hi < 6) || (bm->wi < 22)) return -1;
  /* create icons pixmaps, unique for all gauges */
  if (batt_icons == (Pixmap) NULL)
  {
    dh_pixmap(&batt_icons,NULL,battery_xpm);
  }
  return 0;
}



/* draw battery histogram */
void batt_draw(struct battery_monitor *bm) {

  int ux;

  dh_color(dh_background());
  dh_fillrect(bm->xb,bm->yb,bm->wb,bm->hb);
  if (bm->battery_percentage < 0) return;    /* no battery */
  ux = (bm->battery_percentage * bm->wb) / 100;
  if (bm->battery_percentage <= bm->percent)
    dh_color(bm->low_color);
  else
    dh_color(bm->norm_color);
  dh_fillrect(bm->xb,bm->yb,ux,bm->hb);
}



/* draw icon battery */
void batt_icon_draw(struct battery_monitor *bm) {

  switch (bm->ac_line_status) {
  case 0:   /* no ac -> battery */
    bm->acindex = 6 * DH_ICONX;
    break;
  case 1:   /* on ac, no charge */
    bm->acindex = 4 * DH_ICONX;
    break;
  default:  /* unknown */
    bm->acindex = 6;
  }
  switch (bm->battery_status) {
  case 0:   /* full - norm */
    bm->battindex = 0;
    break;
  case 1:   /* low charge - ignored */
    bm->battindex = 0;
    break;
  case 2:   /* critical low - red */
    bm->battindex = 7 * DH_ICONX;
    break;
  case 3:   /* charging */
    bm->battindex = bm->battanim * DH_ICONX;
    bm->battanim = (bm->battanim + 1) % 4;
    if (bm->battery_percentage == 100) {
      /* buggy APM BIOS that reports battery in charge */
      /* even if fully charged */
      bm->battindex = 0;
    }
    break;
  case 0xff:	/* no battery */
    bm->battindex = 8 * DH_ICONX;
    break;
  default:  /* unknown */
    bm->battindex = 5 * DH_ICONX;
  }
  /* set ac power icon */
  dh_copyarea(batt_icons,bm->acindex,0,bm->xi,bm->yi,DH_ICONX,DH_ICONY);
  /* set batt status icon */
  dh_copyarea(batt_icons,bm->battindex,0,bm->xi+DH_ICONX+2,bm->yi,DH_ICONX,DH_ICONY);
}



