
/*

    filesys.c - file system histograms for dock applets
    Copyright (C) 2002-2004 Mario Pascucci <ilpettegolo (a) yahoo it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
    or
    visit the URL http://www.fsf.org/

*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/vfs.h>

#include "dockhelper.h"
#include "filesys.h"



/* reads filesys status */
void filesys_update(struct fsys_monitor *fmon) {
  
  struct statfs sfs;
  
  if (fmon->fsflag1) 
  {
    if (statfs(fmon->fsys1, &sfs)) 
    {
      fprintf(stderr, "Path: %s\n",fmon->fsys1);
      perror("Read filesys status failed.");
      return;
    }
    fmon->fstotal1 = sfs.f_blocks;
    fmon->fsfree1 = sfs.f_bfree;
  }
  if (fmon->fsflag2) 
  {
    if (statfs(fmon->fsys2, &sfs)) 
    {
      fprintf(stderr, "Path: %s\n",fmon->fsys2);
      perror("Read filesys status failed.");
      return;
    }
    fmon->fstotal2 = sfs.f_blocks;
    fmon->fsfree2 = sfs.f_bfree;
  }
  if (fmon->fsflag3) 
  {
    if (statfs(fmon->fsys3, &sfs)) 
    {
      fprintf(stderr, "Path: %s\n",fmon->fsys3);
      perror("Read filesys status failed.");
      return;
    }
    fmon->fstotal3 = sfs.f_blocks;
    fmon->fsfree3 = sfs.f_bfree;
  }
}



/* set size for filesystem histogram */
int filesys_size(struct fsys_monitor *fmon) {

  dh_getsize(fmon->handler, &(fmon->xf), &(fmon->yf), &(fmon->wf), &(fmon->hf));
  /* if size is less than 2x10 pixel is unusable */
  if ((fmon->hf < 2) || (fmon->wf < 10)) return -1;
  switch (fmon->fstotals) 
  {
    case 1:
      fmon->hi = fmon->hf;
      break;
    case 2:
      fmon->hi = fmon->hf / 2 - 1;
      fmon->y2 = fmon->yf + fmon->hi + 2;
      break;
    case 3:
      fmon->hi = fmon->hf / 3 - 1;
      fmon->y2 = fmon->yf + fmon->hi + 2;
      fmon->y3 = fmon->yf + fmon->hi + fmon->hi + 4;
      break;
  }
  return 0;
}



/* checks paths and flags for filesys histogram */
int filesys_setpath(struct fsys_monitor *fmon) 
{

  fmon->fstotals = 0;
  if (strlen(fmon->fsys1) != 0) 
  {
    fmon->fsflag1 = -1;
    fmon->fstotals++;
  }
  else fmon->fsflag1 = 0;
  if (strlen(fmon->fsys2) != 0) 
  {
    fmon->fsflag2 = -1;
    fmon->fstotals++;
  }
  else fmon->fsflag2 = 0;
  if (strlen(fmon->fsys3) != 0) 
  {
    fmon->fsflag3 = -1;
    fmon->fstotals++;
  }
  else fmon->fsflag3 = 0;
  return fmon->fstotals;
}



/* draw filesys histogram */
void filesys_draw(struct fsys_monitor *fmon) {

  int ux1=0,ux2=0,ux3=0;
  int p1=0,p2=0,p3=0;

  if (fmon->fsflag1)
  {
    if (fmon->fstotal1 == 0)
    {
      ux1 = 0;
      p1 = 0;
    }
    else
    {
      ux1 = ((fmon->fstotal1-fmon->fsfree1) * fmon->wf) / fmon->fstotal1;
      p1 = (fmon->fsfree1 * 100) / fmon->fstotal1;
    }
  }
  if (fmon->fsflag2)
  {
    if (fmon->fstotal2 == 0)
    {
      ux2 = 0;
      p2 = 0;
    }
    else
    {
      ux2 = ((fmon->fstotal2-fmon->fsfree2) * fmon->wf) / fmon->fstotal2;
      p2 = (fmon->fsfree2 * 100) / fmon->fstotal2;
    }
  }
  if (fmon->fsflag3)
  {
    if (fmon->fstotal3 == 0)
    {
      ux3 = 0;
      p3 = 0;
    }
    else
    {
      ux3 = ((fmon->fstotal3-fmon->fsfree3) * fmon->wf) / fmon->fstotal3;
      p3 = (fmon->fsfree3 * 100) / fmon->fstotal3;
    }
  }
  dh_color(dh_background());
  dh_fillrect(fmon->xf,fmon->yf,fmon->wf,fmon->hf);
  /* draw individual filesys histogram */
  if (fmon->fsflag1)
  {
    if (p1 <= fmon->perc1)
      dh_color(fmon->crit_color);
    else
      dh_color(fmon->used_color);
    dh_fillrect(fmon->xf,fmon->yf,ux1,fmon->hi);
  }
  if (fmon->fsflag2)
  {
    if (p2 <= fmon->perc2)
      dh_color(fmon->crit_color);
    else
      dh_color(fmon->used_color);
    dh_fillrect(fmon->xf,fmon->y2,ux2,fmon->hi);
  }
  if (fmon->fsflag3)
  {
    if (p3 <= fmon->perc3)
      dh_color(fmon->crit_color);
    else
      dh_color(fmon->used_color);
    dh_fillrect(fmon->xf,fmon->y3,ux3,fmon->hi);
  }
}



