/*********************************************************************
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ===================================================================
 * Revision History ::
 * YYYY.MM.DD   Change ID       Developer
 *              Description
 * -------------------------------------------------------------------
 * 2002.04.25                   Vlad Skarzhevskyy
 *              Initial implementation.
 *
 * ===================================================================
 *
 ********************************************************************/

#ifndef __unix

#include "pure-sfv.h"

#include <stddef.h>
#include <ctype.h>
#include <string.h>

#include "platform.h"

char*  optarg;
int    optind = 1;

/*
  FUNCTION: getopt()

      Get next command line option and parameter.
      Simple implementation to emulate unix getopt

  PARAMETERS:

      argc - count of command line arguments
      argv - array of command line argument strings
      validOpts - string of valid, case-sensitive option characters,
                  a colon ':' following a given character means that
                  option can take a parameter

      optarg - pointer to a pointer to a string for output
      optind - last scaned argument

  RETURNS:

      If valid option is found, the character value of that option
          is returned, and optarg points to the parameter if given,
          or is NULL if no param
      If standalone parameter (with no option) is found, 1 is returned,
          and optarg points to the standalone parameter
      If option is found, but it is not in the list of valid options,
          0 is returned
      When end of argument list is reached, EOF is returned.
*/

static int nextchar = 0;

int getopt(int argc, char** argv, const char* validOpts)
{
    char chOpt = 0, specifier;
    char* parg = NULL;
    char* popt = NULL;
    optarg = NULL;

    if (optind >= argc)
    {
        /* end of argument list */
        return EOF;
    }

    parg = &(argv[optind][0]);
    if ((*parg != '-') && (*parg != '/') && (!nextchar)) {
        /*
        fprintf(stderr, "Error: standalone arg %c given with no option specifier\n", *parg);
        return 0;
        */
        return EOF;
    }
        
    /* we have an option specifier */
    specifier = argv[optind][nextchar + 1];
    if ((!isalnum(specifier)) && !(ispunct(specifier))) {
        fprintf(stderr, "Error: though option specifier was given %c,\n", specifier);
        fprintf(stderr, "       option character is not alpha or was was not specified\n");
        return 0;
    }
        
    // we have an option character
    popt = strchr(validOpts, specifier);
    if (popt == NULL) {
        fprintf(stderr, "Error: option specified %c is not in list of valid options\n", specifier);
        return 0;
    }
    
    
    // option is valid, we want to return chOpt
    chOpt = specifier;
    if (popt[1] != ':')
    {
        // option is alone, has no parameter
        parg = &(argv[optind][nextchar + 2]);
        if (*parg == '\0')
        {
            optind++; 
            nextchar = 0;
        } 
        else
        {
            nextchar ++;
        }
    } 
    else
    {
        // option should have a parameter
        parg = &(argv[optind][nextchar + 2]);
        if (*parg != '\0')
        {
            // somthing is attached to option
            fprintf(stderr, "Error: the %c option requires an argument\n", chOpt);
            chOpt = 0;
        }
        else
        {
            nextchar = 0;
            optind++;
            // must look at next argv for param
            if (optind < argc)
            {
                parg = &(argv[optind][0]);
                if (*parg == '-' || *parg == '/')
                {
                    // next argv is a new option, so param
                    // not given for current option
                    fprintf(stderr, "Error: the %c option requires an argument\n", chOpt);
                    chOpt = 0;

                }
                else
                {
                    // next argv is the param
                    optind++;
                    optarg = parg;
                }
            }
            else
            {
                // reached end of args looking for param
                fprintf(stderr, "Error: the %c option requires an argument\n", chOpt);
                chOpt = 0;
            }

        }
    }
    
    return (chOpt);
}

#endif
/* EOF */
