// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief gamma gamma -> K*K*
  class ARGUS_2000_I511512 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(ARGUS_2000_I511512);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // Initialise and register projections
      declare(FinalState(), "FS");
      declare(UnstableParticles(), "UFS");
      // book histos
      raiseBeamErrorIf(!inRange(sqrtS()/GeV, 1.5, 3.0));
      // loop over tables in paper
      book(_h["K0K0_5"],      3, 1, 1);
      book(_h["K0K0_3"],      4, 1, 1);
      book(_h["K0K0_1"],      5, 1, 1);
      book(_h["K0K0_KSKSpp"], 9, 1, 1);
      book(_h["K0K0_KSK+pp"], 6, 1, 2);
      book(_h["K0K0"],       10, 1, 1);

      book(_h["KpKm_KSK+pp"], 6, 1, 1);
      book(_h["KpKm_KSKSpp"], 8, 1, 1);
      book(_h["KpKm_K+K-pp"], 9, 1, 2);
      book(_h["K+K-"],       10, 1, 2);

      book(_h["rhoPi_5"], 3, 1, 3);
      book(_h["rhoPi_3"], 4, 1, 3);
      book(_h["rhoPi_1"], 5, 1, 3);

      book(_h["K0K+-_5"], 3, 1, 2);
      book(_h["K0K+-_3"], 4, 1, 2);
      book(_h["K0K+-_1"], 5, 1, 2);

      book(_h["K+KS-l-"],  7, 1, 1);
      book(_h["K+KS-p-"],  8, 1, 2);
      book(_h["K+K-p0"],   7, 1, 2);
      book(_h["KSKSp-p0"], 7, 1, 3);
      book(_h["KSKSp+p-"], 8, 1, 3);

      book(_h["K+K-p+p-_5"], 3, 1, 4);
      book(_h["K+K-p+p-_3"], 4, 1, 4);
      book(_h["K+K-p+p-_1"], 5, 1, 4);

      book(_h["num"], "TMP/nMeson_num", refData(11, 1, 1));
      book(_h["den"], "TMP/nMeson_den", refData(11, 1, 1));
    }

    void findChildren(const Particle& p, map<long,int>& nRes, int& ncount) const {
      for (const Particle& child : p.children()) {
        if (child.children().empty()) {
          nRes[child.pid()]-=1;
          --ncount;
        }
        else {
          findChildren(child,nRes,ncount);
        }
      }
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const FinalState& fs = apply<FinalState>(event, "FS");
      // find the final-state particles
      map<long,int> nCount;
      int ntotal(0);
      for (const Particle& p : fs.particles()) {
        nCount[p.pid()] += 1;
        ++ntotal;
      }
      if (ntotal==4) {
        if (nCount[PID::KPLUS]==1 && nCount[PID::KMINUS]==1 &&
            nCount[PID::PIPLUS]==1 && nCount[PID::PIMINUS]==1) {
          _h["K+K-p+p-_5"]->fill(sqrtS()/GeV);
          _h["K+K-p+p-_3"]->fill(sqrtS()/GeV);
          _h["K+K-p+p-_1"]->fill(sqrtS()/GeV);
        }
        else if (nCount[PID::K0S]==2 && nCount[PID::PIPLUS]==1 &&
                 nCount[PID::PIMINUS]==1) {
          _h["KSKSp+p-"]->fill(sqrtS()/GeV);
        }
        else if (nCount[PID::K0S]==1 && nCount[PID::PI0]==1 &&
               ((nCount[PID::KPLUS ]==1 && nCount[PID::PIMINUS]==1) ||
                (nCount[PID::KMINUS]==1 && nCount[PID::PIPLUS ]==1))) {
          _h["KSKSp-p0"]->fill(sqrtS()/GeV);
        }

      }
      const UnstableParticles& ufs = apply<UnstableParticles>(event, "UFS");
      // find any K* mesons
      Particles Kstar=ufs.particles(Cuts::abspid==313 or Cuts::abspid==323);
      for (size_t ix=0; ix<Kstar.size(); ++ix) {
       	if (Kstar[ix].children().empty()) continue;
       	map<long,int> nRes=nCount;
       	int ncount = ntotal;
       	findChildren(Kstar[ix],nRes,ncount);
        int sign = Kstar[ix].pid()/Kstar[ix].abspid();
        // three body intermediate states
        if (ncount==2) {
          // K*0 K- pi+ +ccd
          if (Kstar[ix].abspid()==313) {
            bool matched=true;
            for (const auto& val : nRes) {
              if (val.first==sign*211 || val.first==-sign*321) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
              }
              else {
                if (val.second!=0) {
                  matched = false;
                  break;
                }
              }
            }
            if (matched) {
              _h["K0K+-_5"]->fill(sqrtS()/GeV);
              _h["K0K+-_3"]->fill(sqrtS()/GeV);
              _h["K0K+-_1"]->fill(sqrtS()/GeV);
            }
          }
          else {
            bool matched=false;
            // K*+ K0S pi- + cc
            for (const auto& val : nRes) {
              if (val.first==-sign*211 || val.first==PID::K0S) {
                if (val.second!=1) {
                  matched = false;
                  break;
                }
              }
              else {
                if (val.second!=0) {
                  matched = false;
                  break;
                }
              }
            }
            if (matched) {
              _h["K+KS-l-"]->fill(sqrtS()/GeV);
              _h["K+KS-p-"]->fill(sqrtS()/GeV);
            }
            else {
              // K*+ K- pi0 +cc
              matched=false;
              for (const auto& val : nRes) {
                if (val.first==-sign*321 || val.first==PID::PI0) {
                  if (val.second!=1) {
                    matched = false;
                    break;
                  }
                }
                else {
                  if (val.second!=0) {
                    matched = false;
                    break;
                  }
                }
              }
              if (matched)  _h["K+K-p0"]->fill(sqrtS()/GeV);
            }
          }
        }

        // K*K*
        for (size_t iy=ix+1; iy<Kstar.size(); ++iy) {
          if (Kstar[iy].children().empty()) continue;
          if (Kstar[ix].pid()!=-Kstar[iy].pid()) continue;
          map<long,int> nRes2=nRes;
          int ncount2 = ncount;
          findChildren(Kstar[iy],nRes2,ncount2);
          if (ncount2 !=0 ) continue;
          bool matched2 = true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched2 = false;
              break;
            }
          }
          if (matched2) {
            if (Kstar[ix].abspid()==313) {
              _h["K0K0_5"]->fill(sqrtS()/GeV);
              _h["K0K0_3"]->fill(sqrtS()/GeV);
              _h["K0K0_1"]->fill(sqrtS()/GeV);
              _h["K0K0_KSKSpp"]->fill(sqrtS()/GeV);
              _h["K0K0_KSK+pp"]->fill(sqrtS()/GeV);
              _h["K0K0"]->fill(sqrtS()/GeV);
              _h["den"]->fill(sqrtS()/GeV);
            }
            else {
              _h["KpKm_KSK+pp"]->fill(sqrtS()/GeV);
              _h["KpKm_KSKSpp"]->fill(sqrtS()/GeV);
              _h["KpKm_K+K-pp"]->fill(sqrtS()/GeV);
              _h["K+K-"]->fill(sqrtS()/GeV);
              _h["num"]->fill(sqrtS()/GeV);
            }
            break;
          }
        }
      }
      // finally the rho phi intermediate states
      for (const Particle& p1 : ufs.particles(Cuts::pid==PID::RHO0)) {
       	if (p1.children().empty()) continue;
       	map<long,int> nRes=nCount;
       	int ncount = ntotal;
       	findChildren(p1,nRes,ncount);
        for (const Particle& p2 : ufs.particles(Cuts::pid==PID::PHI)) {
          if (p2.children().empty()) continue;
          map<long,int> nRes2=nRes;
          int ncount2 = ncount;
          findChildren(p2,nRes2,ncount2);
          if (ncount2 !=0 ) continue;
          bool matched = true;
          for (const auto& val : nRes2) {
            if (val.second!=0) {
              matched = false;
              break;
            }
          }
          if (matched) {
            _h["rhoPi_5"]->fill(sqrtS()/GeV);
            _h["rhoPi_3"]->fill(sqrtS()/GeV);
            _h["rhoPi_1"]->fill(sqrtS()/GeV);
            break;
          }
        }
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_h, crossSection()/nanobarn/sumOfWeights());
      // finally the ratio K*+/K*0
      Estimate1DPtr mult;
      book(mult, 11, 1, 1);
      divide(_h["num"], _h["den"], mult);
    }

    /// @}


    /// @name Histograms
    /// @{
    map<string,Histo1DPtr> _h;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(ARGUS_2000_I511512);

}
