// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package registry

import (
	"context"

	"github.com/drone/drone/core"
	"github.com/drone/drone/logger"

	"github.com/sirupsen/logrus"
)

// Combine combines the registry services, allowing the
// system to source registry credential from multiple sources.
func Combine(services ...core.RegistryService) core.RegistryService {
	return &combined{services}
}

type combined struct {
	sources []core.RegistryService
}

func (c *combined) List(ctx context.Context, req *core.RegistryArgs) ([]*core.Registry, error) {
	var all []*core.Registry
	for _, source := range c.sources {
		list, err := source.List(ctx, req)
		if err != nil {
			return all, err
		}
		all = append(all, list...)
	}
	// if trace level debugging is enabled we print
	// all registry credentials retrieved from the
	// various registry sources.
	logger := logger.FromContext(ctx)
	if logrus.IsLevelEnabled(logrus.TraceLevel) {
		if len(all) == 0 {
			logger.Traceln("registry: no registry credentials loaded")
		}
		for _, registry := range all {
			logger.WithField("address", registry.Address).
				Traceln("registry: registry credentials loaded")
		}
	}
	return all, nil
}
