// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package registry

import (
	"testing"

	"github.com/drone/drone-yaml/yaml"
	"github.com/drone/drone/core"
	"github.com/google/go-cmp/cmp"
)

var mockDockerAuthConfig = `{
	"auths": {
		"https://index.docker.io/v1/": {
			"auth": "b2N0b2NhdDpjb3JyZWN0LWhvcnNlLWJhdHRlcnktc3RhcGxl"
		}
	}
}`

func TestStatic(t *testing.T) {
	secrets := []*core.Secret{
		{
			Name: "dockerhub",
			Data: mockDockerAuthConfig,
		},
	}

	manifest, err := yaml.ParseString("kind: pipeline\nimage_pull_secrets: [ dockerhub ]")
	if err != nil {
		t.Error(err)
		return
	}

	args := &core.RegistryArgs{
		Build:    &core.Build{Event: core.EventPush},
		Conf:     manifest,
		Pipeline: manifest.Resources[0].(*yaml.Pipeline),
	}
	service := Static(secrets)
	got, err := service.List(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}

	want := []*core.Registry{
		{
			Address:  "https://index.docker.io/v1/",
			Username: "octocat",
			Password: "correct-horse-battery-staple",
		},
	}
	if diff := cmp.Diff(got, want); diff != "" {
		t.Errorf(diff)
		return
	}
}

func TestStatic_NoMatch(t *testing.T) {
	secrets := []*core.Secret{
		{
			Name: "dockerhub",
			Data: mockDockerAuthConfig,
		},
	}

	manifest, err := yaml.ParseString("kind: pipeline\nimage_pull_secrets: [ unknown ]")
	if err != nil {
		t.Error(err)
		return
	}

	args := &core.RegistryArgs{
		Build:    &core.Build{Event: core.EventPush},
		Conf:     manifest,
		Pipeline: manifest.Resources[0].(*yaml.Pipeline),
	}
	service := Static(secrets)
	got, err := service.List(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if len(got) != 0 {
		t.Errorf("Expect no results")
	}
}

func TestStatic_DisablePullRequest(t *testing.T) {
	secrets := []*core.Secret{
		{
			Name:        "dockerhub",
			Data:        mockDockerAuthConfig,
			PullRequest: false,
		},
	}

	manifest, err := yaml.ParseString("kind: pipeline\nimage_pull_secrets: [ dockerhub ]")
	if err != nil {
		t.Error(err)
		return
	}

	args := &core.RegistryArgs{
		Build:    &core.Build{Event: core.EventPullRequest},
		Conf:     manifest,
		Pipeline: manifest.Resources[0].(*yaml.Pipeline),
	}
	service := Static(secrets)
	got, err := service.List(noContext, args)
	if err != nil {
		t.Error(err)
		return
	}
	if len(got) != 0 {
		t.Errorf("Expect no results")
	}
}
