// Copyright 2019 Drone.IO Inc. All rights reserved.
// Use of this source code is governed by the Drone Non-Commercial License
// that can be found in the LICENSE file.

package logs

import (
	"bytes"
	"context"
	"database/sql"
	"io/ioutil"
	"testing"

	"github.com/drone/drone/store/shared/db/dbtest"
	"github.com/drone/drone/core"
	"github.com/drone/drone/store/build"
	"github.com/drone/drone/store/repos"
	"github.com/drone/drone/store/step"
)

var noContext = context.TODO()

func TestLogs(t *testing.T) {
	conn, err := dbtest.Connect()
	if err != nil {
		t.Error(err)
		return
	}
	defer func() {
		dbtest.Reset(conn)
		dbtest.Disconnect(conn)
	}()

	// seed with a dummy repository
	arepo := &core.Repository{UID: "1", Slug: "octocat/hello-world"}
	repos := repos.New(conn)
	repos.Create(noContext, arepo)

	// seed with a dummy stage
	stage := &core.Stage{Number: 1}
	stages := []*core.Stage{stage}

	// seed with a dummy build
	abuild := &core.Build{Number: 1, RepoID: arepo.ID}
	builds := build.New(conn)
	builds.Create(noContext, abuild, stages)

	// seed with a dummy step
	astep := &core.Step{Number: 1, StageID: stage.ID}
	steps := step.New(conn)
	steps.Create(noContext, astep)

	store := New(conn).(*logStore)
	t.Run("Create", testLogsCreate(store, astep))
	t.Run("Find", testLogsFind(store, astep))
	t.Run("Update", testLogsUpdate(store, astep))
	t.Run("Delete", testLogsDelete(store, astep))
}

func testLogsCreate(store *logStore, step *core.Step) func(t *testing.T) {
	return func(t *testing.T) {
		buf := bytes.NewBufferString("hello world")
		err := store.Create(noContext, step.ID, buf)
		if err != nil {
			t.Error(err)
		}
	}
}

func testLogsFind(store *logStore, step *core.Step) func(t *testing.T) {
	return func(t *testing.T) {
		r, err := store.Find(noContext, step.ID)
		if err != nil {
			t.Error(err)
			return
		}
		data, err := ioutil.ReadAll(r)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := string(data), "hello world"; got != want {
			t.Errorf("Want log output stream %q, got %q", want, got)
		}
	}
}

func testLogsUpdate(store *logStore, step *core.Step) func(t *testing.T) {
	return func(t *testing.T) {
		buf := bytes.NewBufferString("hola mundo")
		err := store.Update(noContext, step.ID, buf)
		if err != nil {
			t.Error(err)
			return
		}
		r, err := store.Find(noContext, step.ID)
		if err != nil {
			t.Error(err)
			return
		}
		data, err := ioutil.ReadAll(r)
		if err != nil {
			t.Error(err)
			return
		}
		if got, want := string(data), "hola mundo"; got != want {
			t.Errorf("Want updated log output stream %q, got %q", want, got)
		}
	}
}

func testLogsDelete(store *logStore, step *core.Step) func(t *testing.T) {
	return func(t *testing.T) {
		err := store.Delete(noContext, step.ID)
		if err != nil {
			t.Error(err)
			return
		}
		_, err = store.Find(noContext, step.ID)
		if got, want := sql.ErrNoRows, err; got != want {
			t.Errorf("Want sql.ErrNoRows, got %v", got)
			return
		}
	}
}
