// Licensed to the Software Freedom Conservancy (SFC) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The SFC licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

package org.openqa.selenium.bidi.network;

import java.util.Map;
import org.openqa.selenium.internal.Require;
import org.openqa.selenium.json.JsonInput;

/**
 * @see <a href="https://www.w3.org/TR/webdriver-bidi/#type-network-Header">BiDi spec</a>
 */
public class Header {
  private final String name;
  private final BytesValue value;

  public Header(String name, BytesValue value) {
    this.name = name;
    this.value = value;
  }

  public static Header fromJson(JsonInput input) {
    String name = null;
    BytesValue value = null;

    input.beginObject();
    while (input.hasNext()) {
      switch (input.nextName()) {
        case "name":
          name = input.read(String.class);
          break;
        case "value":
          value = input.read(BytesValue.class);
          break;
        default:
          input.skipValue();
      }
    }

    input.endObject();

    return new Header(Require.nonNull("name", name), Require.nonNull("value", value));
  }

  public String getName() {
    return name;
  }

  public BytesValue getValue() {
    return value;
  }

  public Map<String, Object> toMap() {
    return Map.of("name", this.name, "value", this.value.toMap());
  }
}
