module CSharpLanguageServer.State.ServerState

open System
open System.Threading
open System.Threading.Tasks

open Ionide.LanguageServerProtocol.Types
open Ionide.LanguageServerProtocol
open Microsoft.Extensions.Logging
open Newtonsoft.Json.Linq

open CSharpLanguageServer.Logging
open CSharpLanguageServer.Roslyn.Conversions
open CSharpLanguageServer.Roslyn.Solution
open CSharpLanguageServer.Lsp.Workspace
open CSharpLanguageServer.Lsp.WorkspaceFolder
open CSharpLanguageServer.Types
open CSharpLanguageServer.Util
open CSharpLanguageServer.Lsp

type ServerRequestMode =
    | ReadOnly
    | ReadWrite

type RequestMetrics =
    { Count: int
      TotalDuration: TimeSpan
      MaxDuration: TimeSpan
      ImpactedRequestsCount: int
      TotalImpactedWaitingTime: TimeSpan }

    static member Zero =
        { Count = 0
          TotalDuration = TimeSpan.Zero
          MaxDuration = TimeSpan.Zero
          ImpactedRequestsCount = 0
          TotalImpactedWaitingTime = TimeSpan.Zero }

type RequestDetails =
    { Name: string
      Mode: ServerRequestMode
      Priority: int  // 0 is the highest priority, 1 is lower prio, etc.
    // priority is used to order pending R/O requests and is ignored wrt R/W requests
    }

type QueuedRequest =
    { Details: RequestDetails
      Id: int
      Semaphore: SemaphoreSlim
      StartedProcessing: option<DateTime>
      Enqueued: DateTime }

type ServerState =
    { Settings: ServerSettings
      LspClient: ILspClient option
      ClientCapabilities: ClientCapabilities
      Workspace: LspWorkspace
      LastRequestId: int
      PendingRequests: QueuedRequest list
      RunningRequests: Map<int, QueuedRequest>
      WorkspaceReloadPending: DateTime option
      PushDiagnosticsDocumentBacklog: string list
      PushDiagnosticsCurrentDocTask: (string * Task) option
      RequestStats: Map<string, RequestMetrics>
      LastStatsDumpTime: DateTime }

    static member Empty =
        { Settings = ServerSettings.Default
          LspClient = None
          ClientCapabilities = emptyClientCapabilities
          Workspace = LspWorkspace.Empty
          LastRequestId = 0
          PendingRequests = []
          RunningRequests = Map.empty
          WorkspaceReloadPending = None
          PushDiagnosticsDocumentBacklog = []
          PushDiagnosticsCurrentDocTask = None
          RequestStats = Map.empty
          LastStatsDumpTime = DateTime.MinValue }

let tryPullNextRunnableRequest state =
    let noRWRequestRunning =
        state.RunningRequests
        |> Seq.tryFind (fun r -> r.Value.Details.Mode = ReadWrite)
        |> Option.isNone

    let canPullNextRequest = noRWRequestRunning && state.WorkspaceReloadPending.IsNone

    match canPullNextRequest, state.PendingRequests with
    | false, _ -> None, state

    | true, [] -> None, state

    | true, nonEmptyRequestQueue ->
        let requestIsReadOnly (r: QueuedRequest) = (r.Details.Mode = ReadOnly)

        // here we will try to take non-interrupted r/o request sequence at the front,
        // order it by priority and run the most prioritized one first
        let nextRoRequestByPriorityMaybe =
            nonEmptyRequestQueue
            |> Seq.takeWhile requestIsReadOnly
            |> Seq.sortBy _.Details.Priority
            |> Seq.tryHead

        // otherwise, if no r/o request by priority was found then we should just take the first request
        let nextRequest =
            nextRoRequestByPriorityMaybe
            |> Option.defaultValue (nonEmptyRequestQueue |> Seq.head)

        let newPendingRequests = nonEmptyRequestQueue |> List.except [ nextRequest ]

        let nextRunnableRequest =
            { nextRequest with
                StartedProcessing = Some DateTime.Now }

        let newRunningRequests =
            state.RunningRequests |> Map.add nextRunnableRequest.Id nextRunnableRequest

        let newState =
            { state with
                RunningRequests = newRunningRequests
                PendingRequests = newPendingRequests }

        Some nextRunnableRequest, newState

type ServerStateEvent =
    | ClientCapabilityChange of ClientCapabilities
    | ClientChange of ILspClient option
    | DocumentClosed of string
    | DocumentOpened of string * int * DateTime
    | DocumentTouched of string * DateTime
    | DumpAndResetRequestStats
    | FinishRequest of int
    | GetState of AsyncReplyChannel<ServerState>
    | PeriodicTimerTick
    | ProcessRequestQueue
    | PushDiagnosticsDocumentBacklogUpdate
    | PushDiagnosticsDocumentDiagnosticsResolution of Result<(string * int option * Diagnostic array), Exception>
    | PushDiagnosticsProcessPendingDocuments
    | SettingsChange of ServerSettings
    | StartRequest of RequestDetails * AsyncReplyChannel<int * SemaphoreSlim>
    | WorkspaceConfigurationChanged of WorkspaceFolder list
    | WorkspaceFolderChange of LspWorkspaceFolder
    | WorkspaceReloadRequested of TimeSpan

let processFinishRequest postServerEvent state request =
    request.Semaphore.Dispose()

    let newRequestStats =
        let requestExecutionDuration: TimeSpan =
            match request.StartedProcessing with
            | Some startTime -> DateTime.Now - startTime
            | None -> DateTime.Now - request.Enqueued

        let updateRequestStats (stats: RequestMetrics option) : RequestMetrics option =
            match stats with
            | None ->
                { RequestMetrics.Zero with
                    Count = 1
                    TotalDuration = requestExecutionDuration
                    MaxDuration = requestExecutionDuration }
                |> Some
            | Some s ->
                let (impactedCount, totalImpactedWait) =
                    if request.Details.Mode = ReadWrite then
                        let blockingStartTime =
                            request.StartedProcessing |> Option.defaultValue request.Enqueued

                        let aggregateBlockedRequestStats (count, totalWait) pendingRequest =
                            let waitStartTime = max blockingStartTime pendingRequest.Enqueued
                            let waitDurationForPending = DateTime.Now - waitStartTime

                            (count + 1, totalWait + waitDurationForPending)

                        state.PendingRequests
                        |> List.fold aggregateBlockedRequestStats (0, TimeSpan.Zero)
                    else
                        (0, TimeSpan.Zero)

                Some
                    { s with
                        Count = s.Count + 1
                        TotalDuration = s.TotalDuration + requestExecutionDuration
                        MaxDuration = max s.MaxDuration requestExecutionDuration
                        ImpactedRequestsCount = s.ImpactedRequestsCount + impactedCount
                        TotalImpactedWaitingTime = s.TotalImpactedWaitingTime + totalImpactedWait }

        match state.Settings.DebugMode with
        | true -> state.RequestStats |> Map.change request.Details.Name updateRequestStats
        | false -> state.RequestStats

    let newRunningRequests = state.RunningRequests |> Map.remove request.Id

    let newState =
        { state with
            RunningRequests = newRunningRequests
            RequestStats = newRequestStats }

    postServerEvent ProcessRequestQueue
    newState

let processDumpAndResetRequestStats (logger: ILogger) state =
    let formatStats stats =
        let calculateRequestStatsMetrics (name, metrics) =
            let avgDurationMs =
                if metrics.Count > 0 then
                    metrics.TotalDuration.TotalMilliseconds / float metrics.Count
                else
                    0.0

            let avgImpactedWaitMs =
                if metrics.ImpactedRequestsCount > 0 then
                    metrics.TotalImpactedWaitingTime.TotalMilliseconds
                    / float metrics.ImpactedRequestsCount
                else
                    0.0

            (name, metrics, avgImpactedWaitMs, avgDurationMs)

        let sortedStats =
            stats
            |> Map.toList
            |> List.map calculateRequestStatsMetrics
            |> List.sortByDescending (fun (_, _, _, avgDuration) -> avgDuration)

        let formatStatsRowWithImpact (name, metrics, avgImpactedWaitMs: float, avgDurationMs: float) =
            [ $"\"{name}\""
              metrics.Count |> string
              avgDurationMs.ToString("F2")
              metrics.MaxDuration.TotalMilliseconds.ToString("F2")
              sprintf "%d (%s ms on avg)" metrics.ImpactedRequestsCount (avgImpactedWaitMs.ToString("F2")) ]

        let headerRow =
            [ "Name"; "Count"; "AvgDuration (ms)"; "MaxDuration (ms)"; "ImpactedRequests" ]

        let dataRows = sortedStats |> List.map formatStatsRowWithImpact

        formatInColumns (headerRow :: dataRows)

    if not (Map.isEmpty state.RequestStats) then
        logger.LogDebug("--------- Request Stats ---------")
        logger.LogDebug("{stats}", (state.RequestStats |> formatStats))
        logger.LogDebug("---------------------------------")
    else
        logger.LogDebug("------- No request stats  -------")

    { state with
        RequestStats = Map.empty
        LastStatsDumpTime = DateTime.Now }

let processServerEvent (logger: ILogger) state postServerEvent ev : Async<ServerState> = async {
    match ev with
    | SettingsChange newSettings ->
        let newState: ServerState = { state with Settings = newSettings }

        let solutionChanged =
            not (state.Settings.SolutionPath = newState.Settings.SolutionPath)

        if solutionChanged then
            postServerEvent (WorkspaceReloadRequested(TimeSpan.FromMilliseconds(int64 250)))

        return newState

    | GetState replyChannel ->
        replyChannel.Reply(state)
        return state

    | StartRequest(requestDetails, replyChannel) ->
        postServerEvent ProcessRequestQueue

        let newRequest =
            { Details = requestDetails
              Id = state.LastRequestId + 1
              Semaphore = new SemaphoreSlim(0, 1)
              StartedProcessing = None
              Enqueued = DateTime.Now }

        replyChannel.Reply((newRequest.Id, newRequest.Semaphore))

        return
            { state with
                LastRequestId = newRequest.Id
                PendingRequests = state.PendingRequests @ [ newRequest ] }

    | FinishRequest requestId ->
        let request = state.RunningRequests |> Map.tryFind requestId

        match request with
        | Some request -> return processFinishRequest postServerEvent state request

        | None ->
            logger.LogWarning(
                "serverEventLoop/FinishRequest#{requestId}: request not found in state.RunningRequests",
                requestId
            )

            return state

    | ProcessRequestQueue ->
        let nextRunnableRequest, newState = tryPullNextRunnableRequest state

        match nextRunnableRequest with
        | Some requestToRun ->
            // try to process next msg from the remainder, if possible, later
            postServerEvent ProcessRequestQueue

            // unblock this request to run by sending it current state
            requestToRun.Semaphore.Release() |> ignore

        | None -> ()

        return newState

    | WorkspaceConfigurationChanged workspaceFolders ->
        let newWorkspace = workspaceFrom workspaceFolders
        return { state with Workspace = newWorkspace }

    | ClientChange lspClient -> return { state with LspClient = lspClient }

    | ClientCapabilityChange cc ->
        let experimentalCapsBoolValue boolPropName =
            cc.Experimental
            |> Option.map _.SelectToken(boolPropName)
            |> Option.bind Option.ofObj
            |> Option.map (fun t ->
                let v = t :?> JValue
                v.Value :?> bool)

        let oldSettings = state.Settings

        let newSettings =
            { oldSettings with
                UseMetadataUris =
                    experimentalCapsBoolValue "csharp.metadataUris"
                    |> Option.defaultValue oldSettings.UseMetadataUris }

        return
            { state with
                ClientCapabilities = cc
                Settings = newSettings }

    | WorkspaceFolderChange updatedWf ->
        let updatedWorkspaceFolderList =
            state.Workspace.Folders
            |> List.map (fun wf -> if wf.Uri = updatedWf.Uri then updatedWf else wf)

        // request queue may have been blocked due to workspace folder(s)
        // not having solution loaded yet
        postServerEvent ProcessRequestQueue

        let newWorkspace =
            { state.Workspace with
                Folders = updatedWorkspaceFolderList }

        let newState = { state with Workspace = newWorkspace }

        return newState

    | DocumentOpened(uri, ver, timestamp) ->
        postServerEvent PushDiagnosticsDocumentBacklogUpdate

        let openDocInfo = { Version = ver; Touched = timestamp }
        let newOpenDocs = state.Workspace.OpenDocs |> Map.add uri openDocInfo

        return
            { state with
                Workspace =
                    { state.Workspace with
                        OpenDocs = newOpenDocs } }

    | DocumentClosed uri ->
        postServerEvent PushDiagnosticsDocumentBacklogUpdate

        let newOpenDocVersions = state.Workspace.OpenDocs |> Map.remove uri

        return
            { state with
                Workspace =
                    { state.Workspace with
                        OpenDocs = newOpenDocVersions } }

    | DocumentTouched(uri, timestamp) ->
        postServerEvent PushDiagnosticsDocumentBacklogUpdate

        let openDocInfo = state.Workspace.OpenDocs |> Map.tryFind uri

        match openDocInfo with
        | None -> return state
        | Some openDocInfo ->
            let updatedOpenDocInfo = { openDocInfo with Touched = timestamp }
            let newOpenDocVersions = state.Workspace.OpenDocs |> Map.add uri updatedOpenDocInfo

            return
                { state with
                    Workspace =
                        { state.Workspace with
                            OpenDocs = newOpenDocVersions } }

    | WorkspaceReloadRequested reloadNoLaterThanIn ->
        // we need to wait a bit before starting this so we
        // can buffer many incoming requests at once
        let newSolutionReloadDeadline =
            let suggestedDeadline = DateTime.Now + reloadNoLaterThanIn

            match state.WorkspaceReloadPending with
            | Some currentDeadline ->
                if suggestedDeadline < currentDeadline then
                    suggestedDeadline
                else
                    currentDeadline
            | None -> suggestedDeadline

        return
            { state with
                WorkspaceReloadPending = newSolutionReloadDeadline |> Some }

    | PushDiagnosticsDocumentBacklogUpdate ->
        // here we build new backlog for background diagnostics processing
        // which will consider documents by their last modification date
        // for processing first
        let newBacklog =
            state.Workspace.OpenDocs
            |> Seq.sortByDescending (fun kv -> kv.Value.Touched)
            |> Seq.map (fun kv -> kv.Key)
            |> List.ofSeq

        return
            { state with
                PushDiagnosticsDocumentBacklog = newBacklog }

    | PushDiagnosticsProcessPendingDocuments ->
        match state.PushDiagnosticsCurrentDocTask with
        | Some _ ->
            // another document is still being processed, do nothing
            return state
        | None ->
            // try pull next doc from the backlog to process
            let nextDocUri, newBacklog =
                match state.PushDiagnosticsDocumentBacklog with
                | [] -> (None, [])
                | uri :: remainder -> (Some uri, remainder)

            // push diagnostic is enabled only if pull diagnostics is
            // not reported to be supported by the client
            let diagnosticPullSupported =
                state.ClientCapabilities.TextDocument
                |> Option.map _.Diagnostic
                |> Option.map _.IsSome
                |> Option.defaultValue false

            match diagnosticPullSupported, nextDocUri with
            | false, Some docUri ->
                let newState =
                    { state with
                        PushDiagnosticsDocumentBacklog = newBacklog }

                let wf, docForUri = docUri |> workspaceDocument state.Workspace AnyDocument
                let wfPathToUri = workspaceFolderPathToUri wf.Value

                match wf, docForUri with
                | Some wf, None ->
                    let cshtmlPath = workspaceFolderUriToPath wf docUri |> _.Value

                    match! solutionGetRazorDocumentForPath wf.Solution.Value cshtmlPath with
                    | Some(_, compilation, cshtmlTree) ->
                        let semanticModelMaybe = compilation.GetSemanticModel cshtmlTree |> Option.ofObj

                        match semanticModelMaybe with
                        | None ->
                            Error(Exception "could not GetSemanticModelAsync")
                            |> PushDiagnosticsDocumentDiagnosticsResolution
                            |> postServerEvent

                        | Some semanticModel ->
                            let diagnostics =
                                semanticModel.GetDiagnostics()
                                |> Seq.map (Diagnostic.fromRoslynDiagnostic (workspaceFolderPathToUri wf))
                                |> Seq.filter (fun (_, uri) -> uri = docUri)
                                |> Seq.map fst
                                |> Array.ofSeq

                            Ok(docUri, None, diagnostics)
                            |> PushDiagnosticsDocumentDiagnosticsResolution
                            |> postServerEvent

                    | None ->
                        // could not find document for this enqueued uri
                        logger.LogDebug(
                            "PushDiagnosticsProcessPendingDocuments: could not find document w/ uri \"{docUri}\"",
                            string docUri
                        )

                        ()

                    return newState

                | Some wf, Some doc ->
                    let resolveDocumentDiagnostics () : Task = task {
                        let! semanticModelMaybe = doc.GetSemanticModelAsync()

                        match semanticModelMaybe |> Option.ofObj with
                        | None ->
                            Error(Exception("could not GetSemanticModelAsync"))
                            |> PushDiagnosticsDocumentDiagnosticsResolution
                            |> postServerEvent

                        | Some semanticModel ->
                            let diagnostics =
                                semanticModel.GetDiagnostics()
                                |> Seq.map (Diagnostic.fromRoslynDiagnostic wfPathToUri)
                                |> Seq.map fst
                                |> Array.ofSeq

                            Ok(docUri, None, diagnostics)
                            |> PushDiagnosticsDocumentDiagnosticsResolution
                            |> postServerEvent
                    }

                    let newTask = Task.Run(resolveDocumentDiagnostics)

                    let newState =
                        { newState with
                            PushDiagnosticsCurrentDocTask = Some(docUri, newTask) }

                    return newState

                | _, _ -> return newState

            | _, _ ->
                // backlog is empty or pull diagnostics is enabled instead,--nothing to do
                return state

    | PushDiagnosticsDocumentDiagnosticsResolution result ->
        // enqueue processing for the next doc on the queue (if any)
        postServerEvent PushDiagnosticsProcessPendingDocuments

        let newState =
            { state with
                PushDiagnosticsCurrentDocTask = None }

        match result with
        | Error exn ->
            logger.LogDebug("PushDiagnosticsDocumentDiagnosticsResolution: {exn}", exn)
            return newState

        | Ok(docUri, version, diagnostics) ->
            match state.LspClient with
            | None -> return newState

            | Some lspClient ->
                let resolvedDocumentDiagnostics =
                    { Uri = docUri
                      Version = version
                      Diagnostics = diagnostics }

                do! lspClient.TextDocumentPublishDiagnostics(resolvedDocumentDiagnostics)
                return newState

    | PeriodicTimerTick ->
        postServerEvent PushDiagnosticsProcessPendingDocuments

        let statsDumpDeadline = state.LastStatsDumpTime + TimeSpan.FromMinutes(1.0)

        if state.Settings.DebugMode && statsDumpDeadline < DateTime.Now then
            postServerEvent DumpAndResetRequestStats

        let solutionReloadDeadline =
            state.WorkspaceReloadPending |> Option.defaultValue (DateTime.Now.AddDays 1)

        match solutionReloadDeadline < DateTime.Now with
        | true ->
            let! updatedWorkspace =
                workspaceWithSolutionsLoaded
                    state.Settings
                    state.LspClient.Value
                    state.ClientCapabilities
                    state.Workspace

            return
                { state with
                    Workspace = updatedWorkspace
                    WorkspaceReloadPending = None }

        | false -> return state

    | DumpAndResetRequestStats -> return processDumpAndResetRequestStats logger state
}

let serverEventLoop initialState (inbox: MailboxProcessor<ServerStateEvent>) =
    let logger = Logging.getLoggerByName "serverEventLoop"

    let rec loop state = async {
        let! msg = inbox.Receive()

        try
            let! newState = msg |> processServerEvent logger state inbox.Post
            return! loop newState
        with ex ->
            logger.LogError(ex, "serverEventLoop: crashed with {exception}", string ex)
            raise ex
    }

    loop initialState
