/*
 *	Ported to Linux's Second Extended File System as part of the
 *	dump and restore backup suit
 *	Remy Card <card@Linux.EU.Org>, 1994-1997
 *	Stelian Pop <stelian@popies.net>, 1999-2000
 *	Stelian Pop <stelian@popies.net> - Alcôve <www.alcove.com>, 2000-2002
 */

/*
 * Copyright (c) 1980, 1993
 *	The Regents of the University of California.  All rights reserved.
 * (c) UNIX System Laboratories, Inc.
 * All or some portions of this file are derived from material licensed
 * to the University of California by American Telephone and Telegraph
 * Co. or Unix System Laboratories, Inc. and are reproduced herein with
 * the permission of UNIX System Laboratories, Inc.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef _PROTOCOLS_DUMPRESTORE_H_
#define _PROTOCOLS_DUMPRESTORE_H_

#include "config.h" // IWYU pragma: keep
#include <stdint.h>
#include "../bsdcompat.h"

/*
 * TP_BSIZE is the size of file blocks on the dump tapes.
 * Note that TP_BSIZE must be a multiple of DEV_BSIZE.
 *
 * NTREC is the number of TP_BSIZE blocks that are written
 * in each tape record. HIGHDENSITYTREC is the number of
 * TP_BSIZE blocks that are written in each tape record on
 * 6250 BPI or higher density tapes.
 *
 * TP_NINDIR is the number of indirect pointers in a TS_INODE
 * or TS_ADDR record. Note that it must be a power of two.
 */
typedef u_int32_t	dump_ino_t;

#define TP_BSIZE	1024
#define NTREC   	10
#define HIGHDENSITYTREC	32
#define TP_NINDIR	(TP_BSIZE/2)
#define TP_NINOS	(TP_NINDIR / sizeof (dump_ino_t))
#define LBLSIZE		16
#define NAMELEN		64

#define OFS_MAGIC   	(int)60011
#define NFS_MAGIC   	(int)60012
#define FS_UFS2_MAGIC   (int)0x19540119
#define CHECKSUM	(int)84446


extern union u_data {
	uint8_t	s_addrs[TP_NINDIR];	/* low bit: 1 => data; 0 => hole in inode
					 * high bits encode count of blocks
					 * see get_s_addr_length */
	dump_ino_t	s_inos[TP_NINOS];	/* table of first inode on each volume */
} u_data;
uint32_t get_s_addr_length(const uint8_t*);

extern union u_spcl {
	char dummy[TP_BSIZE];
	struct	s_spcl {
		int32_t	c_type;                /* record type (see below) */
		int32_t	c_date;                /* date of this dump */
		int32_t	c_ddate;               /* date of previous dump */
		int32_t	c_volume;              /* dump volume number */
		u_int32_t c_tapea_lo;          /* logical block of this record */
		dump_ino_t c_inumber;          /* number of inode */
		int32_t	c_magic;               /* magic number (see above) */
		int32_t	c_checksum;            /* record checksum */
		struct	new_bsd_inode c_dinode;
		int32_t	c_count;               /* number of valid c_addr entries */
		union u_data c_data;           /* see above */
		char	c_label[LBLSIZE];      /* dump label */
		int32_t	c_level;               /* level of this dump */
		char	c_filesys[NAMELEN];    /* name of dumpped file system */
		char	c_dev[NAMELEN];        /* name of dumpped device */
		char	c_host[NAMELEN];       /* name of dumpped host */
		int32_t	c_flags;               /* additional information */
		u_int32_t	c_firstrec_lo; /* first record on volume */
		int32_t	c_ntrec;               /* blocksize on volume */
		int32_t	c_extattributes;       /* additional inode info */
		u_int32_t	c_tapea_hi;    /* top 32 bits of c_tapea */
		u_int32_t	c_firstrec_hi; /* top 32 bits of c_firstrec */
		int32_t	c_spare[28];           /* reserved for future uses */
	} s_spcl;
} u_spcl;
#define spcl u_spcl.s_spcl
#define c_addr c_data.s_addrs
#define c_inos c_data.s_inos

/* hacky compile-time check on sizeof */
__attribute__((unused)) static void u_spcl_size_assert(int x) {
	union u_spcl us;
	struct s_spcl ss;
	struct new_bsd_inode nbi;
	struct dinode_tape dit;

	switch(x) {
		case 0:
			break;

		case sizeof(us)  == 1024 ? 1 : 0:
		case sizeof(ss)  == 1024 ? 2 : 0:
		case sizeof(nbi) ==  128 ? 3 : 0:
		case sizeof(dit) ==  128 ? 4 : 0:

// These are written to tape and so should not move.
		case ((char*)&ss.c_type                 - (char*)&ss.c_type) ==   0 ? 11 : 0:
		case ((char*)&ss.c_date                 - (char*)&ss.c_type) ==   4 ? 12 : 0:
		case ((char*)&ss.c_ddate                - (char*)&ss.c_type) ==   8 ? 13 : 0:
		case ((char*)&ss.c_volume               - (char*)&ss.c_type) ==  12 ? 14 : 0:
		case ((char*)&ss.c_tapea_lo             - (char*)&ss.c_type) ==  16 ? 15 : 0:
		case ((char*)&ss.c_inumber              - (char*)&ss.c_type) ==  20 ? 16 : 0:
		case ((char*)&ss.c_magic                - (char*)&ss.c_type) ==  24 ? 17 : 0:
		case ((char*)&ss.c_checksum             - (char*)&ss.c_type) ==  28 ? 18 : 0:
		case ((char*)&ss.c_dinode               - (char*)&ss.c_type) ==  32 ? 19 : 0:
		case ((char*)&ss.c_count                - (char*)&ss.c_type) == 160 ? 20 : 0:
		case ((char*)&ss.c_data                 - (char*)&ss.c_type) == 164 ? 21 : 0:
		case ((char*)&ss.c_label                - (char*)&ss.c_type) == 676 ? 22 : 0:
		case ((char*)&ss.c_level                - (char*)&ss.c_type) == 692 ? 23 : 0:
		case ((char*)&ss.c_filesys              - (char*)&ss.c_type) == 696 ? 24 : 0:
		case ((char*)&ss.c_dev                  - (char*)&ss.c_type) == 760 ? 25 : 0:
		case ((char*)&ss.c_host                 - (char*)&ss.c_type) == 824 ? 26 : 0:
		case ((char*)&ss.c_flags                - (char*)&ss.c_type) == 888 ? 27 : 0:
		case ((char*)&ss.c_firstrec_lo          - (char*)&ss.c_type) == 892 ? 28 : 0:
		case ((char*)&ss.c_ntrec                - (char*)&ss.c_type) == 896 ? 29 : 0:
		case ((char*)&ss.c_extattributes        - (char*)&ss.c_type) == 900 ? 30 : 0:
		case ((char*)&ss.c_tapea_hi             - (char*)&ss.c_type) == 904 ? 31 : 0:
		case ((char*)&ss.c_firstrec_hi          - (char*)&ss.c_type) == 908 ? 32 : 0:
		case ((char*)&ss.c_spare                - (char*)&ss.c_type) == 912 ? 99 : 0:

		case ((char*)&nbi.di_mode               - (char*)&nbi.di_mode) ==   0 ? 50 : 0:
		case ((char*)&nbi.di_nlink              - (char*)&nbi.di_mode) ==   2 ? 51 : 0:
		case ((char*)&nbi.di_u                  - (char*)&nbi.di_mode) ==   4 ? 52 : 0:
		case ((char*)&nbi.di_size               - (char*)&nbi.di_mode) ==   8 ? 53 : 0:
		case ((char*)&nbi.di_atime              - (char*)&nbi.di_mode) ==  16 ? 54 : 0:
		case ((char*)&nbi.di_mtime              - (char*)&nbi.di_mode) ==  24 ? 55 : 0:
		case ((char*)&nbi.di_ctime              - (char*)&nbi.di_mode) ==  32 ? 56 : 0:
		case ((char*)&nbi.di_db                 - (char*)&nbi.di_mode) ==  40 ? 57 : 0:
		case ((char*)&nbi.di_ib                 - (char*)&nbi.di_mode) ==  88 ? 58 : 0:
		case ((char*)&nbi.di_flags              - (char*)&nbi.di_mode) == 100 ? 59 : 0:
		case ((char*)&nbi.di_blocks             - (char*)&nbi.di_mode) == 104 ? 60 : 0:
		case ((char*)&nbi.di_gen                - (char*)&nbi.di_mode) == 108 ? 61 : 0:
		case ((char*)&nbi.di_uid                - (char*)&nbi.di_mode) == 112 ? 62 : 0:
		case ((char*)&nbi.di_gid                - (char*)&nbi.di_mode) == 116 ? 63 : 0:
		case ((char*)&nbi.di_spare              - (char*)&nbi.di_mode) == 120 ? 64 : 0:

		case ((char*)&dit.di_mode               - (char*)&dit.di_mode) ==   0 ? 70 : 0:
		case ((char*)&dit.di_uid                - (char*)&dit.di_mode) ==   2 ? 71 : 0:
		case ((char*)&dit.di_size               - (char*)&dit.di_mode) ==   4 ? 72 : 0:
		case ((char*)&dit.di_atime              - (char*)&dit.di_mode) ==   8 ? 73 : 0:
		case ((char*)&dit.di_ctime              - (char*)&dit.di_mode) ==  12 ? 74 : 0:
		case ((char*)&dit.di_mtime              - (char*)&dit.di_mode) ==  16 ? 75 : 0:
		case ((char*)&dit.di_dtime              - (char*)&dit.di_mode) ==  20 ? 76 : 0:
		case ((char*)&dit.di_gid                - (char*)&dit.di_mode) ==  24 ? 77 : 0:
		case ((char*)&dit.di_nlink              - (char*)&dit.di_mode) ==  26 ? 78 : 0:
		case ((char*)&dit.di_blocks             - (char*)&dit.di_mode) ==  28 ? 79 : 0:
		case ((char*)&dit.di_flags              - (char*)&dit.di_mode) ==  32 ? 80 : 0:
		case ((char*)&dit.di_reserved1          - (char*)&dit.di_mode) ==  36 ? 81 : 0:
		case ((char*)&dit.di_db                 - (char*)&dit.di_mode) ==  40 ? 82 : 0:
		case ((char*)&dit.di_ib                 - (char*)&dit.di_mode) ==  88 ? 83 : 0:
		case ((char*)&dit.di_gen                - (char*)&dit.di_mode) == 100 ? 84 : 0:
		case ((char*)&dit.di_file_acl           - (char*)&dit.di_mode) == 104 ? 85 : 0:
		case ((char*)&dit.di_dir_acl            - (char*)&dit.di_mode) == 108 ? 86 : 0:
		case ((char*)&dit.di_faddr              - (char*)&dit.di_mode) == 112 ? 87 : 0:
		case ((char*)&dit.di_frag               - (char*)&dit.di_mode) == 116 ? 88 : 0:
		case ((char*)&dit.di_fsize              - (char*)&dit.di_mode) == 117 ? 89 : 0:
		case ((char*)&dit.di_pad1               - (char*)&dit.di_mode) == 118 ? 90 : 0:
		case ((char*)&dit.di_uidhigh            - (char*)&dit.di_mode) == 120 ? 91 : 0:
		case ((char*)&dit.di_gidhigh            - (char*)&dit.di_mode) == 122 ? 92 : 0:
		case ((char*)&dit.di_spare              - (char*)&dit.di_mode) == 124 ? 93 : 0:
			break;
	}
}

extern long long u_spcl_c_tapea(void);
extern void u_spcl_c_tapea_add(uint64_t inc);
extern void u_spcl_c_tapea_set(uint64_t v);
extern long long u_spcl_c_firstrec(const struct s_spcl *buf);
extern void u_spcl_c_firstrec_set(uint64_t v);

/*
 * special record types
 */
#define TS_TAPE 	1	/* dump tape header */
#define TS_INODE	2	/* beginning of file record */
#define TS_ADDR 	4	/* continuation of file record */
#define TS_BITS 	3	/* map of inodes on tape */
#define TS_CLRI 	6	/* map of inodes deleted since last dump */
#define TS_END  	5	/* end of volume marker */

/*
 * flag values
 */
#define DR_NEWHEADER	0x0001	/* new format tape header */
#define DR_NEWINODEFMT	0x0002	/* new format inodes on tape */
#define DR_COMPRESSED	0x0080	/* dump tape is compressed */
#define DR_METAONLY	0x0100	/* only the metadata of the inode has
				   been dumped */
#define DR_EXTATTRIBUTES	0x8000

/*
 * extattributes inode info
 */
#define EXT_REGULAR		0
#define EXT_MACOSFNDRINFO	1
#define EXT_MACOSRESFORK	2
#define EXT_XATTR		3

/*
 * compression flags for the tapebuf header.
 */
#define COMPRESS_ZLIB	0
#define COMPRESS_BZLIB	1
#define COMPRESS_LZO	2

/* used for compressed dump tapes */
/* Note that the order of the bits in a bitfield is implementation defined. */
#if 0
// Historically this was written this way. gcc on linux puts compressed in the
// lsb, then flags, and length in the highest bits
// the getters and setters below maintain this order
struct tapebuf {
	uint32_t	compressed:1;
	uint32_t	flags:3;
	uint32_t	length:28;
	char		buf[0];	/* the data */
};
#endif
struct tapebuf {
	uint32_t	hdr;
	char		buf[0];	/* the data */
};
extern int tapebuf_getcompressed(const struct tapebuf* tb);
extern int tapebuf_getflags(const struct tapebuf* tb);
extern int tapebuf_getlength(const struct tapebuf* tb);

extern void tapebuf_setcompressed(struct tapebuf* tb, uint32_t v);
extern void tapebuf_setflags(struct tapebuf* tb, uint32_t v);
extern void tapebuf_setlength(struct tapebuf* tb, uint32_t v);

/* used for EA on tape */
#define EXT2_GOOD_OLD_INODE_SIZE	128
#define EXT2_XATTR_MAGIC		0xEA020000	/* block EA */
#define EXT2_XATTR_MAGIC2		0xEA020001	/* in inode EA */

#endif /* !_DUMPRESTORE_H_ */
