/* Program Copyright Nathanael Hoyle.  Please read both README and LICENSE files. */

/** @file cdstatus.c main source file for cdstatus.
  * This file contains all the code to do pretty much anything not related to
  * cddb.  It handles program arguments (and config file parameters), does all
  * I/O with the cd/drive, and creates and writes the output files. It calls
  * routines in cdstatus_cddb.c to handle cddb related tasks when applicable.
  */

#include "gimme_config_h.h"

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif

#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif

#ifdef HAVE_LINUX_CDROM_H
#include <linux/cdrom.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif

#ifdef HAVE_ERRNO_H
#include <errno.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_LINUX_TYPES_H
#include <linux/types.h>
#endif

#include "cdstatus.h"
#include "cdstatus_cddb.h"
#include "handle_args.h"
#include "basic_info.h"
#include "read_toc.h"
#include "cd_toc_info.h"
#include "build_output_filename.h"
#include "open_output_file.h"
#include "track_listing.h"
#include "cddb_calc_discid.h"
#include "cdstatus_output.h"
#include "close_output_file.h"

#define HG_COMMIT_DATE "Sat, 17 Apr 2010 18:36:56 -0400"

/** Horribly overgrown function that does all the real work of reading a cd.
  * This function retrieves info about cd "mode" (type of cd) and handles 
  * almost all the work of extracting audio data for the actual ripping process.
  * Probably should be broken up. 
  * @param[in] cdsargs program options/settings
  * @see cdstatus_args 
  */
static int process_audio(const cdstatus_args * cdsargs);


/** Perform an internal driver reset operation.
  * This routine issues an ioctl() to the optical drive telling it to perform
  * a reset. Exactly how much good this does depends on the drive and what it
  * thinks a reset is, as well as whether anything was wrong (and what). Can
  * be useful to unlock a locked drive tray, etc.  Doesn't really need any
  * info except the path to the device, but currently takes pointer to full
  * program options struct.  Usually requires root priviledges to actually
  * perform reset operation. It is run by specifying --reset command-line
  * option, and may not be used in conjunction with any other options (aside
  * from drive path).
  * @param[in] cdsargs program options/settings
  */
static int doReset(const cdstatus_args * cdsargs);

/** outputs wav header to file.
  * Takes currently open file handle for output file and writes the wav header
  * portion of the file, after which the rest of the output to file is performed
  * elsewhere. 
  * @param[in] readframes The number of frames of audio in the track.  Needed to
  *                   calculate values filled in in header.
  * @param[in] audio_out Pre-opened file * for output file to write to.
  * @return nothing (probably should return success/failure)
  */
static void writeWavHeader(unsigned int, FILE *);

/** main routine which does almost no work, calling other functions to act.
  * Displays basic "banner" type output and calls other functions, based on
  * what program options were set.
  * @param[in] argc the system provided argument count
  * @param[in] argv the system provided argument strings
  * @return 0 on success, some other value otherwise
  */
int main(int argc, char *argv[])
{
	/* TODO: rework with popt to avoid all the ugly option handling */
	int type;
	cdstatus_args cdsargs;

	/*take care of these up front */
	if (argc == 2)
	{
		if ( (strcmp(argv[1], "--version")==0) || (strcmp(argv[1], "-V")==0))
		{
			conditional_printf(NORMAL, "\ncdstatus (C) Nathanael Hoyle, version %s, committed on %s\n", VERSION, HG_COMMIT_DATE);
			return 0;
		}
		else if (strcmp(argv[1], "--help")==0)
		{
			printHelp();
			return 0;
		}
	}
	/* set default options and override as specified */
	memset((void *) &cdsargs, 0, (size_t) sizeof(cdsargs));
	handleArgs(argc, argv, &cdsargs);

	conditional_puts(NORMAL, "\nCD Drive Status and CD Analyzer / Reader by Nathanael Hoyle");
	conditional_printf(NORMAL, "Version %s, committed at %s\n\n", VERSION, HG_COMMIT_DATE);

	if (cdsargs.reset!=0)
	{
		return doReset(&cdsargs);
	}

	type = basicInfo(&cdsargs);
	if ((type == CDS_AUDIO) || (type == CDS_MIXED))
	{
		return process_audio(&cdsargs);
	}

	return EXIT_SUCCESS;
}

static int process_audio(const cdstatus_args * cdsargs)
{
	int drive;
	struct cdrom_read_audio cdra;
	struct cdrom_tochdr cdtochdr;
	cd_toc_info cdtocinfo[100];
	int status;
	unsigned int counter, counter2, readframes, retry;
	FILE *audio_out;
	unsigned int hashes;
	unsigned int oldhashes = 0;
	unsigned int start_track, end_track;
	unsigned int discid;
	char filename[CDSTATUS_PATH_MAX];
	void * identical_read_buffer;
	unsigned int current_identical_reads;
	unsigned int current_read_set;
   
	/* safety initialization */
	filename[0] = '\0';
	memset(cdtocinfo, 0, sizeof(cdtocinfo));
	memset(&cdtochdr, 0, sizeof(cdtochdr));
	memset(&cdra, 0, sizeof(cdra));

	drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
	if (drive == -1)
	{
		snprintf(output_buffer, OUTPUT_BUFFSIZE, "Error opening drive %s.\n", cdsargs->drivename);
		conditional_perror(CRITICAL, output_buffer);
		return -1;
	}

	if (readTOC(drive, cdtocinfo, &cdtochdr) != 0)
	{
		conditional_puts(CRITICAL, "Error reading TOC.  Exiting.");
		return -1;
	}

	if (cdsargs->cddb!=0)
	{
		discid = calcDiscId(cdtochdr.cdth_trk1, (const cd_toc_info *) cdtocinfo);
		conditional_printf(DEBUG, "CDDB disc id: %08x\n", discid);

		/*do cddb query */
		cddb_query(cdtochdr.cdth_trk1, (const cd_toc_info *) cdtocinfo, cdsargs->noMangle, cdsargs->cddb_site, cdsargs->default_first_match);

		conditional_printf(NORMAL, "Track listing found for cd %s by %s:\n",album_name,artist_name);
		for (counter = 1; counter <= cddb_tracks; ++counter)
		{
			conditional_printf(NORMAL, "Track %u: ", counter);
			if (trackinfo[counter].artist[0] != '\0')
			{
				conditional_printf(NORMAL, "%s by %s\n", trackinfo[counter].title, trackinfo[counter].artist);
			}
			else
			{
				conditional_puts(NORMAL, trackinfo[counter].title);
			}
		}
	}

	if (cdsargs->readtest!=0)
	{
		if (cdsargs->start <= 1)
		{
			start_track = cdtochdr.cdth_trk0;
		}
		else
		{
			start_track = (unsigned int)cdsargs->start;
		}
		if ((cdsargs->stop >= cdtochdr.cdth_trk1) || (cdsargs->stop == 0))
		{
			end_track = cdtochdr.cdth_trk1;
		}
		else
		{
			end_track = (unsigned int)cdsargs->stop;
		}

		{
			if (start_track == end_track)
			{
				conditional_printf(NORMAL, "\n\nBeginning digital audio extraction of track %u.\n", start_track);
			}
			else
			{
				conditional_printf(NORMAL, "\n\nBeginning digital audio extraction of tracks %u through %u.\n", start_track, end_track);
			}
		}

		cdra.buf = malloc(CD_FRAMESIZE_RAW * (cdsargs->read_chunk_size));
		if (cdra.buf == NULL)
		{
			conditional_perror(CRITICAL, 
				"Unable to allocate enough memory for buffer to hold cd data "
				"(cdra.buf). Try decreasing cdsargs->read_chunk_size and try again"
			);
			return -1;
		}
		if(cdsargs->identical_reads>=2)
		{
			identical_read_buffer = malloc(CD_FRAMESIZE_RAW * (cdsargs->read_chunk_size));
			if(identical_read_buffer==NULL)
			{
				conditional_perror(CRITICAL, "Error allocating buffer for identical re-reading.");
				free(cdra.buf);
				return -1;
			}
		}
		else
		{
			identical_read_buffer=NULL;
		}

		for (counter = start_track; counter <= end_track; ++counter)
		{
			hashes = oldhashes = 0;
			if (cdtocinfo[counter].data!=0) 
            {
			    conditional_printf(NORMAL, "Skipping track %u.  (Data track).\n", counter);
				continue;
			}
			conditional_printf(NORMAL, "Beginning track %u: %s.  Progress:\n", counter, trackinfo[counter].title);

			buildOutputFilename(cdsargs, counter, filename);
			conditional_printf(DEBUG, "Returned filename: %s\n", filename);

			audio_out = openOutputFile(filename,cdsargs->encoder, counter,
				cdsargs->encopts);
			if(audio_out==NULL)
			{
				conditional_puts(CRITICAL, "Got null output file handle, this shouldn't happen... exiting.");
				exit(EXIT_FAILURE);
			}

			cdra.nframes = cdsargs->read_chunk_size;
			cdra.addr_format = CDROM_MSF;

			/* This calculates the length of the track it needs to rip by 
             * subtracting its starting offset from the starting offset of 
             * the next track. Only, in the case of the last track, you have 
             * to subtract its offset from that of the leadout track, which 
             * is actually stored in track 0. The difference in offsets, in 
             * 'frames' is assigned to readframes. */
			if (counter == cdtochdr.cdth_trk1)
			{
				readframes = cdtocinfo[0].frame_global - 
					cdtocinfo[counter].frame_global;
			}
			else
			{
				readframes = cdtocinfo[counter + 1].frame_global - 
					cdtocinfo[counter].frame_global;
                /* Fucking experimental as hell, supposed to correct for 
				 * incorrect end-of-track detection with Mixed-Mode "Enhanced"
				 * music cds. Fix sent in by Robert Woods.
                 */
                if(cdtocinfo[counter+1].data==4)
                {
                    conditional_puts(DEBUG, "Doing really wierd adjustment for Enhanced CD mode disc.");
                    readframes-=11400;
                }
			}
            conditional_printf(DEBUG, "Readframes: %d\nReading frame %d through %d\n", readframes,
				cdtocinfo[counter].frame_global, (cdtocinfo[counter].frame_global+readframes)-1);

			cdra.addr.msf.minute = cdtocinfo[counter].min;
			cdra.addr.msf.second = cdtocinfo[counter].sec;
			cdra.addr.msf.frame = cdtocinfo[counter].frame_local;

			/* If they have asked for wav file output (the default at this
			 * point), go ahead and write the wav file header and come back. */
			if (cdsargs->encoder!=ENC_RAW)
			{
				writeWavHeader(readframes, audio_out);
			}
			/* counter2 represents position of next frame to be read */
			for (counter2 = 0; counter2 < readframes; counter2 += (cdsargs->read_chunk_size))
			{
				if ((unsigned int)((cdsargs->read_chunk_size) + counter2) > readframes)
				{
					cdra.nframes = readframes - counter2;
				}
				current_identical_reads = 0;
			
				/* It's just a simple for loop...
				 * This thing is the beast that controls looping for 'secure'
				 * extraction. For so long as we haven't exhausted our quota
				 * of 'start from the top' tries, AND we have not yet reached
				 * the desired number of back-to-back identical reads, we
				 * keep going. Note that this does NOT in any way turn off the
				 * automatic re-reads of failed sectors. So we MAY be rereading
				 * failed sectors AND trying to get the same value multiple
				 * times... multiple times. Make sense? Hope so... */
				for(
					current_read_set = 1,
					current_identical_reads = 0; 
					(
						((cdsargs->max_read_sets == 0) || (current_read_set <= cdsargs->max_read_sets)) &&
						(current_identical_reads < cdsargs->identical_reads)
					)
					;
				)
				{
					status = ioctl(drive, CDROMREADAUDIO, &cdra);
					if (status == -1)
					{
						if (errno == EOPNOTSUPP)
						{
							conditional_puts(CRITICAL, "Error, the drive does not support reading in this mode. You will not be able to "
								"use the --readtest (--rip) option with this drive, sorry."
							);
							free(cdra.buf);
							closeOutputFile(audio_out, cdsargs->encoder);
							exit(EXIT_FAILURE);
						}
						snprintf(output_buffer, OUTPUT_BUFFSIZE, "\nError trying to read track %u at minute: %d, second: %d, frame %d: ", 
							counter, cdra.addr.msf.minute, cdra.addr.msf.second, cdra.addr.msf.frame);
						conditional_perror(NORMAL, output_buffer);
						
						for (retry = 1; (retry <= (unsigned int) (cdsargs->max_retries)) && (status == -1); retry++)
						{
							conditional_puts(NORMAL, "Retrying...");
							status = ioctl(drive, CDROMREADAUDIO, &cdra);
						}
						if (status == -1)
						{
							conditional_printf(NORMAL, "All %d retries failed. Expect tiny skip in audio.\n", cdsargs->max_retries);
						}
						else
						{
							conditional_printf(NORMAL, "Succeeded on retry %u.\n", retry - 1);
						}
						(void)fflush(stdout);
					}

					/* Entering the body here means that we have performed a
					 * read that the drive says is good, whether on the initial
					 * read attempt, or on a retry. If we're doing the 'secure'
					 * stuff though, then that just isn't good enough, and we
					 * have to keep at it.
					 */
					if (status != -1)
					{
						if(cdsargs->identical_reads > 1)
						{
							if(current_identical_reads == 0)
							{
								/* First successful read in a set. Therefore it
								 * doesn't have to match anything from before. 
								 * Copy it to the backup buffer for later use. */
								current_identical_reads = 1;
								memcpy(identical_read_buffer, cdra.buf, (size_t)(CD_FRAMESIZE_RAW * cdra.nframes));
							}
							else
							{
								/* Here we actually care if it matches. */
								if(memcmp(cdra.buf, identical_read_buffer, (size_t)(CD_FRAMESIZE_RAW * cdra.nframes))==0)
								{
									/* We matched our former read */
									++current_identical_reads;
								}
								else
								{
									/* Oops! We got a 'successful' read, but it
									 * didn't match what we got last time. */
									current_identical_reads = 0;
									++current_read_set;
									(void)putchar('X');
									fflush(stdout);
									/* TODO: make this configurable */
									if(current_read_set > cdsargs->max_read_sets)
									{
										exit(EXIT_FAILURE);
									}
								}
							}
						}
						else
						{
							current_identical_reads = 1;
						}
						if( (cdsargs->identical_reads <= 1) || (current_identical_reads >= cdsargs->identical_reads) )
						{
							/* we've now gotten the same thing enough
							 * to trust it... output it */
							if(
								fwrite(
									(const void *) cdra.buf, (size_t) CD_FRAMESIZE_RAW,
									(size_t) cdra.nframes, audio_out
								) 
								!= (size_t)cdra.nframes
							)
							{
								conditional_perror(CRITICAL, "Error writing audio frames to output file.  Exiting");
								closeOutputFile(audio_out, cdsargs->encoder);
								exit(EXIT_FAILURE);
							}
						}
					}
				}
	
				cdra.addr.msf.frame += cdra.nframes;
				if (cdra.addr.msf.frame >= CD_FRAMES)
				{
					cdra.addr.msf.second += cdra.addr.msf.frame / CD_FRAMES;
					cdra.addr.msf.frame = cdra.addr.msf.frame % CD_FRAMES;
					if (cdra.addr.msf.second >= 60)
					{
						cdra.addr.msf.minute += cdra.addr.msf.second / 60;
						cdra.addr.msf.second = cdra.addr.msf.second % 60;
					}
				}

				hashes = ((counter2 * 78) / readframes) + 1;
				if (hashes > oldhashes)
				{
					hashes -= oldhashes;
					oldhashes += hashes;
					for (; hashes!=0; --hashes)
					{
						conditional_printf(NORMAL, "#");
					}
					(void)fflush(stdout);
				}
			}
			/* done reading track. finish up and close out */
			closeOutputFile(audio_out, cdsargs->encoder);
			conditional_printf(NORMAL, "\nTrack %u completed.\n", counter);
		}
	}

	free(cdra.buf);
	return 0;
}

static int doReset(const cdstatus_args * cdsargs)
{
   /** file handle for optical device to reset */
	int drive;

   /** the status code returned from the ioctl() */
	int status;

	conditional_puts(DEBUG, "Attempting to reset drive.");
	drive = open(cdsargs->drivename, O_RDONLY | O_NONBLOCK);
	if (drive == -1)
	{
		conditional_perror(CRITICAL, "Error opening drive device");
		return -1;
	}
	status = ioctl(drive, CDROMRESET);
	if (status == -1)
	{
		if (errno == EACCES)
		{
			conditional_puts(CRITICAL, "Permission denied error received.  Note that most systems will require root access to reset a drive.");
		}
		else
		{
			conditional_perror(CRITICAL, "Error performing drive reset.");
		}
		return -1;
	}
	else
	{
		conditional_puts(NORMAL, "Drive successfully reset.");
	}
	return 0;
}

static void writeWavHeader(unsigned int readframes, FILE * audio_out)
{
	long int chunksize;
	typedef struct _wavHeader
	{
		int32_t RIFF_header;
		int32_t total_size;
		int32_t WAVE;
		int32_t fmt;
		int32_t subchunk_size;
		int16_t audio_format;
		int16_t number_channels;
		int32_t sampling_rate;
		int32_t byte_rate;
		int16_t block_align;
		int16_t bits_per_sample;
	} wavHeader;

	wavHeader wHeader;

	/* "RIFF" */
	wHeader.RIFF_header = 0x46464952;

	chunksize = readframes * CD_FRAMESIZE_RAW;
	wHeader.total_size = (int32_t)(chunksize + sizeof(wavHeader));

	/* "WAVEfmt " */
	wHeader.WAVE = 0x45564157;
	wHeader.fmt = 0x20746D66;

	wHeader.subchunk_size = 16;
	wHeader.audio_format = 1;
	wHeader.number_channels = 2;
	wHeader.sampling_rate = 44100;
	wHeader.byte_rate = 176400;
	wHeader.block_align = 4;
	wHeader.bits_per_sample = 16;

	if(fwrite((const void *) &wHeader, sizeof(wavHeader), (size_t) 1, audio_out)!=1)
	{
		conditional_perror(CRITICAL, "Error writing wav file header");
		if(fclose(audio_out)!=0)
		{
			conditional_perror(WARNING, "Error closing output wav file");
		}
		exit(EXIT_FAILURE);
	}
	if(fprintf(audio_out, "data")!=4)
	{
		conditional_perror(CRITICAL, "Error writing data header in wav file");
		if(fclose(audio_out)!=0)
		{
			conditional_perror(WARNING, "Error closing output wav file");
		}
		exit(EXIT_FAILURE);
	}
	if(fwrite((const void *) &chunksize, sizeof(long int), (size_t) 1, audio_out)!=1)
	{
		conditional_perror(CRITICAL, "Error writing wav file chunksize header");
		if(fclose(audio_out)!=0)
		{
			conditional_perror(WARNING, "Error closing output wav file");
		}
		exit(EXIT_FAILURE);
	}

	return;
}
