/*******************************************************************************
*                                                                              *
* Copyright (C) 1997 Steve Haehn                                               *
*                                                                              *
* This is free software; you can redistribute it and/or modify it under the    *
* terms of the GNU General Public License as published by the Free Software    *
* Foundation; either version 2 of the License, or (at your option) any later   *
* version.                                                                     *
*                                                                              *
* This software is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License *
* for more details.                                                            *
*                                                                              *
* You should have received a copy of the GNU General Public License along with *
* software; if not, write to the Free Software Foundation, Inc., 59 Temple     *
* Place, Suite 330, Boston, MA  02111-1307 USA                                 *
*                                                                              *
*******************************************************************************/

/*-----------------------------------------------------------------------------*
**
** EXPANDER.C
** 
** Abbreviation expansion facility for macro enabled text editors.
** 
** Functions:    
**
**        adjust_indent,      check_keyword,       chop_keyword,
**        eval_environ_names, expand_auto_fields,  expand_date,
**        expand_datetime,    expand_definition,   expand_time,
**        expand_user,        expand_year,         faccess,
**        file_open,
**        get_directory,      get_keyword,         get_line,
**        get_new_delimiters, include_file,        include_template_file,
**        is_defined,         is_language_present, locate_definition,
**        main,               present_time_with,   print_nl_and_indent,
**        remove_file_path,   show_usage,          strtok_r
**
** Revision History:
**
**      Date       Author/Comment
**   Feb 18, 2003  Steven Haehn
**                 Added access to environment variables within definitions.
**
**   Sep 20, 2001  Steven Haehn
**                 Fixed tabbing problem when actual hardware tabs were used
**                 requiring EXP_USE_TABS to be defined. Hardware tabs were
**                 always being assumed to be of length 8, but if they are
**                 being simulated at a different size, the indent calculation
**                 in chop_keyword ignored this.
**
**   Jan 04, 2001  Joor Loohuis, Steve Haehn
**                 Added -i option for overriding indent level in environment.
**
**   Dec 04, 2000  Steven Haehn
**                 Altered to satisfy some picky compiler warnings.
**                 (fgets used exclusively instead of gets)
**
**   Dec 22, 1998  Steven Haehn
**                 Porting to Windows NT (yechh). Added ctype.h for isdigit.
**                 Removed all tabs from file. Removed unused variables.
**
**   Mar 11, 1998  Steven Haehn
**                 Fixed recursive indentation problem found within 
**                 expand_definition. Now constantly updating a 
**                 nested_indent variable in case recursion occurs.
**
**   Mar 02, 1998  Steven Haehn
**                 Added automatic expansion of "auto-fill" fields in
**                 included template files and temporary definitions
**                 so that editors can supply such items as 'filename'.
**
**   Feb 12, 1998  Steven Haehn
**                 Required that BLANX be always a part of the keyword 
**                 delimiter set, no matter what.
**
**   Feb 11, 1998  Steven Haehn
**                 Added EXP_NO_AUTOFIELD and EXP_KEYWORD_DELIM
**                 environment variables. 
**
**   Feb 09, 1998  Steven Haehn
**                 Added print_nl_and_indent courtesy of Joor Loohuis.
**                 Changed to EXP_TAB_SPACING to EXP_INDENT_SIZE.
**                 Added EXP_USE_TABS environment variable.
**
**   Feb 02, 1998  Steven Haehn
**                 Expander now takes a 'language' argument which has
**                 precedence over EXP_LANGUAGE.
**
**   Jan 26, 1998  Steven Haehn
**                 Added @include file mechanism, which allows nested
**                 definition files.
**
**   Dec 12, 1997  Steven Haehn
**                 Fixed problem of getting keyword expansion when
**                 keyword was non-existent
**
**   Jan 16, 1997  Steven Haehn   
**                 Added fields, templates, abbreviation expansion,
**                 usage, and date & time expansion.
**
**   Feb 15, 1996  Marc Verdiesen
**                 Original version.  
**
**----------------------------------------------------------------------------*/

/*============================================================================*/
/*====================  OPERATING SYSTEM INCLUDE FILES  ======================*/
/*============================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>

/*==========================================================================*/
/*                              MACRO DEFINITIONS                           */
/*==========================================================================*/

#define FILE_P(x)       (x)->file
#define FILE_NM(x)      (x)->filename
#define AT_LINE(x)      (x)->line_number
#define COMMENT_CHAR(x) (x)->comment_indicator
#define ISSPACE(a) ((a)==' ' || (a)=='\t')

/*===========================================================================*/
/*                             SYMBOL DEFINITIONS                            */
/*===========================================================================*/

#define WHITESPACE " \t"
#define SKIP_BLANX strspn( expansion, WHITESPACE )

#ifdef FALSE
#undef FALSE
#endif /* FALSE */
#define FALSE   (bool_t)0

#ifdef TRUE
#undef TRUE
#endif /* TRUE */
#define TRUE    (bool_t)!FALSE

#define MAX_TDEFS 10
#define MAX_LEVEL 20
#define MAX_LINE 255
#define MAXBUF   MAX_LINE*10
#define EXP_DEFAULT_DATE_FORMAT "%b %d, %Y"
#define EXP_DEFAULT_TIME_FORMAT "%H:%M:%S"
#define EXP_DEFAULT_DTIME_FORMAT "%b %d, %Y %H:%M:%S"
#define EXP_DEFAULT_TAB_SIZE 8
#define EXP_DEFAULT_KEYWORD_DELIM BLANX
#define AUTO_FIELD_IND "|>^"
#define END_FIELD        "<|"
#define FIELD_MARKER   "|><|"
#define EOS '\0'
#define ESCAPE_CHARACTER '\\'
#define DEFINITION_REFERENCE '@'
#define FORWARD_REFERENCE '>'
#define BACKWARD_REFERENCE '<'
#define CMNT_IND          '!'
#define BLANX            " \011\n"
#define EOL               "\n"
#define LINE_CONTINUE    "\\\n"
#define FILE_IND         '@'
#define ENVIRON_IND      "$"
#define ENV_NAME "_ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789abcdefghijklmnopqrstuvwxyz"
#define DIR_DELIM '/'
#define DEF_DELIM '='


/*==========================================================================*/
/*                        VARIABLE TYPE DEFINITIONS                         */
/*==========================================================================*/

typedef int bool_t;           /* boolean, legal values are TRUE and FALSE */

typedef struct Ln_ctxt
{
    FILE * file;              /* data stream associated with 'filename' */
    char * filename;          /* name of file being accessed            */
    int    line_size;         /* maximum storage area for line in file  */
    int    line_number;       /* number of current line being accessed  */
    char   comment_indicator; /* comment marker for file                */
    
} Ln_ctxt;

typedef struct keyWord
{
   char     *key_name;          /* word to be expanded explicitly by expander */
   bool_t  (*expand_func)(int); /* function which actually performs expansion */
   
} keyWord;

typedef struct Indent
{
    int minimum;     /* first non-whitespace position in given input */
    int maximum;     /* position of keyword being expanded, can be   */
                     /*  equal to minimum value.                     */
} Indent;

/*============================================================================*/
/*                             PROGRAM PROTOTYPES                             */
/*============================================================================*/

#ifdef NEED_GETOPT_PROTO_TYPE
int   getopt( int, char **, char * );
#endif

bool_t locate_definition(

    Ln_ctxt *info,
    char    *word,
    Indent  *indentation
);

int check_keyword(

    const char *keyword_string,
    int *table_index
);

/*----------------------------------------------------------------------------*
-- KEYWORD EXPANSION FUNCTIONS
-- 
-- Each function accepts a single parameter, the indentation level to use.
-- Rules/assumptions for each of the expansion functions are :
--   - all text before the detected keyword has already been written
--   - the keyword has not been written yet
------------------------------------------------------------------------------*/
bool_t expand_date     (int indent);
bool_t expand_time     (int indent);
bool_t expand_datetime (int indent);
bool_t expand_year     (int indent);
bool_t expand_user     (int indent);

/*==========================================================================*/
/*                           VARIABLE DECLARATIONS                          */
/*==========================================================================*/

keyWord keyword_table[] =
{ 
  { "date",      expand_date     },    /* current date             */
  { "time",      expand_time     },    /* current time             */
  { "dtime",     expand_datetime },    /* current date & time      */
  { "year",      expand_year     },    /* current year             */
  { "user",      expand_user     },    /* current year             */
  { NULL,        NULL            }     /* MUST BE THE LAST!!       */
};

int    indent_size       = 4;   /* size when EXP_INDENT_SIZE undefined */
int    tab_size          = 0;   /* size when EXP_USE_TABS undefined    */
char * language_context  = "none";
char * field_marker      = FIELD_MARKER;
char * keyword_delimiter = EXP_DEFAULT_KEYWORD_DELIM;

char *temp_definitions[ MAX_TDEFS ];
int   tdef_index = -1;        /* indicates no temporary definitions provided */

extern int   optind;
extern char *optarg;

/*===========================================================================*/
/*================================= PROGRAMS ================================*/
/*===========================================================================*/

char * remove_file_path(

    char * inp_file_path
)
{
    char * file_name = strrchr( inp_file_path, DIR_DELIM );
    
    return ( file_name ) ? file_name+1 : inp_file_path;
}

#ifdef NEED_STRTOK_R

/*==============================================================================
**
** Re-entrant version of strtok
**
*/
#if 0
** REVISION HISTORY:
**
**      Date       Author/Comment
**  Aug 25, 1994   Steve Haehn
**                 Original version.
**==============================================================================
#endif
 
char * strtok_r(

    char  *inoutp_src,       /* Source string for tokenizing, or NULL */
    char  *inp_separators,   /* List of token separators              */
    char **inoutpp_next_pos  /* Remainder of initial source string    */
)
{
    char *crnt_token, *next_separator;
    int   sepLen;
 
    /*------------------------------------------
    * Use given source string, or when NULL use
    * remaining string from previous call.
    *------------------------------------------*/
    crnt_token = (inoutp_src) ? inoutp_src : *inoutpp_next_pos;
    
    /*---------------------------------------------------
    * When there is a possibility of a token present ...
    *---------------------------------------------------*/
    if (crnt_token != NULL)
    {
        /*-------------------------------------------
        * ... skip over all leading token separators 
        *     prior to determining position of token.
        *-------------------------------------------*/
        sepLen = strspn(crnt_token, inp_separators);
        
        if (crnt_token[sepLen] == EOS)
            crnt_token = *inoutpp_next_pos = NULL;
        else
            crnt_token = crnt_token + sepLen;
 
        /*---------------------------------------------------------
        * When there is a token present, locate the next separator
        * (if any) and prepare for retrieval of subsequent tokens.
        *--------------------------------------------------------*/
        if (crnt_token != NULL)
        {
            next_separator = strpbrk(crnt_token, inp_separators);
    
            if (next_separator == NULL)
                *inoutpp_next_pos = NULL;  /* no more tokens */
            else
            {
                *inoutpp_next_pos = next_separator + 1;
                *next_separator   = EOS;   /* terminate token string */
            }
        }
    }
 
    return ( crnt_token );
}

#endif /* NEED_STRTOK_R */

/*==============================================================================
**
** Build new list of delimiters
**
** DESCRIPTION:  
**
**  This routine builds a new list of keyword delimiters to be used by expander.
**  The need for the routine is so that users can describe whitespace characters
**  with 'C' style escape sequence syntax notation.
**
** RETURNS:
**
**  Newly allocated keyword delimiter string.
**
*/
#if 0
** REVISION HISTORY:
**
**   Date          Author/Comment
**   Feb 11, 1998  Steven Haehn
**                 Original version.
**==============================================================================
#endif

char * get_new_delimiters(

    char * in_new_delimiters
)
{
    int    length     = strlen( in_new_delimiters );
    int    def_key_sz = strlen( EXP_DEFAULT_KEYWORD_DELIM );
    char * new_delims = calloc( 1, length + def_key_sz );
    char * cp         = in_new_delimiters;
    char * ndp        = new_delims;
    
    strcpy( ndp, EXP_DEFAULT_KEYWORD_DELIM );  /* require these always */
    ndp += def_key_sz;
    
    while( *cp != EOS && length > 0 )
    {
        if( *cp != ESCAPE_CHARACTER )
        {
            *ndp = *cp;
        }
        else
        {
            cp++;     /* skip over escape character */
            length--;
            
            switch( *cp )
            {
              case 'b': *ndp = '\b';  break;
              case 'f': *ndp = '\f';  break;
              case 'n': *ndp = '\n';  break;
              case 'r': *ndp = '\r';  break;
              case 't': *ndp = '\t';  break;

              default:  *ndp = *cp;   break;
            }
        }
        
        cp++; ndp++; length--;
    }
    
    return new_delims;
}

/*----------------------------------------------------------------------------*/

char * get_username()
{
    static char * userName = NULL;

    /*-------------------------------------------------------
    * Make sure that there is an EXP_USER in the environment.
    * (Assign it a typical Unix default, hopefully).
    * First attempt LOGNAME, then USER, then error note.
    *-------------------------------------------------------*/
    if( userName == NULL )
    {
        userName = getenv( "EXP_USER" );
        
        if( userName == NULL )
        {
            char nameBuffer[ MAX_LINE ];

            userName = getenv( "LOGNAME" );

            if( userName == NULL )
            {
                userName = getenv( "USER" );

                if( userName == NULL )
                {
                    userName = "EXP_USER environment variable is not set.";
                }
            }

            sprintf( nameBuffer, "EXP_USER=%s", userName );
            putenv( strdup( nameBuffer ) );
        }
    }
    
    return userName;
}

/*----------------------------------------------------------------------------*/

char * get_directory(

    char * filename,  /* original name from which to extract directory path */
    char * directory  /* place to store the extracted directory             */
)
{
    char * result = NULL;
    
    if( filename != NULL )
    {
        char * end_p = strrchr( filename, DIR_DELIM );

        if( end_p != NULL )
        {
            result = directory;

            while( filename <= end_p )
                *(directory++) = *(filename++);
            
            *directory = EOS;
        }
    }
    
    return result;
}

/*----------------------------------------------------------------------------*/

FILE * file_open(

    char * inp_filename,        /* name of file to be accessed       */
    char * inp_access           /* access method accaptable to fopen */
)
{
    FILE * stream  = fopen( inp_filename, inp_access );
    
    /*---------------------------------------------------------------
    * When the file fails to open AND no directory path is specified 
    * in the filename use the path to the root definitions file as a 
    * default and attempt the open again.
    *---------------------------------------------------------------*/
    if( stream == NULL && strchr( inp_filename, DIR_DELIM ) == NULL )
    {
        char   filename_buffer[ MAX_LINE ];
        char * definitions_file = getenv( "EXP_DEFINITIONS" );
        char * path = get_directory( definitions_file, filename_buffer );
        
        if( path != NULL )
        {
            strcat( path, inp_filename );
            stream = fopen( path, "r" );
        }
    }
    
    return stream;
}

/*----------------------------------------------------------------------------*/

bool_t faccess(

    char    * inp_filename,        /* name of file to be accessed  */
    Ln_ctxt * inoutp_info,         /* file context information     */
    int       in_line_sz,          /* maximum line size to be read */
    char      in_comment_ind       /* comment character            */
)
{
    bool_t status = FALSE;
    
    inoutp_info->file = file_open( inp_filename, "r" );
    
    /*--------------------------------------------------
    * Upon successful open, initialize the file context.
    *--------------------------------------------------*/
    if( inoutp_info->file != NULL )
    {
        inoutp_info->line_size         = in_line_sz;
        inoutp_info->line_number       = 0;
        inoutp_info->comment_indicator = in_comment_ind;
        inoutp_info->filename          = inp_filename;
        status = TRUE;
    }
    
    return status;
}

/*----------------------------------------------------------------------------*/

void get_line(

    char    * inoutp_line,         /* Place to store text from file      */
    Ln_ctxt * inoutp_info          /* necessary file context information */
)
{
    int line_size = inoutp_info->line_size;
    
    fgets( inoutp_line, line_size, FILE_P( inoutp_info ) );
    inoutp_info->line_number++;

    /*--------------------------------------------
    * Until there are no more lines in the file...
    *--------------------------------------------*/
    while( ! feof( FILE_P( inoutp_info ) ))
    {
        char *lp = inoutp_line;

        /*-------------------------------------------------
        * Locate comments and remove them by replacing
        * comment indicator with an end-of-line character.
        * (In this case, comment indicator must be located
        * in character position 1 of a line.
        *------------------------------------------------*/
        if( COMMENT_CHAR( inoutp_info ) )
        {
            if( *lp == COMMENT_CHAR( inoutp_info ) ) 
                strcpy(lp, EOL);
        }            

        /*--------------------
        * Ignore empty lines.
        *--------------------*/
        if( strspn( inoutp_line, BLANX ) == strlen( inoutp_line )) /* Empty? */
        {
            fgets( inoutp_line, line_size, FILE_P( inoutp_info ) );
            inoutp_info->line_number++;
        }
        
        /*-----------------------
        * Is this line continued?
        *-----------------------*/
        else if( (lp = strstr( inoutp_line, LINE_CONTINUE )) )
        {
            char line[MAX_LINE], *bp;
            
            /*--------------------------------------------------
            * Place continuation line in separate buffer and
            * remove leading white space to maximize line usage.
            *--------------------------------------------------*/
            fgets( line, MAX_LINE, FILE_P( inoutp_info ) );
            inoutp_info->line_number++;
            bp = line +  strspn( line, BLANX );
            line_size  -= strlen( inoutp_line );
            
            if( (int) strlen( bp ) > (line_size-1) )
            {
                strcpy( inoutp_line, EOL );
                printf( "**ERROR** Line too long in abbreviation definitions\n" );
                printf( "          file (%s)\n", FILE_NM( inoutp_info ));
                printf( "          at line number (%d).\n", AT_LINE( inoutp_info ));
            }
            else
            {
                /*---------------------------------------------
                * Glue together the current and continued line,
                * separating non-white space info with a space.
                *---------------------------------------------*/
                inoutp_line = lp;
                strcpy( inoutp_line, " " );
                strcat( inoutp_line, bp );
            }
        }
        else
            break;                       /* We now have a "valid" line */
    }
}

/*----------------------------------------------------------------------------*/

int adjust_indent(

    char ** character_p,
    int     indentation
)
{
    char *cp = *character_p;
    
    if( isdigit( *(cp+1) ) )
    {
        char temp[2];  /* hold digit and EOS */

        temp[0]      = *(++cp);
        temp[1]      = EOS;
        indentation += atoi( temp ) * indent_size;
        *character_p = cp;
        
        if( indentation < 0 )  indentation = 0;
    }
    
    return indentation;
}

/*----------------------------------------------------------------------------*/

void print_nl_and_indent(

    int indent_sz
)
{
    /*----------------------------------------
    * When not using tabs, indent using spaces.
    *-----------------------------------------*/
    if( tab_size == 0)
        printf( "\n%*s", indent_sz, "" );

    else
    {
        /*-----------------------------------
        * Maximize use of tabs in indent area
        * and pad leftover with spaces.
        *-----------------------------------*/
        int i;
        
        printf( "\n" );
        
        for( i = 0; i < (indent_sz / tab_size); i++ )
            printf( "\t" );
            
        if( indent_sz % tab_size > 0 )
        {
            printf( "%*s", indent_sz % tab_size, "" );
        }
    }
}

/*----------------------------------------------------------------------------*/

bool_t is_defined(

    char   * word,
    char   * definitions_file,
    Indent * indentation
)
{
    bool_t found_word = FALSE;
    
    /*------------------------------------
    * Only attempt to expand the word when
    * a definitions file is available.
    *------------------------------------*/
    if( definitions_file != NULL )
    {
        Ln_ctxt info;
        
        if( faccess( definitions_file, &info, MAXBUF, CMNT_IND ) )
        {
            found_word = locate_definition( &info, word, indentation );
            fclose( FILE_P( &info ) );
        }
    }
    
    return found_word;
}

/*----------------------------------------------------------------------------*/

char * expand_auto_fields(

    char * line,
    bool_t only_once
)
{
    char * cp         = line;
    char * auto_begin = strstr( cp, AUTO_FIELD_IND );
    
    /*-----------------------------------------------------
    * This will handle multiple auto fill fields on a line.
    *-----------------------------------------------------*/
    while( auto_begin != NULL )
    {
        char * auto_end = strstr( auto_begin, END_FIELD );

        /*-----------------------------
        * If this is a malformed field, 
        * or not on one line, bail out.
        *-----------------------------*/
        if( auto_end == NULL )
        {
            auto_begin = NULL;
        }
        else
        {
            char * word    = auto_begin + strlen( AUTO_FIELD_IND );
            char   a_begin = *auto_begin;
            int    table_index;

            *auto_begin = EOS;
            *auto_end   = EOS;

            printf( "%s", cp );

            if( check_keyword( word, &table_index ))
            {
                (*(keyword_table[table_index].expand_func)) (0);
            }
            else
            {
                Indent indentation = { 0, 0 };
                char * definitions_file = getenv( "EXP_DEFINITIONS" );

                if( ! is_defined( word, definitions_file, &indentation ))
                {
                    /*----------------------
                    * Restore unknown field.
                    *----------------------*/
                    *auto_begin = a_begin;
                    printf( "%s%s", auto_begin, END_FIELD );
                }
            }
    
            cp = auto_end + strlen( END_FIELD );
            auto_begin = (only_once) ? NULL : strstr( cp, AUTO_FIELD_IND );
        }
    }
    
    /*--------------------------------------
    * Put out any remaining portion of line.
    *--------------------------------------*/
    if( *cp != EOS && ! only_once )
        printf( "%s", cp );
    
    return cp;
}

/*----------------------------------------------------------------------------*/

bool_t expand_definition(

    Ln_ctxt *info,
    Indent  *indentation,
    char    *line
)
{
    static bool_t found_field = FALSE;
    char * cp                 = line;
    int    indent_kw          = indentation->maximum;
    int    indent_ws          = indentation->minimum;
    Indent nested_indent;
    
    nested_indent.minimum = indent_ws;
    nested_indent.maximum = indent_kw;
    
    while( cp && *cp != EOS )
    {
        if( *cp == ESCAPE_CHARACTER )  /* skip over the escape character */
        {
            int indent_sz;
            
            cp++;
            
            /*---------------------------
            * Convert special characters.
            *---------------------------*/
            switch( *cp )
            {
              case 'n':     /* auto-indent to keyword */
                indent_sz = adjust_indent( &cp, indent_kw );
                *cp = EOS;
                print_nl_and_indent( indent_sz );
                nested_indent.maximum = indent_sz;
                break;
              
              case 'N':     /* auto-indent to keyword, next level */
                *cp = EOS;
                print_nl_and_indent( indent_kw+indent_size );
                nested_indent.maximum = indent_kw+indent_size;
                break;

              case '.':     /* conditional newline, reset indent */
                *cp = ( indent_kw > 0 && cp == line+1 ) ? '\n' : EOS;
                indent_kw = indent_ws = 0;
                break;

              case 'r':     /* auto-indent to initial whitespace */
                indent_sz = adjust_indent( &cp, indent_ws );
                *cp = EOS;
                print_nl_and_indent( indent_sz );
                nested_indent.minimum = indent_sz;
                break;
              
              case 'R':     /* auto-indent to initial whitespace, next level */
                *cp = EOS;
                print_nl_and_indent( indent_ws+indent_size );
                nested_indent.minimum = indent_ws+indent_size;
                break;

              case 't':  *cp = '\t';  break;
              case 'b':  *cp = '\b';  break;
              case 'f':  *cp = '\f';  break;
            }
        }
        
        else if( *cp == DEFINITION_REFERENCE )
        {
            if( *(cp+1) == BACKWARD_REFERENCE || *(cp+1) == FORWARD_REFERENCE )
            {
                char * word         = (cp+2);
                char * ref_end      = strchr(word, DEFINITION_REFERENCE );
                long   where_are_we = ftell( FILE_P( info ));
                
                if( ref_end == NULL )
                {
                    printf( "** ERROR ** Missing definition reference terminator (%c).\n",
                      DEFINITION_REFERENCE );
                    printf( "Check %s near line %d\n", FILE_NM(info), AT_LINE(info));
                    exit(1);
                }
                
                *ref_end = EOS;  /* Whack end-of-reference symbol */
                
                if( *(cp+1) == BACKWARD_REFERENCE )
                    rewind( FILE_P( info ));
                
                /*------------------------------------------
                * If the definition is successfully located,
                * we want to skip over reference to it.
                * Otherwise, just put out what was given.
                *------------------------------------------*/
                if( locate_definition( info, word, &nested_indent ))
                    cp = ref_end;
                else
                    *ref_end = DEFINITION_REFERENCE;

                fseek( FILE_P(info), where_are_we, 0 );
            }
        }
        
        /*-----------------------------------
        * Is there a field in the expansion?
        *-----------------------------------*/
        else if( *cp == '|' && *(cp+1) == '>' )
        {
            found_field = TRUE;

            /*---------------------------------------
            * When this is an automatic fill field...
            *---------------------------------------*/
            if( strncmp( cp, AUTO_FIELD_IND, strlen( AUTO_FIELD_IND )) == 0 )
            {
                char * original_cp = cp;
                
                cp = expand_auto_fields( original_cp, TRUE );
                
                /*----------------------------------------------------------
                * ... the current position will already point past
                *     the field and the new character must go thru the loop.
                *----------------------------------------------------------*/
                if( ! (cp == original_cp) )
                {
                    found_field = FALSE;
                    continue;
                }
            }
        }
        
        if( *cp != EOS )
            putchar( *cp );
        
        cp++;
    }
    
    return found_field;
}

/*----------------------------------------------------------------------------*/

bool_t include_template_file (

    Indent  * indentation,
    char    * hdr_template
)
{
    bool_t template_supplied = FALSE;    
    
    /*---------------------------------------
    * If name of file for template found, 
    * try to open it and display it. 
    *---------------------------------------*/
    if( hdr_template != NULL )
    {
        FILE * usr_tpl = file_open( hdr_template, "r" );
        
        if( usr_tpl != NULL )
        {
            char line[ MAX_LINE ];

            template_supplied = TRUE;
            
            if (indentation->maximum > 0)
            {
                printf ("\n");
            }

            fgets( line, MAX_LINE, usr_tpl );
            
            while( !feof( usr_tpl ) )
            {
                expand_auto_fields( line, FALSE );
                fgets( line, MAX_LINE, usr_tpl );
            }
            
            fclose( usr_tpl );
        }
    }
    
    return template_supplied;
}

/*==============================================================================
**
** Evaluate environment variables in given string
**
** DESCRIPTION:  
**
**  This function is used to evaluate environment variables within the
**  given string. Environment variables are in the form:
**
**       $ [ '{' | '(' ] <envName> '[' } | ')' ]
**
**  where variable markers in square brackets are optional.
**  For example, the following are legal environment variables.
**
**       $LEGAL, ${LEGAL}, $(LEGAL)
**
**  If an environment variable does not have a value, it is removed
**  from the result. To keep the '$' character from being interpreted
**  as an environment variable marker, escape it with '\'.
*/
#if 0
** REVISION HISTORY:
**
**       Date       Author/Comment
**   Feb 18, 2003   Steven Haehn
**                  Can now have environment variables of the form
**                  ${USER} or $(USER).
**
**   Jan 27, 1998   Steven Haehn
**                  Original version.
**==============================================================================
#endif

void eval_environ_names(

    char * inp_string,
    char * outp_string,
    int    in_name_size
)
{
    char * cp    = inp_string;
    char * out_p = outp_string;
    char * end_p = out_p + in_name_size;
    
    /*----------------------------------
    * Examine the entire given string...
    *----------------------------------*/
    while( cp && *cp != EOS )
    {
        char * found_env = strpbrk( cp, ENVIRON_IND );
        
        /*-------------------------------------
        * When no environment indicator found..
        *-------------------------------------*/
        if( found_env == NULL )
        {
            /*-------------------------------------------
            * ... copy remainder of given name to result.
            *-------------------------------------------*/
            while( *cp != EOS )
                *(out_p++) = *(cp++);
        }
        
        /*-------------------------------------
        * When indicator is escaped (eg. \$)...
        *-------------------------------------*/
        else if( *(found_env-1) == ESCAPE_CHARACTER )
        {
            /*-------------------------------------------
            * ... copy characters up to escape indicator.
            *-------------------------------------------*/
            while( cp < found_env-1 && out_p < end_p )
                *(out_p++) = *(cp++);
                
            *(out_p++) = *found_env;  /* copy '$' to output */
            cp = found_env+1;
        }
        else
        {
            char   env_name[MAX_LINE];
            char * envValue;
            int i;
            
            /*-----------------------------------------------------
            * Copy characters up to environment variable indicator.
            *-----------------------------------------------------*/
            while( cp < found_env && out_p < end_p )
                *(out_p++) = *(cp++);

            /*------------------------------------------
            * Environment variables are in the form:
            * 
            *     $ [ '{' | '(' ] <envName> '[' } | ')' ]
            *
            * where variable markers in square brackets
            * are optional. For example, the following
            * are legal environment variables.
            *
            *     $LEGAL, ${LEGAL}, $(LEGAL)
            *------------------------------------------*/
            
            found_env++; /* skip over '$' */

            /*-----------------------------------
            * Skip over optional variable marker.
            *-----------------------------------*/
            if( *(found_env) == '{' ||
                *(found_env) == '('    ) found_env++;

            cp = found_env + strspn( found_env, ENV_NAME );
            
            /*----------------------------------
            * Extract environment variable name.
            *----------------------------------*/
            for( i=0; found_env < cp; i++ )
                env_name[i] = *(found_env++);

            env_name[i] = EOS;

            /*-----------------------------------
            * Skip over optional variable marker.
            *-----------------------------------*/
            if( *cp == '}' || *cp == ')' ) cp++;
            
            /*----------------------------------------------
            * When a value is not found for the environment 
            * variable, it is effectively removed from 
            * the original string.
            *----------------------------------------------*/
            if( (envValue = getenv( env_name )) != NULL )
            {
                /*---------------------------------
                * Copy environment value to result.
                *---------------------------------*/
                while( *envValue != EOS && out_p < end_p )
                    *(out_p++) = *(envValue++);
            }
        }
    }
    
    *out_p = EOS;
}

/*==============================================================================
**
** Determine if given language present in pre-loaded tokenizer
**
** RETURNS:
**
**  TRUE when given language is present, FALSE when language is not present.
**
*/
#if 0
** REVISION HISTORY:
**
**   Date          Author/Comment
**   Jan 27, 1998  Steven Haehn
**                 Original version.
**==============================================================================
#endif

bool_t is_language_present(

    char * language,
    char * line
)
{
    char * context = strtok_r( NULL, BLANX, &line );
    bool_t language_present = FALSE;

    while( context != NULL )
    {
        if( strcmp( language, context ) == 0 )
        {
            language_present = TRUE;
            break;
        }

        context = strtok_r( NULL, BLANX, &line );
    }
    
    return language_present;
}

/*==============================================================================
**
** Search file (from name in pre-loaded tokenizer) for definition
**
** RETURNS:
**
**  TRUE means definition found, FALSE means definition not found.
**
*/
#if 0
** REVISION HISTORY:
**
**   Date          Author/Comment
**   Dec 22, 1998  Steven Haehn
**                 Removed unused variable named 'include'.
**
**   Jan 27, 1998  Steven Haehn
**                 Original version.
**==============================================================================
#endif

bool_t include_file( 

    char * file_to_include,
    char    *word,
    Indent  *indentation
)
{
    char  * file = strtok_r( NULL, BLANX, &file_to_include );
    bool_t  found_word = FALSE;

    if( file != NULL )
    {
        char filename[MAX_LINE];

        eval_environ_names( file, filename, MAX_LINE );
        found_word = is_defined( word, filename, indentation );
    }
    
    return found_word;
}

/*==============================================================================
**
** Locate and expand given abbreviation.
**
** DESCRIPTION:  
**
**  This function searches a definitions file for the given word. If the word
**  is found, it is expanded according to its definition. Conditional
**  language context is provided through the use the environment variable
**  EXP_LANGUAGE. Its value is checked against words that can occur on a "@if"
**  line. For example,
**
**    @if perl C_code
**
**     ... definitions ...
**
**    @fi
**
**  This allows the same abbreviation in the definitions file have different
**  definitions, depending on the language context.
**
**  The abbreviation definitions file lines must have the following form:
**
**     <Definition> <Expansion>
**
**  where <Definition> must start at first character position
**  of the line. The <Definition> cannot contain any white space,
**  which terminates the <Definition>. The <Expansion> follows
**  with the first non-whitespace character to the end of the line.
**  (See show_usage for complete rules). The following are some
**  definition-expansion examples.
**
**    me       Steven Haehn
**    if       if( |><| )\n@>block@
**    ef       elsif( |><| )\n@>block@
**    else     else\n@>block@
**    while    while( |><| )\n@>block@
**    for      for( |><|; |><|; |><| )\n@>block@
**    fe       foreach |><| ( |><| )\n@>block@
**    file     @@$EXPANDER_DIR/file.tpl
**    header   @@$EXPANDER_DIR/function.tpl
**    func     \.@>header@\n@>proto@
**    proto    \.|>function-prototype<|(\n\N|><|\n)\n@>block@
**    {        {\R|><|\r}
**    block    {\R|><|\r}
**
** RETURNS:
**
**  The function returns TRUE if the abbreviated word was found and expanded.
**
*/
#if 0
** REVISION HISTORY:
**
**   Date          Author/Comment
**   Mar 02, 1998  Steven Haehn
**                 Added expansion of temporary definitions.
**
**   Jan 27, 1998  Steven Haehn
**                 Added data definition file inclusion mechanism.
**
**   Jan 30, 1997  Steven Haehn
**                 Original version.
**==============================================================================
#endif

bool_t locate_definition(

    Ln_ctxt *info,
    char    *word,
    Indent  *indentation
)
{
    bool_t found_word        = FALSE;
    bool_t ignore_definition = FALSE;
    int    i;
    char   line[ MAXBUF ];
    static level = 0;
    
    /*---------------------
    * Recursion protection.
    *---------------------*/
    if( ++level > MAX_LEVEL )
    {
        printf( "** ERROR ** Recursion level greater than %d\n", MAX_LEVEL );
        printf( "Check %s for infinite recursive definitions.\n", 
           FILE_NM( info ));
        exit( 1 );
    }
    
    /*----------------------------------------------------------
    * Locate expansion among any temporary definitions supplied.
    *----------------------------------------------------------*/
    for( i = 0; i <= tdef_index; i ++ )
    {
        char * eq_sign  = strchr( temp_definitions[ i ], DEF_DELIM );
        int    def_size = eq_sign - temp_definitions[ i ];
        
        if( strncmp( word, temp_definitions[ i ], def_size ) == 0 )
        {
            char tempValue[ MAX_LINE ];
            
            eval_environ_names( eq_sign+1, tempValue, MAX_LINE );
            printf( "%s", tempValue );
            return TRUE;
        }
    }
    
    get_line( line, info );

    while( !feof( FILE_P( info ) ) && ! found_word )
    {
        char * remainder;
        char * abbreviation;

        abbreviation = strtok_r( line, BLANX, &remainder );

        if( abbreviation != NULL )
        {
            /*---------------------------------
            * Ignore definition lines until the
            * end of the conditional region.
            *---------------------------------*/
            if( ignore_definition )
            {
                if( strcmp( abbreviation, "@fi" ) == 0 )
                {
                    ignore_definition = FALSE;
                }
            }

            /*------------------------------------
            * Check for conditional usage keyword.
            *------------------------------------*/
            else if( strcmp( abbreviation, "@if" ) == 0 )
            {
                ignore_definition = 
                   ! is_language_present( language_context, remainder );
            }

            /*----------------------------------------------------
            * Check for conditional usage - file inclusion keyword
            * ( eg. @when C_code @use C_code.def)
            *----------------------------------------------------*/
            else if( strcmp( abbreviation, "@when" ) == 0 )
            {
                char * use_part = strstr( remainder, "@use" );
                
                if( use_part == NULL )
                {
                    printf( "Definition file error! Missing @use on @when line.\n" );
                }
                else
                {
                    *use_part = EOS;
                    
                    if( is_language_present( language_context, remainder ) )
                    {
                        remainder = use_part +4; /* skip over "@use" */
                        
                        found_word =
                           include_file( remainder, word, indentation );
                    }
                }
            }
            
            /*---------------------------------------------
            * Check for possible definition file inclusion.
            *---------------------------------------------*/
            else if( strcmp( abbreviation, "@include" ) == 0 )
            {
                found_word = include_file( remainder, word, indentation ); 
            }

            /*----------------------------------------
            * Expand abbreviation (only if it is
            * not the conditional context terminator).
            *----------------------------------------*/
            else if( strstr( abbreviation, word ) == abbreviation &&
                     strcmp( abbreviation, "@fi" ) != 0 )
            {
                char * expansion = strtok_r( NULL, "\n", &remainder );

                expansion = expansion + SKIP_BLANX;

                /*-----------------------------------------------
                * When the file indicator is present as the first
                * character in the expansion, assume the contents
                * of a file will replace the given abbreviation.
                *-----------------------------------------------*/
                if( *expansion == FILE_IND && *(expansion+1) == FILE_IND )
                {
                    char * file;
                    char   filename[MAX_LINE];
                    
                    expansion += 2;
                    file = expansion + SKIP_BLANX;

                    eval_environ_names( file, filename, MAX_LINE );

                    /*-----------------------------------------------
                    * When successful inclusion is made, we are done.
                    * Otherwise, continue search. This allows the
                    * same definition to have multiple values.
                    * Relying on files to contain a field.
                    *-----------------------------------------------*/
                    found_word = include_template_file( indentation, filename );
                }
                else
                {
                    char line_buffer[ MAXBUF ];
                    
                    eval_environ_names( expansion, line_buffer, MAXBUF );
                    
                    /*--------------------------------
                    * Only put out a field when one 
                    * is not present in the expansion.
                    *--------------------------------*/
                    if( ! expand_definition( info, indentation, line_buffer ))
                        printf( "%s", field_marker );

                    found_word = TRUE;
                }
            }
        }

        get_line( line, info );
    }

    level--;
    return found_word;
} 

/*----------------------------------------------------------------------------
* Extract keyword and determine indentation levels.
*
* Inputs   : original line      original input line
* 
* Outputs  : truncated_line     original input line without the keyword
*            keyword_string     keyword taken from the end of the original
*                               line
*            indentation        indentation at which the keyword was found
* 
* Purpose  : Take a possible keyword (any word) from the original line
*            and determine its indentation. There are two indentations
*            values determined; a minimum representing leading blank space,
*            and a maximum which represents the initial position of the
*            keyword.
* 
* Notes    : Preconditions:
*               'truncated_line' and 'keyword_string' are of at least the same
*               size as 'original_line'.
*----------------------------------------------------------------------------*/

void chop_keyword (

    const char *original_line,
    char       *truncated_line,
    char       *keyword_string,
    Indent     *indentation
)
{
    int    org_len   = strlen (original_line);
    int    trunc_len = org_len;
    int    key_len;
    int    i;
    int    indent         = 0;
    int    index          = 0;
    int    tabDent        = ( tab_size ) ? tab_size : 8;
    bool_t indent_min_set = FALSE;
    char * delimiter      = NULL;

    /*--------------------------------------------------
    *  Start truncated line as a copy of original line
    *--------------------------------------------------*/
    strcpy( truncated_line, original_line );

    /*------------------------------------------------------------------
    *  Find the keyword, walking back from the end of the string until
    *  keyword delimiter is found or the start of the string is reached
    *------------------------------------------------------------------*/
    while( trunc_len > 0 )
    {
        delimiter = strchr( keyword_delimiter, truncated_line[trunc_len - 1]);

        if( delimiter != NULL)
            break;

        trunc_len--;
    }

    /*------------------------------------
    * Use delimiter as keyword when it is 
    * the last character of original line.
    *------------------------------------*/
    if( trunc_len == org_len && delimiter != NULL )
        trunc_len--;
    
    /*---------------------------------------------
    *  Remove the keyword from the truncated line
    *---------------------------------------------*/
    truncated_line[trunc_len] = '\0';

    /*------------------------------------------
    *  Copy the keyword from the original line
    *------------------------------------------*/
    key_len = org_len - trunc_len;
    
    for (i = 0; i < key_len; i++)
    {
        keyword_string[i] = original_line[trunc_len + i];
    }
    keyword_string[i] = '\0';

    /*------------------------------------------------------------
    * Determine the indentation of the keyword (minding the tabs).
    *------------------------------------------------------------*/
    indentation->minimum = 0;
    indentation->maximum = 0;
    
    while (truncated_line[index] != '\0')
    {
        if (truncated_line[index] == '\t')
        {
            indent = ((indent / tabDent) + 1) * tabDent;
        }
        else if( !indent_min_set && !ISSPACE( truncated_line[index] ) )
        {
            indentation->minimum = indent++;
            indent_min_set = TRUE;
        }
        else
        {
            indent = indent + 1;
        }
        index++;
    }
    
    indentation->maximum = indent;
    
    /*--------------------------------------------------------
    * When the keyword is preceded completely by whitespace,
    * the minimum and maximum indentations should be the same.
    *--------------------------------------------------------*/
    if( !indent_min_set )
    {
        indentation->minimum = indent;
    }
}

/*--------------------------------------------------------------------
 * Inputs   : keyword_string
 *
 * Outputs  : table_index
 *
 * InOuts   : none
 *
 * Returns  : 0      keyword not found in list of keywords
 *            <> 0   keyword found
 *
 * Purpose  : Check if the keyword_string is the starting string of one of
 *            the keywords in keyword_table.
 *
 * Notes    : Preconditions:
 *               none
 *            Postconditions:
 *               return value =
 *                   0     keyword not found, 'table_index' is undefined.
 *                   <>0   keyword found, 'table_index' is the index of the
 *                         keyword in 'keyword_table'.
 *--------------------------------------------------------------------*/

int check_keyword(

    const char *keyword_string,
    int *table_index
)
{
    char   *full_keyword;
    char   *found_string = NULL;
    int     i = 0;
    bool_t  found = FALSE;

    /*--------------------------------------------------------------------*
    -- Lookup the keyword in the keyword_table.
    -- An empty keyword is caught first, because strstr will always
    -- say it is present in the string to be searched in.
    ----------------------------------------------------------------------*/
    if (keyword_string[0] != '\0')
    {
        while (!found && keyword_table[i].key_name != NULL)
        {
            full_keyword = keyword_table[i].key_name;
            found_string = strstr (full_keyword, keyword_string);
            if (found_string == full_keyword)
            {
                found = 1;
                *table_index = i;
            }
            i++;
        }
    }

    return found;
}

/*-----------------------------------------------------------------------*
-- EXPANSION FUNCTIONS
-------------------------------------------------------------------------*/

void present_time_with (char * format)
{
    time_t now;
    struct tm * currentTime;
    char   formatted_time[MAX_LINE];
    
    time( &now );
    currentTime = localtime( &now );
  
    strftime( formatted_time, MAX_LINE, format, currentTime );
    printf( "%s%s", formatted_time, field_marker );
}

bool_t expand_date (int indentation)
{
    char * format = getenv( "EXP_DATE_FORMAT" );

    if( format == NULL )
        format = EXP_DEFAULT_DATE_FORMAT;
    
    present_time_with( format );
    return TRUE;
}

bool_t expand_time (int indentation)
{
    char * format = getenv( "EXP_TIME_FORMAT" );

    if( format == NULL )
        format = EXP_DEFAULT_TIME_FORMAT;
    
    present_time_with( format );
    return TRUE;
}

bool_t expand_datetime (int indentation)
{
    char * format = getenv( "EXP_DTIME_FORMAT" );

    if( format == NULL )
        format = EXP_DEFAULT_DTIME_FORMAT;
    
    present_time_with( format );
    return TRUE;
}

bool_t expand_year (int indentation)
{
    present_time_with( "%Y" );
    return TRUE;
}

bool_t expand_user ( int indentation )
{
    printf( "%s%s", get_username(), field_marker );
    return TRUE;
}

/*----------------------------------------------------------------------------*/

void get_keyword( char * line, int limit )
{
    char * ignore;
    
    fgets( line, limit, stdin );
    strtok_r( line, "\n", &ignore ); /* make this feel like a gets call */
}

/*----------------------------------------------------------------------------*/

void show_usage(

    char * pgm_name,
    bool_t short_version
)
{
    int i;
       
    printf( "\nUsage: %s [-H] [-f template] [-i indent] [-t definition] [language]\n", 
       pgm_name );
    printf( "\n" );
    printf( "  -f allows the user to specify a template file which is used directly for\n" );
    printf( "     expansion instead of using standard input to get the expansion keyword.\n" );
    printf( "\n" );
    printf( "  -H displays extended program help about environment variables, program\n" );
    printf( "     limits, and abbreviation-definition syntax.\n" );
    printf( "\n" );
    printf( "  -i indent using given numeric value. Overrides EXP_INDENT_SIZE.\n" );
    printf( "\n" );
    printf( "  -t allows temporary definitions to be given to the program for use in the\n" );
    printf( "     expansion. The format of temporary defintions is 'abbreviation=expansion'.\n" );
    printf( "     A maximum of 10 temporary definitions is allowed.\n" );
    printf( "\n" );
    printf( "  The 'language' argument associates an abbreviation with a specific expansion.\n" );
    printf( "\n" );
    printf( "  The following are the keywords that are expanded directly by this program.\n" );
    
    /*------------------------------------------
    * Display all internally handled expansions.
    *------------------------------------------*/
    for( i=0; keyword_table[i].key_name != NULL; i++)
    {
        if( i != 0 )
            printf( ", " );
        
        if( i % 5 == 0 )
            printf( "\n    " );

        printf( "%s", keyword_table[i].key_name );
    }

    printf( "\n" );

    if( ! short_version )
    {
        printf( "\n" );
        printf( "  -------------------------------------------------------------------------\n" );
        printf( "  This program is meant to be used as a filter program to an editor for\n" );
        printf( "  computer language construct expansion. It is used to expand user specified\n" );
        printf( "  abbreviations found within special definitions file. The 'language'\n" );
        printf( "  parameter is used when the given abbreviation pertains to a specific\n" );
        printf( "  language. This allows the same abbreviation to be expanded in different\n" );
        printf( "  ways, depending on the language.\n" );
        printf( "\n" );
        printf( "  The following environment variables affect the results of the program.\n" );
        printf( "\n" );
        printf( "  EXP_DEFINITIONS   name of abbreviation definitions file.\n" );
        printf( "  EXP_DATE_FORMAT   strftime format for 'date' abbreviation.\n" );
        printf( "  EXP_TIME_FORMAT   strftime format for 'time' abbreviation.\n" );
        printf( "  EXP_DTIME_FORMAT  strftime format for 'dtime' abbreviation.\n" );
        printf( "  EXP_INDENT_SIZE   when undefined, default is %d columns.\n",
           indent_size );
        printf( "\n" );
        printf( "  EXP_USE_TABS      when undefined or zero, spaces are used to indent.\n" );
        printf( "                    otherwise, numeric value represents tab size\n" );
        printf( "                    (that is, 1 tab (\\t) character for every 'N' columns).\n" );
        printf( "                    when value is non-numeric (eg. 'yes'), the default\n" );
        printf( "                    %d is used.\n", EXP_DEFAULT_TAB_SIZE );
        printf( "\n" );
        printf( "  EXP_NO_AUTOFIELD  when defined, field marker NOT automatically added\n" );
        printf( "                    to any expansion which is missing one.\n" );
        printf( "\n" );
        printf( "  EXP_KEYWORD_DELIM used to define keyword delimiters, default is ' \\t'.\n" );
        printf( "\n" );
        printf( "  EXP_USER          name of current user, default is defined by\n" );
        printf( "                    the environment variables LOGNAME or USER.\n" );
        printf( "\n" );
        printf( "  The abbreviation definitions file lines must have the following form:\n" );
        printf( "\n" );
        printf( "      <Definition> <Expansion>\n" );
        printf( "\n" );
        printf( "  where <Definition> must start at first character position of the line.\n" );
        printf( "  The <Definition> cannot contain any white space, which terminates the\n" );
        printf( "  <Definition>. The <Expansion> follows with the first non-whitespace\n" );
        printf( "  character to the end of the line.\n" );
        printf( "\n" );
        printf( "  If the first character of the <Expansion> string is an '@@', the\n" );
        printf( "  <Expansion> is assumed to be a name of a template file that will have its\n" );
        printf( "  contents become the <Expansion>. Environment variable names can\n" );
        printf( "  be used as part of the pathname to the expansion file. These expansion\n" );
        printf( "  files are assumed to have at least one embedded field, which is\n" );
        printf( "  delimited by the characters %s.\n", FIELD_MARKER );
        printf( "\n" );
        printf( "  Environment variables in the <Expansion> field will be evaluated.\n" );
        printf( "  Legal environment variable syntax is: $LEGAL ${LEGAL} or $(LEGAL).\n" );
        printf( "  When there is no value for an environment variable, the variable is\n" );
        printf( "  removed from the <Expansion>. To have a '$' appear in the <Expansion>\n" );
        printf( "  it must be escaped with the backslash character (eg. '\\$).'\n" );
        printf( "\n" );
        printf( "  There is the concept of auto-fill fields in template file expansion. An\n" );
        printf( "  auto-fill field has a '^' as the first character in the field before the\n" );
        printf( "  field name. The field name is used as the abbreviation to be automatically\n" );
        printf( "  expanded. If the field name cannot be expanded (for instance, due to\n" );
        printf( "  misspelling) it will remain unchanged in the expansion.\n" );
        printf( "\n" );
        printf( "  Since '\\' is the typical 'C' escape character, to have it appear in a\n" );
        printf( "  resulting expansion, it must appear as '\\\\'. The '\\n' character\n" );
        printf( "  represents an auto-indent newline (indent to abbreviation). The '\\r'\n" );
        printf( "  character represents an auto-indent newline (indent to 1st non-space).\n" );
        printf( "  The '\\N' character represents an auto-indent newline to the next available\n" );
        printf( "  tab position (indent to abbreviation). The '\\R' character represents\n" );
        printf( "  an auto-indent newline to the next available tab position (indent to 1st\n" );
        printf( "  non-space). An optional digit (0-9) may follow the auto-indent characters\n" );
        printf( "  '\\n' and '\\r'. The digit represents the number of indentation levels to\n" );
        printf( "  be applied after the newline character has been inserted ('\\N' = '\\n1',\n" );
        printf( "  '\\R' = '\\r1'). A Leading '\\.' means produce an optional newline and reset\n" );
        printf( "  indentation to zero. Other special characters recognized are\n" );
        printf( "  '\\b' (backspace), '\\t' (tab), and `\\f` (formfeed).\n" );
        printf( "\n" );
        printf( "  The '@>' marks a forward definition reference.\n" );
        printf( "  The '@<' marks a backward definition reference. A definition reference\n" );
        printf( "  is terminated with '@'. For example, in the following definition of\n" );
        printf( "  'if', the forward reference to the definition of 'block' is made.\n" );
        printf( "\n" );
        printf( "      if       if( %s )\\n@>block@\n", FIELD_MARKER );
        printf( "      block    {\\R%s\\r}\n", FIELD_MARKER );
        printf( "\n" );
        printf( "  This indicates that to complete the 'if' expansion, the 'block' definition,\n" );
        printf( "  which comes later in the file, must also be examined.\n" );
        printf( "\n" );
        printf( "  The conditional use statement '@if <name> [<name>...]' is evaluated by the\n" );
        printf( "  expander in conjunction with the environment variable EXP_LANGUAGE.\n" );
        printf( "  Abbreviations outside an '@if - @fi' pair are always available to expander.\n" );
        printf( "  Abbreviations within the '@if - @fi' pair are available only if EXP_LANGUAGE\n" );
        printf( "  has one of the values of <name>. The value of EXP_LANGUAGE can be overridden\n" );
        printf( "  by the program's 'language' parameter.\n" );
        printf( "\n" );
        printf( "  An @include statement allows definition files to be nested. The following\n" );
        printf( "  is an example:\n" );
        printf( "\n" );
        printf( "      @if C_code\n" );
        printf( "      @include C_code.def\n" );
        printf( "      @fi\n" );
        printf( "\n" );
        printf( "  The file 'C_code.def' will only be examined for definitions when EXP_LANGUAGE\n" );
        printf( "  is set to 'C_code'. Be careful with definition references. They will have to\n" );
        printf( "  either explicitly appear in the include file or will have to themselves be\n" );
        printf( "  included. The default place to look for included definitions files will be\n" );
        printf( "  in the same directory where the root definitions file is found (from path\n" );
        printf( "  found in EXP_DEFINITIONS). An explicit use of an environment variable in the\n" );
        printf( "  include statement will override the default (eg. @include $NEWPATH/C_code.def)\n" );
        printf( "  There is also a 'when' statement which is equivalent to the if-include-fi\n" );
        printf( "  combination. Its syntax is: '@when <name> [<name>...] @use <fileName>'\n" );
        printf( "  The above example can be written as:\n" );
        printf( "\n" );
        printf( "      @when C_code @use C_code.def\n" );
        printf( "\n" );
        printf( "  The definitions file may contain comments. The comment line must start with\n" );
        printf( "  an exclamation mark! Multiple lines are allowed for a definition by placing\n" );
        printf( "  the '\\' character as the last character in the line. There is an upper limit\n" );
        printf( "  of %d characters for continued lines.\n", MAXBUF );
    }

    printf( "\n" );
}

/*----------------------------------------------------------------------------
 * Inputs   : none, or 'help'
 *
 * Outputs  : none
 *
 * InOuts   : none
 *
 * Returns  : process status (always 0)
 *
 * Purpose  : 'main' function of the program
 *
 * Notes    : Preconditions:
 *               there is a single line on stdin
 *            Postconditions:
 *               the input line has been processed to one or more output
 *               lines, according to the keyword found on the input line.
 *----------------------------------------------------------------------------*/

int main ( int argc, char** argv )
{
    char   original_line[MAX_LINE];
    char   truncated_line[MAX_LINE];
    char   keyword_string[MAX_LINE];
    Indent indentation = { 0, 0 };
    int    table_index;
    bool_t success   = FALSE;
    char * template_file = NULL;
    char * indent_sz = getenv( "EXP_INDENT_SIZE" );
    char * tabuse    = getenv( "EXP_USE_TABS" );
    char * language  = getenv( "EXP_LANGUAGE" );
    char * no_field  = getenv( "EXP_NO_AUTOFIELD" );
    char * key_delim = getenv( "EXP_KEYWORD_DELIM" );
    int    option;
        
    /*-----------------------------
    * Extract command line options.
    *-----------------------------*/
    while( ( option = getopt( argc, argv, "f:i:hHt:" )) != -1 )
    {
        bool_t short_version = TRUE;
        
        switch( option )
        {
          case 'f':
            template_file = optarg;
            break;
          
          case 'i':                   /* custom indent size */
            indent_sz = optarg;
            break;

          case 'H':                   /* User asking for extended help */
            short_version = FALSE;
            
          default:
          {
            char *pgm_name = remove_file_path( *argv );
            show_usage( pgm_name, short_version );
            exit( 0 );
          }
          
          case 't':                  /* Temporary definition */

            /*-------------------------------------
            * Ignore more than the maximum allowed.
            *-------------------------------------*/
            if( tdef_index < MAX_TDEFS )
            {
                temp_definitions[ ++tdef_index ] = optarg;

                if( strchr( temp_definitions[ tdef_index ], DEF_DELIM ) == NULL )
                {
                    printf( "Temporary definition (%s) missing '%c'\n",
                       temp_definitions[ tdef_index ], DEF_DELIM );
                    exit( 1 );
                }
            }
            break;
        }
    }

    /*------------------------------
    * Is language argument present?
    *------------------------------*/
    if( optind < argc  )
        language = argv[ optind ];
    
    /*---------------------------------------
    * Set user's desired language setting.
    *---------------------------------------*/
    if( language != NULL )
        language_context = language;
    
    /*------------------------------------
    * Obtain user's preferred indent size.
    *------------------------------------*/
    if( indent_sz != NULL )
    {
        int size = atoi( indent_sz );
        
        if( size > 0 )
            indent_size = size;
    }
    
    /*----------------------------------------------
    * Does user want to use real tabs for indenting?
    *----------------------------------------------*/
    if( tabuse != NULL )
    {
        int size = atoi( tabuse );
        
        if( size == 0 && *tabuse != '0' )
        {
            tab_size = EXP_DEFAULT_TAB_SIZE;
        }
        else
        {
            tab_size = size;
        }        
    }

    get_username(); /* make EXP_USER available */

    /*----------------------------------------------------
    * Disable production of automatic fields when missing?
    *----------------------------------------------------*/
    if( no_field != NULL )
        field_marker = "";

    /*---------------------------------------------------
    * Is user overriding the standard keyword delimiters?
    *---------------------------------------------------*/
    if( key_delim != NULL && strlen( key_delim ) > 0 )
        keyword_delimiter = get_new_delimiters( key_delim );
    
    /*---------------------------------------------------------
    * If a template file is explicitly given for expansion
    * purposes there is no need to locate an expansion keyword.
    *---------------------------------------------------------*/
    if( template_file != NULL )
    {
        success = include_template_file( &indentation, template_file );
    }
    else
    {
        /*---------------------------------
        * Read the input line from stdin
        *---------------------------------*/
        *original_line = EOS;
        get_keyword( original_line, MAX_LINE );

        /*------------------------------------------------
        * Remove the keyword from the end of the original 
        * line and save the truncated remaining line.
        * Determine the indentation of the keyword.
        *------------------------------------------------*/
        chop_keyword( 
           original_line, truncated_line, keyword_string, &indentation );

        /*--------------------------
        * Restore the truncated line
        *--------------------------*/
        printf( "%s", truncated_line );

        /*------------------------------------------------------------
        * If the keyword is not known (either internally or via lookup
        * in the user's definitions file), restore it and append a  
        * placeholder (for repositioning the cursor in the editor).
        * Otherwise, expand the keyword to its intended definition.
        *------------------------------------------------------------*/
        if( check_keyword( keyword_string, &table_index ))
        {
            success = 
              (*(keyword_table[table_index].expand_func)) (indentation.maximum);
        }

        /*-------------------------------------
        * No sense in looking up empty keyword.
        *-------------------------------------*/
        else if( *keyword_string != EOS )
        {
            char * definitions_file = getenv( "EXP_DEFINITIONS" );

            success = 
               is_defined( keyword_string, definitions_file, &indentation );
        }
    }
    
    if( ! success )
    {
        printf( "%s%s", keyword_string, FIELD_MARKER );
    }

    return 0;
}
