/*
 * GQradio
 * (C) 2005 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqradio.h"

#include "display.h"
#include "io_radio.h"
#include "mixer.h"
#include "preset.h"
#include "rcfile.h"
#include "skin.h"
#include "ui2_editor.h"
#include "ui2_main.h"
#include "ui2_util.h"
#include "ui_bookmark.h"
#include "ui_fileops.h"
#include "ui_help.h"
#include "ui_misc.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"
#include "window.h"

#include <errno.h>
#include <math.h>
#include <time.h>


/*
 *-----------------------------------------------------------------------------
 * the about window
 *-----------------------------------------------------------------------------
 */

static GenericDialog *about = NULL;

static void about_destroy_cb(GtkWidget *widget, gpointer data)
{
	about = NULL;
}

void show_about(void)
{
	GtkWidget *label;
	GtkWidget *image;
	GdkPixbuf *pixbuf;
	GtkWidget *cred;
	gchar *buf;

	if (about)
		{
		gtk_window_present(GTK_WINDOW(about->dialog));
		return;
		}

	about = generic_dialog_new(_("About - GQradio"),
				   "about", "GQradio",
				   NULL, TRUE, NULL, NULL);
	g_signal_connect(G_OBJECT(about->dialog), "destroy",
			 G_CALLBACK(about_destroy_cb), NULL);
	generic_dialog_add_button(about, GTK_STOCK_CLOSE, NULL, NULL, TRUE);

	pixbuf = gdk_pixbuf_new_from_xpm_data(gqradio_logo_xpm);
	image = gtk_image_new_from_pixbuf(pixbuf);
	gdk_pixbuf_unref(pixbuf);

	gtk_box_pack_start(GTK_BOX(about->vbox), image, FALSE, FALSE, 0);
	gtk_widget_show(image);

	buf = g_strdup_printf("GQradio\n%s\nCopyright (c) %s John Ellis\nwebsite: %s\nemail: %s\n\nReleased under the GNU General Public License",
			      VERSION,
			      "2005",
			      "gqmpeg.sourceforge.net",
			      "gqview@users.sourceforge.net");
	label = pref_label_new(about->vbox, buf);
	g_free(buf);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_CENTER);

	cred = ui_slik_credit();
	gtk_box_pack_start(GTK_BOX(about->vbox), cred, FALSE, FALSE, 0);
	gtk_widget_show(cred);

	window_set_icon(about->dialog, gqradio_logo_xpm, NULL, NULL);
	gtk_widget_show(about->dialog);
}

static GtkWidget *help_window = NULL;

static void help_window_destroy_cb(GtkWidget *window, gpointer data)
{
	help_window = NULL;
}

void help_window_show(const gchar *key)
{
	if (help_window)
		{
		gdk_window_raise(help_window->window);
		if (key) help_window_set_key(help_window, key);
		return;
		}

	help_window = help_window_new(_("Help - GQradio"), "GQradio", "help",
                                      GQRADIO_HELPDIR "/README", key);
	g_signal_connect(G_OBJECT(help_window), "destroy",
			 G_CALLBACK(help_window_destroy_cb), NULL);
}

/*
 *-----------------------------------------------------------------------------
 * misc start-up utils
 *-----------------------------------------------------------------------------
 */

static void keys_load(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", PATH_HISTORY_NAME, NULL);
	history_list_load(path);
	g_free(path);
}

static void keys_save(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", PATH_HISTORY_NAME, NULL);
	history_list_save(path);
	g_free(path);
}

static void states_load(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", RC_STATES_NAME, NULL);
	ui_states_load(path);
	g_free(path);
}

static void states_save(void)
{
	gchar *path;

	path = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", RC_STATES_NAME, NULL);
	ui_states_save(path);
	g_free(path);
}

static void check_for_home_path(gchar *path)
{
	gchar *buf;
	buf = g_strconcat(homedir(), "/", path, NULL);
	if (!isdir(buf))
		{
		printf(_("Creating dir:%s\n"), buf);
		if (!mkdir_utf8(buf, 0755))
			{
			printf(_("Could not create dir:%s\n"), buf);
			}
		}
	g_free(buf);
}

static gchar *determine_skin_path(gchar *file)
{
	gchar *path = NULL;
	gchar *buf;
	gchar *cd_buf;

	if (isdir(file))
		{
		path = g_strdup(file);
		}
	else
		{
		cd_buf = get_current_dir();
		buf = g_strconcat(get_current_dir(), "/", file, NULL);
		g_free(cd_buf);
		if (isdir(buf))
			{
			path = buf;
			}
		else
			{
			g_free(buf);
			buf = g_strconcat(homedir(), "/", 
					GQRADIO_RC_DIR_SKIN, "/", file, NULL);
			if (isdir(buf))
				{
				path = buf;
				}
			else
				{
				g_free(buf);
				buf = g_strconcat(GQRADIO_SKINDIR, "/", file, NULL);
				if (isdir(buf))
					{
					path = buf;
					}
				else
					{
					g_free(buf);
					}
				}
			}
		}

	if (path)
		{
		if (debug_mode) printf (_("Using skin: %s\n"), path);
		}
	else
		{
		printf (_("Skin not found: %s\n"), file);
		}

	return path;
}

/*
 *-----------------------------------------------------------------------------
 * help function
 *-----------------------------------------------------------------------------
 */

static void show_help(void)
{
	printf("GQradio version %s\n", VERSION);
	printf(_("usage: gqradio [option(s)]\n"));
	printf(_(" standard options:\n"));
	printf(_("  -f, --frequency nn.n         start with frequency n\n"));
	printf(_("  -p, --preset n               start with preset n\n"));
	printf(_("  --skin:<skindir>             load specified skin\n"));
	printf(_("  --skinhelp                   print mouse event coordinates\n"));
	printf(_("  --no-seek-mute               do not mute when auto seeking\n"));
	printf(_("  --debug                      enable debug output\n"));
	printf(_("  -h,  --help                  displays this message\n"));
	printf(_("  --crack                      reserved\n"));
}

/*
 *-----------------------------------------------------------------------------
 * exit function
 *-----------------------------------------------------------------------------
 */

void gqradio_exit(gint mute)
{
	gchar *buf;


	g_free(skin_default_path);
	skin_default_path = g_strdup(main_window->skin_path);
	g_free(skin_mode_key);
	skin_mode_key = g_strdup(main_window->skin_mode_key);

	if (mute) radio_shutdown();

	save_options();
	keys_save();
	states_save();

        buf = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", GQRADIO_PRESET_FILE, NULL);
        preset_list_save(buf);
        g_free(buf);

	gtk_main_quit();
}

/*
 *-----------------------------------------------------------------------------
 * main routine
 *-----------------------------------------------------------------------------
 */

int main (int argc, char *argv[])
{
	const gchar *cbuf;
	gchar *buf;
	gchar *startup_skin_file = NULL;
	gint i;
	gint tc = 0;
	gint radio_fail = FALSE;
	gint radio_errno = 0;

	guint32 start_freq = 0;
	gint start_preset = -1;

	/* setup locale, i18n */
	gtk_set_locale();
	bindtextdomain(PACKAGE, GQRADIO_LOCALEDIR);
	bind_textdomain_codeset(PACKAGE, "UTF-8");
	textdomain(PACKAGE);

	/* setup random seed */
	srand(time(NULL));

	/* check that we can find a valid home directory, if not exit semi-gracefully */
	cbuf = homedir();
	if (!cbuf)
		{
		printf(_("Unable to determine the home directory\n"));
		return 0;
		}
	else if (!isdir(cbuf))
		{
		printf(_("Could not find home directory: %s\n"), cbuf);
		return 0;
		}

	/* create dirs */
	check_for_home_path(GQRADIO_RC_DIR);
	check_for_home_path(GQRADIO_RC_DIR_SKIN);

	if (argc > 1)
		{
		i = 1;
		while (i < argc)
			{
			if (strcmp(argv[i],"--debug") == 0)
				{
				debug_mode = TRUE;
				printf("debugging enabled\n");
				}
			else if (strcmp(argv[i],"--skinhelp") == 0)
				{
				debug_skin = TRUE;
				printf("skin debugging enabled\n");
				}
			else if (strcmp(argv[i], "--no-seek-mute") == 0)
				{
				auto_seek_nomute = TRUE;
				}
			else if (strncmp(argv[i],"--skin:", 7) == 0)
				{
				if (startup_skin_file)
					{
					printf(_("Ignoring multiple skins!\n"));
					}
				else
					{
					startup_skin_file = determine_skin_path(argv[i] + 7);
					}
				}
			else if (strcmp(argv[i],"-f") == 0 || strcmp(argv[i],"--frequency") == 0)
				{
				i++;
				if (i < argc)
					{
					start_freq = radio_freq_from_text(argv[i]);
					}
				}
			else if (strcmp(argv[i],"-p") == 0 || strcmp(argv[i],"--preset") == 0)
				{
				i++;
				if (i < argc)
					{
					start_preset = atoi(argv[i]);
					}
				}
			else if (strcmp(argv[i],"-h") == 0 || strcmp(argv[i],"--help") == 0)
				{
				show_help();
				return 0;
				}
			else if (strcmp(argv[i],"--crack") == 0)
				{
				tc++;
				}
			else
				{
				printf(_("unknown option: %s\nuse --help for options\n"),argv[i]);
				return 0;
				}
			i++;
			}
		}

/* */
	/*
	 * the gtk inits are moved after the command line parser
	 * (for speed with command line options, or errors thereof)
	 */
	gtk_init (&argc, &argv);

	/* parse a custom gtkrc */
	buf = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/gtkrc", NULL);
	if (isfile(buf)) gtk_rc_parse(buf);
	g_free(buf);

	radio_custom_device = g_strdup(RADIO_DEVICE);
	preset_list_init(PRESET_LIST_SIZE);

	/* load options */
	load_options();
	keys_load();
	states_load();

	radio_set_device(radio_custom_device);
	if (!radio_test(FALSE))
		{
		printf(_("Error, radio support missing, or failed to init radio device!\n"));
		radio_fail = TRUE;
		radio_errno = errno;
		}

	/* setup mixer */

	if (!mixer_command) mixer_command = g_strdup("xmixer");
	mixer_init(mixer_device_id);

	/* setup freq and preset */

	buf = g_strconcat(homedir(), "/", GQRADIO_RC_DIR, "/", GQRADIO_PRESET_FILE, NULL);
	preset_list_load(buf);
	g_free(buf);

	if (start_preset >= 1 && preset_is_set(start_preset - 1))
		{
		preset = start_preset - 1;
		frequency = preset_get_freq(preset);
		}
	else
		{
		if (start_freq > 0 && start_freq >= radio_limit_get_lower() && start_freq <= radio_limit_get_upper())
			{
			frequency = start_freq;
			}
		else if (frequency == 0)
			{
			frequency = radio_freq_clamp_to_increment((radio_limit_get_lower() + radio_limit_get_upper()) / 2, freq_step);
			}
		preset = preset_find_freq(frequency);
		}

	/* register base dirs to editor */
	buf = g_strconcat(homedir(), "/", GQRADIO_RC_DIR_SKIN, NULL);
	ui_edit_add_skin_resource(buf);
	g_free(buf);
	ui_edit_add_skin_resource(GQRADIO_SKINDIR);

	/* this app does not allow keyboard focus */
	slik_focus_enable = FALSE;

	/* set up main window */
	main_window = window_main_new(startup_skin_file);
	g_free(startup_skin_file);
	startup_skin_file = NULL;

	if (radio_fail)
		{
		buf = g_strdup_printf(_("Failed to start %s"), (radio_custom_device) ? radio_custom_device : RADIO_DEVICE);
		ui_set_message(main_window, main_window->skin_path, "skindata_error",
			       buf, strerror(radio_errno));
		g_free(buf);
		}

	gtk_widget_show(main_window->window);

	while (tc > 0)
		{ 
		tc--;
		ui_distort(main_window);
		}

	main_loop_init();

	/* start-up radio */
	radio_startup(frequency);

	gtk_main ();
	return 0;
}

