/*
 * GQradio
 * (C) 2003 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */

#include "gqradio.h"
#include "window.h"

#include "display.h"
#include "io_radio.h"
#include "preferences.h"
#include "skin.h"
#include "ui_fileops.h"
#include "ui_menu.h"
#include "ui2_display.h"
#include "ui2_editor.h"
#include "ui2_main.h"
#include "ui2_menu.h"
#include "ui2_parse.h"

#include <gdk/gdkkeysyms.h> /* for key values */


/*
 *----------------
 * the callbacks
 *----------------
 */

static void menu_mute_cb(GtkWidget *widget, gpointer data)
{
	mute_toggle();
}

static void menu_mixer_cb(GtkWidget *widget, gpointer data)
{
	mixer_run();
}

static void menu_prefs_cb(GtkWidget *widget, gpointer data)
{
	config_window_show(0);
}

static void menu_editor_cb(GtkWidget *widget, gpointer data)
{
	UIData *ui = data;

	edit_window(ui);
}

static void menu_help_cb(GtkWidget *widget, gpointer data)
{
	help_window_show("documentation");
}

static void menu_about_cb(GtkWidget *widget, gpointer data)
{
	show_about();
}

static void menu_iconify_cb(GtkWidget *widget, gpointer data)
{
	ui_iconify(main_window);
}

static void menu_close_cb(GtkWidget *widget, gpointer data)
{
	UIData *ui = data;

	ui_close(ui);
}

static void menu_exit_no_mute_cb(GtkWidget *widget, gpointer data)
{
	gqradio_exit(FALSE);
}

static void menu_exit_cb(GtkWidget *widget, gpointer data)
{
	gqradio_exit(TRUE);
}

static void menu_preset_edit_cb(GtkWidget *widget, gpointer data)
{
	config_window_show(2);
}

static void menu_preset_clear_cb(GtkWidget *widget, gpointer data)
{
	preset_clear_one(preset);
}

static void menu_preset_clear_all_cb(GtkWidget *widget, gpointer data)
{
	preset_clear_all();
}

static void menu_preset_auto_set_cb(GtkWidget *widget, gpointer data)
{
	preset_auto_scan_click();
}

static void menu_seek_manual_cb(GtkWidget *widget, gpointer data)
{
	mode_set(SEEK_MODE_MANUAL);
}

static void menu_seek_preset_cb(GtkWidget *widget, gpointer data)
{
	mode_set(SEEK_MODE_PRESET);
}

static void menu_seek_auto_cb(GtkWidget *widget, gpointer data)
{
	mode_set(SEEK_MODE_AUTO);
}

/*
 *----------------
 * the menus
 *----------------
 */

static GtkWidget *menu_main(UIData *ui)
{
	GtkWidget *menu;
	GtkWidget *preset_menu;
	GtkWidget *seek_menu;
	GtkWidget *submenu;

	menu = popup_menu_short_lived();

	preset_menu = menu_item_add_stock(menu, _("_Presets"), GTK_STOCK_INDEX, NULL, NULL);
	seek_menu = menu_item_add(menu, _("_Seek Mode"), NULL, NULL);
	menu_item_add_stock(menu, _("Pre_ferences..."), GTK_STOCK_PREFERENCES,
		G_CALLBACK(menu_prefs_cb), NULL);
	menu_item_add_divider(menu);
	menu_item_add_check(menu, _("_Mute"), muted,
		G_CALLBACK(menu_mute_cb), NULL);
	menu_item_add_stock(menu, _("_Run Mixer..."), GTK_STOCK_EXECUTE,
		G_CALLBACK(menu_mixer_cb), NULL);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("Skin _Editor..."),
		G_CALLBACK(menu_editor_cb), ui);
	menu_item_add_stock(menu, _("_Help..."), GTK_STOCK_HELP,
		G_CALLBACK(menu_help_cb), NULL);
	menu_item_add(menu, _("_About..."),
		G_CALLBACK(menu_about_cb), NULL);
	menu_item_add_divider(menu);
	menu_item_add(menu, _("_Iconify"),
		G_CALLBACK(menu_iconify_cb), NULL);
	if (ui != main_window)
		{
		menu_item_add_stock(menu, _("Close _Window"), GTK_STOCK_CLOSE,
			G_CALLBACK(menu_close_cb), ui);
		}
	menu_item_add(menu, _("Exit (no mute)"),
		G_CALLBACK(menu_exit_no_mute_cb), NULL);
	menu_item_add_stock(menu, _("E_xit"), GTK_STOCK_QUIT,
		G_CALLBACK(menu_exit_cb), NULL);

	submenu = gtk_menu_new();
	menu_item_add(submenu, _("_Edit..."),
		G_CALLBACK(menu_preset_edit_cb), NULL);
	menu_item_add_divider(submenu);

	menu_item_add_stock_sensitive(submenu, _("Clear _current"), GTK_STOCK_REMOVE, (preset >= 0),
		G_CALLBACK(menu_preset_clear_cb), NULL);

	menu_item_add_stock(submenu, _("Clear _all"), GTK_STOCK_CLEAR,
		G_CALLBACK(menu_preset_clear_all_cb), NULL);
	menu_item_add_divider(submenu);
	menu_item_add(submenu, _("Auto _set"),
		G_CALLBACK(menu_preset_auto_set_cb), NULL);

	gtk_menu_item_set_submenu(GTK_MENU_ITEM(preset_menu), submenu);

	submenu = gtk_menu_new();
	menu_item_add_check(submenu, _("_Manual"), (seek_mode == SEEK_MODE_MANUAL),
		G_CALLBACK(menu_seek_manual_cb), NULL);
	menu_item_add_check(submenu, _("_Preset"), (seek_mode == SEEK_MODE_PRESET),
		G_CALLBACK(menu_seek_preset_cb), NULL);
	menu_item_add_check(submenu, _("_Auto"), (seek_mode == SEEK_MODE_AUTO),
		G_CALLBACK(menu_seek_auto_cb), NULL);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(seek_menu), submenu);

	return menu;
}

/*
 *----------------
 * the skinned menu
 *----------------
 */

static void menu_skin_mute_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	mute_toggle();
}

static void menu_skin_mixer_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	mixer_run();
}

static void menu_skin_prefs_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	config_window_show(0);
}

static void menu_skin_editor_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	UIData *ui = data;

	edit_window(ui);
}

static void menu_skin_help_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	help_window_show("documentation");
}

static void menu_skin_about_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	show_about();
}

static void menu_skin_iconify_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	ui_iconify(main_window);
}

static void menu_skin_close_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	UIData *ui = data;

	ui_close(ui);
}

static void menu_skin_exit_no_mute_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	gqradio_exit(FALSE);
}

static void menu_skin_exit_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	gqradio_exit(TRUE);
}

static void menu_skin_preset_edit_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	config_window_show(2);
}

static void menu_skin_preset_clear_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	preset_clear_one(preset);
}

static void menu_skin_preset_clear_all_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	preset_clear_all();
}

static void menu_skin_preset_auto_set_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	preset_auto_scan_click();
}

static void menu_skin_seek_manual_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	mode_set(SEEK_MODE_MANUAL);
}

static void menu_skin_seek_preset_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	mode_set(SEEK_MODE_PRESET);
}

static void menu_skin_seek_auto_cb(UIMenuData *md, const gchar *path, gint id, gpointer data)
{
	mode_set(SEEK_MODE_AUTO);
}

#define DIVIDER_TEXT " ------"

static gint menu_main_skinned(gint button, guint32 event_time, UIData *ui)
{
	UIMenuData *md;
	SkinData *skin;
	const gchar *path;
	gchar *datafile;

	if (!skinned_menus_enable) return FALSE;

	if (main_window->skin_path)
		{
		path = main_window->skin_path;
		}
	else
		{
		path = GQRADIO_SKIN_DEFAULT;
		}

	datafile = g_strconcat(path, "/", "skindata_menu", NULL);
	/* must check file existance first, so that skins without
	 * menu support do not print an error message (this is not really an error)
	 */
	if (isfile(datafile))
		{
		skin = skin_parse(path, datafile, FALSE);
		}
	else
		{
		skin = NULL;
		}
	g_free(datafile);

	if (!skin) return FALSE;

	md = ui_menu_new("GQradio", "main_menu");
	ui_group_set_child(main_window, md->ui);

	ui_menu_item_add(md, _("Presets"), "/presets", 0,
			 NULL, NULL);
	ui_menu_item_add(md, _("Seek Mode"), "/seek_mode", 0,
			 NULL, NULL);
	ui_menu_item_add(md, _("Preferences..."), "/prefs", 0,
			 menu_skin_prefs_cb, NULL);
	ui_menu_item_add_sensitive(md, DIVIDER_TEXT, "/div1", 0, FALSE,
			 NULL, NULL);
	ui_menu_item_add(md, (muted) ? _("UnMute") : _("Mute"), "/mute", 0,
			 menu_skin_mute_cb, NULL);
	ui_menu_item_add(md, _("Run Mixer..."), "/mixer", 0,
			 menu_skin_mixer_cb, NULL);
	ui_menu_item_add_sensitive(md, DIVIDER_TEXT, "/div2", 0, FALSE,
			 NULL, NULL);
	ui_menu_item_add(md, _("Skin Editor..."), "/editor", 0,
			 menu_skin_editor_cb, ui);
	ui_menu_item_add(md, _("Help..."), "/help", 0,
			 menu_skin_help_cb, NULL);
	ui_menu_item_add(md, _("About..."), "/about", 0,
			 menu_skin_about_cb, NULL);
	ui_menu_item_add_sensitive(md, DIVIDER_TEXT, "/div3", 0, FALSE,
			 NULL, NULL);
	ui_menu_item_add(md, _("Iconify"), "/iconify", 0,
			 menu_skin_iconify_cb, NULL);
	if (ui != main_window)
		{
		ui_menu_item_add(md, _("Close Window"), "close/", 0,
				 menu_skin_close_cb, NULL);
		}
	ui_menu_item_add(md, _("Exit (no mute)"), "/exit_no_mute", 0,
			 menu_skin_exit_no_mute_cb, NULL);
	ui_menu_item_add(md, _("Exit"), "/exit", 0,
			 menu_skin_exit_cb, NULL);

	ui_menu_item_add(md, _("Edit..."), "/presets/edit", 0,
			 menu_skin_preset_edit_cb, NULL);
	ui_menu_item_add_sensitive(md, DIVIDER_TEXT, "/presets/div1", 0, FALSE,
			 NULL, NULL);
	ui_menu_item_add_sensitive(md, _("Clear current"), "/presets/clear_current", 0, (preset >= 0),
			 menu_skin_preset_clear_cb, NULL);
	ui_menu_item_add(md, _("Clear all"), "/presets/clear_all", 0,
			 menu_skin_preset_clear_all_cb, NULL);
	ui_menu_item_add_sensitive(md, DIVIDER_TEXT, "/presets/div2", 0, FALSE,
			 NULL, NULL);
	ui_menu_item_add(md, _("Auto set"), "/presets/auto_set", 0,
			 menu_skin_preset_auto_set_cb, NULL);

	ui_menu_item_add_check(md, _("Manual"), "/seek_mode/manual", 0, (seek_mode == SEEK_MODE_MANUAL),
			 menu_skin_seek_manual_cb, NULL);
	ui_menu_item_add_check(md, _("Preset"), "/seek_mode/preset", 0, (seek_mode == SEEK_MODE_PRESET),
			 menu_skin_seek_preset_cb, NULL);
	ui_menu_item_add_check(md, _("Auto"), "/seek_mode/auto", 0, (seek_mode == SEEK_MODE_AUTO),
			 menu_skin_seek_auto_cb, NULL);

	ui_skin_set(md->ui, skin, path, "skindata_menu");
	ui_menu_show(md, -1, -1, button, event_time);

	return TRUE;
}

/*
 *----------------
 * the window
 *----------------
 */


void edit_window(UIData *ui)
{
	EditData *ed;

	if (!ui) ui = main_window;

	ed = ui_edit_new(ui, "slik_test", "skin_editor", NULL);

	/* ed is NULL if an edit window is already active for the ui */
	if (!ed) return;
	ui_edit_set_help_path(ed, GQRADIO_SLIK_HELP_PATH);
}

static gchar *number_press_buffer = NULL;
static gint number_press_id = -1;

static void number_buffer_do(gint ignore)
{
	if (number_press_buffer && !ignore)
		{

		if (number_press_buffer[0] == '0')
			{
			/* leading 0 indicates a preset */
			preset_select(atoi(number_press_buffer + 1) - 1);
			}
		else
			{
			guint32 freq;

			freq = radio_freq_from_text(number_press_buffer);
			if (freq >= radio_limit_get_lower() &&
			    freq <= radio_limit_get_upper())
				{
				seek_to_freq(freq);
				}
			}
		}

	if (number_press_buffer)
		{
		display_set_frequency(frequency);
		display_set_description(NULL);
		display_hide_minor_digits(FALSE);
		}

	g_free(number_press_buffer);
	number_press_buffer = NULL;
}

static gint number_buffer_cb(gpointer data)
{
	number_buffer_do(FALSE);
	number_press_id = -1;
	return FALSE;
}

static void number_buffer_add(gint keyval)
{
	gchar *tmp;
	gchar *buf;
	gchar *ptr;

	tmp = number_press_buffer;
	number_press_buffer = g_strdup_printf("%s%c", (tmp) ? tmp : "", (char)keyval);
	g_free(tmp);

	if (number_press_id != -1) g_source_remove(number_press_id);
	number_press_id = -1;

	ptr = number_press_buffer;

	if (*ptr == '0')
		{
		if (strlen(number_press_buffer) >= 3)
			{
			number_buffer_do(FALSE);
			return;
			}
		display_hide_minor_digits(TRUE);

		buf = g_strdup_printf(_("Preset: %s"), number_press_buffer + 1);
		display_set_description(buf);
		g_free(buf);
		}
	else
		{
		while (*ptr != '.' && *ptr != '\0') ptr++;
		if (*ptr == '.')
			{
			ptr++;
			if (ptr[0] != '\0' && ptr[1] != '\0')
				{
				number_buffer_do(FALSE);
				return;
				}
			}
		buf = g_strdup_printf(_("Jump: %s"), number_press_buffer);
		display_set_description(buf);
		g_free(buf);
		}

	display_set_frequency(radio_freq_from_text(number_press_buffer));

	number_press_id = g_timeout_add(1500, number_buffer_cb, NULL);
}

static gint key_pressed(GtkWidget *widget, GdkEventKey *event, gpointer data)
{
	UIData *ui = data;
	gint was_number = FALSE;

	if (debug_mode) printf("key pressed: %d\n", event->keyval);

	if (event->state & GDK_CONTROL_MASK)
		{
		switch (event->keyval)
			{
			case 'O': case 'o':
				config_window_show(0);
				break;
			case 'E': case 'e':
			case 'P': case 'p':
				config_window_show(2);
				break;
			case GDK_Delete: case GDK_KP_Delete:
				preset_clear_all();
				break;
			case '-':
			case GDK_KP_Subtract:
				btn_balance_left_pressed();
				btn_balance_released();
				break;
			case '+': case '=':
			case GDK_KP_Add:
				btn_balance_right_pressed();
				btn_balance_released();
				break;
			default:
				break;
			}
		}
	else
		{
		switch (event->keyval)
			{
			case GDK_Escape:
				if (preset_mode_get())
					{
					preset_cancel();
					}
				break;
			case 'n': case 'N':
			case GDK_space:
				seek_start(TRUE);
				seek_stop();
				break;
			case 'b': case 'B':
			case GDK_BackSpace:
				seek_start(FALSE);
				seek_stop();
				break;
			case GDK_Right: case GDK_KP_Right:
				mode_set(SEEK_MODE_MANUAL);
				seek_start(TRUE);
				seek_stop();
				break;
			case GDK_Left: case GDK_KP_Left:
				mode_set(SEEK_MODE_MANUAL);
				seek_start(FALSE);
				seek_stop();
				break;
			case GDK_Up: case GDK_KP_Up:
				mode_set(SEEK_MODE_AUTO);
				seek_start(TRUE);
				seek_stop();
				break;
			case GDK_Down: case GDK_KP_Down:
				mode_set(SEEK_MODE_AUTO);
				seek_start(FALSE);
				seek_stop();
				break;
			case GDK_Page_Down: case GDK_KP_Page_Down:
				preset_prev();
				break;
			case GDK_Page_Up: case GDK_KP_Page_Up:
				preset_next();
				break;
			case 'M': case 'm':
				mute_toggle();
				break;
			case '-':
			case GDK_KP_Subtract:
				btn_volume_down_pressed();
				btn_volume_released();
				break;
			case '+': case '=':
			case GDK_KP_Add:
				btn_volume_up_pressed();
				btn_volume_released();
				break;
			case GDK_Return:
				mode_toggle();
				break;
			case GDK_Delete: case GDK_KP_Delete:
				preset_clear_one(preset);
				break;
			case 'O': case 'o':
				config_window_show(0);
				break;
			case 'E': case 'e':
				config_window_show(2);
				break;
			case 'P': case 'p':
				preset_button_click();
				break;
			case 'I': case 'i':
				ui_iconify(main_window);
				break;
			case 'Q': case 'q':
				gqradio_exit(!(event->state & GDK_SHIFT_MASK));
				break;
			case '!':
				ui_debug_print_all_keys(main_window);
				break;
			case '~': case '`':
				edit_window(ui);
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
			case '.':
				was_number = TRUE;
				number_buffer_add(event->keyval);
				break;
		/* sorry, keypad numbers got ugly */
			case GDK_KP_0:
				was_number = TRUE;
				number_buffer_add('0');
				break;
			case GDK_KP_1:
				was_number = TRUE;
				number_buffer_add('1');
				break;
			case GDK_KP_2:
				was_number = TRUE;
				number_buffer_add('2');
				break;
			case GDK_KP_3:
				was_number = TRUE;
				number_buffer_add('3');
				break;
			case GDK_KP_4:
				was_number = TRUE;
				number_buffer_add('4');
				break;
			case GDK_KP_5:
				was_number = TRUE;
				number_buffer_add('5');
				break;
			case GDK_KP_6:
				was_number = TRUE;
				number_buffer_add('6');
				break;
			case GDK_KP_7:
				was_number = TRUE;
				number_buffer_add('7');
				break;
			case GDK_KP_8:
				was_number = TRUE;
				number_buffer_add('8');
				break;
			case GDK_KP_9:
				was_number = TRUE;
				number_buffer_add('9');
				break;
			case GDK_KP_Decimal:
				was_number = TRUE;
				number_buffer_add('.');
				break;
			case '[':
				display_scroll_preset_list(TRUE);
				break;
			case ']':
				display_scroll_preset_list(FALSE);
				break;
			default:
				break;
			}
		}

	if (!was_number) number_buffer_do(TRUE);

	return FALSE;
}

static gint window_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	gqradio_exit(TRUE);

	return TRUE;
}

void window_main_set_title(const gchar *title)
{
	if (main_window)
		{
		gchar *buf;

		if (title)
			{
			buf = g_strconcat(title, " - GQradio", NULL);
			}
		else
			{
			buf = g_strdup("GQradio");
			}

		ui_title_set(main_window, buf);
		g_free(buf);
		}
}

static void window_main_mouse_cb(UIData *ui, gint button, guint32 time, gpointer data)
{
	if (button == 3)
		{
		if (!menu_main_skinned(button, time, ui))
			{
			GtkWidget *menu;

			menu = menu_main(ui);
			gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL, button, time);
			}
		}
	else if (button == 4)
		{
		btn_volume_up_pressed();
		btn_volume_released();					  
		}
	else if (button == 5)
		{
		btn_volume_down_pressed();
		btn_volume_released();
		}
}

static gint window_subwindow_delete_cb(GtkWidget *widget, GdkEventAny *event, gpointer data)
{
	UIData *ui = data;

	ui_close(ui);
	return TRUE;
}

static void window_subwindow_cb(UIData *child, const gchar *key, gpointer data)
{
	g_signal_connect(G_OBJECT(child->window), "delete_event",
			 G_CALLBACK(window_subwindow_delete_cb), child);
	g_signal_connect(G_OBJECT(child->display), "key_press_event",
			 G_CALLBACK(key_pressed), child);

	ui_set_icon(child, gqradio_logo_xpm, NULL, NULL);
	ui_title_set(child, _("GQradio"));
}

UIData *window_main_new(const gchar *skin_path)
{
	UIData *ui;

	ui = ui_new("GQradio", "main", wm_decorations, "GQradio");
	ui_set_icon(ui, gqradio_logo_xpm, NULL, NULL);

	ui_set_skin_callback(ui, skin_load_default_cb, NULL);

	g_signal_connect(G_OBJECT(ui->window), "delete_event",
			 G_CALLBACK(window_delete_cb), ui);
	g_signal_connect(G_OBJECT(ui->window), "key_press_event",
			 G_CALLBACK(key_pressed), ui);

	ui_set_mouse_callback(ui, window_main_mouse_cb, ui);
	ui_set_new_window_callback(ui, window_subwindow_cb, NULL);

	/* register the right stuff */
	display_register_widgets(ui);

	if (skin_path)
		{
		printf(_("loading skin..\n"));
		if (!ui_skin_load(ui, skin_path, NULL))
			{
			if (skin_default_path)
				{
				if (!ui_skin_load(ui, skin_default_path, skin_mode_key))
					{
					printf(_("Default skin not loaded, using built-in default\n"));
					ui_skin_load(ui, NULL, NULL);
					}
				}
			else
				{
				printf(_("Skin not loaded, using built-in default\n"));
				ui_skin_load(ui, NULL, NULL);
				}
			}
		}
	else if (skin_default_path)
		{
		if (!ui_skin_load(ui, skin_default_path, skin_mode_key))
			{
			printf(_("Skin in config file not loaded, using built-in default\n"));
			ui_skin_load(ui, NULL, NULL);
			}
		}
	else
		{
		ui_skin_load(ui, NULL, skin_mode_key);
		}

	return ui;
}
