﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#include <aws/cloudformation/model/StackSetStatus.h>
#include <aws/core/Globals.h>
#include <aws/core/utils/EnumParseOverflowContainer.h>
#include <aws/core/utils/HashingUtils.h>

using namespace Aws::Utils;

namespace Aws {
namespace CloudFormation {
namespace Model {
namespace StackSetStatusMapper {

static const int ACTIVE_HASH = HashingUtils::HashString("ACTIVE");
static const int DELETED_HASH = HashingUtils::HashString("DELETED");

StackSetStatus GetStackSetStatusForName(const Aws::String& name) {
  int hashCode = HashingUtils::HashString(name.c_str());
  if (hashCode == ACTIVE_HASH) {
    return StackSetStatus::ACTIVE;
  } else if (hashCode == DELETED_HASH) {
    return StackSetStatus::DELETED;
  }
  EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
  if (overflowContainer) {
    overflowContainer->StoreOverflow(hashCode, name);
    return static_cast<StackSetStatus>(hashCode);
  }

  return StackSetStatus::NOT_SET;
}

Aws::String GetNameForStackSetStatus(StackSetStatus enumValue) {
  switch (enumValue) {
    case StackSetStatus::NOT_SET:
      return {};
    case StackSetStatus::ACTIVE:
      return "ACTIVE";
    case StackSetStatus::DELETED:
      return "DELETED";
    default:
      EnumParseOverflowContainer* overflowContainer = Aws::GetEnumOverflowContainer();
      if (overflowContainer) {
        return overflowContainer->RetrieveOverflow(static_cast<int>(enumValue));
      }

      return {};
  }
}

}  // namespace StackSetStatusMapper
}  // namespace Model
}  // namespace CloudFormation
}  // namespace Aws
