const std = @import("std");
const assert = std.debug.assert;
const codegen = @import("../../codegen.zig");
const link = @import("../../link.zig");
const log = std.log.scoped(.codegen);
const tracking_log = std.log.scoped(.tracking);
const verbose_tracking_log = std.log.scoped(.verbose_tracking);
const wip_mir_log = std.log.scoped(.wip_mir);

const Air = @import("../../Air.zig");
const Allocator = std.mem.Allocator;
const Emit = @import("Emit.zig");
const Liveness = @import("../../Liveness.zig");
const Lower = @import("Lower.zig");
const Mir = @import("Mir.zig");
const Zcu = @import("../../Zcu.zig");
const Module = @import("../../Package/Module.zig");
const InternPool = @import("../../InternPool.zig");
const Type = @import("../../Type.zig");
const Value = @import("../../Value.zig");

const abi = @import("abi.zig");
const bits = @import("bits.zig");
const encoder = @import("encoder.zig");

const Condition = bits.Condition;
const Immediate = bits.Immediate;
const Memory = bits.Memory;
const Register = bits.Register;
const RegisterManager = abi.RegisterManager;
const RegisterLock = RegisterManager.RegisterLock;
const FrameIndex = bits.FrameIndex;

const InnerError = codegen.CodeGenError || error{OutOfRegisters};

/// Set this to `false` to uncover Sema OPV bugs.
/// https://github.com/ziglang/zig/issues/22419
const hack_around_sema_opv_bugs = true;

const err_ret_trace_index: Air.Inst.Index = @enumFromInt(std.math.maxInt(u32));

gpa: Allocator,
pt: Zcu.PerThread,
air: Air,
liveness: Liveness,
bin_file: *link.File,
debug_output: link.File.DebugInfoOutput,
target: *const std.Target,
owner: Owner,
inline_func: InternPool.Index,
mod: *Module,
arg_index: u32,
args: []MCValue,
va_info: union {
    sysv: struct {
        gp_count: u32,
        fp_count: u32,
        overflow_arg_area: bits.FrameAddr,
        reg_save_area: bits.FrameAddr,
    },
    win64: struct {},
},
ret_mcv: InstTracking,
err_ret_trace_reg: Register,
fn_type: Type,
src_loc: Zcu.LazySrcLoc,

eflags_inst: ?Air.Inst.Index = null,

/// MIR Instructions
mir_instructions: std.MultiArrayList(Mir.Inst) = .empty,
/// MIR extra data
mir_extra: std.ArrayListUnmanaged(u32) = .empty,
mir_table: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,

/// Byte offset within the source file of the ending curly.
end_di_line: u32,
end_di_column: u32,

/// The value is an offset into the `Function` `code` from the beginning.
/// To perform the reloc, write 32-bit signed little-endian integer
/// which is a relative jump, based on the address following the reloc.
epilogue_relocs: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,

reused_operands: std.StaticBitSet(Liveness.bpi - 1) = undefined,
const_tracking: ConstTrackingMap = .empty,
inst_tracking: InstTrackingMap = .empty,

// Key is the block instruction
blocks: std.AutoHashMapUnmanaged(Air.Inst.Index, BlockData) = .empty,

register_manager: RegisterManager = .{},

/// Generation of the current scope, increments by 1 for every entered scope.
scope_generation: u32 = 0,

frame_allocs: std.MultiArrayList(FrameAlloc) = .empty,
free_frame_indices: std.AutoArrayHashMapUnmanaged(FrameIndex, void) = .empty,
frame_locs: std.MultiArrayList(Mir.FrameLoc) = .empty,

loops: std.AutoHashMapUnmanaged(Air.Inst.Index, struct {
    /// The state to restore before branching.
    state: State,
    /// The branch target.
    target: Mir.Inst.Index,
}) = .empty,
loop_switches: std.AutoHashMapUnmanaged(Air.Inst.Index, struct {
    start: u31,
    len: u11,
    min: Value,
    else_relocs: union(enum) {
        @"unreachable",
        forward: std.ArrayListUnmanaged(Mir.Inst.Index),
        backward: Mir.Inst.Index,
    },
}) = .empty,

next_temp_index: Temp.Index = @enumFromInt(0),
temp_type: [Temp.Index.max]Type = undefined,

const Owner = union(enum) {
    nav_index: InternPool.Nav.Index,
    lazy_sym: link.File.LazySymbol,

    fn getSymbolIndex(owner: Owner, ctx: *CodeGen) !u32 {
        const pt = ctx.pt;
        switch (owner) {
            .nav_index => |nav_index| if (ctx.bin_file.cast(.elf)) |elf_file| {
                return elf_file.zigObjectPtr().?.getOrCreateMetadataForNav(pt.zcu, nav_index);
            } else if (ctx.bin_file.cast(.macho)) |macho_file| {
                return macho_file.getZigObject().?.getOrCreateMetadataForNav(macho_file, nav_index);
            } else if (ctx.bin_file.cast(.coff)) |coff_file| {
                const atom = try coff_file.getOrCreateAtomForNav(nav_index);
                return coff_file.getAtom(atom).getSymbolIndex().?;
            } else if (ctx.bin_file.cast(.plan9)) |p9_file| {
                return p9_file.seeNav(pt, nav_index);
            } else unreachable,
            .lazy_sym => |lazy_sym| if (ctx.bin_file.cast(.elf)) |elf_file| {
                return elf_file.zigObjectPtr().?.getOrCreateMetadataForLazySymbol(elf_file, pt, lazy_sym) catch |err|
                    ctx.fail("{s} creating lazy symbol", .{@errorName(err)});
            } else if (ctx.bin_file.cast(.macho)) |macho_file| {
                return macho_file.getZigObject().?.getOrCreateMetadataForLazySymbol(macho_file, pt, lazy_sym) catch |err|
                    ctx.fail("{s} creating lazy symbol", .{@errorName(err)});
            } else if (ctx.bin_file.cast(.coff)) |coff_file| {
                const atom = coff_file.getOrCreateAtomForLazySymbol(pt, lazy_sym) catch |err|
                    return ctx.fail("{s} creating lazy symbol", .{@errorName(err)});
                return coff_file.getAtom(atom).getSymbolIndex().?;
            } else if (ctx.bin_file.cast(.plan9)) |p9_file| {
                return p9_file.getOrCreateAtomForLazySymbol(pt, lazy_sym) catch |err|
                    return ctx.fail("{s} creating lazy symbol", .{@errorName(err)});
            } else unreachable,
        }
    }
};

const MaskKind = enum(u1) { sign, all };
const MaskInfo = packed struct { kind: MaskKind, inverted: bool = false, scalar: Memory.Size };

pub const MCValue = union(enum) {
    /// No runtime bits. `void` types, empty structs, u0, enums with 1 tag, etc.
    /// TODO Look into deleting this tag and using `dead` instead, since every use
    /// of MCValue.none should be instead looking at the type and noticing it is 0 bits.
    none,
    /// Control flow will not allow this value to be observed.
    unreach,
    /// No more references to this value remain.
    /// The payload is the value of scope_generation at the point where the death occurred
    dead: u32,
    /// The value is undefined.
    undef,
    /// A pointer-sized integer that fits in a register.
    /// If the type is a pointer, this is the pointer address in virtual address space.
    immediate: u64,
    /// The value resides in the EFLAGS register.
    eflags: Condition,
    /// The value is in a register.
    register: Register,
    /// The value is split across two registers.
    register_pair: [2]Register,
    /// The value is split across three registers.
    register_triple: [3]Register,
    /// The value is split across four registers.
    register_quadruple: [4]Register,
    /// The value is a constant offset from the value in a register.
    register_offset: bits.RegisterOffset,
    /// The value is a tuple { wrapped, overflow } where wrapped value is stored in the GP register.
    register_overflow: struct { reg: Register, eflags: Condition },
    /// The value is a bool vector stored in a vector register with a different scalar type.
    register_mask: struct { reg: Register, info: MaskInfo },
    /// The value is in memory at a hard-coded address.
    /// If the type is a pointer, it means the pointer address is stored at this memory location.
    memory: u64,
    /// The value is in memory at an address not-yet-allocated by the linker.
    /// This traditionally corresponds to a relocation emitted in a relocatable object file.
    load_symbol: bits.SymbolOffset,
    /// The address of the memory location not-yet-allocated by the linker.
    lea_symbol: bits.SymbolOffset,
    /// The value is in memory at a constant offset from the address in a register.
    indirect: bits.RegisterOffset,
    /// The value is in memory.
    /// Payload is a symbol index.
    load_direct: u32,
    /// The value is a pointer to a value in memory.
    /// Payload is a symbol index.
    lea_direct: u32,
    /// The value is in memory referenced indirectly via GOT.
    /// Payload is a symbol index.
    load_got: u32,
    /// The value is a pointer to a value referenced indirectly via GOT.
    /// Payload is a symbol index.
    lea_got: u32,
    /// The value is a threadlocal variable.
    /// Payload is a symbol index.
    load_tlv: u32,
    /// The value is a pointer to a threadlocal variable.
    /// Payload is a symbol index.
    lea_tlv: u32,
    /// The value stored at an offset from a frame index
    /// Payload is a frame address.
    load_frame: bits.FrameAddr,
    /// The address of an offset from a frame index
    /// Payload is a frame address.
    lea_frame: bits.FrameAddr,
    /// Supports integer_per_element abi
    elementwise_regs_then_frame: packed struct { regs: u3, frame_off: i29, frame_index: FrameIndex },
    /// This indicates that we have already allocated a frame index for this instruction,
    /// but it has not been spilled there yet in the current control flow.
    /// Payload is a frame index.
    reserved_frame: FrameIndex,
    air_ref: Air.Inst.Ref,

    fn isModifiable(mcv: MCValue) bool {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .immediate,
            .register_offset,
            .register_mask,
            .eflags,
            .register_overflow,
            .lea_symbol,
            .lea_direct,
            .lea_got,
            .lea_tlv,
            .lea_frame,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .air_ref,
            => false,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .memory,
            .load_symbol,
            .load_got,
            .load_direct,
            .load_tlv,
            .indirect,
            => true,
            .load_frame => |frame_addr| !frame_addr.index.isNamed(),
        };
    }

    // hack around linker relocation bugs
    fn isBase(mcv: MCValue) bool {
        return switch (mcv) {
            .memory, .indirect, .load_frame => true,
            else => false,
        };
    }

    fn isMemory(mcv: MCValue) bool {
        return switch (mcv) {
            .memory, .indirect, .load_frame, .load_symbol => true,
            else => false,
        };
    }

    fn isImmediate(mcv: MCValue) bool {
        return switch (mcv) {
            .immediate => true,
            else => false,
        };
    }

    fn isRegister(mcv: MCValue) bool {
        return switch (mcv) {
            .register => true,
            .register_offset => |reg_off| return reg_off.off == 0,
            else => false,
        };
    }

    fn isRegisterOffset(mcv: MCValue) bool {
        return switch (mcv) {
            .register, .register_offset => true,
            else => false,
        };
    }

    fn getReg(mcv: MCValue) ?Register {
        return switch (mcv) {
            .register => |reg| reg,
            .register_offset, .indirect => |ro| ro.reg,
            .register_overflow => |ro| ro.reg,
            .register_mask => |rm| rm.reg,
            else => null,
        };
    }

    fn getRegs(mcv: *const MCValue) []const Register {
        return switch (mcv.*) {
            .register => |*reg| reg[0..1],
            inline .register_pair,
            .register_triple,
            .register_quadruple,
            => |*regs| regs,
            inline .register_offset,
            .indirect,
            .register_overflow,
            .register_mask,
            => |*pl| (&pl.reg)[0..1],
            else => &.{},
        };
    }

    fn getCondition(mcv: MCValue) ?Condition {
        return switch (mcv) {
            .eflags => |cc| cc,
            .register_overflow => |reg_ov| reg_ov.eflags,
            else => null,
        };
    }

    fn isAddress(mcv: MCValue) bool {
        return switch (mcv) {
            .immediate, .register, .register_offset, .lea_frame => true,
            else => false,
        };
    }

    fn address(mcv: MCValue) MCValue {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .immediate,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .lea_symbol,
            .lea_direct,
            .lea_got,
            .lea_tlv,
            .lea_frame,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .air_ref,
            => unreachable, // not in memory
            .memory => |addr| .{ .immediate = addr },
            .indirect => |reg_off| switch (reg_off.off) {
                0 => .{ .register = reg_off.reg },
                else => .{ .register_offset = reg_off },
            },
            .load_direct => |sym_index| .{ .lea_direct = sym_index },
            .load_got => |sym_index| .{ .lea_got = sym_index },
            .load_tlv => |sym_index| .{ .lea_tlv = sym_index },
            .load_frame => |frame_addr| .{ .lea_frame = frame_addr },
            .load_symbol => |sym_off| .{ .lea_symbol = sym_off },
        };
    }

    fn deref(mcv: MCValue) MCValue {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .eflags,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_overflow,
            .register_mask,
            .memory,
            .indirect,
            .load_direct,
            .load_got,
            .load_tlv,
            .load_frame,
            .load_symbol,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .air_ref,
            => unreachable, // not dereferenceable
            .immediate => |addr| .{ .memory = addr },
            .register => |reg| .{ .indirect = .{ .reg = reg } },
            .register_offset => |reg_off| .{ .indirect = reg_off },
            .lea_direct => |sym_index| .{ .load_direct = sym_index },
            .lea_got => |sym_index| .{ .load_got = sym_index },
            .lea_tlv => |sym_index| .{ .load_tlv = sym_index },
            .lea_frame => |frame_addr| .{ .load_frame = frame_addr },
            .lea_symbol => |sym_index| .{ .load_symbol = sym_index },
        };
    }

    fn offset(mcv: MCValue, off: i32) MCValue {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .air_ref,
            => unreachable, // not valid
            .eflags,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_overflow,
            .register_mask,
            .memory,
            .indirect,
            .load_direct,
            .lea_direct,
            .load_got,
            .lea_got,
            .load_tlv,
            .lea_tlv,
            .load_frame,
            .load_symbol,
            .lea_symbol,
            => switch (off) {
                0 => mcv,
                else => unreachable, // not offsettable
            },
            .immediate => |imm| .{ .immediate = @bitCast(@as(i64, @bitCast(imm)) +% off) },
            .register => |reg| .{ .register_offset = .{ .reg = reg, .off = off } },
            .register_offset => |reg_off| .{
                .register_offset = .{ .reg = reg_off.reg, .off = reg_off.off + off },
            },
            .lea_frame => |frame_addr| .{
                .lea_frame = .{ .index = frame_addr.index, .off = frame_addr.off + off },
            },
        };
    }

    fn mem(mcv: MCValue, function: *CodeGen, mod_rm: Memory.Mod.Rm) !Memory {
        return switch (mcv) {
            .none,
            .unreach,
            .dead,
            .undef,
            .immediate,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .load_direct,
            .lea_direct,
            .load_got,
            .lea_got,
            .load_tlv,
            .lea_tlv,
            .lea_frame,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .lea_symbol,
            => unreachable,
            .memory => |addr| if (std.math.cast(i32, @as(i64, @bitCast(addr)))) |small_addr| .{
                .base = .{ .reg = .ds },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = small_addr + mod_rm.disp,
                } },
            } else .{ .base = .{ .reg = .ds }, .mod = .{ .off = addr } },
            .indirect => |reg_off| .{
                .base = .{ .reg = registerAlias(reg_off.reg, @divExact(function.target.ptrBitWidth(), 8)) },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = reg_off.off + mod_rm.disp,
                } },
            },
            .load_frame => |frame_addr| .{
                .base = .{ .frame = frame_addr.index },
                .mod = .{ .rm = .{
                    .size = mod_rm.size,
                    .index = mod_rm.index,
                    .scale = mod_rm.scale,
                    .disp = frame_addr.off + mod_rm.disp,
                } },
            },
            .load_symbol => |sym_off| {
                assert(sym_off.off == 0);
                return .{
                    .base = .{ .reloc = sym_off.sym_index },
                    .mod = .{ .rm = .{
                        .size = mod_rm.size,
                        .index = mod_rm.index,
                        .scale = mod_rm.scale,
                        .disp = sym_off.off + mod_rm.disp,
                    } },
                };
            },
            .air_ref => |ref| (try function.resolveInst(ref)).mem(function, mod_rm),
        };
    }

    pub fn format(
        mcv: MCValue,
        comptime _: []const u8,
        _: std.fmt.FormatOptions,
        writer: anytype,
    ) @TypeOf(writer).Error!void {
        switch (mcv) {
            .none, .unreach, .dead, .undef => try writer.print("({s})", .{@tagName(mcv)}),
            .immediate => |pl| try writer.print("0x{x}", .{pl}),
            .memory => |pl| try writer.print("[ds:0x{x}]", .{pl}),
            inline .eflags, .register => |pl| try writer.print("{s}", .{@tagName(pl)}),
            .register_pair => |pl| try writer.print("{s}:{s}", .{ @tagName(pl[1]), @tagName(pl[0]) }),
            .register_triple => |pl| try writer.print("{s}:{s}:{s}", .{
                @tagName(pl[2]), @tagName(pl[1]), @tagName(pl[0]),
            }),
            .register_quadruple => |pl| try writer.print("{s}:{s}:{s}:{s}", .{
                @tagName(pl[3]), @tagName(pl[2]), @tagName(pl[1]), @tagName(pl[0]),
            }),
            .register_offset => |pl| try writer.print("{s} + 0x{x}", .{ @tagName(pl.reg), pl.off }),
            .register_overflow => |pl| try writer.print("{s}:{s}", .{
                @tagName(pl.eflags),
                @tagName(pl.reg),
            }),
            .register_mask => |pl| try writer.print("mask({s},{}):{c}{s}", .{
                @tagName(pl.info.kind),
                pl.info.scalar,
                @as(u8, if (pl.info.inverted) '!' else ' '),
                @tagName(pl.reg),
            }),
            .load_symbol => |pl| try writer.print("[sym:{} + 0x{x}]", .{ pl.sym_index, pl.off }),
            .lea_symbol => |pl| try writer.print("sym:{} + 0x{x}", .{ pl.sym_index, pl.off }),
            .indirect => |pl| try writer.print("[{s} + 0x{x}]", .{ @tagName(pl.reg), pl.off }),
            .load_direct => |pl| try writer.print("[direct:{d}]", .{pl}),
            .lea_direct => |pl| try writer.print("direct:{d}", .{pl}),
            .load_got => |pl| try writer.print("[got:{d}]", .{pl}),
            .lea_got => |pl| try writer.print("got:{d}", .{pl}),
            .load_tlv => |pl| try writer.print("[tlv:{d}]", .{pl}),
            .lea_tlv => |pl| try writer.print("tlv:{d}", .{pl}),
            .load_frame => |pl| try writer.print("[{} + 0x{x}]", .{ pl.index, pl.off }),
            .elementwise_regs_then_frame => |pl| try writer.print("elementwise:{d}:[{} + 0x{x}]", .{
                pl.regs, pl.frame_index, pl.frame_off,
            }),
            .lea_frame => |pl| try writer.print("{} + 0x{x}", .{ pl.index, pl.off }),
            .reserved_frame => |pl| try writer.print("(dead:{})", .{pl}),
            .air_ref => |pl| try writer.print("(air:0x{x})", .{@intFromEnum(pl)}),
        }
    }
};

const InstTrackingMap = std.AutoArrayHashMapUnmanaged(Air.Inst.Index, InstTracking);
const ConstTrackingMap = std.AutoArrayHashMapUnmanaged(InternPool.Index, InstTracking);
const InstTracking = struct {
    long: MCValue,
    short: MCValue,

    fn init(result: MCValue) InstTracking {
        return .{ .long = switch (result) {
            .none,
            .unreach,
            .undef,
            .immediate,
            .memory,
            .load_direct,
            .lea_direct,
            .load_got,
            .lea_got,
            .load_tlv,
            .lea_tlv,
            .load_frame,
            .lea_frame,
            .load_symbol,
            .lea_symbol,
            => result,
            .dead,
            .elementwise_regs_then_frame,
            .reserved_frame,
            .air_ref,
            => unreachable,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .indirect,
            => .none,
        }, .short = result };
    }

    fn getReg(self: InstTracking) ?Register {
        return self.short.getReg();
    }

    fn getRegs(self: *const InstTracking) []const Register {
        return self.short.getRegs();
    }

    fn getCondition(self: InstTracking) ?Condition {
        return self.short.getCondition();
    }

    fn spill(self: *InstTracking, cg: *CodeGen, inst: Air.Inst.Index) !void {
        if (std.meta.eql(self.long, self.short)) return; // Already spilled
        // Allocate or reuse frame index
        switch (self.long) {
            .none => self.long = try cg.allocRegOrMem(inst, false),
            .load_frame => {},
            .lea_frame => return,
            .reserved_frame => |index| self.long = .{ .load_frame = .{ .index = index } },
            else => unreachable,
        }
        tracking_log.debug("spill {} from {} to {}", .{ inst, self.short, self.long });
        try cg.genCopy(cg.typeOfIndex(inst), self.long, self.short, .{});
        for (self.short.getRegs()) |reg| if (reg.class() == .x87) try cg.asmRegister(.{ .f_, .free }, reg);
    }

    fn reuseFrame(self: *InstTracking) void {
        self.* = .init(switch (self.long) {
            .none => switch (self.short) {
                .dead => .none,
                else => |short| short,
            },
            .reserved_frame => |index| .{ .load_frame = .{ .index = index } },
            else => |long| long,
        });
    }

    fn trackSpill(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index) !void {
        switch (self.short) {
            .register => |reg| function.register_manager.freeReg(reg),
            inline .register_pair,
            .register_triple,
            .register_quadruple,
            => |regs| for (regs) |reg| function.register_manager.freeReg(reg),
            .register_offset, .indirect => |reg_off| function.register_manager.freeReg(reg_off.reg),
            .register_overflow => |reg_ov| {
                function.register_manager.freeReg(reg_ov.reg);
                function.eflags_inst = null;
            },
            .register_mask => |reg_mask| function.register_manager.freeReg(reg_mask.reg),
            .eflags => function.eflags_inst = null,
            else => {}, // TODO process stack allocation death
        }
        self.reuseFrame();
        tracking_log.debug("{} => {} (spilled)", .{ inst, self.* });
    }

    fn verifyMaterialize(self: InstTracking, target: InstTracking) void {
        switch (self.long) {
            .none,
            .load_frame,
            .reserved_frame,
            => switch (target.long) {
                .none,
                .load_frame,
                .reserved_frame,
                => {},
                else => unreachable,
            },
            .unreach,
            .undef,
            .immediate,
            .memory,
            .load_direct,
            .lea_direct,
            .load_got,
            .lea_got,
            .load_tlv,
            .lea_tlv,
            .lea_frame,
            .load_symbol,
            .lea_symbol,
            => assert(std.meta.eql(self.long, target.long)),
            .dead,
            .eflags,
            .register,
            .register_pair,
            .register_triple,
            .register_quadruple,
            .register_offset,
            .register_overflow,
            .register_mask,
            .indirect,
            .elementwise_regs_then_frame,
            .air_ref,
            => unreachable,
        }
    }

    fn materialize(
        self: *InstTracking,
        function: *CodeGen,
        inst: Air.Inst.Index,
        target: InstTracking,
    ) !void {
        self.verifyMaterialize(target);
        try self.materializeUnsafe(function, inst, target);
    }

    fn materializeUnsafe(
        self: InstTracking,
        function: *CodeGen,
        inst: Air.Inst.Index,
        target: InstTracking,
    ) !void {
        const ty = function.typeOfIndex(inst);
        if ((self.long == .none or self.long == .reserved_frame) and target.long == .load_frame)
            try function.genCopy(ty, target.long, self.short, .{});
        try function.genCopy(ty, target.short, self.short, .{});
    }

    fn trackMaterialize(self: *InstTracking, inst: Air.Inst.Index, target: InstTracking) void {
        self.verifyMaterialize(target);
        // Don't clobber reserved frame indices
        self.long = if (target.long == .none) switch (self.long) {
            .load_frame => |addr| .{ .reserved_frame = addr.index },
            .reserved_frame => self.long,
            else => target.long,
        } else target.long;
        self.short = target.short;
        tracking_log.debug("{} => {} (materialize)", .{ inst, self.* });
    }

    fn resurrect(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index, scope_generation: u32) !void {
        switch (self.short) {
            .dead => |die_generation| if (die_generation >= scope_generation) {
                self.reuseFrame();
                try function.getValue(self.short, inst);
                tracking_log.debug("{} => {} (resurrect)", .{ inst, self.* });
            },
            else => {},
        }
    }

    fn die(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index) !void {
        if (self.short == .dead) return;
        try function.freeValue(self.short);
        if (self.long == .none) self.long = self.short;
        self.short = .{ .dead = function.scope_generation };
        tracking_log.debug("{} => {} (death)", .{ inst, self.* });
    }

    fn reuse(
        self: *InstTracking,
        function: *CodeGen,
        new_inst: ?Air.Inst.Index,
        old_inst: Air.Inst.Index,
    ) void {
        self.short = .{ .dead = function.scope_generation };
        tracking_log.debug("{?} => {} (reuse {})", .{ new_inst, self.*, old_inst });
    }

    fn liveOut(self: *InstTracking, function: *CodeGen, inst: Air.Inst.Index) void {
        for (self.getRegs()) |reg| {
            if (function.register_manager.isRegFree(reg)) {
                tracking_log.debug("{} => {} (live-out)", .{ inst, self.* });
                continue;
            }

            const index = RegisterManager.indexOfRegIntoTracked(reg).?;
            const tracked_inst = function.register_manager.registers[index];
            const tracking = function.getResolvedInstValue(tracked_inst);

            // Disable death.
            var found_reg = false;
            var remaining_reg: Register = .none;
            for (tracking.getRegs()) |tracked_reg| if (tracked_reg.id() == reg.id()) {
                assert(!found_reg);
                found_reg = true;
            } else {
                assert(remaining_reg == .none);
                remaining_reg = tracked_reg;
            };
            assert(found_reg);
            tracking.short = switch (remaining_reg) {
                .none => .{ .dead = function.scope_generation },
                else => .{ .register = remaining_reg },
            };

            // Perform side-effects of freeValue manually.
            function.register_manager.freeReg(reg);

            tracking_log.debug("{} => {} (live-out {})", .{ inst, self.*, tracked_inst });
        }
    }

    pub fn format(
        tracking: InstTracking,
        comptime _: []const u8,
        _: std.fmt.FormatOptions,
        writer: anytype,
    ) @TypeOf(writer).Error!void {
        if (!std.meta.eql(tracking.long, tracking.short)) try writer.print("|{}| ", .{tracking.long});
        try writer.print("{}", .{tracking.short});
    }
};

const FrameAlloc = struct {
    abi_size: u31,
    spill_pad: u3,
    abi_align: InternPool.Alignment,
    ref_count: u16,

    fn init(alloc_abi: struct { size: u64, pad: u3 = 0, alignment: InternPool.Alignment }) FrameAlloc {
        return .{
            .abi_size = @intCast(alloc_abi.size),
            .spill_pad = alloc_abi.pad,
            .abi_align = alloc_abi.alignment,
            .ref_count = 0,
        };
    }
    fn initType(ty: Type, zcu: *Zcu) FrameAlloc {
        return init(.{
            .size = ty.abiSize(zcu),
            .alignment = ty.abiAlignment(zcu),
        });
    }
    fn initSpill(ty: Type, zcu: *Zcu) FrameAlloc {
        const abi_size = ty.abiSize(zcu);
        const spill_size = if (abi_size < 8)
            std.math.ceilPowerOfTwoAssert(u64, abi_size)
        else
            std.mem.alignForward(u64, abi_size, 8);
        return init(.{
            .size = spill_size,
            .pad = @intCast(spill_size - abi_size),
            .alignment = ty.abiAlignment(zcu).maxStrict(
                .fromNonzeroByteUnits(@min(spill_size, 8)),
            ),
        });
    }
};

const StackAllocation = struct {
    inst: ?Air.Inst.Index,
    /// TODO do we need size? should be determined by inst.ty.abiSize(zcu)
    size: u32,
};

const BlockData = struct {
    relocs: std.ArrayListUnmanaged(Mir.Inst.Index) = .empty,
    state: State,

    fn deinit(self: *BlockData, gpa: Allocator) void {
        self.relocs.deinit(gpa);
        self.* = undefined;
    }
};

const CodeGen = @This();

pub fn generate(
    bin_file: *link.File,
    pt: Zcu.PerThread,
    src_loc: Zcu.LazySrcLoc,
    func_index: InternPool.Index,
    air: Air,
    liveness: Liveness,
    code: *std.ArrayListUnmanaged(u8),
    debug_output: link.File.DebugInfoOutput,
) codegen.CodeGenError!void {
    const zcu = pt.zcu;
    const comp = zcu.comp;
    const gpa = zcu.gpa;
    const ip = &zcu.intern_pool;
    const func = zcu.funcInfo(func_index);
    const fn_type: Type = .fromInterned(func.ty);
    const mod = zcu.navFileScope(func.owner_nav).mod;

    var function: CodeGen = .{
        .gpa = gpa,
        .pt = pt,
        .air = air,
        .liveness = liveness,
        .target = &mod.resolved_target.result,
        .mod = mod,
        .bin_file = bin_file,
        .debug_output = debug_output,
        .owner = .{ .nav_index = func.owner_nav },
        .inline_func = func_index,
        .arg_index = undefined,
        .args = undefined, // populated after `resolveCallingConventionValues`
        .va_info = undefined, // populated after `resolveCallingConventionValues`
        .ret_mcv = undefined, // populated after `resolveCallingConventionValues`
        .err_ret_trace_reg = undefined, // populated after `resolveCallingConventionValues`
        .fn_type = fn_type,
        .src_loc = src_loc,
        .end_di_line = func.rbrace_line,
        .end_di_column = func.rbrace_column,
    };
    defer {
        function.frame_allocs.deinit(gpa);
        function.free_frame_indices.deinit(gpa);
        function.frame_locs.deinit(gpa);
        function.loops.deinit(gpa);
        function.loop_switches.deinit(gpa);
        var block_it = function.blocks.valueIterator();
        while (block_it.next()) |block| block.deinit(gpa);
        function.blocks.deinit(gpa);
        function.inst_tracking.deinit(gpa);
        function.const_tracking.deinit(gpa);
        function.epilogue_relocs.deinit(gpa);
        function.mir_instructions.deinit(gpa);
        function.mir_extra.deinit(gpa);
        function.mir_table.deinit(gpa);
    }
    try function.inst_tracking.ensureTotalCapacity(gpa, Temp.Index.max);
    for (0..Temp.Index.max) |temp_index| {
        const temp: Temp.Index = @enumFromInt(temp_index);
        function.inst_tracking.putAssumeCapacityNoClobber(temp.toIndex(), .init(.none));
    }

    wip_mir_log.debug("{}:", .{fmtNav(func.owner_nav, ip)});

    try function.frame_allocs.resize(gpa, FrameIndex.named_count);
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.stack_frame),
        .init(.{ .size = 0, .alignment = .@"1" }),
    );
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.call_frame),
        .init(.{ .size = 0, .alignment = .@"1" }),
    );

    const fn_info = zcu.typeToFunc(fn_type).?;
    var call_info = function.resolveCallingConventionValues(fn_info, &.{}, .args_frame) catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        else => |e| return e,
    };
    defer call_info.deinit(&function);

    function.args = call_info.args;
    function.ret_mcv = call_info.return_value;
    function.err_ret_trace_reg = call_info.err_ret_trace_reg;
    function.frame_allocs.set(@intFromEnum(FrameIndex.ret_addr), .init(.{
        .size = Type.usize.abiSize(zcu),
        .alignment = Type.usize.abiAlignment(zcu).min(call_info.stack_align),
    }));
    function.frame_allocs.set(@intFromEnum(FrameIndex.base_ptr), .init(.{
        .size = Type.usize.abiSize(zcu),
        .alignment = call_info.stack_align.min(
            .fromNonzeroByteUnits(function.target.stackAlignment()),
        ),
    }));
    function.frame_allocs.set(
        @intFromEnum(FrameIndex.args_frame),
        .init(.{
            .size = call_info.stack_byte_count,
            .alignment = call_info.stack_align,
        }),
    );
    function.va_info = switch (fn_info.cc) {
        else => undefined,
        .x86_64_sysv => .{ .sysv = .{
            .gp_count = call_info.gp_count,
            .fp_count = call_info.fp_count,
            .overflow_arg_area = .{ .index = .args_frame, .off = call_info.stack_byte_count },
            .reg_save_area = undefined,
        } },
        .x86_64_win => .{ .win64 = .{} },
    };
    if (call_info.err_ret_trace_reg != .none) {
        function.register_manager.getRegAssumeFree(call_info.err_ret_trace_reg, err_ret_trace_index);
        try function.inst_tracking.putNoClobber(
            gpa,
            err_ret_trace_index,
            .init(.{ .register = call_info.err_ret_trace_reg }),
        );
    }

    function.gen() catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        error.OutOfRegisters => return function.fail("ran out of registers (Zig compiler bug)", .{}),
        else => |e| return e,
    };

    var mir: Mir = .{
        .instructions = function.mir_instructions.toOwnedSlice(),
        .extra = try function.mir_extra.toOwnedSlice(gpa),
        .table = try function.mir_table.toOwnedSlice(gpa),
        .frame_locs = function.frame_locs.toOwnedSlice(),
    };
    defer mir.deinit(gpa);

    var emit: Emit = .{
        .air = function.air,
        .lower = .{
            .bin_file = bin_file,
            .target = function.target,
            .allocator = gpa,
            .mir = mir,
            .cc = fn_info.cc,
            .src_loc = src_loc,
            .output_mode = comp.config.output_mode,
            .link_mode = comp.config.link_mode,
            .pic = mod.pic,
        },
        .atom_index = function.owner.getSymbolIndex(&function) catch |err| switch (err) {
            error.CodegenFail => return error.CodegenFail,
            else => |e| return e,
        },
        .debug_output = debug_output,
        .code = code,
        .prev_di_loc = .{
            .line = func.lbrace_line,
            .column = func.lbrace_column,
            .is_stmt = switch (debug_output) {
                .dwarf => |dwarf| dwarf.dwarf.debug_line.header.default_is_stmt,
                .plan9 => undefined,
                .none => undefined,
            },
        },
        .prev_di_pc = 0,
    };
    emit.emitMir() catch |err| switch (err) {
        error.LowerFail, error.EmitFail => return function.failMsg(emit.lower.err_msg.?),

        error.InvalidInstruction, error.CannotEncode => |e| return function.fail("emit MIR failed: {s} (Zig compiler bug)", .{@errorName(e)}),
        else => |e| return function.fail("emit MIR failed: {s}", .{@errorName(e)}),
    };
}

pub fn generateLazy(
    bin_file: *link.File,
    pt: Zcu.PerThread,
    src_loc: Zcu.LazySrcLoc,
    lazy_sym: link.File.LazySymbol,
    code: *std.ArrayListUnmanaged(u8),
    debug_output: link.File.DebugInfoOutput,
) codegen.CodeGenError!void {
    const comp = bin_file.comp;
    const gpa = comp.gpa;
    // This function is for generating global code, so we use the root module.
    const mod = comp.root_mod;
    var function: CodeGen = .{
        .gpa = gpa,
        .pt = pt,
        .air = undefined,
        .liveness = undefined,
        .target = &mod.resolved_target.result,
        .mod = mod,
        .bin_file = bin_file,
        .debug_output = debug_output,
        .owner = .{ .lazy_sym = lazy_sym },
        .inline_func = undefined,
        .arg_index = undefined,
        .args = undefined,
        .va_info = undefined,
        .ret_mcv = undefined,
        .err_ret_trace_reg = undefined,
        .fn_type = undefined,
        .src_loc = src_loc,
        .end_di_line = undefined, // no debug info yet
        .end_di_column = undefined, // no debug info yet
    };
    defer {
        function.inst_tracking.deinit(gpa);
        function.mir_instructions.deinit(gpa);
        function.mir_extra.deinit(gpa);
        function.mir_table.deinit(gpa);
    }
    try function.inst_tracking.ensureTotalCapacity(gpa, Temp.Index.max);
    for (0..Temp.Index.max) |temp_index| {
        const temp: Temp.Index = @enumFromInt(temp_index);
        function.inst_tracking.putAssumeCapacityNoClobber(temp.toIndex(), .init(.none));
    }

    function.genLazy(lazy_sym) catch |err| switch (err) {
        error.CodegenFail => return error.CodegenFail,
        error.OutOfRegisters => return function.fail("ran out of registers (Zig compiler bug)", .{}),
        else => |e| return e,
    };

    var mir: Mir = .{
        .instructions = function.mir_instructions.toOwnedSlice(),
        .extra = try function.mir_extra.toOwnedSlice(gpa),
        .table = try function.mir_table.toOwnedSlice(gpa),
        .frame_locs = function.frame_locs.toOwnedSlice(),
    };
    defer mir.deinit(gpa);

    var emit: Emit = .{
        .air = function.air,
        .lower = .{
            .bin_file = bin_file,
            .target = function.target,
            .allocator = gpa,
            .mir = mir,
            .cc = .auto,
            .src_loc = src_loc,
            .output_mode = comp.config.output_mode,
            .link_mode = comp.config.link_mode,
            .pic = mod.pic,
        },
        .atom_index = function.owner.getSymbolIndex(&function) catch |err| switch (err) {
            error.CodegenFail => return error.CodegenFail,
            else => |e| return e,
        },
        .debug_output = debug_output,
        .code = code,
        .prev_di_loc = undefined, // no debug info yet
        .prev_di_pc = undefined, // no debug info yet
    };
    emit.emitMir() catch |err| switch (err) {
        error.LowerFail, error.EmitFail => return function.failMsg(emit.lower.err_msg.?),
        error.InvalidInstruction => return function.fail("failed to find a viable x86 instruction (Zig compiler bug)", .{}),
        error.CannotEncode => return function.fail("failed to encode x86 instruction (Zig compiler bug)", .{}),
        else => |e| return function.fail("failed to emit MIR: {s}", .{@errorName(e)}),
    };
}

const FormatNavData = struct {
    ip: *const InternPool,
    nav_index: InternPool.Nav.Index,
};
fn formatNav(
    data: FormatNavData,
    comptime _: []const u8,
    _: std.fmt.FormatOptions,
    writer: anytype,
) @TypeOf(writer).Error!void {
    try writer.print("{}", .{data.ip.getNav(data.nav_index).fqn.fmt(data.ip)});
}
fn fmtNav(nav_index: InternPool.Nav.Index, ip: *const InternPool) std.fmt.Formatter(formatNav) {
    return .{ .data = .{
        .ip = ip,
        .nav_index = nav_index,
    } };
}

const FormatAirData = struct {
    self: *CodeGen,
    inst: Air.Inst.Index,
};
fn formatAir(
    data: FormatAirData,
    comptime _: []const u8,
    _: std.fmt.FormatOptions,
    writer: anytype,
) @TypeOf(writer).Error!void {
    @import("../../print_air.zig").dumpInst(
        data.inst,
        data.self.pt,
        data.self.air,
        data.self.liveness,
    );
}
fn fmtAir(self: *CodeGen, inst: Air.Inst.Index) std.fmt.Formatter(formatAir) {
    return .{ .data = .{ .self = self, .inst = inst } };
}

const FormatWipMirData = struct {
    self: *CodeGen,
    inst: Mir.Inst.Index,
};
fn formatWipMir(
    data: FormatWipMirData,
    comptime _: []const u8,
    _: std.fmt.FormatOptions,
    writer: anytype,
) @TypeOf(writer).Error!void {
    const comp = data.self.bin_file.comp;
    const mod = comp.root_mod;
    var lower: Lower = .{
        .bin_file = data.self.bin_file,
        .target = data.self.target,
        .allocator = data.self.gpa,
        .mir = .{
            .instructions = data.self.mir_instructions.slice(),
            .extra = data.self.mir_extra.items,
            .table = data.self.mir_table.items,
            .frame_locs = (std.MultiArrayList(Mir.FrameLoc){}).slice(),
        },
        .cc = .auto,
        .src_loc = data.self.src_loc,
        .output_mode = comp.config.output_mode,
        .link_mode = comp.config.link_mode,
        .pic = mod.pic,
    };
    var first = true;
    for ((lower.lowerMir(data.inst) catch |err| switch (err) {
        error.LowerFail => {
            defer {
                lower.err_msg.?.deinit(data.self.gpa);
                lower.err_msg = null;
            }
            try writer.writeAll(lower.err_msg.?.msg);
            return;
        },
        error.OutOfMemory, error.InvalidInstruction, error.CannotEncode => |e| {
            try writer.writeAll(switch (e) {
                error.OutOfMemory => "Out of memory",
                error.InvalidInstruction => "CodeGen failed to find a viable instruction.",
                error.CannotEncode => "CodeGen failed to encode the instruction.",
            });
            return;
        },
        else => |e| return e,
    }).insts) |lowered_inst| {
        if (!first) try writer.writeAll("\ndebug(wip_mir): ");
        try writer.print("  | {}", .{lowered_inst});
        first = false;
    }
    if (first) {
        const ip = &data.self.pt.zcu.intern_pool;
        const mir_inst = lower.mir.instructions.get(data.inst);
        try writer.print("  | .{s}", .{@tagName(mir_inst.ops)});
        switch (mir_inst.ops) {
            else => unreachable,
            .pseudo_dbg_prologue_end_none,
            .pseudo_dbg_epilogue_begin_none,
            .pseudo_dbg_enter_block_none,
            .pseudo_dbg_leave_block_none,
            .pseudo_dbg_var_args_none,
            .pseudo_dead_none,
            => {},
            .pseudo_dbg_line_stmt_line_column, .pseudo_dbg_line_line_column => try writer.print(
                " {[line]d}, {[column]d}",
                mir_inst.data.line_column,
            ),
            .pseudo_dbg_enter_inline_func, .pseudo_dbg_leave_inline_func => try writer.print(" {}", .{
                ip.getNav(ip.indexToKey(mir_inst.data.func).func.owner_nav).name.fmt(ip),
            }),
            .pseudo_dbg_local_a => try writer.print(" {}", .{mir_inst.data.a.air_inst}),
            .pseudo_dbg_local_ai_s => try writer.print(" {}, {d}", .{
                mir_inst.data.ai.air_inst,
                @as(i32, @bitCast(mir_inst.data.ai.i)),
            }),
            .pseudo_dbg_local_ai_u => try writer.print(" {}, {d}", .{
                mir_inst.data.ai.air_inst,
                mir_inst.data.ai.i,
            }),
            .pseudo_dbg_local_ai_64 => try writer.print(" {}, {d}", .{
                mir_inst.data.ai.air_inst,
                lower.mir.extraData(Mir.Imm64, mir_inst.data.ai.i).data.decode(),
            }),
            .pseudo_dbg_local_as => {
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .reloc = mir_inst.data.as.sym_index },
                }) };
                try writer.print(" {}, {}", .{ mir_inst.data.as.air_inst, mem_op.fmt(.m) });
            },
            .pseudo_dbg_local_aso => {
                const sym_off = lower.mir.extraData(bits.SymbolOffset, mir_inst.data.ax.payload).data;
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .reloc = sym_off.sym_index },
                    .disp = sym_off.off,
                }) };
                try writer.print(" {}, {}", .{ mir_inst.data.ax.air_inst, mem_op.fmt(.m) });
            },
            .pseudo_dbg_local_aro => {
                const air_off = lower.mir.extraData(Mir.AirOffset, mir_inst.data.rx.payload).data;
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .reg = mir_inst.data.rx.r1 },
                    .disp = air_off.off,
                }) };
                try writer.print(" {}, {}", .{ air_off.air_inst, mem_op.fmt(.m) });
            },
            .pseudo_dbg_local_af => {
                const frame_addr = lower.mir.extraData(bits.FrameAddr, mir_inst.data.ax.payload).data;
                const mem_op: encoder.Instruction.Operand = .{ .mem = .initSib(.qword, .{
                    .base = .{ .frame = frame_addr.index },
                    .disp = frame_addr.off,
                }) };
                try writer.print(" {}, {}", .{ mir_inst.data.ax.air_inst, mem_op.fmt(.m) });
            },
            .pseudo_dbg_local_am => {
                const mem_op: encoder.Instruction.Operand = .{
                    .mem = lower.mir.extraData(Mir.Memory, mir_inst.data.ax.payload).data.decode(),
                };
                try writer.print(" {}, {}", .{ mir_inst.data.ax.air_inst, mem_op.fmt(.m) });
            },
        }
    }
}
fn fmtWipMir(self: *CodeGen, inst: Mir.Inst.Index) std.fmt.Formatter(formatWipMir) {
    return .{ .data = .{ .self = self, .inst = inst } };
}

const FormatTrackingData = struct {
    self: *CodeGen,
};
fn formatTracking(
    data: FormatTrackingData,
    comptime _: []const u8,
    _: std.fmt.FormatOptions,
    writer: anytype,
) @TypeOf(writer).Error!void {
    var it = data.self.inst_tracking.iterator();
    while (it.next()) |entry| try writer.print("\n{} = {}", .{ entry.key_ptr.*, entry.value_ptr.* });
}
fn fmtTracking(self: *CodeGen) std.fmt.Formatter(formatTracking) {
    return .{ .data = .{ .self = self } };
}

fn addInst(self: *CodeGen, inst: Mir.Inst) error{OutOfMemory}!Mir.Inst.Index {
    const gpa = self.gpa;
    try self.mir_instructions.ensureUnusedCapacity(gpa, 1);
    const result_index: Mir.Inst.Index = @intCast(self.mir_instructions.len);
    self.mir_instructions.appendAssumeCapacity(inst);
    if (inst.ops != .pseudo_dead_none) wip_mir_log.debug("{}", .{self.fmtWipMir(result_index)});
    return result_index;
}

fn addExtra(self: *CodeGen, extra: anytype) Allocator.Error!u32 {
    const fields = std.meta.fields(@TypeOf(extra));
    try self.mir_extra.ensureUnusedCapacity(self.gpa, fields.len);
    return self.addExtraAssumeCapacity(extra);
}

fn addExtraAssumeCapacity(self: *CodeGen, extra: anytype) u32 {
    const fields = std.meta.fields(@TypeOf(extra));
    const result: u32 = @intCast(self.mir_extra.items.len);
    inline for (fields) |field| {
        self.mir_extra.appendAssumeCapacity(switch (field.type) {
            u32 => @field(extra, field.name),
            i32, Mir.Memory.Info => @bitCast(@field(extra, field.name)),
            FrameIndex => @intFromEnum(@field(extra, field.name)),
            else => @compileError("bad field type: " ++ field.name ++ ": " ++ @typeName(field.type)),
        });
    }
    return result;
}

fn asmOps(self: *CodeGen, tag: Mir.Inst.FixedTag, ops: [4]Operand) !void {
    return switch (ops[0]) {
        .none => self.asmOpOnly(tag),
        .reg => |reg0| switch (ops[1]) {
            .none => self.asmRegister(tag, reg0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmRegisterRegister(tag, reg0, reg1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterRegister(tag, reg0, reg1, reg2),
                    .reg => |reg3| self.asmRegisterRegisterRegisterRegister(tag, reg0, reg1, reg2, reg3),
                    .imm => |imm3| self.asmRegisterRegisterRegisterImmediate(tag, reg0, reg1, reg2, imm3),
                    else => error.InvalidInstruction,
                },
                .mem => |mem2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterMemory(tag, reg0, reg1, mem2),
                    .reg => |reg3| self.asmRegisterRegisterMemoryRegister(tag, reg0, reg1, mem2, reg3),
                    .imm => |imm3| self.asmRegisterRegisterMemoryImmediate(tag, reg0, reg1, mem2, imm3),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmRegisterRegisterImmediate(tag, reg0, reg1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .mem => |mem1| switch (ops[2]) {
                .none => self.asmRegisterMemory(tag, reg0, mem1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmRegisterMemoryRegister(tag, reg0, mem1, reg2),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmRegisterMemoryImmediate(tag, reg0, mem1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmRegisterImmediate(tag, reg0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .mem => |mem0| switch (ops[1]) {
            .none => self.asmMemory(tag, mem0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmMemoryRegister(tag, mem0, reg1),
                .reg => |reg2| switch (ops[3]) {
                    .none => self.asmMemoryRegisterRegister(tag, mem0, reg1, reg2),
                    else => error.InvalidInstruction,
                },
                .imm => |imm2| switch (ops[3]) {
                    .none => self.asmMemoryRegisterImmediate(tag, mem0, reg1, imm2),
                    else => error.InvalidInstruction,
                },
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmMemoryImmediate(tag, mem0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .imm => |imm0| switch (ops[1]) {
            .none => self.asmImmediate(tag, imm0),
            .reg => |reg1| switch (ops[2]) {
                .none => self.asmImmediateRegister(tag, imm0, reg1),
                else => error.InvalidInstruction,
            },
            .imm => |imm1| switch (ops[2]) {
                .none => self.asmImmediateImmediate(tag, imm0, imm1),
                else => error.InvalidInstruction,
            },
            else => error.InvalidInstruction,
        },
        .inst => |inst0| switch (ops[1]) {
            .none => self.asmReloc(tag, inst0),
            else => error.InvalidInstruction,
        },
    };
}

/// A `cc` of `.z_and_np` clobbers `reg2`!
fn asmCmovccRegisterRegister(self: *CodeGen, cc: Condition, reg1: Register, reg2: Register) !void {
    if (self.hasFeature(.cmov)) _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .cmov,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .rr,
            .z_and_np => .pseudo_cmov_z_and_np_rr,
            .nz_or_p => .pseudo_cmov_nz_or_p_rr,
        },
        .data = .{ .rr = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np, .nz_or_p => ._,
            },
            .r1 = reg1,
            .r2 = reg2,
        } },
    }) else {
        const reloc = try self.asmJccReloc(cc.negate(), undefined);
        try self.asmRegisterRegister(.{ ._, .mov }, reg1, reg2);
        self.performReloc(reloc);
    }
}

/// A `cc` of `.z_and_np` is not supported by this encoding!
fn asmCmovccRegisterMemory(self: *CodeGen, cc: Condition, reg: Register, m: Memory) !void {
    if (self.hasFeature(.cmov)) _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .cmov,
            .z_and_np => unreachable,
            .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .rm,
            .z_and_np => unreachable,
            .nz_or_p => .pseudo_cmov_nz_or_p_rm,
        },
        .data = .{ .rx = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np => unreachable,
                .nz_or_p => ._,
            },
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    }) else {
        const reloc = try self.asmJccReloc(cc.negate(), undefined);
        try self.asmRegisterMemory(.{ ._, .mov }, reg, m);
        self.performReloc(reloc);
    }
}

fn asmSetccRegister(self: *CodeGen, cc: Condition, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .set,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .r,
            .z_and_np => .pseudo_set_z_and_np_r,
            .nz_or_p => .pseudo_set_nz_or_p_r,
        },
        .data = switch (cc) {
            else => .{ .r = .{
                .fixes = .fromCond(cc),
                .r1 = reg,
            } },
            .z_and_np, .nz_or_p => .{ .rr = .{
                .r1 = reg,
                .r2 = (try self.register_manager.allocReg(null, abi.RegisterClass.gp)).to8(),
            } },
        },
    });
}

fn asmSetccMemory(self: *CodeGen, cc: Condition, m: Memory) !void {
    const payload = try self.addExtra(Mir.Memory.encode(m));
    _ = try self.addInst(.{
        .tag = switch (cc) {
            else => .set,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .m,
            .z_and_np => .pseudo_set_z_and_np_m,
            .nz_or_p => .pseudo_set_nz_or_p_m,
        },
        .data = switch (cc) {
            else => .{ .x = .{
                .fixes = .fromCond(cc),
                .payload = payload,
            } },
            .z_and_np, .nz_or_p => .{ .rx = .{
                .r1 = (try self.register_manager.allocReg(null, abi.RegisterClass.gp)).to8(),
                .payload = payload,
            } },
        },
    });
}

fn asmJmpReloc(self: *CodeGen, target: Mir.Inst.Index) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = .j,
        .ops = .inst,
        .data = .{ .inst = .{
            .fixes = ._mp,
            .inst = target,
        } },
    });
}

fn asmJccReloc(self: *CodeGen, cc: Condition, target: Mir.Inst.Index) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = switch (cc) {
            else => .j,
            .z_and_np, .nz_or_p => .pseudo,
        },
        .ops = switch (cc) {
            else => .inst,
            .z_and_np => .pseudo_j_z_and_np_inst,
            .nz_or_p => .pseudo_j_nz_or_p_inst,
        },
        .data = .{ .inst = .{
            .fixes = switch (cc) {
                else => .fromCond(cc),
                .z_and_np, .nz_or_p => ._,
            },
            .inst = target,
        } },
    });
}

fn asmReloc(self: *CodeGen, tag: Mir.Inst.FixedTag, target: Mir.Inst.Index) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .inst,
        .data = .{ .inst = .{
            .fixes = tag[0],
            .inst = target,
        } },
    });
}

fn asmPlaceholder(self: *CodeGen) !Mir.Inst.Index {
    return self.addInst(.{
        .tag = .pseudo,
        .ops = .pseudo_dead_none,
        .data = undefined,
    });
}

const MirTagAir = enum { dbg_local };

fn asmAir(self: *CodeGen, tag: MirTagAir, inst: Air.Inst.Index) !void {
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = switch (tag) {
            .dbg_local => .pseudo_dbg_local_a,
        },
        .data = .{ .a = .{ .air_inst = inst } },
    });
}

fn asmAirImmediate(self: *CodeGen, tag: MirTagAir, inst: Air.Inst.Index, imm: Immediate) !void {
    switch (imm) {
        .signed => |s| _ = try self.addInst(.{
            .tag = .pseudo,
            .ops = switch (tag) {
                .dbg_local => .pseudo_dbg_local_ai_s,
            },
            .data = .{ .ai = .{
                .air_inst = inst,
                .i = @bitCast(s),
            } },
        }),
        .unsigned => |u| _ = if (std.math.cast(u32, u)) |small| try self.addInst(.{
            .tag = .pseudo,
            .ops = switch (tag) {
                .dbg_local => .pseudo_dbg_local_ai_u,
            },
            .data = .{ .ai = .{
                .air_inst = inst,
                .i = small,
            } },
        }) else try self.addInst(.{
            .tag = .pseudo,
            .ops = switch (tag) {
                .dbg_local => .pseudo_dbg_local_ai_64,
            },
            .data = .{ .ai = .{
                .air_inst = inst,
                .i = try self.addExtra(Mir.Imm64.encode(u)),
            } },
        }),
        .reloc => |sym_off| _ = if (sym_off.off == 0) try self.addInst(.{
            .tag = .pseudo,
            .ops = switch (tag) {
                .dbg_local => .pseudo_dbg_local_as,
            },
            .data = .{ .as = .{
                .air_inst = inst,
                .sym_index = sym_off.sym_index,
            } },
        }) else try self.addInst(.{
            .tag = .pseudo,
            .ops = switch (tag) {
                .dbg_local => .pseudo_dbg_local_aso,
            },
            .data = .{ .ax = .{
                .air_inst = inst,
                .payload = try self.addExtra(sym_off),
            } },
        }),
    }
}

fn asmAirRegisterImmediate(
    self: *CodeGen,
    tag: MirTagAir,
    inst: Air.Inst.Index,
    reg: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = switch (tag) {
            .dbg_local => .pseudo_dbg_local_aro,
        },
        .data = .{ .rx = .{
            .r1 = reg,
            .payload = try self.addExtra(Mir.AirOffset{
                .air_inst = inst,
                .off = imm.signed,
            }),
        } },
    });
}

fn asmAirFrameAddress(
    self: *CodeGen,
    tag: MirTagAir,
    inst: Air.Inst.Index,
    frame_addr: bits.FrameAddr,
) !void {
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = switch (tag) {
            .dbg_local => .pseudo_dbg_local_af,
        },
        .data = .{ .ax = .{
            .air_inst = inst,
            .payload = try self.addExtra(frame_addr),
        } },
    });
}

fn asmAirMemory(self: *CodeGen, tag: MirTagAir, inst: Air.Inst.Index, m: Memory) !void {
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = switch (tag) {
            .dbg_local => .pseudo_dbg_local_am,
        },
        .data = .{ .ax = .{
            .air_inst = inst,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmOpOnly(self: *CodeGen, tag: Mir.Inst.FixedTag) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .none,
        .data = .{ .none = .{
            .fixes = tag[0],
        } },
    });
}

fn asmPseudo(self: *CodeGen, ops: Mir.Inst.Ops) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_none"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = undefined,
    });
}

fn asmPseudoRegister(self: *CodeGen, ops: Mir.Inst.Ops, reg: Register) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_r"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .r = .{ .r1 = reg } },
    });
}

fn asmPseudoImmediate(self: *CodeGen, ops: Mir.Inst.Ops, imm: Immediate) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_i_s"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .i = .{ .i = @bitCast(imm.signed) } },
    });
}

fn asmPseudoRegisterRegister(self: *CodeGen, ops: Mir.Inst.Ops, reg1: Register, reg2: Register) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_rr"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .rr = .{ .r1 = reg1, .r2 = reg2 } },
    });
}

fn asmPseudoRegisterImmediate(self: *CodeGen, ops: Mir.Inst.Ops, reg: Register, imm: Immediate) !void {
    assert(std.mem.startsWith(u8, @tagName(ops), "pseudo_") and
        std.mem.endsWith(u8, @tagName(ops), "_ri_s"));
    _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = ops,
        .data = .{ .ri = .{ .r1 = reg, .i = @bitCast(imm.signed) } },
    });
}

fn asmRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .r,
        .data = .{ .r = .{
            .fixes = tag[0],
            .r1 = reg,
        } },
    });
}

fn asmImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, imm: Immediate) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .i_s,
            .unsigned => .i_u,
            .reloc => .rel,
        },
        .data = switch (imm) {
            .reloc => |sym_off| reloc: {
                assert(tag[0] == ._);
                break :reloc .{ .reloc = sym_off };
            },
            .signed, .unsigned => .{ .i = .{
                .fixes = tag[0],
                .i = switch (imm) {
                    .signed => |s| @bitCast(s),
                    .unsigned => |u| @intCast(u),
                    .reloc => unreachable,
                },
            } },
        },
    });
}

fn asmImmediateRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, imm: Immediate, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .ir,
        .data = .{ .ri = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = @as(u8, switch (imm) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .reloc => unreachable,
            }),
        } },
    });
}

fn asmImmediateImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, imm1: Immediate, imm2: Immediate) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .ii,
        .data = .{ .ii = .{
            .fixes = tag[0],
            .i1 = switch (imm1) {
                .signed => |s| @bitCast(@as(i16, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .reloc => unreachable,
            },
            .i2 = switch (imm2) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .reloc => unreachable,
            },
        } },
    });
}

fn asmRegisterRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, reg1: Register, reg2: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rr,
        .data = .{ .rr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
        } },
    });
}

fn asmRegisterImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register, imm: Immediate) !void {
    const ops: Mir.Inst.Ops, const i: u32 = switch (imm) {
        .signed => |s| .{ .ri_s, @bitCast(s) },
        .unsigned => |u| if (std.math.cast(u32, u)) |small|
            .{ .ri_u, small }
        else
            .{ .ri_64, try self.addExtra(Mir.Imm64.encode(imm.unsigned)) },
        .reloc => unreachable,
    };
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = ops,
        .data = .{ .ri = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = i,
        } },
    });
}

fn asmRegisterRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrr,
        .data = .{ .rrr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
        } },
    });
}

fn asmRegisterRegisterRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
    reg4: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrrr,
        .data = .{ .rrrr = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .r4 = reg4,
        } },
    });
}

fn asmRegisterRegisterRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    reg3: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrri,
        .data = .{ .rrri = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .i = switch (imm) {
                .signed => |s| @bitCast(@as(i8, @intCast(s))),
                .unsigned => |u| @intCast(u),
                .reloc => unreachable,
            },
        } },
    });
}

fn asmRegisterRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .rri_s,
            .unsigned => .rri_u,
            .reloc => unreachable,
        },
        .data = .{ .rri = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .i = switch (imm) {
                .signed => |s| @bitCast(s),
                .unsigned => |u| @intCast(u),
                .reloc => unreachable,
            },
        } },
    });
}

fn asmRegisterRegisterMemory(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrm,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterRegisterMemoryRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
    reg3: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrmr,
        .data = .{ .rrrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .r3 = reg3,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemory(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .m,
        .data = .{ .x = .{
            .fixes = tag[0],
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemory(self: *CodeGen, tag: Mir.Inst.FixedTag, reg: Register, m: Memory) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rm,
        .data = .{ .rx = .{
            .fixes = tag[0],
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemoryRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    m: Memory,
    reg2: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rmr,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmRegisterMemoryImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg: Register,
    m: Memory,
    imm: Immediate,
) !void {
    if (switch (imm) {
        .signed => |s| if (std.math.cast(i16, s)) |x| @as(u16, @bitCast(x)) else null,
        .unsigned => |u| std.math.cast(u16, u),
        .reloc => unreachable,
    }) |small_imm| {
        _ = try self.addInst(.{
            .tag = tag[1],
            .ops = .rmi,
            .data = .{ .rix = .{
                .fixes = tag[0],
                .r1 = reg,
                .i = small_imm,
                .payload = try self.addExtra(Mir.Memory.encode(m)),
            } },
        });
    } else {
        const payload = try self.addExtra(Mir.Imm32{ .imm = switch (imm) {
            .signed => |s| @bitCast(s),
            .unsigned => unreachable,
            .reloc => unreachable,
        } });
        assert(payload + 1 == try self.addExtra(Mir.Memory.encode(m)));
        _ = try self.addInst(.{
            .tag = tag[1],
            .ops = switch (imm) {
                .signed => .rmi_s,
                .unsigned => .rmi_u,
                .reloc => unreachable,
            },
            .data = .{ .rx = .{
                .fixes = tag[0],
                .r1 = reg,
                .payload = payload,
            } },
        });
    }
}

fn asmRegisterRegisterMemoryImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    reg1: Register,
    reg2: Register,
    m: Memory,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .rrmi,
        .data = .{ .rrix = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .i = @intCast(imm.unsigned),
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryRegister(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory, reg: Register) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mr,
        .data = .{ .rx = .{
            .fixes = tag[0],
            .r1 = reg,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryImmediate(self: *CodeGen, tag: Mir.Inst.FixedTag, m: Memory, imm: Immediate) !void {
    const payload = try self.addExtra(Mir.Imm32{ .imm = switch (imm) {
        .signed => |s| @bitCast(s),
        .unsigned => |u| @intCast(u),
        .reloc => unreachable,
    } });
    assert(payload + 1 == try self.addExtra(Mir.Memory.encode(m)));
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = switch (imm) {
            .signed => .mi_s,
            .unsigned => .mi_u,
            .reloc => unreachable,
        },
        .data = .{ .x = .{
            .fixes = tag[0],
            .payload = payload,
        } },
    });
}

fn asmMemoryRegisterRegister(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    m: Memory,
    reg1: Register,
    reg2: Register,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mrr,
        .data = .{ .rrx = .{
            .fixes = tag[0],
            .r1 = reg1,
            .r2 = reg2,
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn asmMemoryRegisterImmediate(
    self: *CodeGen,
    tag: Mir.Inst.FixedTag,
    m: Memory,
    reg: Register,
    imm: Immediate,
) !void {
    _ = try self.addInst(.{
        .tag = tag[1],
        .ops = .mri,
        .data = .{ .rix = .{
            .fixes = tag[0],
            .r1 = reg,
            .i = @intCast(imm.unsigned),
            .payload = try self.addExtra(Mir.Memory.encode(m)),
        } },
    });
}

fn gen(self: *CodeGen) InnerError!void {
    const pt = self.pt;
    const zcu = pt.zcu;
    const fn_info = zcu.typeToFunc(self.fn_type).?;
    if (fn_info.cc != .naked) {
        try self.asmRegister(.{ ._, .push }, .rbp);
        try self.asmPseudoImmediate(.pseudo_cfi_adjust_cfa_offset_i_s, .s(8));
        try self.asmPseudoRegisterImmediate(.pseudo_cfi_rel_offset_ri_s, .rbp, .s(0));
        try self.asmRegisterRegister(.{ ._, .mov }, .rbp, .rsp);
        try self.asmPseudoRegister(.pseudo_cfi_def_cfa_register_r, .rbp);
        const backpatch_push_callee_preserved_regs = try self.asmPlaceholder();
        const backpatch_frame_align = try self.asmPlaceholder();
        const backpatch_frame_align_extra = try self.asmPlaceholder();
        const backpatch_stack_alloc = try self.asmPlaceholder();
        const backpatch_stack_alloc_extra = try self.asmPlaceholder();

        switch (self.ret_mcv.long) {
            .none, .unreach => {},
            .indirect => {
                // The address where to store the return value for the caller is in a
                // register which the callee is free to clobber. Therefore, we purposely
                // spill it to stack immediately.
                const frame_index = try self.allocFrameIndex(.initSpill(.usize, zcu));
                try self.genSetMem(
                    .{ .frame = frame_index },
                    0,
                    .usize,
                    self.ret_mcv.long.address().offset(-self.ret_mcv.short.indirect.off),
                    .{},
                );
                self.ret_mcv.long = .{ .load_frame = .{ .index = frame_index } };
                tracking_log.debug("spill {} to {}", .{ self.ret_mcv.long, frame_index });
            },
            else => unreachable,
        }

        if (fn_info.is_var_args) switch (fn_info.cc) {
            .x86_64_sysv => {
                const info = &self.va_info.sysv;
                const reg_save_area_fi = try self.allocFrameIndex(.init(.{
                    .size = abi.SysV.c_abi_int_param_regs.len * 8 +
                        abi.SysV.c_abi_sse_param_regs.len * 16,
                    .alignment = .@"16",
                }));
                info.reg_save_area = .{ .index = reg_save_area_fi };

                for (abi.SysV.c_abi_int_param_regs[info.gp_count..], info.gp_count..) |reg, reg_i|
                    try self.genSetMem(.{ .frame = reg_save_area_fi }, @intCast(reg_i * 8), .usize, .{ .register = reg }, .{});

                try self.asmRegisterImmediate(.{ ._, .cmp }, .al, .u(info.fp_count));
                const skip_sse_reloc = try self.asmJccReloc(.na, undefined);

                const vec_2_f64 = try pt.vectorType(.{ .len = 2, .child = .f64_type });
                for (abi.SysV.c_abi_sse_param_regs[info.fp_count..], info.fp_count..) |reg, reg_i|
                    try self.genSetMem(
                        .{ .frame = reg_save_area_fi },
                        @intCast(abi.SysV.c_abi_int_param_regs.len * 8 + reg_i * 16),
                        vec_2_f64,
                        .{ .register = reg },
                        .{},
                    );

                self.performReloc(skip_sse_reloc);
            },
            .x86_64_win => return self.fail("TODO implement gen var arg function for Win64", .{}),
            else => |cc| return self.fail("{s} does not support var args", .{@tagName(cc)}),
        };

        if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_prologue_end_none);

        try self.genBody(self.air.getMainBody());

        const epilogue = if (self.epilogue_relocs.items.len > 0) epilogue: {
            const epilogue_relocs_last_index = self.epilogue_relocs.items.len - 1;
            for (if (self.epilogue_relocs.items[epilogue_relocs_last_index] == self.mir_instructions.len - 1) epilogue_relocs: {
                _ = self.mir_instructions.pop();
                break :epilogue_relocs self.epilogue_relocs.items[0..epilogue_relocs_last_index];
            } else self.epilogue_relocs.items) |epilogue_reloc| self.performReloc(epilogue_reloc);

            if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_epilogue_begin_none);
            const backpatch_stack_dealloc = try self.asmPlaceholder();
            const backpatch_pop_callee_preserved_regs = try self.asmPlaceholder();
            try self.asmRegister(.{ ._, .pop }, .rbp);
            try self.asmPseudoRegisterImmediate(.pseudo_cfi_def_cfa_ri_s, .rsp, .s(8));
            try self.asmOpOnly(.{ ._, .ret });
            break :epilogue .{
                .backpatch_stack_dealloc = backpatch_stack_dealloc,
                .backpatch_pop_callee_preserved_regs = backpatch_pop_callee_preserved_regs,
            };
        } else null;

        const frame_layout = try self.computeFrameLayout(fn_info.cc);
        const need_frame_align = frame_layout.stack_mask != std.math.maxInt(u32);
        const need_stack_adjust = frame_layout.stack_adjust > 0;
        const need_save_reg = frame_layout.save_reg_list.count() > 0;
        if (need_frame_align) {
            const page_align = @as(u32, std.math.maxInt(u32)) << 12;
            self.mir_instructions.set(backpatch_frame_align, .{
                .tag = .@"and",
                .ops = .ri_s,
                .data = .{ .ri = .{
                    .r1 = .rsp,
                    .i = @max(frame_layout.stack_mask, page_align),
                } },
            });
            if (frame_layout.stack_mask < page_align) {
                self.mir_instructions.set(backpatch_frame_align_extra, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_align_ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = ~frame_layout.stack_mask & page_align,
                    } },
                });
            }
        }
        if (need_stack_adjust) {
            const page_size: u32 = 1 << 12;
            if (frame_layout.stack_adjust <= page_size) {
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .sub,
                    .ops = .ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = frame_layout.stack_adjust,
                    } },
                });
            } else if (frame_layout.stack_adjust <
                page_size * Lower.pseudo_probe_adjust_unrolled_max_insts)
            {
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_unrolled_ri_s,
                    .data = .{ .ri = .{
                        .r1 = .rsp,
                        .i = frame_layout.stack_adjust,
                    } },
                });
            } else {
                const scratch_reg = abi.getCAbiLinkerScratchReg(fn_info.cc);
                self.mir_instructions.set(backpatch_stack_alloc, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_setup_rri_s,
                    .data = .{ .rri = .{
                        .r1 = .rsp,
                        .r2 = scratch_reg,
                        .i = frame_layout.stack_adjust,
                    } },
                });
                self.mir_instructions.set(backpatch_stack_alloc_extra, .{
                    .tag = .pseudo,
                    .ops = .pseudo_probe_adjust_loop_rr,
                    .data = .{ .rr = .{
                        .r1 = .rsp,
                        .r2 = scratch_reg,
                    } },
                });
            }
        }
        if (epilogue) |e| if (need_frame_align or need_stack_adjust) {
            self.mir_instructions.set(e.backpatch_stack_dealloc, switch (-frame_layout.save_reg_list.size(self.target)) {
                0 => .{
                    .tag = .mov,
                    .ops = .rr,
                    .data = .{ .rr = .{
                        .r1 = .rsp,
                        .r2 = .rbp,
                    } },
                },
                else => |disp| .{
                    .tag = .lea,
                    .ops = .rm,
                    .data = .{ .rx = .{
                        .r1 = .rsp,
                        .payload = try self.addExtra(Mir.Memory.encode(.{
                            .base = .{ .reg = .rbp },
                            .mod = .{ .rm = .{
                                .size = .qword,
                                .disp = disp,
                            } },
                        })),
                    } },
                },
            });
        };
        if (need_save_reg) {
            self.mir_instructions.set(backpatch_push_callee_preserved_regs, .{
                .tag = .pseudo,
                .ops = .pseudo_push_reg_list,
                .data = .{ .reg_list = frame_layout.save_reg_list },
            });
            if (epilogue) |e| self.mir_instructions.set(e.backpatch_pop_callee_preserved_regs, .{
                .tag = .pseudo,
                .ops = .pseudo_pop_reg_list,
                .data = .{ .reg_list = frame_layout.save_reg_list },
            });
        }
    } else {
        if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_prologue_end_none);
        try self.genBody(self.air.getMainBody());
        if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_epilogue_begin_none);
    }

    // Drop them off at the rbrace.
    if (self.debug_output != .none) _ = try self.addInst(.{
        .tag = .pseudo,
        .ops = .pseudo_dbg_line_line_column,
        .data = .{ .line_column = .{
            .line = self.end_di_line,
            .column = self.end_di_column,
        } },
    });
}

fn checkInvariantsAfterAirInst(self: *CodeGen) void {
    assert(!self.register_manager.lockedRegsExist());

    if (std.debug.runtime_safety) {
        // check consistency of tracked registers
        var it = self.register_manager.free_registers.iterator(.{ .kind = .unset });
        while (it.next()) |index| {
            const tracked_inst = self.register_manager.registers[index];
            const tracking = self.getResolvedInstValue(tracked_inst);
            for (tracking.getRegs()) |reg| {
                if (RegisterManager.indexOfRegIntoTracked(reg).? == index) break;
            } else unreachable; // tracked register not in use
        }
    }
}

fn genBodyBlock(self: *CodeGen, body: []const Air.Inst.Index) InnerError!void {
    if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_enter_block_none);
    try self.genBody(body);
    if (self.debug_output != .none) try self.asmPseudo(.pseudo_dbg_leave_block_none);
}

fn genBody(cg: *CodeGen, body: []const Air.Inst.Index) InnerError!void {
    @setEvalBranchQuota(13_600);
    const pt = cg.pt;
    const zcu = pt.zcu;
    const ip = &zcu.intern_pool;
    const air_tags = cg.air.instructions.items(.tag);
    const air_datas = cg.air.instructions.items(.data);
    const use_old = cg.target.ofmt == .coff;

    cg.arg_index = 0;
    for (body) |inst| switch (air_tags[@intFromEnum(inst)]) {
        .arg => {
            wip_mir_log.debug("{}", .{cg.fmtAir(inst)});
            verbose_tracking_log.debug("{}", .{cg.fmtTracking()});

            cg.reused_operands = .initEmpty();
            try cg.inst_tracking.ensureUnusedCapacity(cg.gpa, 1);

            try cg.airArg(inst);

            try cg.resetTemps();
            cg.checkInvariantsAfterAirInst();
        },
        else => break,
    };

    if (cg.arg_index == 0) try cg.airDbgVarArgs();
    cg.arg_index = 0;
    for (body) |inst| {
        if (cg.liveness.isUnused(inst) and !cg.air.mustLower(inst, ip)) continue;
        wip_mir_log.debug("{}", .{cg.fmtAir(inst)});
        verbose_tracking_log.debug("{}", .{cg.fmtTracking()});

        cg.reused_operands = .initEmpty();
        try cg.inst_tracking.ensureUnusedCapacity(cg.gpa, 1);
        switch (air_tags[@intFromEnum(inst)]) {
            // zig fmt: off
            .shr, .shr_exact => try cg.airShlShrBinOp(inst),
            .shl, .shl_exact => try cg.airShlShrBinOp(inst),

            .mul_wrap,
            => |air_tag| try cg.airMulDivBinOp(inst, air_tag),

            .add_sat         => try cg.airAddSat(inst),
            .sub_sat         => try cg.airSubSat(inst),
            .mul_sat         => try cg.airMulSat(inst),
            .shl_sat         => try cg.airShlSat(inst),

            .add_with_overflow => try cg.airAddSubWithOverflow(inst),
            .sub_with_overflow => try cg.airAddSubWithOverflow(inst),
            .mul_with_overflow => try cg.airMulWithOverflow(inst),
            .shl_with_overflow => try cg.airShlWithOverflow(inst),

            .bitcast          => try cg.airBitCast(inst),

            .splat            => try cg.airSplat(inst),
            .select           => try cg.airSelect(inst),
            .shuffle          => try cg.airShuffle(inst),
            .reduce           => try cg.airReduce(inst),
            .reduce_optimized => try cg.airReduce(inst),
            .aggregate_init   => try cg.airAggregateInit(inst),
            // zig fmt: on

            .arg => if (cg.debug_output != .none) {
                // skip zero-bit arguments as they don't have a corresponding arg instruction
                var arg_index = cg.arg_index;
                while (cg.args[arg_index] == .none) arg_index += 1;
                cg.arg_index = arg_index + 1;

                const name = air_datas[@intFromEnum(inst)].arg.name;
                if (name != .none) try cg.genLocalDebugInfo(inst, cg.getResolvedInstValue(inst).short);
                if (cg.liveness.isUnused(inst)) try cg.processDeath(inst);

                for (cg.args[arg_index + 1 ..]) |arg| {
                    if (arg != .none) break;
                } else try cg.airDbgVarArgs();
            },
            .add, .add_optimized, .add_wrap => |air_tag| if (use_old) try cg.airBinOp(inst, switch (air_tag) {
                else => unreachable,
                .add, .add_optimized => .add,
                .add_wrap => .add_wrap,
            }) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .int = .byte }, .{ .int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm8, .none } },
                        .{ .src = .{ .imm8, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .imm8, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm16, .none } },
                        .{ .src = .{ .imm16, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm32, .none } },
                        .{ .src = .{ .imm32, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .simm32, .none } },
                        .{ .src = .{ .simm32, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .mut_mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .add, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memsia(.src0d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp1d, .memsia(.src1d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0d, .@"4", .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_q, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .add, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5d, .leasi(.tmp1d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .adc, .tmp5d, .leasi(.tmp2d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3d, .@"4", .tmp4), .tmp5d, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .add, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .add, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .add, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .add, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .add, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .add, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .add, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .add, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .add, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .add, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .add, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .add, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .add, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .add, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .add, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                switch (air_tag) {
                    else => unreachable,
                    .add, .add_optimized => {},
                    .add_wrap => res[0].wrapInt(cg) catch |err| switch (err) {
                        error.SelectFailed => return cg.fail("failed to select wrap {} {} {}", .{
                            cg.typeOf(bin_op.lhs).fmt(pt),
                            ops[0].tracking(cg),
                            ops[1].tracking(cg),
                        }),
                        else => |e| return e,
                    },
                }
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .add_safe => unreachable,
            .sub, .sub_optimized, .sub_wrap => |air_tag| if (use_old) try cg.airBinOp(inst, switch (air_tag) {
                else => unreachable,
                .sub, .sub_optimized => .sub,
                .sub_wrap => .sub_wrap,
            }) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .int = .byte }, .{ .int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm8, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm8, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm16, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm16, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .imm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .imm32, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mut_mem, .simm32, .none } },
                        .{ .src = .{ .to_mut_gpr, .simm32, .none } },
                        .{ .src = .{ .mut_mem, .to_gpr, .none } },
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .sub, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp0, .add_size), .tmp1q, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memsia(.src0d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1d, .memsia(.src1d, .@"4", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0d, .@"4", .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_b, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1b, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_q, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_q, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_q, .sub, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_u64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_q, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .sub, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5q, .leasi(.tmp2q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp5q, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_unaligned_size_add_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4p, .sa(.src0, .sub_elem_size_div_4), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp5d, .leasi(.tmp1d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5d, .leasi(.tmp2d, .@"4", .tmp4), ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3d, .@"4", .tmp4), .tmp5d, ._, ._ },
                        .{ ._, ._c, .in, .tmp4p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .sub, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .sub, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .sub, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .sub, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subhf3" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .sub, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .sub, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .sub, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .sub, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .sub, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .sub, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .sub, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .sub, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .sub, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .sub, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .sub, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .subr, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .sub, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .sub, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__subtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                switch (air_tag) {
                    else => unreachable,
                    .sub, .sub_optimized => {},
                    .sub_wrap => res[0].wrapInt(cg) catch |err| switch (err) {
                        error.SelectFailed => return cg.fail("failed to select wrap {} {} {}", .{
                            cg.typeOf(bin_op.lhs).fmt(pt),
                            ops[0].tracking(cg),
                            ops[1].tracking(cg),
                        }),
                        else => |e| return e,
                    },
                }
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .sub_safe => unreachable,
            .mul, .mul_optimized => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, .mul) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                const ty = cg.typeOf(bin_op.lhs);
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{ty}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .al }, .mem, .none } },
                        .{ .src = .{ .mem, .{ .to_reg = .al }, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .{ .to_reg = .al }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .src1b, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .al }, .mem, .none } },
                        .{ .src = .{ .mem, .{ .to_reg = .al }, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .{ .to_reg = .al }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mul, .src1b, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .imm16, .none } },
                        .{ .src = .{ .imm16, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_gpr, .imm16, .none } },
                        .{ .src = .{ .imm16, .to_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .general_purpose } }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0w, .src0w, .src1w, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .word }, .{ .int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .imm32, .none } },
                        .{ .src = .{ .imm32, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_gpr, .imm32, .none } },
                        .{ .src = .{ .imm32, .to_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .general_purpose } }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0d, .src0d, .src1d, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .int = .dword }, .{ .int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .simm32, .none } },
                        .{ .src = .{ .simm32, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_gpr, .simm32, .none } },
                        .{ .src = .{ .simm32, .to_gpr, .none }, .commute = .{ 0, 1 } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .general_purpose } }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0q, .src0q, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .qword }, .{ .int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .int = .xword }, .{ .int = .xword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, ._, .mul, .src1q, ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0q, .tmp0q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .src0q, ._, ._ },
                        .{ ._, .i_, .mul, .tmp0q, .memd(.src1q, 8), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp0q, ._, ._ },
                        .{ ._, ._, .mov, .tmp0q, .src1q, ._, ._ },
                        .{ ._, .i_, .mul, .tmp0q, .memd(.src0q, 8), ._, ._ },
                        .{ ._, ._, .add, .tmp1q, .tmp0q, ._, ._ },
                        .{ ._, ._, .mov, .memd(.dst0q, 8), .tmp1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .adx, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .sia(-8, .src0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.src1), ._, ._ },
                        .{ .@"0:", ._, .xor, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp2q, .memi(.src0q, .tmp0), ._, ._ },
                        .{ ._, ._z, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .leaad(.tmp0, .sub_src0_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4d, .tmp4d, ._, ._ },
                        .{ .@"1:", ._x, .mul, .tmp6q, .tmp5q, .leai(.tmp1q, .tmp3), ._ },
                        .{ ._, ._x, .adc, .tmp5q, .tmp4q, ._, ._ },
                        .{ ._, ._, .mov, .memiad(.dst0q, .tmp3, .add_size, -8), .tmp5q, ._, ._ },
                        .{ ._, ._rcxz, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._x, .ado, .tmp6q, .memia(.dst0q, .tmp3, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp6q, ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .lead(.tmp3, 8), ._, ._ },
                        .{ ._, ._mp, .j, .@"1b", ._, ._, ._ },
                        .{ .@"2:", ._, .mov, .memi(.dst0q, .tmp0), .tmp2q, ._, ._ },
                        .{ .@"1:", ._, .lea, .tmp1p, .lead(.tmp1, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp0d, .si(8), ._, ._ },
                        .{ ._, ._ae, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .slow_incdec, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .memd(.src1, 8), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .leaa(.tmp0, .sub_src0_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._, .xor, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._, .xor, .tmp5d, .tmp5d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp3q, .memsi(.src0q, .@"8", .tmp0), ._, ._ },
                        .{ ._, ._nz, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memsi(.dst0q, .@"8", .tmp0), .tmp3q, ._, ._ },
                        .{ ._, ._mp, .j, .@"3f", ._, ._, ._ },
                        .{ .@"1:", ._, .adc, .tmp7q, .memsia(.dst0q, .@"8", .tmp2, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp4b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .tmp7q, ._, ._ },
                        .{ ._, ._l, .sh, .tmp4b, .ui(4), ._, ._ },
                        .{ .@"2:", ._x, .mul, .tmp7q, .tmp6q, .leasi(.tmp1q, .@"8", .tmp2), ._ },
                        .{ ._, ._, .adc, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp2, .add_size), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp2p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ .@"3:", ._, .lea, .tmp1p, .lead(.tmp1, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp0d, .si(1), ._, ._ },
                        .{ ._, ._ae, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .memd(.src1, 8), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .leaa(.tmp0, .sub_src0_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._, .xor, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._, .xor, .tmp5d, .tmp5d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp3q, .memsi(.src0q, .@"8", .tmp0), ._, ._ },
                        .{ ._, ._nz, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memsi(.dst0q, .@"8", .tmp0), .tmp3q, ._, ._ },
                        .{ ._, ._mp, .j, .@"3f", ._, ._, ._ },
                        .{ .@"1:", ._, .adc, .tmp7q, .memsia(.dst0q, .@"8", .tmp2, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp4b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .tmp7q, ._, ._ },
                        .{ ._, ._l, .sh, .tmp4b, .ui(4), ._, ._ },
                        .{ .@"2:", ._x, .mul, .tmp7q, .tmp6q, .leasi(.tmp1q, .@"8", .tmp2), ._ },
                        .{ ._, ._, .adc, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp2, .add_size), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp2p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ .@"3:", ._, .lea, .tmp1p, .lead(.tmp1, 8), ._, ._ },
                        .{ ._, ._c, .de, .tmp0d, ._, ._, ._ },
                        .{ ._, ._ns, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .memd(.src1, 8), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .leaa(.tmp0, .sub_src0_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._, .xor, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._, .xor, .tmp5d, .tmp5d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp3q, .memsi(.src0q, .@"8", .tmp0), ._, ._ },
                        .{ ._, ._nz, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memsi(.dst0q, .@"8", .tmp0), .tmp3q, ._, ._ },
                        .{ ._, ._mp, .j, .@"3f", ._, ._, ._ },
                        .{ .@"1:", ._, .adc, .tmp7q, .memsia(.dst0q, .@"8", .tmp2, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp4b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .tmp7q, ._, ._ },
                        .{ .@"2:", ._, .mov, .tmp6q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mul, .leasi(.tmp1q, .@"8", .tmp2), ._, ._, ._ },
                        .{ ._, ._l, .sh, .tmp4b, .ui(4), ._, ._ },
                        .{ ._, ._, .adc, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp2, .add_size), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp2p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ .@"3:", ._, .lea, .tmp1p, .lead(.tmp1, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp0d, .si(1), ._, ._ },
                        .{ ._, ._ae, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0d, .sia(-1, .src0, .add_size_div_8), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .memd(.src1, 8), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp2p, .leaa(.tmp0, .sub_src0_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp3d, ._, ._ },
                        .{ ._, ._, .xor, .tmp4d, .tmp4d, ._, ._ },
                        .{ ._, ._, .xor, .tmp5d, .tmp5d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp3q, .memsi(.src0q, .@"8", .tmp0), ._, ._ },
                        .{ ._, ._nz, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._, .mov, .memsi(.dst0q, .@"8", .tmp0), .tmp3q, ._, ._ },
                        .{ ._, ._mp, .j, .@"3f", ._, ._, ._ },
                        .{ .@"1:", ._, .adc, .tmp7q, .memsia(.dst0q, .@"8", .tmp2, .add_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp4b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .tmp7q, ._, ._ },
                        .{ .@"2:", ._, .mov, .tmp6q, .tmp3q, ._, ._ },
                        .{ ._, ._, .mul, .leasi(.tmp1q, .@"8", .tmp2), ._, ._, ._ },
                        .{ ._, ._l, .sh, .tmp4b, .ui(4), ._, ._ },
                        .{ ._, ._, .adc, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._, .mov, .memsia(.dst0q, .@"8", .tmp2, .add_size), .tmp6q, ._, ._ },
                        .{ ._, ._c, .in, .tmp2p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"1b", ._, ._, ._ },
                        .{ .@"3:", ._, .lea, .tmp1p, .lead(.tmp1, 8), ._, ._ },
                        .{ ._, ._c, .de, .tmp0d, ._, ._, ._ },
                        .{ ._, ._ns, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .movsxb, .dst0x, .src0q, ._, ._ },
                        .{ ._, .vp_w, .movsxb, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .vp_w, .mull, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .vp_b, .ackssw, .dst0x, .dst0x, .dst0x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .movsxb, .dst0x, .src0q, ._, ._ },
                        .{ ._, .p_w, .movsxb, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .p_w, .mull, .dst0x, .tmp0x, ._, ._ },
                        .{ ._, .p_b, .ackssw, .dst0x, .dst0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .to_mut_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .xor, .tmp0x, .tmp0x, ._, ._ },
                        .{ ._, .p_b, .cmpgt, .tmp0x, .src0x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .dst0x, .tmp0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp0x, .tmp0x, ._, ._ },
                        .{ ._, .p_b, .cmpgt, .tmp0x, .src1x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .src1x, .tmp0x, ._, ._ },
                        .{ ._, .p_w, .mull, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .ackssw, .dst0x, .dst0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_16_i16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .movsxb, .dst0y, .src0x, ._, ._ },
                        .{ ._, .vp_w, .movsxb, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .vp_w, .mull, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .vp_b, .ackssw, .dst0y, .dst0y, .dst0y, ._ },
                        .{ ._, .v_q, .perm, .dst0y, .dst0y, .ui(0b10_00_10_00), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .movzxb, .dst0x, .src0q, ._, ._ },
                        .{ ._, .vp_w, .movzxb, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .vp_w, .mull, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .vp_b, .ackusw, .dst0x, .dst0x, .dst0x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .movzxb, .dst0x, .src0q, ._, ._ },
                        .{ ._, .p_w, .movzxb, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .p_w, .mull, .dst0x, .tmp0x, ._, ._ },
                        .{ ._, .p_b, .ackusw, .dst0x, .dst0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .to_mut_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_, .xor, .tmp0x, .tmp0x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .dst0x, .tmp0x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .src1x, .tmp0x, ._, ._ },
                        .{ ._, .p_w, .mull, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_b, .ackusw, .dst0x, .dst0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_16_u16, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .movzxb, .dst0y, .src0x, ._, ._ },
                        .{ ._, .vp_w, .movzxb, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .vp_w, .mull, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .vp_b, .ackusw, .dst0y, .dst0y, .dst0y, ._ },
                        .{ ._, .v_q, .perm, .dst0y, .dst0y, .ui(0b10_00_10_00), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_w, .movsxb, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .movsxb, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .vp_b, .ackssw, .tmp1y, .tmp1y, .tmp1y, ._ },
                        .{ ._, .v_q, .perm, .tmp1y, .tmp1y, .ui(0b10_00_10_00), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_w, .movsxb, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .movsxb, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1x, .tmp1x, .tmp2x, ._ },
                        .{ ._, .vp_b, .ackssw, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .v_q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_w, .movsxb, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .movsxb, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .mull, .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .ackssw, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .tmp2x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .cmpgt, .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .tmp3x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_b, .cmpgt, .tmp1x, .tmp3x, ._, ._ },
                        .{ ._, .p_, .unpcklbw, .tmp3x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .mull, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, .p_b, .ackssw, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .reg = .al } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .reg = .al } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_w, .movzxb, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .movzxb, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .vp_b, .ackusw, .tmp1y, .tmp1y, .tmp1y, ._ },
                        .{ ._, .v_q, .perm, .tmp1y, .tmp1y, .ui(0b10_00_10_00), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_w, .movzxb, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .movzxb, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1x, .tmp1x, .tmp2x, ._ },
                        .{ ._, .vp_b, .ackusw, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .v_q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_w, .movzxb, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .movzxb, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .mull, .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, .p_b, .ackusw, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .tmp2x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .unpcklbw, .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, ._q, .mov, .tmp3x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_, .unpcklbw, .tmp3x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .mull, .tmp2x, .tmp3x, ._, ._ },
                        .{ ._, .p_b, .ackusw, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._q, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .reg = .al } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mul, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .reg = .al } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mul, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .mull, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .mull, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .mull, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_w, .mull, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_w, .mull, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .tmp1w, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .mull, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .mull, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .mull, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .mull, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .vp_d, .mull, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .p_d, .mull, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .tmp1d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .mul, .tmp1q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .adx, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .isize, .kind = .{ .reg = .rcx } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src1, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-8, .src0, .add_elem_size), ._, ._ },
                        .{ .@"1:", ._, .xor, .tmp5d, .tmp5d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp5q, .leai(.tmp1q, .tmp4), ._, ._ },
                        .{ ._, ._z, .j, .@"3f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp6p, .leaad(.tmp4, .sub_src0_elem_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp7d, .tmp7d, ._, ._ },
                        .{ .@"2:", ._x, .mul, .tmp9q, .tmp8q, .leai(.tmp2q, .tmp6), ._ },
                        .{ ._, ._x, .adc, .tmp8q, .tmp7q, ._, ._ },
                        .{ ._, ._, .mov, .leaiad(.tmp3q, .tmp6, .add_dst0_elem_size, -8), .tmp8q, ._, ._ },
                        .{ ._, ._rcxz, .j, .@"2f", ._, ._, ._ },
                        .{ ._, ._x, .ado, .tmp9q, .leaia(.tmp3q, .tmp6, .add_dst0_elem_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp7q, .tmp9q, ._, ._ },
                        .{ ._, ._, .lea, .tmp6p, .lead(.tmp6, 8), ._, ._ },
                        .{ ._, ._mp, .j, .@"2b", ._, ._, ._ },
                        .{ .@"3:", ._, .mov, .leai(.tmp3q, .tmp4), .tmp5q, ._, ._ },
                        .{ .@"2:", ._, .lea, .tmp2p, .lead(.tmp2, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp4d, .si(8), ._, ._ },
                        .{ ._, ._ae, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, .slow_incdec, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_size, 8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .lea, .tmp5p, .leaa(.tmp4, .sub_src0_elem_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp6d, ._, ._ },
                        .{ ._, ._, .xor, .tmp7d, .tmp7d, ._, ._ },
                        .{ ._, ._, .xor, .tmp8d, .tmp8d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp6q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._nz, .j, .@"3f", ._, ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp6q, ._, ._ },
                        .{ ._, ._mp, .j, .@"4f", ._, ._, ._ },
                        .{ .@"2:", ._, .adc, .tmp10q, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp7b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp8q, .tmp10q, ._, ._ },
                        .{ ._, ._l, .sh, .tmp7b, .ui(4), ._, ._ },
                        .{ .@"3:", ._x, .mul, .tmp10q, .tmp9q, .leasi(.tmp2q, .@"8", .tmp5), ._ },
                        .{ ._, ._, .adc, .tmp9q, .tmp8q, ._, ._ },
                        .{ ._, ._, .mov, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), .tmp9q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"2b", ._, ._, ._ },
                        .{ .@"4:", ._, .lea, .tmp2p, .lead(.tmp2, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp4d, .si(1), ._, ._ },
                        .{ ._, ._ae, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .bmi2, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_size, 8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .lea, .tmp5p, .leaa(.tmp4, .sub_src0_elem_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp6d, ._, ._ },
                        .{ ._, ._, .xor, .tmp7d, .tmp7d, ._, ._ },
                        .{ ._, ._, .xor, .tmp8d, .tmp8d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp6q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._nz, .j, .@"3f", ._, ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp6q, ._, ._ },
                        .{ ._, ._mp, .j, .@"4f", ._, ._, ._ },
                        .{ .@"2:", ._, .adc, .tmp10q, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp7b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp8q, .tmp10q, ._, ._ },
                        .{ ._, ._l, .sh, .tmp7b, .ui(4), ._, ._ },
                        .{ .@"3:", ._x, .mul, .tmp10q, .tmp9q, .leasi(.tmp2q, .@"8", .tmp5), ._ },
                        .{ ._, ._, .adc, .tmp9q, .tmp8q, ._, ._ },
                        .{ ._, ._, .mov, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), .tmp9q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"2b", ._, ._, ._ },
                        .{ .@"4:", ._, .lea, .tmp2p, .lead(.tmp2, 8), ._, ._ },
                        .{ ._, ._c, .de, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ns, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_size, 8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .lea, .tmp5p, .leaa(.tmp4, .sub_src0_elem_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp6d, ._, ._ },
                        .{ ._, ._, .xor, .tmp7d, .tmp7d, ._, ._ },
                        .{ ._, ._, .xor, .tmp8d, .tmp8d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp6q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._nz, .j, .@"3f", ._, ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp6q, ._, ._ },
                        .{ ._, ._mp, .j, .@"4f", ._, ._, ._ },
                        .{ .@"2:", ._, .adc, .tmp10q, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp7b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp8q, .tmp10q, ._, ._ },
                        .{ .@"3:", ._, .mov, .tmp9q, .tmp6q, ._, ._ },
                        .{ ._, ._, .mul, .leasi(.tmp2q, .@"8", .tmp5), ._, ._, ._ },
                        .{ ._, ._l, .sh, .tmp7b, .ui(4), ._, ._ },
                        .{ ._, ._, .adc, .tmp9q, .tmp8q, ._, ._ },
                        .{ ._, ._, .mov, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), .tmp9q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"2b", ._, ._, ._ },
                        .{ .@"4:", ._, .lea, .tmp2p, .lead(.tmp2, 8), ._, ._ },
                        .{ ._, ._, .sub, .tmp4d, .si(1), ._, ._ },
                        .{ ._, ._ae, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_remainder_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.src0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memiad(.src1, .tmp0, .add_size, 8), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.dst0, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sia(-1, .src0, .add_elem_size_div_8), ._, ._ },
                        .{ .@"1:", ._, .lea, .tmp5p, .leaa(.tmp4, .sub_src0_elem_size_div_8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp6d, ._, ._ },
                        .{ ._, ._, .xor, .tmp7d, .tmp7d, ._, ._ },
                        .{ ._, ._, .xor, .tmp8d, .tmp8d, ._, ._ },
                        .{ ._, ._, .@"or", .tmp6q, .leasi(.tmp1q, .@"8", .tmp4), ._, ._ },
                        .{ ._, ._nz, .j, .@"3f", ._, ._, ._ },
                        .{ ._, ._, .mov, .leasi(.tmp3q, .@"8", .tmp4), .tmp6q, ._, ._ },
                        .{ ._, ._mp, .j, .@"4f", ._, ._, ._ },
                        .{ .@"2:", ._, .adc, .tmp10q, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), ._, ._ },
                        .{ ._, ._, .adc, .tmp7b, .si(0), ._, ._ },
                        .{ ._, ._, .mov, .tmp8q, .tmp10q, ._, ._ },
                        .{ .@"3:", ._, .mov, .tmp9q, .tmp6q, ._, ._ },
                        .{ ._, ._, .mul, .leasi(.tmp2q, .@"8", .tmp5), ._, ._, ._ },
                        .{ ._, ._l, .sh, .tmp7b, .ui(4), ._, ._ },
                        .{ ._, ._, .adc, .tmp9q, .tmp8q, ._, ._ },
                        .{ ._, ._, .mov, .leasia(.tmp3q, .@"8", .tmp5, .add_src0_elem_size), .tmp9q, ._, ._ },
                        .{ ._, ._c, .in, .tmp5p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"2b", ._, ._, ._ },
                        .{ .@"4:", ._, .lea, .tmp2p, .lead(.tmp2, 8), ._, ._ },
                        .{ ._, ._c, .de, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ns, .j, .@"1b", ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .mul, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__mulhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .mul, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .mul, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .mul, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__mulhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__mulhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__mulhf3" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__mulhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .mul, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .mul, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .mul, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .mul, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .mul, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .mul, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mul, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .mul, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .mul, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .mul, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .mul, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .mul, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .mul, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .mul, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mul, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .mul, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .mul, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .mul, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .mul, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__multf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__multf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__multf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__multf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        ty.fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .mul_safe => unreachable,
            .div_float, .div_float_optimized, .div_exact, .div_exact_optimized => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, switch (air_tag) {
                else => unreachable,
                .div_float, .div_float_optimized => .div_float,
                .div_exact, .div_exact_optimized => .div_exact,
            }) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                const ty = cg.typeOf(bin_op.lhs);
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                (if (cg.floatBits(ty.scalarType(zcu))) |_| cg.select(&res, &.{ty}, &ops, comptime &.{ .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .div, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .div, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .div, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .div, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0q, ._, ._, ._ },
                        .{ ._, .f_, .div, .src1q, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .div, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .div, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_x87, .mem, .none }, .commute = .{ 0, 1 } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .divr, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .to_x87, .none } },
                        .{ .src = .{ .to_x87, .to_x87, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .x87 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .div, .tmp0t, .src1t, ._, ._ },
                        .{ ._, .f_p, .st, .dst0t, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) else err: {
                    assert(air_tag == .div_exact);
                    res[0] = ops[0].divTruncInts(&ops[1], cg) catch |err| break :err err;
                }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        ty.fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .div_trunc => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, air_tag) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                const ty = cg.typeOf(bin_op.lhs);
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                (if (cg.floatBits(ty.scalarType(zcu))) |_| cg.select(&res, &.{ty}, &ops, comptime &.{ .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .div, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0x, .dst0q, ._, ._ },
                        .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = .zero, .precision = .inexact }) },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__trunch" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0x, .dst0q, ._, ._ },
                        .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0y, .dst0x, ._, ._ },
                        .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .tmp1x, .tmp1y, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp1y, .tmp1x, ._, ._ },
                        .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__trunch" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__trunch" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__trunch" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp5d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp5w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__trunch" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp6d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .div, .dst0x, .src0x, .src1d, ._ },
                        .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = .zero, .precision = .inexact }) },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                        .{ ._, ._ss, .round, .dst0x, .dst0d, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .div, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ss, .div, .tmp1x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .round, .tmp1x, .tmp1x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .div, .dst0x, .src0x, .src1q, ._ },
                        .{ ._, .v_sd, .round, .dst0x, .dst0x, .dst0q, .rm(.{ .direction = .zero, .precision = .inexact }) },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                        .{ ._, ._sd, .round, .dst0x, .dst0q, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "trunc" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "trunc" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .v_pd, .round, .dst0x, .dst0x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .div, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._pd, .round, .dst0x, .dst0x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .v_pd, .round, .dst0y, .dst0y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .round, .tmp1x, .tmp1x, .rm(.{ .direction = .zero, .precision = .inexact }), ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "trunc" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._sd, .div, .tmp1x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, ._sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "trunc" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._ps, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._ps, .movl, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .movl, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ps, .movl, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__truncx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__truncx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp3t), ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "truncq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) else err: {
                    res[0] = ops[0].divTruncInts(&ops[1], cg) catch |err| break :err err;
                }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        ty.fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .div_trunc_optimized, .div_floor_optimized => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, switch (air_tag) {
                else => unreachable,
                .div_trunc_optimized => .div_trunc,
                .div_floor_optimized => .div_floor,
            }) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, switch (@as(bits.RoundMode.Direction, switch (air_tag) {
                    else => unreachable,
                    .div_trunc_optimized => .zero,
                    .div_floor_optimized => .down,
                })) {
                    else => unreachable,
                    inline .zero, .down => |direction| comptime &.{ .{
                        .required_features = .{ .f16c, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .word, .is = .word } },
                            .{ .scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                            .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                            .{ ._, .v_ss, .div, .dst0x, .dst0x, .tmp0d, ._ },
                            .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = direction, .precision = .inexact }) },
                            .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .word, .is = .word } },
                            .{ .scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__trunch",
                                .down => "__floorh",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .f16c, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .qword, .is = .word } },
                            .{ .scalar_float = .{ .of = .qword, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .mem, .mem, .none } },
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .mem, .to_sse, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                            .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                            .{ ._, .v_ps, .div, .dst0x, .dst0x, .tmp0x, ._ },
                            .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        } },
                    }, .{
                        .required_features = .{ .f16c, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .word } },
                            .{ .scalar_float = .{ .of = .xword, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .mem, .mem, .none } },
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .mem, .to_sse, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                            .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                            .{ ._, .v_ps, .div, .dst0y, .dst0y, .tmp0y, ._ },
                            .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                        } },
                    }, .{
                        .required_features = .{ .f16c, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                            .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .tmp2y, ._ },
                            .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__trunch",
                                .down => "__floorh",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                            .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                            .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                            .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__trunch",
                                .down => "__floorh",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                            .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                            .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                            .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                            .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse2, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__trunch",
                                .down => "__floorh",
                            } } } },
                            .{ .type = .f16, .kind = .{ .reg = .ax } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                            .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                            .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                            .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, .p_w, .extr, .tmp5d, .tmp1x, .ui(0), ._ },
                            .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp5w, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f16, .kind = .{ .reg = .ax } },
                            .{ .type = .f32, .kind = .mem },
                            .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__trunch",
                                .down => "__floorh",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                            .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                            .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                            .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                            .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp6d, ._, ._, ._ },
                            .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                            .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                            .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ss, .div, .dst0x, .src0x, .src1d, ._ },
                            .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = direction, .precision = .inexact }) },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mut_sse, .mem, .none } },
                            .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                            .{ ._, ._ss, .round, .dst0x, .dst0d, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncf",
                                .down => "floorf",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                            .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ps, .div, .dst0x, .src0x, .src1x, ._ },
                            .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mut_sse, .mem, .none } },
                            .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, ._ps, .div, .dst0x, .src1x, ._, ._ },
                            .{ ._, ._ps, .round, .dst0x, .dst0x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                            .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncf",
                                .down => "floorf",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._ss, .div, .tmp1x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                            .{ ._, ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                            .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_ps, .div, .dst0y, .src0y, .src1y, ._ },
                            .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                            .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                            .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._ps, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._ps, .round, .tmp1x, .tmp1x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_sd, .div, .dst0x, .src0x, .src1q, ._ },
                            .{ ._, .v_sd, .round, .dst0x, .dst0x, .dst0q, .rm(.{ .direction = direction, .precision = .inexact }) },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mut_sse, .mem, .none } },
                            .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                            .{ ._, ._sd, .round, .dst0x, .dst0q, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse2, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "trunc",
                                .down => "floor",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                            .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "trunc",
                                .down => "floor",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_pd, .div, .dst0x, .src0x, .src1x, ._ },
                            .{ ._, .v_pd, .round, .dst0x, .dst0x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mut_sse, .mem, .none } },
                            .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, ._pd, .div, .dst0x, .src1x, ._, ._ },
                            .{ ._, ._pd, .round, .dst0x, .dst0x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                            .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_sse, .mem, .none } },
                            .{ .src = .{ .to_sse, .to_sse, .none } },
                        },
                        .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                        .each = .{ .once = &.{
                            .{ ._, .v_pd, .div, .dst0y, .src0y, .src1y, ._ },
                            .{ ._, .v_pd, .round, .dst0y, .dst0y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                            .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, .v_pd, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                            .{ ._, .v_pd, .round, .tmp1y, .tmp1y, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse4_1, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._pd, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._pd, .round, .tmp1x, .tmp1x, .rm(.{ .direction = direction, .precision = .inexact }), ._ },
                            .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse2, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "trunc",
                                .down => "floor",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._sd, .div, .tmp1x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                            .{ ._, ._sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                            .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "trunc",
                                .down => "floor",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                            .{ ._, ._ps, .xor, .tmp2x, .tmp2x, ._, ._ },
                            .{ ._, ._ps, .movl, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._ps, .movl, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, ._ps, .movl, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .x87, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .mem, .mem, .none } },
                        },
                        .call_frame = .{ .size = 16, .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .f80, .kind = .{ .reg = .st6 } },
                            .{ .type = .f80, .kind = .{ .reg = .st7 } },
                            .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__truncx",
                                .down => "__floorx",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .reg = .st0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                            .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                            .{ ._, .f_p, .div, ._, ._, ._, ._ },
                            .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .x87, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_x87, .mem, .none } },
                        },
                        .call_frame = .{ .size = 16, .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .f80, .kind = .{ .reg = .st7 } },
                            .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__truncx",
                                .down => "__floorx",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .reg = .st0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                            .{ ._, .f_, .divr, .tmp0t, .src1t, ._, ._ },
                            .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                            .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .x87, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .mem, .to_x87, .none } },
                            .{ .src = .{ .to_x87, .to_x87, .none } },
                        },
                        .call_frame = .{ .size = 16, .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .f80, .kind = .{ .reg = .st7 } },
                            .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__truncx",
                                .down => "__floorx",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .reg = .st0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                            .{ ._, .f_, .div, .tmp0t, .src1t, ._, ._ },
                            .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                            .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .x87, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .size = 16, .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f80, .kind = .{ .reg = .st6 } },
                            .{ .type = .f80, .kind = .{ .reg = .st7 } },
                            .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "__truncx",
                                .down => "__floorx",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                            .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                            .{ ._, .f_p, .div, ._, ._, ._, ._ },
                            .{ ._, .f_p, .st, .mem(.tmp3t), ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                            .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                            .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncq",
                                .down => "floorq",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .{ .ref = .src0 }, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .avx, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncq",
                                .down => "floorq",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse2, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncq",
                                .down => "floorq",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    }, .{
                        .required_features = .{ .sse, null, null, null },
                        .src_constraints = .{
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                            .any,
                        },
                        .patterns = &.{
                            .{ .src = .{ .to_mem, .to_mem, .none } },
                        },
                        .call_frame = .{ .alignment = .@"16" },
                        .extra_temps = .{
                            .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                            .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                            .{ .type = .usize, .kind = .{ .symbol = &.{ .name = switch (direction) {
                                else => unreachable,
                                .zero => "truncq",
                                .down => "floorq",
                            } } } },
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                            .unused,
                        },
                        .dst_temps = .{ .mem, .unused },
                        .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                        .each = .{ .once = &.{
                            .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                            .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                            .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                            .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                            .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                            .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                            .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                        } },
                    } },
                }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .div_floor => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, air_tag) else fallback: {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                if (cg.floatBits(cg.typeOf(bin_op.lhs).scalarType(zcu)) == null) break :fallback try cg.airMulDivBinOp(inst, air_tag);
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .div, .dst0x, .dst0x, .tmp0d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0x, .dst0q, ._, ._ },
                        .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = .down, .precision = .inexact }) },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .{ .scalar_float = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_4_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0x, .src1q, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0x, .dst0x, .tmp0x, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0x, .dst0q, ._, ._ },
                        .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .vector_8_f32, .kind = .{ .mut_rc = .{ .ref = .src1, .rc = .sse } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .cvtph2, .dst0y, .src0x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp0y, .src1x, ._, ._ },
                        .{ ._, .v_ps, .div, .dst0y, .dst0y, .tmp0y, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .dst0y, .dst0x, ._, ._ },
                        .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0x, .dst0y, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .cvtph2, .tmp1y, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp2y, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .tmp2y, ._ },
                        .{ ._, .v_, .cvtps2ph, .tmp1x, .tmp1y, .rm(.{}), ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp1y, .tmp1x, ._, ._ },
                        .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, .v_, .cvtps2ph, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1y, .rm(.{}), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp5d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp5w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divhf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp6d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ss, .div, .dst0x, .src0x, .src1d, ._ },
                        .{ ._, .v_ss, .round, .dst0x, .dst0x, .dst0d, .rm(.{ .direction = .down, .precision = .inexact }) },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                        .{ ._, ._ss, .round, .dst0x, .dst0d, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .div, .dst0x, .src1d, ._, ._ },
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .v_ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .div, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._ps, .round, .dst0x, .dst0x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ss, .div, .tmp1x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_ps, .div, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .v_ps, .round, .dst0y, .dst0y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ps, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ps, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_ps, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, .v_ps, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .round, .tmp1x, .tmp1x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_sd, .div, .dst0x, .src0x, .src1q, ._ },
                        .{ ._, .v_sd, .round, .dst0x, .dst0x, .dst0q, .rm(.{ .direction = .down, .precision = .inexact }) },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                        .{ ._, ._sd, .round, .dst0x, .dst0q, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floor" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._sd, .div, .dst0x, .src1q, ._, ._ },
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floor" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0x, .src0x, .src1x, ._ },
                        .{ ._, .v_pd, .round, .dst0x, .dst0x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._pd, .div, .dst0x, .src1x, ._, ._ },
                        .{ ._, ._pd, .round, .dst0x, .dst0x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .mut_rc = .{ .ref = .src0, .rc = .sse } }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .v_pd, .div, .dst0y, .src0y, .src1y, ._ },
                        .{ ._, .v_pd, .round, .dst0y, .dst0y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .yword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_pd, .mova, .tmp1y, .memia(.src0y, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_pd, .div, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_unaligned_size), ._ },
                        .{ ._, .v_pd, .round, .tmp1y, .tmp1y, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, .v_pd, .mova, .memia(.dst0y, .tmp0, .add_unaligned_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_2_f64, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._pd, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .div, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._pd, .round, .tmp1x, .tmp1x, .rm(.{ .direction = .down, .precision = .inexact }), ._ },
                        .{ ._, ._pd, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floor" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._sd, .div, .tmp1x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, ._sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divdf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floor" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._ps, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._ps, .movl, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .movl, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ps, .movl, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .f_, .ld, .src0t, ._, ._, ._ },
                        .{ ._, .f_, .ld, .src1t, ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .x87, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .st6 } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__floorx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .f_, .ld, .memia(.src0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_, .ld, .memia(.src1t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .div, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp3t), ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__divtf3" } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "floorq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .rem, .rem_optimized => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, .rem) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .ah }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .movsx, .dst0d, .src0b, ._, ._ },
                        .{ ._, .i_, .div, .src1b, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .mem, .mem, .none } },
                        .{ .src = .{ .to_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_gpr, .none } },
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .ah }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .movzx, .dst0d, .src0b, ._, ._ },
                        .{ ._, ._, .div, .src1b, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .ax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .ax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .dx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cwd, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .src1w, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .ax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .ax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .dx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .dst0d, .dst0d, ._, ._ },
                        .{ ._, ._, .div, .src1w, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .eax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .eax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .edx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cdq, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .src1d, ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .eax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .eax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .edx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .dst0d, .dst0d, ._, ._ },
                        .{ ._, ._, .div, .src1d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .rax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .rax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .rdx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cqo, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .src1q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .rax }, .mem, .none } },
                        .{ .src = .{ .{ .to_reg = .rax }, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .reg = .rdx }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .xor, .dst0q, .dst0q, ._, ._ },
                        .{ ._, ._, .div, .src1q, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .signed_int = .xword }, .{ .signed_int = .xword }, .any },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_gpr_pair = .{ .cc = .ccc, .index = 0 } },
                            .{ .to_param_gpr_pair = .{ .cc = .ccc, .index = 2 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__modti3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ret_gpr_pair = .{ .cc = .ccc, .index = 0 } }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .xword }, .{ .unsigned_int = .xword }, .any },
                    .patterns = &.{
                        .{ .src = .{
                            .{ .to_param_gpr_pair = .{ .cc = .ccc, .index = 0 } },
                            .{ .to_param_gpr_pair = .{ .cc = .ccc, .index = 2 } },
                            .none,
                        } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__umodti3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ret_gpr_pair = .{ .cc = .ccc, .index = 0 } }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mem, .to_mut_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__modei4" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.src0), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .mem(.src1), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .sa(.src0, .add_8_size), ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .remainder_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .remainder_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__umodei4" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.src0), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .mem(.src1), ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .sa(.src0, .add_8_size), ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .reg = .ah } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .div, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .reg = .ah } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .i_, .div, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .reg = .ah } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .div, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .reg = .ah } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .div, .memia(.src1b, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_unaligned_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .reg = .ax } },
                        .{ .type = .i16, .kind = .{ .reg = .dx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cwd, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp2w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .reg = .ax } },
                        .{ .type = .u16, .kind = .{ .reg = .dx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .div, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp2w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .reg = .eax } },
                        .{ .type = .i32, .kind = .{ .reg = .edx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cdq, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp2d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .reg = .eax } },
                        .{ .type = .u32, .kind = .{ .reg = .edx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .div, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp2d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i64, .kind = .{ .reg = .rax } },
                        .{ .type = .i64, .kind = .{ .reg = .rdx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .cqo, ._, ._, ._, ._ },
                        .{ ._, .i_, .div, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .reg = .rax } },
                        .{ .type = .u64, .kind = .{ .reg = .rdx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .xor, .tmp2d, .tmp2d, ._, ._ },
                        .{ ._, ._, .div, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp2q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .i64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .i64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__modti3" } } },
                        .{ .type = .u64, .kind = .{ .ret_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .memiad(.src0q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp6q, ._, ._ },
                        .{ ._, ._, .mov, .memiad(.dst0q, .tmp0, .add_unaligned_size, 8), .tmp3q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .u64, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__umodti3" } } },
                        .{ .type = .u64, .kind = .{ .ret_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .memiad(.src0q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp6q, ._, ._ },
                        .{ ._, ._, .mov, .memiad(.dst0q, .tmp0, .add_unaligned_size, 8), .tmp3q, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mem, .to_mut_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__modei4" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.dst0, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src0, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.src1, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sa(.src0, .add_8_elem_size), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{
                        .{ .scalar_remainder_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_remainder_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mem, .to_mut_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 0 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 1 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 2 } } },
                        .{ .type = .usize, .kind = .{ .param_gpr = .{ .cc = .ccc, .index = 3 } } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__umodei4" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .memia(.dst0, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp2p, .memia(.src0, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .lea, .tmp3p, .memia(.src1, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sa(.src0, .add_8_elem_size), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .sa(.src0, .add_elem_size), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp2x, .tmp2x, .tmp2x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp2x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, .vp_w, .insr, .tmp2x, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, .p_w, .insr, .tmp2x, .memia(.src1w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .p_w, .extr, .tmp4d, .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp4w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ss, .mov, .tmp2x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ss, .mov, .tmp2x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_sd, .mov, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._sd, .mov, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._ps, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._ps, .movl, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .movl, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .movl, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, .x87, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, .x87, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .x87, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, .x87, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, .x87, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .x87, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .call, .tmp0d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .mod, .mod_optimized => |air_tag| if (use_old) try cg.airMulDivBinOp(inst, .mod) else fallback: {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                if (cg.floatBits(cg.typeOf(bin_op.lhs).scalarType(zcu)) == null) break :fallback try cg.airMulDivBinOp(inst, .mod);
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .required_features = .{ .f16c, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .src0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .src1x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .add, .dst0x, .src0x, .src1d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .src0x, .src0q, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .src1x, .src1q, ._, ._ },
                        .{ ._, .v_ss, .add, .dst0x, .src0x, .src1d, ._ },
                        .{ ._, .v_, .cvtps2ph, .dst0q, .dst0x, .rm(.{}), ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .mov, .mem(.tmp0d), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .mem(.tmp0d), ._, ._ },
                        .{ ._, ._, .@"and", .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._ss, .mov, .src1x, .mem(.tmp0d), ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .{ .scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .mov, .mem(.tmp0d), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .mem(.tmp0d), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp3w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._ss, .mov, .src1x, .mem(.tmp0d), ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp3x, .tmp3x, ._, ._ },
                        .{ ._, .v_ss, .add, .tmp1x, .tmp1x, .tmp3d, ._ },
                        .{ ._, .v_, .cvtps2ph, .tmp1q, .tmp1x, .rm(.{}), ._ },
                        .{ .@"1:", .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .f16c, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .v_ps, .cvtph2, .tmp3x, .tmp3x, ._, ._ },
                        .{ ._, .v_ss, .add, .tmp1x, .tmp1x, .tmp3d, ._ },
                        .{ ._, .v_, .cvtps2ph, .tmp1q, .tmp1x, .rm(.{}), ._ },
                        .{ .@"1:", .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .vp_, .xor, .tmp1x, .tmp1x, .tmp1x, ._ },
                        .{ ._, .vp_w, .insr, .tmp1x, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0) },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", .vp_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", .p_w, .extr, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1x, .ui(0), ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp5w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f16, .kind = .{ .reg = .dx } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .p_, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, .p_w, .insr, .tmp1x, .memia(.src0w, .tmp0, .add_unaligned_size), .ui(0), ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp2d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp3x, .tmp2d, ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp5w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .fast_imm16, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp6d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .@"and", .tmp1w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .mem(.tmp6d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp6d), .tmp3x, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp1d, .mem(.tmp6d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_float = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f16, .kind = .{ .reg = .ax } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f16, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f16, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodh" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addhf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0x, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp3x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .movzx, .tmp1d, .memia(.src1w, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .mov, .mem(.tmp2d), .tmp1d, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp6d), .tmp3x, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp1d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .xor, .tmp1d, .mem(.tmp6d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._ss, .mov, .tmp4x, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp6d), .tmp3x, ._, ._ },
                        .{ .@"1:", ._, .mov, .tmp1d, .mem(.tmp6d), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_unaligned_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .{ .reg = .edx } },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, .v_ss, .add, .dst0x, .src0x, .src1d, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .{ .reg = .edx } },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._d, .mov, .tmp0d, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp2d, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .tmp2d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .src1x, .tmp0d, ._, ._ },
                        .{ ._, ._ss, .add, .dst0x, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ss, .mov, .mem(.tmp0d), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp2d), .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp3d, .tmp0d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp3d, .mem(.tmp2d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp3d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._ss, .add, .dst0x, .mem(.tmp0d), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .f32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .{ .reg = .edx } },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_ss, .mov, .tmp2x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_d, .mov, .tmp3d, .tmp2x, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp3d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_d, .mov, .tmp2x, .tmp3d, ._, ._ },
                        .{ ._, .v_ss, .add, .tmp1x, .tmp1x, .tmp2d, ._ },
                        .{ .@"1:", .v_ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .f32, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .{ .reg = .edx } },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ss, .mov, .tmp2x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._d, .mov, .tmp3d, .tmp2x, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp5d, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6d, .tmp3d, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp6d, .tmp5d, ._, ._ },
                        .{ ._, ._, .cmp, .tmp6d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._d, .mov, .tmp2x, .tmp3d, ._, ._ },
                        .{ ._, ._ss, .add, .tmp1x, .tmp2d, ._, ._ },
                        .{ .@"1:", ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_float = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f32, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodf" } } },
                        .{ .type = .f32, .kind = .mem },
                        .{ .type = .f32, .kind = .{ .reg = .eax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ss, .mov, .tmp1x, .memia(.src0d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ss, .mov, .tmp2x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._ss, .mov, .mem(.tmp4d), .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp5d, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5d, .mem(.tmp4d), ._, ._ },
                        .{ ._, ._, .cmp, .tmp5d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._ss, .add, .tmp1x, .memia(.src1d, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ .@"1:", ._ss, .mov, .memia(.dst0d, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .{ .reg = .rcx } },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_q, .mov, .tmp0q, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .v_q, .mov, .tmp3q, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp0q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._, .cmp, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_q, .mov, .src1q, .tmp0q, ._, ._ },
                        .{ ._, .v_sd, .add, .dst0x, .src0x, .src1q, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .{ .reg = .rcx } },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._q, .mov, .tmp0q, .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._q, .mov, .tmp3q, .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp0q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._, .cmp, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._q, .mov, .src1x, .tmp0q, ._, ._ },
                        .{ ._, ._sd, .add, .dst0x, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f64, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .mem },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .movl, .mem(.tmp0q), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp3q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._ps, .movl, .mem(.tmp2q), .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp0q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .mem(.tmp2q), ._, ._ },
                        .{ ._, ._, .cmp, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._na, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .mem(.tmp2q), ._, ._, ._ },
                        .{ ._, .f_, .add, .mem(.tmp0q), ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2q), ._, ._, ._ },
                        .{ ._, ._ps, .xor, .dst0x, .dst0x, ._, ._ },
                        .{ ._, ._ps, .movl, .dst0x, .mem(.tmp2q), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .f64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .{ .reg = .rcx } },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_sd, .mov, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_q, .mov, .tmp3q, .tmp2x, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .v_q, .mov, .tmp6q, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp7q, .tmp3q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp7q, .tmp5q, ._, ._ },
                        .{ ._, ._, .xor, .tmp7q, .tmp6q, ._, ._ },
                        .{ ._, ._, .cmp, .tmp7q, .tmp5q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_q, .mov, .tmp2x, .tmp3q, ._, ._ },
                        .{ ._, .v_sd, .add, .tmp1x, .tmp1x, .tmp2q, ._ },
                        .{ .@"1:", .v_sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .f64, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .{ .reg = .rcx } },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._sd, .mov, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._sd, .mov, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._q, .mov, .tmp3q, .tmp2x, ._, ._ },
                        .{ ._, ._, .call, .tmp4d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._q, .mov, .tmp6q, .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp7q, .tmp3q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp7q, .tmp5q, ._, ._ },
                        .{ ._, ._, .xor, .tmp7q, .tmp6q, ._, ._ },
                        .{ ._, ._, .cmp, .tmp7q, .tmp5q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._q, .mov, .tmp2x, .tmp3q, ._, ._ },
                        .{ ._, ._sd, .add, .tmp1x, .tmp2q, ._, ._ },
                        .{ .@"1:", ._sd, .mov, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .{ .multiple_scalar_float = .{ .of = .qword, .is = .qword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f64, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmod" } } },
                        .{ .type = .f64, .kind = .{ .reg = .rdx } },
                        .{ .type = .f64, .kind = .mem },
                        .{ .type = .f64, .kind = .{ .reg = .rax } },
                        .{ .type = .f64, .kind = .{ .reg = .st6 } },
                        .{ .type = .f64, .kind = .{ .reg = .st7 } },
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._ps, .xor, .tmp2x, .tmp2x, ._, ._ },
                        .{ ._, ._ps, .movl, .tmp1x, .memia(.src0q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .movl, .tmp2x, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._ps, .movl, .mem(.tmp5q), .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6q, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .@"and", .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._, .xor, .tmp6q, .mem(.tmp5q), ._, ._ },
                        .{ ._, ._, .cmp, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._na, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .mem(.tmp5q), ._, ._, ._ },
                        .{ ._, .f_, .add, .memia(.src1q, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp5q), ._, ._, ._ },
                        .{ ._, ._ps, .xor, .tmp1x, .tmp1x, ._, ._ },
                        .{ ._, ._ps, .movl, .tmp1x, .mem(.tmp5q), ._, ._ },
                        .{ .@"1:", ._ps, .movl, .memia(.dst0q, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(8), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp4d, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, .x87, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp4d, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, .x87, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp4d, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, .x87, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .reg = .st0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src0x), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp1x), .tmp0x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp0x, .mem(.src1x), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp1x, 16), .tmp0x, ._, ._ },
                        .{ ._, ._, .call, .tmp2d, ._, ._, ._ },
                        .{ ._, .f_, .ld, .dst0t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp1t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp4w, .memd(.tmp1w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4w, .memd(.tmp1w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp1q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp1t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp5d, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, .x87, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp5d, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, .x87, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, .x87, .fast_imm16 },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .movzx, .tmp5d, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, .x87, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .tbyte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .size = 16 * 2, .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f80, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f80, .kind = .{ .frame = .call_frame } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__fmodx" } } },
                        .{ .type = .f80, .kind = .{ .reg = .st7 } },
                        .{ .type = .f80, .kind = .{ .reg = .rax } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp2x), .tmp1x, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp1x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .memd(.tmp2x, 16), .tmp1x, ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ .pseudo, .f_cstp, .de, ._, ._, ._, ._ },
                        .{ ._, .f_, .ld, .tmp4t, ._, ._, ._ },
                        .{ ._, .f_p, .st, .mem(.tmp2t), ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp5d, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._, .@"and", .tmp5w, .memd(.tmp2w, 16 + 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp5w, .memd(.tmp2w, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp2q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp5w, .sa(.src0, .add_smin), ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .f_, .ld, .memd(.tmp2t, 16), ._, ._, ._ },
                        .{ ._, .f_p, .add, ._, ._, ._, ._ },
                        .{ .@"1:", .f_p, .st, .memia(.dst0t, .tmp0, .add_unaligned_size), ._, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, .v_dqa, .mov, .mem(.tmp0x), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .vp_q, .extr, .tmp3q, .dst0x, .ui(1), ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .memd(.tmp0q, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, .v_q, .mov, .tmp3q, .dst0x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .src1x, .mem(.tmp0x), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse4_1, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .mem(.tmp0x), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .p_q, .extr, .tmp3q, .dst0x, .ui(1), ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .memd(.tmp0q, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._q, .mov, .tmp3q, .dst0x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .src1x, .mem(.tmp0x), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .mem(.tmp0x), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .p_d, .shuf, .src1x, .dst0x, .ui(0b11_10_11_10), ._ },
                        .{ ._, ._q, .mov, .tmp3q, .src1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .memd(.tmp0q, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .tmp3q, ._, ._ },
                        .{ ._, ._q, .mov, .tmp3q, .dst0x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp3q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .src1x, .mem(.tmp0x), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, null, null },
                    .src_constraints = .{
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .{ .to_reg = .xmm0 }, .{ .to_reg = .xmm1 }, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._ps, .mova, .mem(.tmp0x), .src1x, ._, ._ },
                        .{ ._, ._, .call, .tmp1d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp2q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp3x), .dst0x, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp4q, .memd(.tmp0q, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp4q, .memd(.tmp3q, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp3q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp4q, .tmp2q, ._, ._ },
                        .{ ._, ._nae, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._ps, .mova, .src1x, .mem(.tmp0x), ._, ._ },
                        .{ ._, ._, .call, .tmp5d, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .avx, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .vp_q, .extr, .tmp5q, .tmp1x, .ui(1), ._ },
                        .{ ._, ._, .mov, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, .v_q, .mov, .tmp5q, .tmp1x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp5q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, .v_dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse4_1, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .p_q, .extr, .tmp5q, .tmp1x, .ui(1), ._ },
                        .{ ._, ._, .mov, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._q, .mov, .tmp5q, .tmp1x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp5q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse2, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rcx } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, .p_d, .shuf, .tmp2x, .tmp1x, .ui(0b11_10_11_10), ._ },
                        .{ ._, ._q, .mov, .tmp5q, .tmp2x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6q, .tmp5q, ._, ._ },
                        .{ ._, ._q, .mov, .tmp5q, .tmp1x, ._, ._ },
                        .{ ._, ._, .cmp, .tmp5q, .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", ._dqa, .mov, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .sse, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .{ .multiple_scalar_float = .{ .of = .xword, .is = .xword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .call_frame = .{ .alignment = .@"16" },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm0 } },
                        .{ .type = .f128, .kind = .{ .reg = .xmm1 } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "fmodq" } } },
                        .{ .type = .f128, .kind = .{ .reg = .rdx } },
                        .{ .type = .f128, .kind = .mem },
                        .{ .type = .f128, .kind = .{ .reg = .rax } },
                        .{ .type = .usize, .kind = .{ .symbol = &.{ .name = "__addtf3" } } },
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true, .caller_preserved = .ccc },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_unaligned_size), ._, ._ },
                        .{ .@"0:", ._ps, .mova, .tmp1x, .memia(.src0x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp3d, ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp4q, .ua(.src0, .add_smin), ._, ._ },
                        .{ ._, ._ps, .mova, .mem(.tmp5x), .tmp1x, ._, ._ },
                        .{ ._, ._, .mov, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._, .@"and", .tmp6q, .memiad(.src1q, .tmp0, .add_unaligned_size, 8), ._, ._ },
                        .{ ._, ._, .xor, .tmp6q, .memd(.tmp5q, 8), ._, ._ },
                        .{ ._, ._, .cmp, .mem(.tmp5q), .si(1), ._, ._ },
                        .{ ._, ._, .sbb, .tmp6q, .tmp4q, ._, ._ },
                        .{ ._, ._nae, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._ps, .mova, .tmp2x, .memia(.src1x, .tmp0, .add_unaligned_size), ._, ._ },
                        .{ ._, ._, .call, .tmp7d, ._, ._, ._ },
                        .{ .@"1:", ._ps, .mova, .memia(.dst0x, .tmp0, .add_unaligned_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                };
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .ptr_add => |air_tag| if (use_old) try cg.airPtrArithmetic(inst, air_tag) else {
                const ty_pl = air_datas[@intFromEnum(inst)].ty_pl;
                const bin_op = cg.air.extraData(Air.Bin, ty_pl.payload).data;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                try ops[0].toSlicePtr(cg);
                var res: [1]Temp = undefined;
                if (!hack_around_sema_opv_bugs or ty_pl.ty.toType().elemType2(zcu).hasRuntimeBitsIgnoreComptime(zcu)) cg.select(&res, &.{ty_pl.ty.toType()}, &ops, comptime &.{ .{
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .simm32, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leaa(.src0, .add_src0_elem_size_mul_src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 2 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"2", .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 2 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"2", .src1), ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 4 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"4", .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 4 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"4", .src1), ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .dst_constraints = .{ .{ .elem_size_is = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"8", .src1), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .dst_constraints = .{ .{ .elem_size_is = 8 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"8", .src1), ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .po2_elem_size, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._l, .sh, .src1p, .sa(.none, .add_log2_src0_elem_size), ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .src1), ._, ._ },
                    } },
                }, .{
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0p, .src1p, .sa(.none, .add_src0_elem_size), ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                } else res[0] = ops[0];
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .ptr_sub => |air_tag| if (use_old) try cg.airPtrArithmetic(inst, air_tag) else {
                const ty_pl = air_datas[@intFromEnum(inst)].ty_pl;
                const bin_op = cg.air.extraData(Air.Bin, ty_pl.payload).data;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                try ops[0].toSlicePtr(cg);
                var res: [1]Temp = undefined;
                if (!hack_around_sema_opv_bugs or ty_pl.ty.toType().elemType2(zcu).hasRuntimeBitsIgnoreComptime(zcu)) cg.select(&res, &.{ty_pl.ty.toType()}, &ops, comptime &.{ .{
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .simm32, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leaa(.src0, .sub_src0_elem_size_mul_src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .neg, .src1p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 2 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .neg, .src1p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"2", .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 2 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"2", .src1), ._, ._ },
                        .{ ._, ._, .neg, .dst0p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 4 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .neg, .src1p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"4", .src1), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .{ .elem_size_is = 4 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"4", .src1), ._, ._ },
                        .{ ._, ._, .neg, .dst0p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .dst_constraints = .{ .{ .elem_size_is = 8 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .neg, .src1p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leasi(.src0, .@"8", .src1), ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .dst_constraints = .{ .{ .elem_size_is = 8 + 1 }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .dst0p, .leasi(.src1, .@"8", .src1), ._, ._ },
                        .{ ._, ._, .neg, .dst0p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                }, .{
                    .dst_constraints = .{ .po2_elem_size, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_mut_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src1 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._l, .sa, .src1p, .sa(.none, .add_log2_src0_elem_size), ._, ._ },
                        .{ ._, ._, .neg, .src1p, ._, ._, ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .src1), ._, ._ },
                    } },
                }, .{
                    .patterns = &.{
                        .{ .src = .{ .to_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .general_purpose }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, .i_, .mul, .dst0p, .src1p, .sa(.none, .sub_src0_elem_size), ._ },
                        .{ ._, ._, .lea, .dst0p, .leai(.src0, .dst0), ._, ._ },
                    } },
                } }) catch |err| switch (err) {
                    error.SelectFailed => return cg.fail("failed to select {s} {} {} {}", .{
                        @tagName(air_tag),
                        cg.typeOf(bin_op.lhs).fmt(pt),
                        ops[0].tracking(cg),
                        ops[1].tracking(cg),
                    }),
                    else => |e| return e,
                } else res[0] = ops[0];
                try res[0].finish(inst, &.{ bin_op.lhs, bin_op.rhs }, &ops, cg);
            },
            .max => |air_tag| if (use_old) try cg.airBinOp(inst, air_tag) else {
                const bin_op = air_datas[@intFromEnum(inst)].bin_op;
                var ops = try cg.tempsFromOperands(inst, .{ bin_op.lhs, bin_op.rhs });
                var res: [1]Temp = undefined;
                cg.select(&res, &.{cg.typeOf(bin_op.lhs)}, &ops, comptime &.{ .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0b, .src1b, ._, ._ },
                        .{ ._, ._l, .cmov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .byte }, .{ .signed_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0b, .src1b, ._, ._ },
                        .{ ._, ._nl, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0b, .src1b, ._, ._ },
                        .{ ._, ._b, .cmov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .byte }, .{ .unsigned_int = .byte }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0b, .src1b, ._, ._ },
                        .{ ._, ._nb, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0b, .src1b, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0w, .src1w, ._, ._ },
                        .{ ._, ._l, .cmov, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .word }, .{ .signed_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0w, .src1w, ._, ._ },
                        .{ ._, ._nl, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0w, .src1w, ._, ._ },
                        .{ ._, ._b, .cmov, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .word }, .{ .unsigned_int = .word }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0w, .src1w, ._, ._ },
                        .{ ._, ._nb, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0w, .src1w, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0d, .src1d, ._, ._ },
                        .{ ._, ._l, .cmov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .signed_int = .dword }, .{ .signed_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0d, .src1d, ._, ._ },
                        .{ ._, ._nl, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0d, .src1d, ._, ._ },
                        .{ ._, ._b, .cmov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{ .{ .unsigned_int = .dword }, .{ .unsigned_int = .dword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0d, .src1d, ._, ._ },
                        .{ ._, ._nb, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0d, .src1d, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0q, .src1q, ._, ._ },
                        .{ ._, ._l, .cmov, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .signed_int = .qword }, .{ .signed_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0q, .src1q, ._, ._ },
                        .{ ._, ._nl, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0q, .src1q, ._, ._ },
                        .{ ._, ._b, .cmov, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .{ .unsigned_int = .qword }, .{ .unsigned_int = .qword }, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_gpr, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_gpr, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_gpr, .to_gpr, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .cmp, .src0q, .src1q, ._, ._ },
                        .{ ._, ._nb, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .mov, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .any_signed_int, .any_signed_int, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rsi } },
                        .{ .type = .u64, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memad(.src1q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src0), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.src1), ._, ._ },
                        .{ ._, ._l, .cmov, .tmp0p, .tmp1p, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .mov, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .any_signed_int, .any_signed_int, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rsi } },
                        .{ .type = .u64, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sia(1, .src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsiad(.src0q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsiad(.src1q, .@"8", .tmp0, .add_size, -8), ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1q, .memad(.src0q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memad(.src1q, .add_size, -8), ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src0), ._, ._ },
                        .{ ._, ._nl, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src1), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .mov, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", .cmov, null, null },
                    .src_constraints = .{ .any_unsigned_int, .any_unsigned_int, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rsi } },
                        .{ .type = .u64, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src0), ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.src1), ._, ._ },
                        .{ ._, ._b, .cmov, .tmp0p, .tmp1p, ._, ._ },
                        .{ ._, ._, .lea, .tmp1p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .mov, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .@"64bit", null, null, null },
                    .src_constraints = .{ .any_unsigned_int, .any_unsigned_int, .any },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .reg = .rsi } },
                        .{ .type = .u64, .kind = .{ .reg = .rdi } },
                        .{ .type = .u64, .kind = .{ .reg = .rcx } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size_div_8), ._, ._ },
                        .{ ._, ._c, .cl, ._, ._, ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1q, .memsia(.src0q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .sbb, .tmp1q, .memsia(.src1q, .@"8", .tmp0, .add_size), ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src0), ._, ._ },
                        .{ ._, ._nb, .j, .@"0f", ._, ._, ._ },
                        .{ ._, ._, .lea, .tmp0p, .mem(.src1), ._, ._ },
                        .{ .@"0:", ._, .lea, .tmp1p, .mem(.dst0), ._, ._ },
                        .{ ._, ._, .mov, .tmp2d, .sa(.src0, .add_size_div_8), ._, ._ },
                        .{ ._, .@"rep _sq", .mov, ._, ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .maxs, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .maxs, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_b, .cmpgt, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"and", .src0x, .dst0x, ._, ._ },
                        .{ ._, .p_, .andn, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"or", .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .maxs, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_b, .maxs, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_b, .maxs, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_b, .maxs, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_16_i8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, .p_b, .cmpgt, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_, .@"and", .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .andn, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_, .@"or", .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .movsx, .tmp2d, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._l, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .movsx, .tmp2d, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._l, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nl, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_signed_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nl, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .qword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .maxu, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .maxu, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_b, .maxu, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_b, .maxu, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_32_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_b, .maxu, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_b, .maxu, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u8, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_b, .maxu, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, .slow_incdec, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._b, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .movzx, .tmp2d, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .tmp2b, ._, ._ },
                        .{ ._, ._b, .cmov, .tmp1d, .tmp2d, ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .slow_incdec, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nb, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(1), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .byte, .is = .byte } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u8, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nb, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1b, .memia(.src1b, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0b, .tmp0, .add_size), .tmp1b, ._, ._ },
                        .{ ._, ._c, .in, .tmp0p, ._, ._, ._ },
                        .{ ._, ._nz, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse, .mmx, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .qword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_mmx, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_mmx, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_mmx, .to_mmx, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .maxs, .dst0q, .src1q, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .maxs, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .maxs, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .maxs, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_w, .maxs, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_w, .maxs, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_i16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_w, .maxs, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._l, .cmov, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_signed_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movsx, .tmp1d, .memia(.src0w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nl, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .maxu, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .maxu, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_w, .subus, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_w, .add, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_w, .maxu, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_16_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_w, .maxu, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_w, .maxu, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_w, .maxu, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u16, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_w, .subus, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_w, .add, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._b, .cmov, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0w, .tmp0, .add_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .word, .is = .word } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u16, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .movzx, .tmp1d, .memia(.src0w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nb, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1w, .memia(.src1w, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0w, .tmp0, .add_size), .tmp1w, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(2), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxs, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .maxs, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_or_exclusive_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_signed_or_exclusive_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._dqa, .mov, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"and", .src0x, .dst0x, ._, ._ },
                        .{ ._, .p_, .andn, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"or", .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxs, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_d, .maxs, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_d, .maxs, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_d, .maxs, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_or_exclusive_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_signed_or_exclusive_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_i32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_, .@"and", .tmp2x, .tmp1x, ._, ._ },
                        .{ ._, .p_, .andn, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_, .@"or", .tmp1x, .tmp2x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._l, .cmov, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_signed_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .i32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._nl, .j, .@"1f", ._, ._, ._ },
                        .{ ._, ._, .mov, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ .@"1:", ._, .mov, .memia(.dst0d, .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxu, .dst0x, .src0x, .src1x, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .ref = .src0 }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .p_d, .maxu, .dst0x, .src1x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mut_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_mut_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_mut_sse, .to_sse, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .usize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .dst0x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .dst0x, ._, ._ },
                        .{ ._, .p_, .xor, .dst0x, .src0x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp2x, .src1x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .dst0x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .@"and", .src0x, .dst0x, ._, ._ },
                        .{ ._, .p_, .andn, .dst0x, .src1x, ._, ._ },
                        .{ ._, .p_, .@"or", .dst0x, .src0x, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .{ .scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_sse, .mem, .none } },
                        .{ .src = .{ .mem, .to_sse, .none }, .commute = .{ 0, 1 } },
                        .{ .src = .{ .to_sse, .to_sse, .none } },
                    },
                    .dst_temps = .{ .{ .rc = .sse }, .unused },
                    .each = .{ .once = &.{
                        .{ ._, .vp_d, .maxu, .dst0y, .src0y, .src1y, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .yword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_8_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1y, .memia(.src0y, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_d, .maxu, .tmp1y, .tmp1y, .memia(.src1y, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0y, .tmp0, .add_size), .tmp1y, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(32), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .avx, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", .v_dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .vp_d, .maxu, .tmp1x, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._ },
                        .{ ._, .v_dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse4_1, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .xword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp1x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, .p_d, .maxu, .tmp1x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp1x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .sse2, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .{ .multiple_scalar_exact_unsigned_int = .{ .of = .xword, .is = 32 } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .vector_4_u32, .kind = .{ .smin_mem = .{ .ref = .src0, .vectorize_to = .xword } } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .{ .type = .vector_4_u32, .kind = .{ .rc = .sse } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .lea, .tmp0p, .mem(.tmp1), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp2x, .lea(.tmp0x), ._, ._ },
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._dqa, .mov, .tmp3x, .memia(.src0x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp4x, .memia(.src1x, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp5x, .tmp3x, ._, ._ },
                        .{ ._, ._dqa, .mov, .tmp6x, .tmp4x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp5x, .tmp2x, ._, ._ },
                        .{ ._, .p_, .xor, .tmp6x, .tmp2x, ._, ._ },
                        .{ ._, .p_d, .cmpgt, .tmp5x, .tmp6x, ._, ._ },
                        .{ ._, .p_, .@"and", .tmp3x, .tmp5x, ._, ._ },
                        .{ ._, .p_, .andn, .tmp5x, .tmp4x, ._, ._ },
                        .{ ._, .p_, .@"or", .tmp3x, .tmp5x, ._, ._ },
                        .{ ._, ._dqa, .mov, .memia(.dst0x, .tmp0, .add_size), .tmp3x, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(16), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .required_features = .{ .cmov, null, null, null },
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind = .{ .rc = .general_purpose } },
                        .{ .type = .u32, .kind = .{ .rc = .general_purpose } },
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                        .unused,
                    },
                    .dst_temps = .{ .mem, .unused },
                    .clobbers = .{ .eflags = true },
                    .each = .{ .once = &.{
                        .{ ._, ._, .mov, .tmp0p, .sa(.src0, .sub_size), ._, ._ },
                        .{ .@"0:", ._, .mov, .tmp1d, .memia(.src0d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .cmp, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._b, .cmov, .tmp1d, .memia(.src1d, .tmp0, .add_size), ._, ._ },
                        .{ ._, ._, .mov, .memia(.dst0d, .tmp0, .add_size), .tmp1d, ._, ._ },
                        .{ ._, ._, .add, .tmp0p, .si(4), ._, ._ },
                        .{ ._, ._nc, .j, .@"0b", ._, ._, ._ },
                    } },
                }, .{
                    .src_constraints = .{
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .{ .multiple_scalar_unsigned_int = .{ .of = .dword, .is = .dword } },
                        .any,
                    },
                    .patterns = &.{
                        .{ .src = .{ .to_mem, .to_mem, .none } },
                    },
                    .extra_temps = .{
                        .{ .type = .isize, .kind 