#include "config.h"

#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <sys/file.h>
#include <fcntl.h>
#include "uostr.h"
#include "uoio.h"
#include "uogetopt.h"
#include "smtp.h"
#include "smtptools.h"
#include "mtalib.h"

char *myhostname=0;

/* user changeable options */
char *greeting=NULL;
static const char *inputfname=NULL;
static const char *delivered_to;
static const char *return_path;

static struct uogetopt myopts[]={
	{'d',"delivered-to", UOGO_STRING,&delivered_to,0, 
			"content of delivered-to line\n"
			"default: no such line is written\n"
			"useful: an address, maybe with prefix","ADDRESS"},
	{'g',"greeting", UOGO_STRING,&greeting,0, "name to use for HELO","MYHOSTNAME"},
	{'i',"input", UOGO_STRING,&inputfname,0, "file to read message from (instead of stdin)","FILENAME"},
	{'r',"return-path", UOGO_STRING,&return_path,0, 
		"content of return-path line\n"
		"default: no such line is written\n"
		"useful: an address, or empty", "ADDRESS"},
	{0,0}
};

static int 
translate_exitcode(int exitcode)
{
	switch(exitcode) {
	case SMTP_SUCCESS: return (0);
	case SMTP_DEFER: return (111);
	case SMTP_FATAL: return (100);
	default: return (111); /* should never arrive here */
	}
}

int 
main(int argc, char **argv)
{
	char buf[128];
	uoio_t rop;
	int ifd;
	const char *argv0;
	mta_t *mta;
	mta_ret_t mta_ret;
	uostr_t fname;fname.data=0;
	argv0=strrchr(argv[0],'/'); if (!argv0) argv0=argv[0]; else argv0++;

	if (gethostname(buf,sizeof(buf)-1)==-1) {
		uostr_t s;s.data=0;
		uostr_xdup_cstrmulti(&s,argv0,": hostname too long, gethostname failed: ",strerror(errno),"\n",0);
		write(2,s.data,s.len);
		_exit(translate_exitcode(SMTP_DEFER));
	} else {
		buf[sizeof(buf)-1]=0;
		myhostname=malloc(strlen(buf)+1);
		if (!myhostname) {
			write(2,argv0,strlen(argv0));
			write(2,": out of memory\n",16);
			_exit(translate_exitcode(SMTP_DEFER));
		}
		memcpy(myhostname,buf,strlen(buf)+1);
	}
	greeting=myhostname;

	uogetopt(argv0,PACKAGE,VERSION,&argc,argv,uogetopt_out,
		"usage: tomaildir [options] MAILDIR",myopts,0);

	if (!inputfname)
		ifd=0;
	else {
		ifd=open(inputfname,O_RDONLY);
		if (ifd==-1) {
			uostr_t s;s.data=0;
			uostr_xadd_cstrmulti(&s,argv0,": cannot open ",inputfname,": ",strerror(errno),"\n",0);
			write(2,s.data,s.len);
			_exit(translate_exitcode(SMTP_FATAL));
		}
	}
	
	if (!argv[1]) {
		uostr_t s;s.data=0;
		uostr_xadd_cstrmulti(&s,argv0,": need argument (maildir)\n",0);
		write(2,s.data,s.len);
		_exit(translate_exitcode(SMTP_FATAL));
	}
	if (-1==chdir(argv[1])) {
		uostr_t s;s.data=0;
		uostr_xadd_cstrmulti(&s,argv0,": cannot chdir(",argv[1],") : ",strerror(errno),"\n",0);
		write(2,s.data,s.len);
		_exit(translate_exitcode(SMTP_DEFER));
	}
	/* layering violation, but i really do not want to link in the whole mta*.c here ... */
	mta_ret=mta_maildir.mta_open(&mta,0,0,0);
	switch (mta_ret) {
	case MTA_SUCCESS: break;
	case MTA_DEFER: {
			uostr_t s;s.data=0;
			uostr_xadd_cstrmulti(&s,argv0,": ",mta_errtxt,"\n",0);
			write(2,s.data,s.len);
			_exit(translate_exitcode(SMTP_DEFER));
		}
	case MTA_FATAL: {
			uostr_t s;s.data=0;
			uostr_xadd_cstrmulti(&s,argv0,": ",mta_errtxt,"\n",0);
			write(2,s.data,s.len);
			_exit(translate_exitcode(SMTP_FATAL));
		}
	}

	if (return_path) {
		mta_ret=mta->mta_write(mta,"Return-Path: <",14);
		if (mta_ret==MTA_SUCCESS) mta_ret=mta->mta_write(mta,return_path,strlen(return_path));
		if (mta_ret==MTA_SUCCESS) mta_ret=mta->mta_write(mta,">\n",2);
		if (mta_ret!=MTA_SUCCESS) { goto bailoutlocalerr; }
	}
	if (delivered_to) {
		mta_ret=mta->mta_write(mta,"Delivered-To: ",14);
		if (mta_ret==MTA_SUCCESS) mta_ret=mta->mta_write(mta,delivered_to,strlen(delivered_to));
		if (mta_ret==MTA_SUCCESS) mta_ret=mta->mta_write(mta,"\n",1);
		if (mta_ret!=MTA_SUCCESS) goto bailoutlocalerr;
	}

	uoio_assign_r(&rop,ifd,read,0);
	while (1) {
		ssize_t len;
		char *s;
		len=uoio_getdelim_zc(&rop,&s,'\n');
		if (len==-1) {
			uostr_t m;m.data=0;
			uostr_xadd_cstrmulti(&m,"cannot read message: ",strerror(errno),"\n",0);
			write(2,m.data,m.len);
			mta->mta_abort(mta);
			_exit(translate_exitcode(SMTP_DEFER));
		}
		if (len==0) break; /* EOF */
		mta_ret=mta->mta_write(mta,s,len);
		if (mta_ret!=MTA_SUCCESS) goto bailoutlocalerr;
	}
	uoio_destroy(&rop);
	/* it's maildir, it cannot have fatal errors here */
	if (MTA_SUCCESS!=mta->mta_close(mta)) {
		uostr_t s;s.data=0;
		uostr_xadd_cstrmulti(&s,"cannot save message to maildir: ",mta_errtxt,"\n",0);
		write(2,s.data,s.len);
		_exit(translate_exitcode(SMTP_DEFER));
	}
	_exit(translate_exitcode(SMTP_SUCCESS));
  bailoutlocalerr: {
		uostr_t s;s.data=0;
		uostr_xadd_cstrmulti(&s,argv0,": ",mta_errtxt,"\n",0);
		write(2,s.data,s.len);
		mta->mta_abort(mta);
		_exit(translate_exitcode(mta_ret==MTA_DEFER ? SMTP_DEFER : SMTP_FATAL));
	}
}
