//  bbacpi.cc for bbacpi - a tool to display acpi info in X11.
//
//  Copyright (c) 2004 by Max Lekeux, m_lekeux@hotmail.com
//
//  This program is heavily based on bbsload by John Kennis, jkennis@chello.nl
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
// (See the included file COPYING / GPL-2.0)
//

#include "bbacpi.hh"

CheckLoad::CheckLoad(ToolWindow *toolwindow) 
{
  bbtool=toolwindow;
  timer=new BTimer(bbtool->getCurrentScreenInfo()->getBaseDisplay(),this) ;
  user.delay = bbtool->getResource()->report.check_delay;
  user.xmessage_delay = bbtool->getResource()->report.xmessage_delay;
  user.minlim =  bbtool->getResource()->report.minlim;
  timer->setTimeout(user.delay*1000);
  timer->start();

  task.running=0;
  task.total=0;
    
  GetFileNames();

  ReadStaticInfo();
}


void CheckLoad::GetFileNames() 
{  
  acpi_path = strdup(ACPI_PATH);
  
  if (!acpi_path) {
    fprintf(stderr, "Out of memory\n");
    exit(-1);
  }
  
  /* Batteries */
  bool bat1 = (!GetName(names.battery1state, "battery", "state", 0) 
	       && !checkbatteryfile(names.battery1state, "battery 1")
	       && !GetName(names.battery1info, "battery", "info", 0)
	       && !checkbatteryfile(names.battery1info, "battery 1"));
  bool bat2 = (!GetName(names.battery2state, "battery", "state", 1) 
	       && !checkbatteryfile(names.battery2state, "battery 2")
	       && !GetName(names.battery2info, "battery", "info", 1)
	       && !checkbatteryfile(names.battery2info, "battery 2"));
  
  if (bbtool->getResource()->show.battery_number == 1) {
    if (!bat1 && !bat2)
      bbtool->hideBattery();
  }
  else
    if ((!bat1 && bat2) || (bat1 && !bat2))
      bbtool->hide1Battery();
    else
      if (!bat1) 
	bbtool->hideBattery();
  
  /* Adapter */
  if (GetName(names.adapter, "ac_adapter", "state", 0) 
      || checkfile(names.adapter, "ac_adapter")) 
    bbtool->hideAdapter();
  
  /* Temperature */
  if (GetName(names.temperature, "thermal_zone", "temperature", 0)
      || checkfile(names.temperature, "thermal_zone")
      || GetName(names.temptrip, "thermal_zone", "trip_points", 0)
      || checkfile(names.temptrip, "thermal_zone"))
    bbtool->hideTemperature();
}


int CheckLoad::GetName(char* filepath, char *device, char *file, int number) 
{
  DIR *d;
  struct dirent *de;

  /* Go to acpi directory */
  if (chdir(acpi_path) < 0) {
    fprintf(stderr, 
	    "No ACPI support in kernel, or incorrect acpi_path (\"%s\").\n", 
	    acpi_path);
    exit(1);
  }
  
  /* Go to device directory */
  if (chdir(device) < 0) {
    fprintf(stderr, "No support for device type: %s\n", device);
    return 1;
  }

  d = opendir(".");
  if (!d) {
    fprintf(stderr, "No support for device type: %s\n", device);
    return 1;
  }

  /* Go to the good element */
  int i = 0;
  while (i < number) {
    de = readdir(d);
    if (de == NULL) {
    fprintf(stderr, "No support for device type: %s\n", device);
    return 1;
    }
    if (!ignore_directory_entry(de)) i++;
  }
  
  /* Read the correct file */
  do {
    de = readdir(d);
    if (de == NULL) {
      fprintf(stderr, "No support for device type: %s\n", device);
      return 1;
    }
  } while (ignore_directory_entry(de));

  /* Return file name */
  sprintf(filepath, "%s/%s/%s/%s", acpi_path, device, de->d_name, file);

  return 0;
}


int CheckLoad::ignore_directory_entry(struct dirent *de)
{
  return !strcmp(de->d_name, ".") || \
    !strcmp(de->d_name, "..");
}


bool CheckLoad::checkfile (char* name, char* device)
{
  FILE *fp;
  bool ret;

  if (ret = (fp = fopen(name, "r")))
    fclose(fp);
  else
    fprintf(stderr, "No support for device type: %s\n", device);

  return !ret;
}


bool CheckLoad::checkbatteryfile (char* name, char* device)
{
  FILE *fp;
  bool ret;

  if (ret = (fp = fopen(name, "r"))) {
    char buffer[100];
    char present[20];
    fgets(buffer,128,fp);
    sscanf(buffer,"present:                 %s", present);
    if (strncmp(present, "yes", 3) != 0) {
      fprintf(stderr, "Device %s not present.\n", device);
      ret = 0;
    }

    fclose(fp);
  }
  else
    fprintf(stderr, "No support for device type: %s\n", device);
  
  return !ret;
}


void CheckLoad::ReadStaticInfo() 
{
  /* Reading battery info */
  FILE *fp;
  char buffer[64];
  int offset = 25;
  int found;
  
  if ((fp = fopen(names.battery1info, "r")))
    {
      /* Read data in any order */
      found = 0;
      while (found < 11 && (fgets(buffer, 63, fp) != NULL)) {
	/* Present */
	if (strncmp(buffer, "present:", 7) == 0 
	    || strncmp(buffer, "Present:", 7) == 0) {
	  if (strncmp(buffer+offset, "yes", 3) == 0) 
	    battery1.present = True;
	  else battery1.present = False;
	  found++;
	}
	/* Design capacity */
	if (strncmp(buffer, "design capacity:", 16) == 0 
	    || strncmp(buffer, "Design Capacity:", 16) == 0) {
	  sscanf(buffer+offset, "%u", &battery1.design_capacity);
	  found++;
	}
	/* Last full charge */
	if (strncmp(buffer, "last full capacity:", 19) == 0 
	    || strncmp(buffer, "Last Full Capacity:", 19) == 0) {
	  sscanf(buffer+offset, "%u", &battery1.full_capacity);
	  found++;
	}
	/* Battery technology */
	if (strncmp(buffer, "battery technology:", 19) == 0 
	    || strncmp(buffer, "Battery Technology:", 19) == 0) {
	  sscanf(buffer+offset, "%s", battery1.battery_technology);
	  found++;
	}
	/* Design Voltage */
	if (strncmp(buffer, "design voltage:", 15) == 0 
	    || strncmp(buffer, "Design Voltage:", 15) == 0) {
	  sscanf(buffer+offset, "%u", &battery1.design_voltage);
	  found++;
	}
	/* Capacity warning */
	if (strncmp(buffer, "design capacity warning:", 25) == 0 
	    || strncmp(buffer, "Design Capacity Warning:", 25) == 0) {
	  sscanf(buffer+offset, "%u", &battery1.capacity_warning);
	  found++;
	}
	/* Capacity low */
	if (strncmp(buffer, "design capacity low:", 20) == 0 
	    || strncmp(buffer, "Design Capacity Low:", 20) == 0) {
	  sscanf(buffer+offset, "%u", &battery1.capacity_low);
	  found++;
	}
	/* Battery model */
	if (strncmp(buffer, "model number:", 13) == 0 
	    || strncmp(buffer, "Model Number:", 13) == 0) {
	  sscanf(buffer+offset, "%s", battery1.model);
	  found++;
	}
	/* Battery Serial */
	if (strncmp(buffer, "serial number:", 14) == 0 
	    || strncmp(buffer, "Serial Number:", 14) == 0) {
	  sscanf(buffer+offset, "%s", battery1.serial);
	  found++;
	}
	/* Battery Type */
	if (strncmp(buffer, "battery type:", 13) == 0 
	    || strncmp(buffer, "Battery Type:", 13) == 0) {
	  sscanf(buffer+offset, "%s", battery1.type);
	  found++;
	}
	/* OEM info */
	if (strncmp(buffer, "OEM info:", 9) == 0 
	    || strncmp(buffer, "OEM Info:", 9) == 0) {
	  sscanf(buffer+offset, "%s", battery1.OEM_info);
	  found++;
	}
      }
      fclose(fp);
    }
  else {
    battery1.present = False;
  }
  
  /* Reading second battery if needed */
  if (!battery1.present || bbtool->getResource()->show.battery_number == 2) {
    if ((fp = fopen(names.battery2info, "r")))
      {
	/* Read data in any order */
	found = 0;
	while (found < 11  && (fgets(buffer, 63, fp) != NULL)) {
	  /* Present */
	  if (strncmp(buffer, "present:", 7) == 0 
	      || strncmp(buffer, "Present:", 7) == 0) {
	    if (strncmp(buffer+offset, "yes", 3) == 0)
	      battery2.present = True;
	    else battery2.present = False;
	    found++;
	  }
	  /* Design capacity */
	  if (strncmp(buffer, "design capacity:", 16) == 0 
	      || strncmp(buffer, "Design Capacity:", 16) == 0) {
	    sscanf(buffer+offset, "%u", &battery2.design_capacity);
	    found++;
	  }
	  /* Last full charge */
	  if (strncmp(buffer, "last full capacity:", 19) == 0 
	      || strncmp(buffer, "Last Full Capacity:", 19) == 0) {
	    sscanf(buffer+offset, "%u", &battery2.full_capacity);
	    found++;
	  }
	  /* Battery technology */
	  if (strncmp(buffer, "battery technology:", 19) == 0 
	      || strncmp(buffer, "Battery Technology:", 19) == 0) {
	    sscanf(buffer+offset, "%s", battery2.battery_technology);
	    found++;
	  }
	  /* Design Voltage */
	  if (strncmp(buffer, "design voltage:", 15) == 0 
	      || strncmp(buffer, "Design Voltage:", 15) == 0) {
	    sscanf(buffer+offset, "%u", &battery2.design_voltage);
	    found++;
	  }
	  /* Capacity warning */
	  if (strncmp(buffer, "design capacity warning:", 24) == 0 
	      || strncmp(buffer, "Design Capacity Warning:", 24) == 0) {
	    sscanf(buffer+offset, "%u", &battery2.capacity_warning);
	    found++;
	  }
	  /* Capacity low */
	  if (strncmp(buffer, "design capacity low:", 20) == 0 
	      || strncmp(buffer, "Design Capacity Low:", 20) == 0) {
	    sscanf(buffer+offset, "%u", &battery2.capacity_low);
	    found++;
	  }
	  /* Battery model */
	  if (strncmp(buffer, "model number:", 13) == 0 
	      || strncmp(buffer, "Model Number:", 13) == 0) {
	    sscanf(buffer+offset, "%s", battery2.model);
	    found++;
	  }
	  /* Battery Serial */
	  if (strncmp(buffer, "serial number:", 14) == 0 
	      || strncmp(buffer, "Serial Number:", 14) == 0) {
	    sscanf(buffer+offset, "%s", battery2.serial);
	    found++;
	  }
	  /* Battery Type */
	  if (strncmp(buffer, "battery type:", 13) == 0 
	      || strncmp(buffer, "Battery Type:", 13) == 0) {
	    sscanf(buffer+offset, "%s", battery2.type);
	    found++;
	  }
	  /* OEM info */
	  if (strncmp(buffer, "OEM info:", 9) == 0 
	      || strncmp(buffer, "OEM Info:", 9) == 0) {
	    sscanf(buffer+offset, "%s", battery2.OEM_info);
	    found++;
	  }
	}
	fclose(fp);
      }
    else {
      battery2.present = False;
    }
  }
  else {
    battery2.present = False;
  }
  
  /* Temperature passive and critical */
  if ((fp = fopen(names.temptrip, "r")))
    {
      found = 0;
      while (found < 2 && (fgets(buffer,128,fp) != NULL)) {
	/* Critical */
	if (strncmp(buffer, "critical", 8) == 0 
	    || strncmp(buffer, "Critical", 8) == 0) {
	  sscanf(buffer+offset,"%d", &temperature.critical);
	  found++;
	}
	/* Passive */
	if (strncmp(buffer, "passive", 7) == 0 
	    || strncmp(buffer, "Passive", 7) == 0) {
	  sscanf(buffer+offset,"%d", &temperature.passive);
	  found++;
	}
      }
      fclose(fp);
    } 
  return;
}


void CheckLoad::getBattery1 (struct BATTERYINFO *bat)
{
  strcpy(bat->state, battery1.state);
  bat->present = battery1.present;
  bat->present_rate = battery1.present_rate;
  bat->remaining_capacity = battery1.remaining_capacity;
  bat->present_voltage = battery1.present_voltage;
  bat->full_capacity = battery1.full_capacity;
  bat->capacity_warning = battery1.capacity_warning;
  bat->capacity_low = battery1.capacity_low;
}


void CheckLoad::getBattery2 (struct BATTERYINFO *bat)
{
  strcpy(bat->state, battery2.state);
  bat->present = battery2.present;
  bat->present_rate = battery2.present_rate;
  bat->remaining_capacity = battery2.remaining_capacity;
  bat->present_voltage = battery2.present_voltage;
  bat->full_capacity = battery2.full_capacity;
  bat->capacity_warning = battery2.capacity_warning;
  bat->capacity_low = battery2.capacity_low;
}


void CheckLoad::Reconfigure() 
{
  FILE *fp = NULL;
  char buffer[64];
  struct stat file_status;
  static time_t last_mtime;
  bool redraw = False;
  int found;
  int offset = 25;
  
  if (battery1.present) 
    if ((fp = fopen(names.battery1state, "r")))
      {
	fstat(fileno(fp),&file_status);
	
	if  (file_status.st_mtime != last_mtime)
	  {
	    found = 0;
	    while (found < 4 && (fgets(buffer, 63, fp) != NULL)) {
	      /* Charging state */
	      if (strncmp(buffer, "charging state:", 15) == 0 
		  || strncmp(buffer, "Charging State:", 15) == 0) {
		sscanf(buffer+offset,"%s", battery1.state);
		found++;
	      }
	      /* Present rate */
	      if (strncmp(buffer, "present rate:", 13) == 0 
		  || strncmp(buffer, "Present Rate:", 13) == 0) {
		sscanf(buffer+offset,"%u", &battery1.present_rate);
		found++;
	      }
	      /* Remaining voltage */
	      if (strncmp(buffer, "remaining capacity:", 19) == 0 
		  || strncmp(buffer, "Remaining Capacity:", 19) == 0) {
		sscanf(buffer+offset,"%u", &battery1.remaining_capacity);
		found++;
	      }
	      /* Present Voltage */
	      if (strncmp(buffer, "present voltage:", 16) == 0 
		  || strncmp(buffer, "Present Voltage:", 16) == 0) {
		sscanf(buffer+offset,"%u", &battery1.present_voltage);
		found++;
	      }
	    }
	    fclose(fp);
	    redraw = True;
	  }
      }
  
  if (!battery1.present || bbtool->getResource()->show.battery_number == 2) 
    if ((fp = fopen(names.battery2state, "r")))
      {
	fstat(fileno(fp),&file_status);
	if  (file_status.st_mtime != last_mtime)
	  {
	    found = 0;
	    while (found < 4 && (fgets(buffer, 63, fp) != NULL)) {
	      /* Charging state */
	      if (strncmp(buffer, "charging state:", 15) == 0 
		  || strncmp(buffer, "Charging State:", 15) == 0) {
		sscanf(buffer+offset,"%s", battery2.state);
		found++;
	      }
	      /* Present rate */
	      if (strncmp(buffer, "present rate:", 13) == 0 
		  || strncmp(buffer, "Present Rate:", 13) == 0) {
		sscanf(buffer+offset,"%u", &battery2.present_rate);
		found++;
	      }
	      /* Remaining voltage */
	      if (strncmp(buffer, "remaining capacity:", 19) == 0 
		  || strncmp(buffer, "Remaining Capacity:", 19) == 0) {
		sscanf(buffer+offset,"%u", &battery2.remaining_capacity);
		found++;
	      }
	      /* Present Voltage */
	      if (strncmp(buffer, "present voltage:", 16) == 0 
		  || strncmp(buffer, "Present Voltage:", 16) == 0) {
		sscanf(buffer+offset,"%u", &battery2.present_voltage);
		found++;
	      }
	    }
	    fclose(fp);
	    redraw = True;
	  }
      }
  
  if ((fp = fopen(names.adapter, "r")))
    {
      found = 0;
      while (found < 1 && (fgets(buffer, 63,fp) != NULL)) {
	if (strncmp(buffer, "state:", 6) == 0 
	    || strncmp(buffer, "State:", 6) == 0) {
	  sscanf(buffer+offset,"%s", adapter.state);
	  found++;
	}
      }
      fclose(fp);
      redraw = True;
    }
  
  if ((fp = fopen(names.temperature, "r")))
    {
      found = 0;
      while (found < 1 && (fgets(buffer, 63,fp) != NULL)) {
	if (strncmp(buffer, "temperature:", 12) == 0 
	    || strncmp(buffer, "Temperature:", 12) == 0) {
	  sscanf(buffer+offset,"%u", &temperature.celcius);
	  found++;
	}
      }
      fclose(fp);
      redraw = True;
    }
  
  if (redraw)
    bbtool->Redraw();
}


void CheckLoad::timeout()
{
  bbtool->xmessageCount--;
  Reconfigure();
}


ToolWindow::ToolWindow(int argc,char **argv,struct CMDOPTIONS *options):
  Basewindow(argc,argv,options)
{
  /* initialize xrm database */
  XrmInitialize();

  resource = new Resource(this);

  xmessageCount = 0;
  motion = false;
  first_motion = true;
  offset_x = 0;
  offset_y = 0;
  
  load = new GEOM [resource->gauge.number_of_bars];
  
  check_load = new CheckLoad(this);
  
  MakeWindow(False);
  
  check_load->Reconfigure();
  
  Redraw();
  
  eventLoop();
}

ToolWindow::~ToolWindow()
{
    XUnmapWindow(getXDisplay(),framewin);

    /* destroy pixmaps */
    if (pixmap.frame) getImageControl()->removeImage(pixmap.frame);
    if (pixmap.label) getImageControl()->removeImage(pixmap.label);
    if (pixmap.loadbar_active)
        getImageControl()->removeImage(pixmap.loadbar_active);
    if (pixmap.loadbar_inactive)
        getImageControl()->removeImage(pixmap.loadbar_inactive);

    if (frameGC) XFreeGC(getXDisplay(),frameGC);
    if (menuGC) XFreeGC(getXDisplay(),menuGC);
    if (menuHiBGGC) XFreeGC(getXDisplay(),menuHiBGGC);
    if (menuHiGC) XFreeGC(getXDisplay(),menuHiGC);
    if (menuFrameGC)   XFreeGC(getXDisplay(),menuFrameGC);
    XDestroyWindow(getXDisplay(),labelwin);
    XDestroyWindow(getXDisplay(),framewin);
    delete resource;
    return;
}


void ToolWindow::hide1Battery (void)
{
  resource->show.battery_number = 1;
  return;
}


void ToolWindow::hideBattery (void)
{
  resource->show.gauge_battery = False;
  resource->show.label_battery = False;
  resource->show.label_time = False;
  resource->show.xmessage = False;
  return;
}


void ToolWindow::hideAdapter (void)
{
  resource->show.label_adapter = False;
  resource->show.gfx_adapter = False;
  return;
}


void ToolWindow::hideTemperature (void)
{
  resource->show.label_temperature = False;
  resource->show.gauge_temperature = False;
  return;
}


void ToolWindow::printBattery1 (void)
{
  char xmessageBuffer[200];
  struct BATTERYINFO *battery1 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
  
  /* Reconfigure the check_load */
  check_load->Reconfigure();
  
  /* Init batteries */
  check_load->getBattery1(battery1);
  if (!battery1->present) 
    check_load->getBattery2(battery1);
  
  /* Battery 1 */
  sprintf(xmessageBuffer, "%s%s%s%s%d%s%d%s%d%s%d%s%d%s%d%s", 
          "xmessage -timeout 20 -center \"Acpi core information\n\n",
          "Battery 1\nState: ", battery1->state,
          "\nPresent rate: ", battery1->present_rate,
          " mW\nRemaining capacity: ", battery1->remaining_capacity,
          " mWh\nPresent Voltage: ", battery1->present_voltage,
          " mV\nFull capacity: ", battery1->full_capacity,
          " mWh\nLow battery level: ", battery1->capacity_low,
          " mWh\nWarning battery level: ", battery1->capacity_warning, 
          " mWh\" &");
  
  system(xmessageBuffer);
}


void ToolWindow::printBattery2 (void)
{
  char xmessageBuffer[200];
  struct BATTERYINFO *battery1 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
  struct BATTERYINFO *battery2 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
  int nbbatteries = resource->show.battery_number;
  
  /* Reconfigure the check_load */
  check_load->Reconfigure();
  
  /* Init batteries */
  check_load->getBattery1(battery1);
  if (battery1->present) {
    if (nbbatteries == 2)
      check_load->getBattery2(battery2);
  }
  else 
    return;
  
  /* Battery 2 */
  sprintf(xmessageBuffer, "%s%s%s%s%d%s%d%s%d%s%d%s%d%s%d%s%s", 
          "xmessage -timeout 20 -center \"Acpi core information\n\n",
          "Battery 2\nState: ", battery2->state,
          "\nPresent rate: ", battery2->present_rate,
          " mW\nRemaining capacity: ", battery2->remaining_capacity,
          " mWh\nPresent Voltage: ", battery2->present_voltage,
          " mV\nFull capacity: ", battery2->full_capacity,
          " mWh\nLow battery level: ", battery2->capacity_low,
          " mWh\nWarning battery level: ", battery2->capacity_warning, 
          " mWh\n\n", "\" &");
  
  system(xmessageBuffer);
}


void ToolWindow::printTemperature (void)
{
  char xmessageBuffer[200];
  int temp, passive, critical;
  char degree;

  if (mbmenu->dcelcius()) {
    temp = check_load->getTemperature().celcius;
    passive = check_load->getTemperature().passive;
    critical = check_load->getTemperature().critical;
    degree = 'C';
  }
  else {
    temp = (int)((((double)check_load->getTemperature().celcius * 1.8) + 32));
    passive = (int)((((double)check_load->getTemperature().passive * 1.8) + 32));
    critical = (int)((((double)check_load->getTemperature().critical * 1.8) + 32));
    degree = 'F';
  }
  sprintf(xmessageBuffer, "%s%s%d%s%c%s%d%s%c%s%d%s%c%s",
          "xmessage -timeout 20 -center \"Acpi core information\n\n",
	  "Temperature\nCelcius: ", temp, " ", degree,
	  "\nPassive: ", passive, " ", degree,
	 "\nCritical: ", critical, " ", degree,
	 "\" &");

  system(xmessageBuffer);
}


void ToolWindow::printAdapter (void)
{
  char xmessageBuffer[200];
  sprintf(xmessageBuffer, "%s%s%s%s",
          "xmessage -timeout 20 -center \"Acpi core information\n\n",
	 "Adapter: ", check_load->getAdapter().state, "\" &");

  system(xmessageBuffer);
}


void ToolWindow::printACPI (void)
{
  char xmessageBuffer[500], xmsg_tmp[200];
  struct BATTERYINFO *battery1 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
  struct BATTERYINFO *battery2 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
  int nbbatteries = resource->show.battery_number;

  /* Reconfigure the check_load */
  check_load->Reconfigure();

  /* Init batteries */
  check_load->getBattery1(battery1);
  if (battery1->present) {
    if (nbbatteries == 2)
      check_load->getBattery2(battery2);
  }
  else {
    check_load->getBattery2(battery1);
    battery2->present = False;
  }
  
  /* Init xmessage buffer */
  sprintf(xmessageBuffer, "%s", "xmessage -timeout 30 -center \"ACPI core information\n\n");

  /* Battery 1 */
  sprintf(xmsg_tmp, "%s%s%s%d%s%d%s%d%s%d%s%d%s%d%s", 
	 "Battery 1\nState: ", battery1->state,
	 "\nPresent rate: ", battery1->present_rate,
	 " mW\nRemaining capacity: ", battery1->remaining_capacity,
	 " mWh\nPresent Voltage: ", battery1->present_voltage,
	 " mV\nFull capacity: ", battery1->full_capacity,
	 " mWh\nLow battery level: ", battery1->capacity_low,
	 " mWh\nWarning battery level: ", battery1->capacity_warning, " mWh\n\n");
  strcat(xmessageBuffer,xmsg_tmp); 

  /* Battery 2 */
  if (nbbatteries == 2) {
    sprintf(xmsg_tmp, "%s%s%s%d%s%d%s%d%s%d%s%d%s%d%s", 
	    "Battery 2\nState: ", battery2->state,
	    "\nPresent rate: ", battery2->present_rate,
	    " mW\nRemaining capacity: ", battery2->remaining_capacity,
	    " mWh\nPresent Voltage: ", battery2->present_voltage,
	    " mV\nFull capacity: ", battery2->full_capacity,
	    " mWh\nLow battery level: ", battery2->capacity_low,
	    " mWh\nWarning battery level: ", battery2->capacity_warning, "mWh\n\n");
    strcat(xmessageBuffer,xmsg_tmp); 
  }
    
  /* Temperature */
  int temp, passive, critical;
  char degree;

  if (mbmenu->dcelcius()) {
    temp = check_load->getTemperature().celcius;
    passive = check_load->getTemperature().passive;
    critical = check_load->getTemperature().critical;
    degree = 'C';
  }
  else {
    temp = (int)((((double)check_load->getTemperature().celcius * 1.8) + 32));
    passive = (int)((((double)check_load->getTemperature().passive * 1.8) + 32));
    critical = (int)((((double)check_load->getTemperature().critical * 1.8) + 32));
    degree = 'F';
  }
  sprintf(xmsg_tmp, "%s%d%s%c%s%d%s%c%s%d%s%c%s",
	  "Temperature\nCelcius: ", temp, " ", degree,
	  "\nPassive: ", passive, " ", degree,
	 "\nCritical: ", critical, " ", degree,
	 "\n\n");
  strcat(xmessageBuffer,xmsg_tmp); 

  /* Adapter */
  sprintf(xmsg_tmp, "%s%s%s",
	 "Adapter: ", check_load->getAdapter().state, "\" &"); 
  strcat(xmessageBuffer,xmsg_tmp);
	 

  system(xmessageBuffer);
}


void ToolWindow::reconfigure(void)
{
  /* destroy pixmaps */
  getImageControl()->removeImage(pixmap.frame);
  getImageControl()->removeImage(pixmap.label);
#ifdef __IMLIB
  getImageControl()->removeImage(pixmap.graphics);
#endif
  getImageControl()->removeImage(pixmap.loadbar_active);
  getImageControl()->removeImage(pixmap.loadbar_inactive);
  delete [] load;
  mbmenu->ClearMenu();
  resource->Reload();
  mbmenu->Reconfigure();
  
  load = new GEOM [resource->gauge.number_of_bars];
  memset(load,0,sizeof(GEOM)*resource->gauge.number_of_bars);
  
  MakeWindow(True);
  
  if (resource->show.label)
    XClearWindow(getXDisplay(), labelwin);
#ifdef __IMLIB
  if (resource->show.gfx_adapter)
    XClearWindow(getXDisplay(), graphicswin);
#endif
  XClearWindow(getXDisplay(), framewin);
  check_load->Reconfigure();
  Redraw();
}


void ToolWindow::saveMenuSearch(Window window,Basemenu *menu)
{
  menuwin = window;
}


void ToolWindow::removeMenuSearch(Window window)
{
  menuwin = (Window )NULL;
}


void ToolWindow::MakeWindow(bool reconfigure)
{
  XSetWindowAttributes attrib;
  XWMHints wmhints;
  int i;

  unsigned long create_mask = CWBackPixmap|CWCursor|CWEventMask ;

  chargingCount1 = 0;
  chargingCount2 = 0;

  if (!resource->show.vertical)
    frame.height = resource->frame.font->ascent + resource->frame.font->descent +
      4*resource->frame.bevelWidth;
  else
    frame.height = (resource->frame.font->ascent + resource->frame.font->descent +
		    4*resource->frame.bevelWidth)*2;

  frame.height -= 2;

  label.separator_width = label.space_width = 0;

  /* Checking the number of batteries */
  int nbbatteries = 0;
  if (resource->show.gauge_battery || resource->show.label_battery) {
    struct BATTERYINFO *battery = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
    
    check_load->getBattery1(battery);
    if (battery->present) {
      nbbatteries++;
      check_load->getBattery2(battery);
      if (resource->show.battery_number == 2 && battery->present)
	nbbatteries++;
    }
    else {
      check_load->getBattery2(battery);
      if (battery->present) 
	nbbatteries++;
    }
  }

  /* Calculate label total width */
  for (i = 0; i < 19; i++) label.width[i] = 0;
  int extra_space = 0;
  int label_counter = 0;

  if (resource->show.label) {
    if (resource->show.label_battery) {
      int i;
      for (i = 0; i < nbbatteries; i++)
	label.width[label_counter++] =  XTextWidth(resource->label.font,"100%",strlen("100%"));
    }
    if (resource->show.label_time) {
      label.width[label_counter++] =  XTextWidth(resource->label.font,"0h00",strlen("0h00")); 
      if (nbbatteries == 2) 
	label.width[label_counter++] =  XTextWidth(resource->label.font,"0h00",strlen("0h00")); 
    }
    if (resource->show.label_temperature) {
      if (resource->show.label_temperature_celcius)
	label.width[label_counter++] =  XTextWidth(resource->label.font,"00C",strlen("00C"));
      else
	label.width[label_counter++] =  XTextWidth(resource->label.font,"000F",strlen("000F"));
    }
    if (resource->show.label_adapter) {
      label.width[label_counter++] =  XTextWidth(resource->label.font,"Plg",strlen("Plg"));
    }


    label.separator_width = XTextWidth(resource->label.font,"/",strlen("/"));
    label.space_width = XTextWidth(resource->label.font," ",strlen(" ")) +
                       resource->frame.bevelWidth;
    if (!resource->show.vertical) 
      label.height=frame.height-2*resource->frame.bevelWidth;
    else
       label.height=(frame.height-2*resource->frame.bevelWidth)/2;

     
    extra_space=resource->frame.bevelWidth;
  }

  label.total_width=(label_counter-1)*label.separator_width+2*label.space_width;
  for (i=0;i<15;i++) 
    label.total_width+=label.width[i];
    
  /* Calculate loadars total width */
  int number_of_loadbars=0;

  /* Batteries */
  if (resource->show.gauge_battery)
    number_of_loadbars += nbbatteries;

  /* Temperature */
  if (resource->show.gauge_temperature) number_of_loadbars++;

  /* Final calculation of frame width */
  if (!resource->show.vertical) 
    frame.width =label.total_width + extra_space+ 
                resource->frame.bevelWidth*2 + 
                (resource->frame.bevelWidth+ resource->gauge.width)*
                number_of_loadbars;
  else {
    if ((label.total_width+extra_space)>
        (resource->frame.bevelWidth+ resource->gauge.width)*number_of_loadbars)
      frame.width=label.total_width+extra_space + 
                  resource->frame.bevelWidth*2;
    else
      frame.width=resource->frame.bevelWidth*2 + 
                  (resource->frame.bevelWidth+ resource->gauge.width)*
                   number_of_loadbars;
  }
  if (label.total_width > 0) frame.width += 3;

  /* Graphics */
#ifdef __IMLIB
  if (resource->show.gfx_adapter)
    frame.width += 22;
#endif    

  int label_x_position;
  
  if ((number_of_loadbars!=0)&(!resource->show.vertical))
    label_x_position= 2*resource->frame.bevelWidth+
                     (resource->frame.bevelWidth+
                      resource->gauge.width)*number_of_loadbars + 3;
  else 
    label_x_position= resource->frame.bevelWidth+extra_space/2;
    
  if (resource->position.mask & XNegative)
    resource->position.x = getCurrentScreenInfo()->getWidth()+
                           resource->position.x-frame.width;

  if (resource->position.mask & YNegative)
    resource->position.y = getCurrentScreenInfo()->getHeight()+
                           resource->position.y-frame.height;

  if (!withdrawn) {
    /*
    frame.x = resource->position.x + offset_x;
    frame.y = resource->position.y + offset_y;
    */
    resource->position.x += offset_x;
    resource->position.y += offset_y;
    frame.x = resource->position.x;
    frame.y = resource->position.y;
  }
  
  attrib.background_pixmap = ParentRelative;

  pixmap.frame = getImageControl()->renderImage(frame.width, frame.height, 
                                                  &resource->frame.texture);

  if (withdrawn)
    wmhints.initial_state = WithdrawnState;
  else
    wmhints.initial_state = NormalState;

  attrib.cursor = getSessionCursor();
  attrib.event_mask = ButtonPressMask | ButtonReleaseMask | ExposureMask |
                      FocusChangeMask | KeyPressMask | StructureNotifyMask | PointerMotionMask;


  if	(!reconfigure) {
    framewin = XCreateWindow(getXDisplay(),
                             getCurrentScreenInfo()->getRootWindow(), 
                             resource->position.x,resource->position.y, 
                             frame.width,frame.height, 0, 
                             getCurrentScreenInfo()->getDepth(),InputOutput,
                             getCurrentScreenInfo()->getVisual(), 
                             create_mask, &attrib);
  }
  else if (!withdrawn)
    XMoveResizeWindow(getXDisplay(),framewin,resource->position.x,
                      resource->position.y, frame.width,
                       frame.height);
  else
    XResizeWindow(getXDisplay(),framewin,frame.width,
                      frame.height);

 	wmhints.flags = IconWindowHint | StateHint;
	wmhints.icon_window = framewin;

  XTextProperty windowname;
  XClassHint classhints;
  
  char *name=BBTOOL;
  XSizeHints sizehints;

  classhints.res_name=BBTOOL;
  classhints.res_class="bbtools";
          
  sizehints.x=resource->position.x;
  sizehints.y=resource->position.y;
  sizehints.max_width=sizehints.min_width=frame.width;
  sizehints.max_height=sizehints.min_height=frame.height;
  sizehints.flags=USPosition|PMinSize|PMaxSize;

  XStringListToTextProperty(&name,1,&windowname);
  XSetWMProperties(getXDisplay(),framewin,&windowname,NULL,getArgv(),getArgc(),
                  &sizehints,&wmhints,&classhints);
  Atom wmproto[2];
  wmproto[0]=wm_delete_window;
  wmproto[1]=getBlackboxStructureMessagesAtom();
  XSetWMProtocols(getXDisplay(), framewin,wmproto, 2);

 if (!decorated && !withdrawn) {
    BlackboxHints bb_hints;
    bb_hints.decoration = DecorNone;
    bb_hints.attrib = AttribOmnipresent;
    bb_hints.flags = AttribDecoration|AttribOmnipresent;
    XChangeProperty(getXDisplay(), framewin, getBlackboxHintsAtom(),
		    getBlackboxHintsAtom(), 32, PropModeReplace,
		    (unsigned char *) &bb_hints,  PropBlackboxHintsElements);
  }

  if (!shape)
    XSetWindowBackgroundPixmap(getXDisplay(), framewin, pixmap.frame);

  /* Labels */
  if (resource->show.label) {
    pixmap.label =
            getImageControl()->renderImage(label.total_width, label.height,
                                       &resource->label.texture);

    int label_y_position;
    if (!resource->show.vertical)
      label_y_position=resource->frame.bevelWidth;
    else
      label_y_position=frame.height/2+resource->frame.bevelWidth;

    label_x_position += 3;
    
    if (!reconfigure)
      labelwin = XCreateWindow(getXDisplay(), framewin,label_x_position, 
                               label_y_position,
                               label.total_width,label.height, 0,
                               getCurrentScreenInfo()->getDepth(),InputOutput,
                               getCurrentScreenInfo()->getVisual(),create_mask, 
                               &attrib);
    else
      XMoveResizeWindow(getXDisplay(),labelwin,
                        label_x_position,
                        label_y_position,
                        label.total_width,label.height);
    if (!resource->label.transparent)
      XSetWindowBackgroundPixmap(getXDisplay(), labelwin, pixmap.label);
  }

  /* Graphics */
#ifdef __IMLIB
  if (resource->show.gfx_adapter) {
    gfx_adapter = 0;
    gfx_battery = 0;
    id = (ImlibData *) Imlib_init(getXDisplay());
    
    pixmap.graphics = 
      getImageControl()->renderImage(14, 16, &resource->label.texture);

    int gfx_y_position;
    if (!resource->show.vertical)
      gfx_y_position=resource->frame.bevelWidth ;
    else
      gfx_y_position=frame.height/2+resource->frame.bevelWidth;
    
    if ((frame.height - 20)/2 > 0)
      gfx_y_position += (frame.height - 20)/2;

    if (!reconfigure)
      graphicswin = XCreateWindow(getXDisplay(), framewin, label_x_position+label.total_width+3,
				  gfx_y_position, 14, 16, 0,
				  getCurrentScreenInfo()->getDepth(),InputOutput,
				  getCurrentScreenInfo()->getVisual(),create_mask, 
				  &attrib);
    else
      XMoveResizeWindow(getXDisplay(),graphicswin,
			label_x_position+label.total_width+3,gfx_y_position, 14, 16);
    
    
    if (!resource->label.transparent)
      XSetWindowBackgroundPixmap(getXDisplay(), graphicswin, pixmap.graphics);
  }
#endif

  /* Loadbars */
  if (resource->show.loadbar) {
    int height;
    if (!resource->show.vertical)
      height=frame.height;
    else
      height=frame.height/2;

    pixmap.loadbar_active =	
          getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                     &resource->gauge.active_texture);
    pixmap.loadbar_inactive =
           getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                      &resource->gauge.inactive_texture);

    pixmap.loadbar_background = 
            getImageControl()->renderImage(resource->gauge.width+4,height+4,
                                       &resource->gauge.background_texture);
    if (resource->gauge.number_of_bars!=0) {
      if (!resource->show.vertical)
        gauge_height=frame.height-2*resource->frame.bevelWidth;
      else
        gauge_height=frame.height/2-2*resource->frame.bevelWidth;
    }

    int space=0;
    if (resource->gauge.tickmarks) {
      loadbar_height=(gauge_height-(resource->gauge.number_of_bars-1))/
                     (resource->gauge.number_of_bars);
      space=1;
    } 
    else
      loadbar_height=gauge_height/(resource->gauge.number_of_bars);
       
    for (i=0;i<resource->gauge.number_of_bars;i++) {
      load[i].x= (int) (resource->gauge.left_step_width*i
                        +resource->frame.bevelWidth);
      if (!resource->show.vertical)
        load[i].y = frame.height 
	  - (loadbar_height+space)*(i+1) 
	  - resource->frame.bevelWidth;
      else
        load[i].y = frame.height/2
	  - (loadbar_height+space)*(i+1)
	  - resource->frame.bevelWidth;;

      load[i].width=(int)(resource->gauge.width-
                         (resource->gauge.left_step_width+
                          resource->gauge.right_step_width)*i);
      if (!resource->gauge.levels_defined)
        resource->gauge.level[i]=
              (i+1)*(resource->gauge.max_load/resource->gauge.number_of_bars);
    }
  }

  if (!reconfigure) {
    if (resource->show.label) {
      gcv.font = resource->label.font->fid;
      frameGC = XCreateGC(getXDisplay(), framewin,GCFont, &gcv);

      gcv.font = resource->menu.font->fid;
      gcv.foreground = resource->menu.texture.getColor()->getPixel();
      menuGC = XCreateGC(getXDisplay(), framewin,GCFont|GCForeground, &gcv);
      
      gcv.foreground = resource->menu.highlightColor.getPixel();
      gcv.arc_mode = ArcChord;
      gcv.fill_style = FillSolid;
      menuHiBGGC = XCreateGC(getXDisplay(), framewin,GCForeground|
			     GCFillStyle|GCArcMode, &gcv);

      gcv.foreground = resource->menu.hitextColor.getPixel();
      menuHiGC = XCreateGC(getXDisplay(), framewin,	GCFont|GCForeground, &gcv);
      
      gcv.foreground = resource->menu.textColor.getPixel();
      menuFrameGC = XCreateGC(getXDisplay(), framewin,GCFont|GCForeground, &gcv);

      /* Main menu */
      mbmenu = new Menu(this);
      mbmenu->Update();
    }
    else
      frameGC = XCreateGC(getXDisplay(), framewin,0, &gcv);
  } else {
    if (resource->show.label) {
      gcv.font = resource->label.font->fid;
      XChangeGC(getXDisplay(), frameGC,GCFont, &gcv);
    }
    else
      XChangeGC(getXDisplay(), frameGC,0, &gcv);
  }


  if (resource->gauge.use_simple_color) {
    Pixmap dummy;
    for (i=0;i<resource->gauge.number_of_bars;i++) {
      dummy=getImageControl()->renderImage(resource->gauge.width+4,
                      frame.height+4,&resource->gauge.segment_texture[i]);
      XCopyArea(getXDisplay(),dummy,pixmap.loadbar_active,frameGC,
                2,2,resource->gauge.width,loadbar_height,2,load[i].y-1);
      getImageControl()->removeImage(dummy);
    }
  }
  
  if (!reconfigure) {
    XClearWindow(getXDisplay(), framewin);
    XMapSubwindows(getXDisplay(), framewin);
    XMapWindow(getXDisplay(), framewin);
    mapped=true;
    if (!decorated&&!withdrawn)
      XMoveResizeWindow(getXDisplay(),framewin,frame.x,
			frame.y, frame.width,
			frame.height);
  }  
}


void ToolWindow::RedrawGauge(double level, int offset, bool reverse)
{
  int i, loadi;
  
  for (i=0;i<resource->gauge.number_of_bars;i++)
    {
      loadi = (reverse) ? resource->gauge.number_of_bars-i-1 : i;
      if (level>=resource->gauge.level[i])
	XCopyArea(getXDisplay(),pixmap.loadbar_active,framewin,frameGC,
		  2,load[loadi].y-1,load[i].width,loadbar_height,
		  load[i].x+5+offset,load[i].y-1);
      else
	XCopyArea(getXDisplay(),pixmap.loadbar_inactive,framewin,frameGC,
		  2,load[i].y-1,load[i].width,loadbar_height,
		  load[i].x+5+offset,load[i].y-1);
    }
  if (resource->gauge.background)
    {
      for (i=1;i<resource->gauge.number_of_bars;i++)
	{
	  XCopyArea(getXDisplay(),pixmap.loadbar_background,framewin,frameGC,
		    2,load[i].y-1,load[i].width,1,
		    load[i].x+5+offset,load[i].y+loadbar_height-1);
	}
    }
}


void ToolWindow::RedrawLabel(char* t,BColor textColor, bool first)
{
  
  if (!first)
  {
    char tt[6];
    XSetForeground(getXDisplay(),frameGC,
                   resource->label.separator_textColor.getPixel());
    sprintf(tt, "/");
    XDrawString(getXDisplay(), labelwin, frameGC, xposition,
                (label.height+resource->label.font->ascent
                 -resource->label.font->descent) / 2,
                 tt, strlen(tt));
    xposition += XTextWidth(resource->label.font,"/",strlen("/"));
  } 
 
  XSetForeground(getXDisplay(),frameGC, textColor.getPixel());
  XDrawString(getXDisplay(), labelwin, frameGC, xposition,
             (label.height+resource->label.font->ascent-
             resource->label.font->descent) / 2,
             t, strlen(t));

  xposition += XTextWidth(resource->label.font,t,strlen(t));
}


#ifdef __IMLIB
void ToolWindow::ReDrawGraphics (bool adapter)
{
  if (adapter) {
    if (!gfx_adapter) {
      Imlib_free_pixmap(id,p);
      im=Imlib_load_image(id, GLOBAL_DATA"/bbacpi-adapter.png");
      Imlib_apply_image(id, im, graphicswin);
      
      gfx_adapter = True;
      gfx_battery = False;
    }
  }
  else 
    if (!gfx_battery) {
      Imlib_free_pixmap(id,p);
      im=Imlib_load_image(id, GLOBAL_DATA"/bbacpi-battery.png");
      Imlib_apply_image(id, im, graphicswin);
      
      gfx_adapter = False;
      gfx_battery = True;
    }

  xposition += 16;
}
#endif


void ToolWindow::Redraw()
{
    int offset=0;
    int nbbatteries = resource->show.battery_number;
    struct BATTERYINFO *battery1 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));
    struct BATTERYINFO *battery2 = (struct BATTERYINFO *) malloc (sizeof(struct BATTERYINFO));

    /* Init batteries */
    check_load->getBattery1(battery1);

    if (battery1->present) {
      if (nbbatteries == 2)
	check_load->getBattery2(battery2);
    }
    else {
      check_load->getBattery2(battery1);
      battery2->present = False;
    }

    /* Xmessage */
    if (resource->show.xmessage) {

      if ((xmessageCount <= 0) && (strcmp(battery1->state, "discharging") == 0 )) {
	/* Set xmessageCount back*/
	xmessageCount = check_load->getUser().xmessage_delay / check_load->getUser().delay;

	unsigned int seconds, minutes, hours;
	seconds = (int)(3600 * (float)battery1->remaining_capacity / battery1->present_rate);
	
	hours = seconds / 3600;
	seconds -= hours * 3600;
	minutes = seconds / 60;
      
	if (hours > 24 || hours < 0) hours = 0;
	if (minutes > 59 || minutes < 0) minutes = 0;
	
	double procent = (double)battery1->remaining_capacity / battery1->full_capacity;
	if ((procent < 0.5)
	    && (minutes <= check_load->getUser().minlim)
	    && (hours == 0 )) {
	  char xmessageBuffer[200];
	  if (minutes == 0) {
	    sprintf(xmessageBuffer, "%s%s",
		    "xmessage -timeout 20 -center \"The battery level is at lowest possible level\n",
		    "Turn off or plug adapter immediatly\" &");
	  }
	  else {
	    sprintf(xmessageBuffer, "%s%.0f%s%d%s%s",
		    "xmessage -timeout 20 -center \"The battery level is low [",
		    procent * 100,
		    "%]\nApproximately ",
		    minutes,
		    " minutes left\nYou should plug the adapter\n",
		    "or switch off the computer\" &"); 
	  }
	  system(xmessageBuffer);
	}
      }
    }

    /* Gauges */
    int i;
    for (i = 1; i < 3; i++) {
      if (resource->order.gauge_battery == i && resource->show.gauge_battery)
	{
	  if (battery1->present) {
	    if (strcmp(battery1->state, "charging") 
		|| !resource->show.gauge_battery_special_charging) {
	      RedrawGauge((((double)battery1->remaining_capacity 
			    / (double)battery1->full_capacity)+0.05),
			  offset,resource->reverse.gauge_battery);  
	      offset += load[0].width+resource->frame.bevelWidth;
	    }
	    else {
	      int max = (int) ((((double)battery1->remaining_capacity 
				 / (double)battery1->full_capacity)+0.05)
			       * (double)resource->gauge.number_of_bars);
	      if (++chargingCount1 > max) {
		chargingCount1 = 0;
		RedrawGauge(0.0,offset,resource->reverse.gauge_battery);
	      }
	      else {
		int i = 0;
		float val = (float)chargingCount1/(float)resource->gauge.number_of_bars;
		while (resource->gauge.level[i] < val) {i++;}
		RedrawGauge(resource->gauge.level[i],offset,resource->reverse.gauge_battery);
	      }
	    }
	  }
	  
	  if (nbbatteries == 2 && battery2->present) {
	    if (strcmp(battery2->state, "charging") 
		|| !resource->show.gauge_battery_special_charging) {
	      RedrawGauge((((double)battery2->remaining_capacity 
			    / (double)battery2->full_capacity)+0.05),
			  offset,resource->reverse.gauge_battery);  
	      offset += load[0].width+resource->frame.bevelWidth;
	    }
	    else {
	      int max = (int) ((((double)battery2->remaining_capacity 
				 / (double)battery2->full_capacity)+0.05)
			       * (double)resource->gauge.number_of_bars);
	      if (++chargingCount2 > max) {
		chargingCount2 = 0;
		RedrawGauge(0.0,offset,resource->reverse.gauge_battery);
	      }
	      else {
		int i = 0;
		float val = (float)chargingCount2/(float)resource->gauge.number_of_bars;
		while (resource->gauge.level[i] < val) {i++;}
		RedrawGauge(resource->gauge.level[i],offset,resource->reverse.gauge_battery);
	      }
	    }
	  }
	}
      
      if (resource->order.gauge_temperature == i && resource->show.gauge_temperature)
	{
	  double temp;
	  int tempe = check_load->getTemperature().celcius;
	  
	  if (tempe >= (int) check_load->getTemperature().critical) 
	    temp = 1;
	  else if (tempe >= (int) check_load->getTemperature().passive) 
	    temp = 0.9;
	  else
	    temp = (double) tempe / (double) check_load->getTemperature().passive;
	  
	  RedrawGauge(temp,offset,resource->reverse.gauge_temperature); 
	  offset += load[0].width+resource->frame.bevelWidth;
	}
    }

    /*  Labels */
    bool first = true;

    if (resource->show.label)
      {
	char t[6];
	xposition=label.space_width; 
	XClearWindow(getXDisplay(),labelwin);
	
	for (i = 1; i < 5; i++) {
	  /* Battery label */
	  if (resource->order.label_battery == i && resource->show.label_battery)
	    {
	      int tp = (int)(((battery1->remaining_capacity * 100) / 
			      battery1->full_capacity)+1);

	      if (tp > 100) tp = 100;

	      sprintf(t, "%2d%%", tp);
	      
	      RedrawLabel(t, resource->label.battery_textColor, first);
	      first = false;
	      
	      if (resource->show.battery_number == 2) {
		tp = (int)(((battery2->remaining_capacity * 100) / 
			    battery2->full_capacity)+1);

		if (tp > 100) tp = 100;

		sprintf(t, "%2d%%", tp);
	      
		RedrawLabel(t, resource->label.battery_textColor, first);
	      }
	    }
	  
	  /* Time label */
	  if (resource->order.label_time == i && resource->show.label_time)
	    {
	      if (strcmp(battery1->state, "unknown") == 0 || strcmp(battery1->state, "charged") == 0) {
		strcpy (t, "idle");
		RedrawLabel(t, resource->label.time_textColor, first);
	      }
	      else {
		unsigned int seconds, minutes, hours;
		bool charging = false;

		if (strcmp(battery1->state, "charging") == 0) {
		  seconds = (int)(3600 * (double)(battery1->full_capacity - battery1->remaining_capacity) 
				  / battery1->present_rate);
		  charging = true;
		}
		else
		  seconds = (int)(3600 * (float)battery1->remaining_capacity / battery1->present_rate);
		
		hours = seconds / 3600;
		seconds -= hours * 3600;
		minutes = seconds / 60;
		
                /* Correct printing if remaining > full */
                if (battery1->full_capacity < battery1->remaining_capacity && !charging) {
                  sprintf(t, "OhOO");
                  RedrawLabel(t, resource->label.time_textColor, first);
                }
                else {
                  if (hours > 24 || hours < 0) hours = 0;
                  if (minutes > 59 || minutes < 0) minutes = 0;
                  
                  if (minutes < 10)
                    sprintf(t, "%1dh0%1d", hours, minutes);
                  else sprintf(t, "%1dh%2d", hours, minutes);
                  
                  if (hours == 0 && minutes <= check_load->getUser().minlim && !charging)
                    RedrawLabel(t, resource->label.time_short_textColor, first);
                  else RedrawLabel(t, resource->label.time_textColor, first);
                }
	      }
	      first = false;

	      /* Second battery */
	      if (nbbatteries == 2) {
		if (strcmp(battery2->state, "unknown") == 0 || strcmp(battery2->state, "charged") == 0) {
		  strcpy (t, "idle");
		  RedrawLabel(t, resource->label.time_textColor, first);
		}
		else {
		  unsigned int seconds, minutes, hours;
		  bool charging = false;
		  
		  if (strcmp(battery2->state, "charging") == 0) {
		    seconds = (int)(3600 * (double)(battery2->full_capacity - battery2->remaining_capacity) 
				    / battery2->present_rate);
		    charging = true;
		  }
		  else
		    seconds = (int)(3600 * (float)battery2->remaining_capacity / battery2->present_rate);
		  
		  hours = seconds / 3600;
		  seconds -= hours * 3600;
		  minutes = seconds / 60;
		  
                  /* Correct printing if remaining > full */
                  if (battery2->full_capacity < battery2->remaining_capacity && !charging) {
                    sprintf(t, "OhOO");
                    RedrawLabel(t, resource->label.time_textColor, first);
                  }
                  else {
                    if (hours > 24 || hours < 0) hours = 0;
                    if (minutes > 59 || minutes < 0) minutes = 0;
                    
                    if (minutes < 10)
                      sprintf(t, "%1dh0%1d", hours, minutes);
                    else sprintf(t, "%1dh%2d", hours, minutes);
                    
                    if (hours == 0 && minutes <= check_load->getUser().minlim && !charging)
                      RedrawLabel(t, resource->label.time_short_textColor, first);
                    else RedrawLabel(t, resource->label.time_textColor, first);
                  }
                }
	      }
	    }
	  
	  /* Temperature label */
	  if (resource->order.label_temperature == i && resource->show.label_temperature)
	    {
	      if (mbmenu->dcelcius())
		sprintf(t, "%2dC", check_load->getTemperature().celcius);
	      else 
		sprintf(t, "%2dF", (int)((((double)check_load->getTemperature().celcius * 1.8) + 32)));
	      
	      RedrawLabel(t, resource->label.temperature_textColor, first);
	      first = false;
	    }
	  
	  /* Adapter label */
	  if (resource->order.label_adapter == i && resource->show.label_adapter)
	    {
	      if (strcmp (check_load->getAdapter().state, "on-line") == 0) 
		strcpy(t, "Plg");
	      else
		strcpy(t, "unP");
	      RedrawLabel(t, resource->label.adapter_textColor, first);
	      first = false;
	    }
	}  
      }

#ifdef __IMLIB
    /* Graphics */
    if (resource->show.gfx_adapter) 
      ReDrawGraphics (strcmp (check_load->getAdapter().state, "off-line"));
#endif

    /* Checking there is something to show */
    if (!resource->show.gauge_battery && !resource->show.gauge_temperature 
	&& !resource->show.label_battery && !resource->show.label_temperature
	&& !resource->show.label_adapter && !resource->show.label_time
	&& !resource->show.gfx_adapter) {
      fprintf(stderr, "\nNo device is able to be shown.\n");
      fprintf(stderr, "Check in the settings file for bbacpi.show.**** .\n");
      exit(-1);
    }
}

void ToolWindow::CheckConfig()
{
    struct stat file_status;

    if (stat(config_filename,&file_status)!=0)
    {
        fprintf(stderr,"Could not config file %s\n",
                config_filename);
    }
        else if(file_status.st_mtime != resource->style.mtime)
    {
        resource->style.mtime=file_status.st_mtime;
        reconfigure();
    }
}


void ToolWindow::process_event(XEvent *Event) 
{
  switch (Event->type) {
  case ClientMessage: 
    {
      if ((unsigned)Event->xclient.data.l[0] == wm_delete_window) shutdown();
    }
    break;
  case Expose: 
    {
      mbmenu->exposeEvent(&Event->xexpose);
      Redraw();
    }
    break;
  case ButtonPress:
    {
      /* If the move option is on */
      if (mbmenu->dmove()) {
	motion = true;
      }

      /* If the mouse is on the menu */
      if (Event->xbutton.window == menuwin) {
	mbmenu->buttonPressEvent(&Event->xbutton);        
      } 
      /* Left button pressed */
      else if (Event->xbutton.button == LEFT_BUTTON) {
	if (!(raised)) {
	  XRaiseWindow(getXDisplay(),framewin);
	  raised=True;
	}
      } 
      /* Middle button pressed */
      else if (Event->xbutton.button == MIDDLE_BUTTON) {
	if (raised) {
	  XLowerWindow(getXDisplay(),framewin);
	  raised=False;
	}
      } 
      /* Right button pressed */
      else if (Event->xbutton.button == RIGHT_BUTTON) {
	if (mbmenu->isVisible())
	  mbmenu->hide();
	else {
	  mbmenu->Move(frame.x,frame.y,withdrawn);
	  mbmenu->show();
	}
      }
    }
    break;
  case ButtonRelease: 
    {
      /* If the move option is on */
      if (mbmenu->dmove()) {
	/* Check if the user moved the window */
	if (first_motion == false) {
	  offset_x += last_x - motion_x;
	  offset_y += last_y - motion_y;
	  
	  /* Reset first_motion */
	  first_motion = true;
	  reconfigure();
	}
	motion = false;
      }

      /* If the mouse is on the menu */
      if (Event->xbutton.window == menuwin) {
	mbmenu->buttonReleaseEvent(&Event->xbutton);
      } 
    }
    break;
  case MotionNotify:
    {
      /* If the move option is on */
      if (mbmenu->dmove()) {
	/* If the user is moving the window */
	if (motion) {
	  if (first_motion) {
	    motion_x = Event->xmotion.x;
	    motion_y = Event->xmotion.y;
	    first_motion = false;
	  }
	  else {
	    last_x = Event->xmotion.x;
	    last_y = Event->xmotion.y;
	  }
	}
      }
      
      /* If the mouse is on the menu */
      if (Event->xmotion.window == menuwin) 
	mbmenu->motionNotifyEvent(&Event->xmotion);
    }
    break;
  case EnterNotify:
    {
      /* If the mouse is on the menu */
      if (Event->xcrossing.window == menuwin)
	mbmenu->enterNotifyEvent(&Event->xcrossing);
    }
    break;
  case LeaveNotify:
    {
      /* If the mouse is on the menu */
      if (Event->xcrossing.window == menuwin)
	mbmenu->leaveNotifyEvent(&Event->xcrossing);
    }
    break;
  case ConfigureNotify: 
    {
      if (Event->xconfigure.window == framewin && Event->xconfigure.send_event) {
	
	if (withdrawn) {
	  reconfigure();
	}
	int parent_x,parent_y;
	Window parent_root;
	unsigned int parent_width;
	unsigned int parent_height;
	unsigned int parent_border_width;
	unsigned int parent_depth;
	if (withdrawn) {
	  XGetGeometry(getXDisplay(),Event->xconfigure.above,&parent_root,
		       &parent_x,&parent_y,&parent_width,&parent_height,
		       &parent_border_width,&parent_depth);
	  frame.x = Event->xconfigure.x+parent_x;
	  frame.y = Event->xconfigure.y+parent_y;
	} else {
	  frame.x = Event->xconfigure.x;
	  frame.y = Event->xconfigure.y;
	  if (decorated) {
	    if (position == NULL) position = new char [13];
	    sprintf(position,"+%i+%i",frame.x,frame.y);
	  }
	}
      }
    }
  }
}
