//==============================================================================
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Library General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//==============================================================================

//==============================================================================
// File: cRocket.cpp
// Project: Shooting Star
// Author: Jarmo Hekkanen <jarski@2ndpoint.fi>
// Copyrights (c) 2003 2ndPoint ry (www.2ndpoint.fi)
//------------------------------------------------------------------------------
// Revision history
//==============================================================================

//==============================================================================
// Includes
#include "cRocket.hpp"
#include "cPlayer.hpp"
#include "cRocketLauncher.hpp"
#include "cParticleSystem.hpp"
#include "cTextureManager.hpp"
#include "cWorld.hpp"
#include "cMixer.hpp"
#include "Debug.hpp"
#include "cExplosionSystem.hpp"
#include "cHurtable.hpp"
//------------------------------------------------------------------------------
// Namespaces
using namespace ShootingStar;
//==============================================================================


//! Constructor
cRocket::cRocket (void):
mSmokeSystem (0),
mSound (cMixer::GetInstance ().LoadSound ("explosion.wav")),
mChannel (-2)
{
	SetCollisionModel (CollisionModel_Ray);
	SetContactModel (ContactModel_Stop);
	
	SetEmittingAngle (180.0f);
	SetEmittingDistance (0.0f);
};

//! Destructor
cRocket::~cRocket (void)
{
	// Empty
};

//! Create a smoke particle system
void 
cRocket::CreateSmokeSystem (void)
{
	dbg::assertion (DBG_ASSERTION (mSmokeSystem == 0));
	
	cParticleSystem *pSystem = new cParticleSystem (50);
	pSystem->SetTexture (cTextureManager::GetInstance ().LoadTexture ("smoke.png"));
	pSystem->SetAngleVariation (0.0f);
	pSystem->SetSpeed (0.01f, 0.0f);
	pSystem->SetSize (16.0f, 128.0f);
	pSystem->SetEnergy (500, 0.3f);
	pSystem->SetBlending (true);
	pSystem->SetEmitDelay (10);
	pSystem->SetEmitter (this);
	pSystem->SetStartColor (0.5f, 0.5f, 0.5f, 1.0f);
	pSystem->SetEndColor (1.0f, 1.0f, 1.0f, 0.0f);
	pSystem->SetLayer (-1);
	cWorld::GetInstance ().SpawnObject (pSystem);
	
	mSmokeSystem = pSystem->GetID ();
}

void 
cRocket::OnMapCollision (const cVector2f &begin, const cVector2f &end)
{
	Explode ();
}

void 
cRocket::OnObjectCollision (cCollidable *pOther)
{
	cHurtable *pHurtable = dynamic_cast<cHurtable*>(pOther);
	if ( pHurtable != NULL )
	{
		Explode ();
		pHurtable->DoDamage (50, cVector2f (0.0f, 0.0f), GetID ());
	}
}

void 
cRocket::Explode ()
{
	// Kill the smoke system
	cParticleSystem *pSmoke = NULL;
	if ( mSmokeSystem != 0 )
		pSmoke = dynamic_cast<cParticleSystem*> (cWorld::GetInstance ().GetObject (mSmokeSystem));
	if ( pSmoke != NULL )
	{
		pSmoke->SetEmitDelay (0);
		pSmoke->KillEmptySystem (true);
	}
	
	if ( mChannel >= 0 )
	{
		cMixer::GetInstance ().StopSound(mChannel,250);
		mChannel = -2;
	}
	mChannel = cMixer::GetInstance ().PlaySound (mSound);
	
	// Spawn an explosion
	cExplosionSystem *pSystem = new cExplosionSystem;
	pSystem->SetEmitter (this);
	pSystem->EmitAll ();
	pSystem->SetOwner (GetOwner ());
	cWorld::GetInstance ().SpawnObject (pSystem);
	
	// Kill the rocket
	Kill ();
}



//==============================================================================
// EOF
//==============================================================================
