//Copyright (c) HashiCorp, Inc.
//SPDX-License-Identifier: MPL-2.0

package commonids

import (
	"fmt"
	"strings"

	"github.com/hashicorp/go-azure-helpers/resourcemanager/resourceids"
)

var _ resourceids.ResourceId = &DevCenterId{}

// DevCenterId is a struct representing the Resource ID for a Dev Center
type DevCenterId struct {
	SubscriptionId    string
	ResourceGroupName string
	DevCenterName     string
}

// NewDevCenterID returns a new DevCenterId struct
func NewDevCenterID(subscriptionId string, resourceGroupName string, devCenterName string) DevCenterId {
	return DevCenterId{
		SubscriptionId:    subscriptionId,
		ResourceGroupName: resourceGroupName,
		DevCenterName:     devCenterName,
	}
}

// ParseDevCenterID parses 'input' into a DevCenterId
func ParseDevCenterID(input string) (*DevCenterId, error) {
	parser := resourceids.NewParserFromResourceIdType(&DevCenterId{})
	parsed, err := parser.Parse(input, false)
	if err != nil {
		return nil, fmt.Errorf("parsing %q: %+v", input, err)
	}

	id := DevCenterId{}
	if err := id.FromParseResult(*parsed); err != nil {
		return nil, err
	}

	return &id, nil
}

// ParseDevCenterIDInsensitively parses 'input' case-insensitively into a DevCenterId
// note: this method should only be used for API response data and not user input
func ParseDevCenterIDInsensitively(input string) (*DevCenterId, error) {
	parser := resourceids.NewParserFromResourceIdType(&DevCenterId{})
	parsed, err := parser.Parse(input, true)
	if err != nil {
		return nil, fmt.Errorf("parsing %q: %+v", input, err)
	}

	id := DevCenterId{}
	if err = id.FromParseResult(*parsed); err != nil {
		return nil, err
	}

	return &id, nil
}

func (id *DevCenterId) FromParseResult(input resourceids.ParseResult) error {
	var ok bool

	if id.SubscriptionId, ok = input.Parsed["subscriptionId"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "subscriptionId", input)
	}

	if id.ResourceGroupName, ok = input.Parsed["resourceGroupName"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "resourceGroupName", input)
	}

	if id.DevCenterName, ok = input.Parsed["devCenterName"]; !ok {
		return resourceids.NewSegmentNotSpecifiedError(id, "devCenterName", input)
	}

	return nil
}

func ValidateDevCenterID(input interface{}, key string) (warnings []string, errors []error) {
	v, ok := input.(string)
	if !ok {
		errors = append(errors, fmt.Errorf("expected string, got %T", input))
		return
	}

	if _, err := ParseDevCenterID(v); err != nil {
		errors = append(errors, err)
	}

	return
}

// ID returns the formatted dev center ID
func (id DevCenterId) ID() string {
	fmtString := "/subscriptions/%s/resourceGroups/%s/providers/Microsoft.DevCenter/devCenters/%s"
	return fmt.Sprintf(fmtString, id.SubscriptionId, id.ResourceGroupName, id.DevCenterName)
}

// String returns a human-readable representation of the dev center ID
func (id DevCenterId) String() string {
	components := []string{
		fmt.Sprintf("Subscription ID %q", id.SubscriptionId),
		fmt.Sprintf("Resource Group Name %q", id.ResourceGroupName),
		fmt.Sprintf("Dev Center Name %q", id.DevCenterName),
	}
	return fmt.Sprintf("Dev Center (%s)", strings.Join(components, "\n"))
}

// Segments returns a slice of Resource ID Segments which comprise this Dev Center ID
func (id DevCenterId) Segments() []resourceids.Segment {
	return []resourceids.Segment{
		resourceids.StaticSegment("staticSubscriptions", "subscriptions", "subscriptions"),
		resourceids.SubscriptionIdSegment("subscriptionId", "12345678-1234-9876-4563-123456789012"),
		resourceids.StaticSegment("staticResourceGroups", "resourceGroups", "resourceGroup"),
		resourceids.ResourceGroupSegment("resourceGroupName", "example-resource-group"),
		resourceids.StaticSegment("staticProviders", "providers", "providers"),
		resourceids.ResourceProviderSegment("staticMicrosoftDevCenter", "Microsoft.DevCenter", "Microsoft.DevCenter"),
		resourceids.StaticSegment("staticDevCenters", "devCenters", "devCenters"),
		resourceids.UserSpecifiedSegment("devCenterName", "devCenterValue"),
	}
}
