# Copyright (C) 2011-2012 - Curtis Hovey <sinzui.is at verizon.net>
# This software is licensed under the MIT license (see the file COPYING).

from textwrap import dedent

from gi.repository import (
    Gio,
    Gtk,
    )

import gdpcomplete
from gdpcomplete import CompletePlugin
from gdp import (
    Config,
    config,
    )
from gdp.complete import Completer
from testing import GeditTestCase


class CompletePluginTestCase(GeditTestCase):

    def test_init(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        self.assertIs(None, plugin.controller)
        self.assertIs(None, plugin.window)

    def test_do_activate(self):
        # do_activate sets up the menus and gets the controller.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertIsInstance(plugin.controller, Completer)
        action = plugin.action_group.get_action('CompleteWord')
        self.assertEqual('Complete _word', action.get_property('label'))
        self.assertEqual(
            ['active-tab-changed', 'tab-added'],
            sorted(plugin.signal_ids.keys()))

    def test_do_update_state(self):
        # do_update_state() is implemented, but it does nothing.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertIs(None, plugin.do_update_state())

    def test_do_update_state_with_config(self):
        # do_update_state() is implemented, but it does nothing.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        config.do_update_state = True
        plugin.do_activate()
        self.assertIs(None, plugin.do_update_state())
        self.assertIs(False, config.do_update_state)

    def test_do_deactivate(self):
        # do_deactivate removes the menus and the controller.
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        plugin.do_deactivate()
        self.assertIs(None, plugin.controller)

    def test_on_tab_added_or_changed(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        self.assertEqual(
            ['active-tab-changed', 'tab-added'],
            sorted(plugin.signal_ids.keys()))
        location = Gio.file_new_for_path('plugins/gdp/data/snark.py')
        window.create_tab_from_location(location, None, 0, 0, False, True)
        new_view = window.get_active_view()
        self.assertIsNot(view, new_view)
        self.assertIs(new_view, plugin.controller.view)

    def test_actions(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        action = plugin.actions(plugin.controller)[0]
        self.assertEqual('CompleteWord', action[0])
        self.assertEqual(config.get('completer', 'show_accel'), action[3])
        action = plugin.actions(plugin.controller)[1]
        self.assertEqual('SuggestCompletions', action[0])
        self.assertEqual(
            config.getboolean('completer', 'suggest_completions'), action[6])

    def test_do_create_configure_widget(self):
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        widget = plugin.do_create_configure_widget()
        self.assertIsInstance(widget, Gtk.Grid)

    def test_on_completer_entry_changed(self):
        # Setup a test config so that the singleton is not mutated.
        config_file = self.make_config(gdpcomplete, Config)
        window, view, document = self.make_gedit(
            'plugins/gdp/data/snark12.txt')
        plugin = CompletePlugin()
        plugin.window = window
        plugin.do_activate()
        widgets = plugin._get_configure_widgets()
        entry = widgets.get_object('shortcut_entry')
        entry.set_text('<Control>a')
        entry.emit('focus-out-event', None)
        self.assertEqual(
            '<Control>a',
            gdpcomplete.config.get('completer', 'show_accel'))
        text = dedent("""\
            [completer]
            show_accel = <Control>a
            suggest_completions = False

            """)
        self.assertIn(text, config_file.read())
