/***************************************************************************
 *   Copyright (C) 2008-2011 by Marcel Hasler                              *
 *   mahasler@gmail.com                                                    *
 *                                                                         *
 *   This file is part of KOSD.                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the          *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.  *
 ***************************************************************************/

#include "mixer.h"

#include <alsa/asoundlib.h>

static snd_mixer_t *mixer; 
static snd_mixer_selem_id_t *mixerSelemId;
static snd_mixer_elem_t *mixerElem;
static long volumeMax, volumeMin, volumeRange, volumeStep;
static QString controlName = "Master";

void Mixer::setControlName(const QString &name)
{
    controlName = name;
}


int Mixer::increaseVolume()
{
    open();

    long lVol, rVol;
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &lVol);
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_RIGHT, &rVol);
    
    int volume = (lVol+rVol)/2;
    volume += volumeStep;
    snd_mixer_selem_set_playback_volume_all(mixerElem, (volume > volumeMax) ? volumeMax : volume);
    snd_mixer_selem_set_playback_switch_all(mixerElem, 1);

    close();

    return 100 * (((float)(volume - volumeMin))/volumeRange);
}


int Mixer::decreaseVolume()
{
    open();

    long lVol, rVol;
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &lVol);
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_RIGHT, &rVol);
    
    int volume = (lVol+rVol)/2;
    volume -= volumeStep;
    snd_mixer_selem_set_playback_volume_all(mixerElem, (volume < 0) ? 0 : volume);
    snd_mixer_selem_set_playback_switch_all(mixerElem, 1);

    close();

    return 100 * (((float)(volume - volumeMin))/volumeRange);
}


int Mixer::volume()
{
    open();

    long lVol, rVol;
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &lVol);
    snd_mixer_selem_get_playback_volume(mixerElem, SND_MIXER_SCHN_FRONT_RIGHT, &rVol);
    
    close();

    int volume = (lVol+rVol)/2;
    return 100 * (((float)(volume - volumeMin))/volumeRange);
}


void Mixer::setMuted(bool muted)
{
    open();
    snd_mixer_selem_set_playback_switch_all(mixerElem, muted ? 0: 1);
    close();
}


bool Mixer::isMuted()
{
    int unmuted;

    open();
    snd_mixer_selem_get_playback_switch(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &unmuted);
    close();

    return (unmuted == 0);
}


bool Mixer::toggleMuted()
{
    open();

    int unmuted;
    snd_mixer_selem_get_playback_switch(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &unmuted);
    snd_mixer_selem_set_playback_switch_all(mixerElem, unmuted ? 0: 1);
    snd_mixer_selem_get_playback_switch(mixerElem, SND_MIXER_SCHN_FRONT_LEFT, &unmuted);

    close();

    return unmuted ? false : true;
}


void Mixer::open()
{
    snd_mixer_open(&mixer, 0);
    snd_mixer_attach(mixer, "default");
    snd_mixer_selem_register(mixer, NULL, NULL);
    snd_mixer_load(mixer);

    snd_mixer_selem_id_alloca(&mixerSelemId);
    snd_mixer_selem_id_set_name(mixerSelemId, controlName.toAscii());
    mixerElem = snd_mixer_find_selem(mixer, mixerSelemId);

    if (mixerElem == NULL)
        mixerElem = snd_mixer_first_elem(mixer);

    snd_mixer_selem_get_playback_volume_range(mixerElem, &volumeMin, &volumeMax);
    volumeRange = volumeMax - volumeMin;
    volumeStep = long(5.0f * ((float)volumeRange/100.0f));
    if (volumeStep < 1)
        volumeStep = 1;
}


void Mixer::close()
{
    snd_mixer_detach(mixer, "default");
    snd_mixer_free(mixer);
}
