/*
 * Copyright (C) 2010  Enrico Zini <enrico@enricozini.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#include <buffy/config/config.h>
#include <wibble/exception.h>
#include <wibble/string.h>

#include <glib.h>

//#define TRACE

using namespace std;
using namespace wibble;

namespace buffy {
namespace config {

namespace {
struct Parser
{
	Config& cfg;
	vector<string> path;
	string name;
	size_t namelev;

	Parser(Config& cfg) : cfg(cfg), namelev(0) {}

	string fullpath() const { return str::join(path.begin(), path.end(), "/"); }

	bool parseBool(const std::string& val)
	{
		return val == "true";
	}

	int parseInt(const std::string& val)
	{
		return strtoul(val.c_str(), 0, 10);
	}

	void acquire_view(const std::string& val, size_t pos)
	{
		if (path.size() < pos + 1) return;

		if (path[pos] == "empty")
			cfg.view().setEmpty(parseBool(val));
		else if (path[pos] == "read")
			cfg.view().setRead(parseBool(val));
		else if (path[pos] == "important")
			cfg.view().setImportant(parseBool(val));
	}

	void acquire_locations(const std::string& val, size_t pos)
	{
		if (path.size() < pos + 1) return;
		cfg.location(val);
	}

	void acquire_programs(const std::string& val, size_t pos)
	{
		if (path.size() < pos + 2) return;
		++pos;
		string newname;
		string type;
		if (name == "Mutt")
		{
			newname = "mutt";
			type = "gui";
		}
		else if (name == "Mutt-gui")
		{
			newname = "mutt";
			type = "gui";
		}
		else if (name == "Mutt-text")
		{
			newname = "mutt";
			type = "text";
		}
		else
		{
			newname = name;
			type = "gui";
		}

		if (path[pos] == "selected" && val == "true")
			cfg.selectMailProgram(newname);
		else if (path[pos] == "command")
			cfg.mailProgram(newname).setCommand(type, val);
	}

	void acquire_applications(const std::string& val, size_t pos)
	{
		if (path.size() < pos + 2) return;
		cfg.application(name).set(path[pos + 1], val);
	}

	void acquire_folders(const std::string& val, size_t pos)
	{
		if (path.size() < pos + 1) return;

		if (path[pos + 1] == "forceview")
			cfg.folder(name).setForceView(parseBool(val));
		else if (path[pos + 1] == "forcehide")
			cfg.folder(name).setForceHide(parseBool(val));
	}

	void acquire(const std::string& val)
	{
		if (path.size() < 1 || path[0] != "buffy") return;
		if (path.size() < 2) return;
		if (path[1] == "applications")
		{
			acquire_applications(val, 2);
		} else if (path[1] == "general") {
			if (path.size() < 3) return;
			if (path[2] == "interval")
				cfg.general().setInterval(parseInt(val));
			else if (path[2] == "view")
				acquire_view(val, 3);
			else if (path[2] == "locations")
				acquire_locations(val, 3);
			else if (path[2] == "programs")
				acquire_programs(val, 3);
		} else if (path[1] == "folders") {
			acquire_folders(val, 2);
		}
	}

	static void start_element(GMarkupParseContext *context,
			const gchar         *element_name,
			const gchar        **attribute_names,
			const gchar        **attribute_values,
			gpointer             user_data,
			GError             **error)
	{
		Parser* p = (Parser*)user_data;
		p->path.push_back(element_name);
		for (const gchar** i = attribute_names; *i; ++i)
			if (string(*i) == "name")
			{
				p->name = attribute_values[i-attribute_names];
				p->namelev = p->path.size();
				break;
			}
		//cerr << "START " << p->fullpath() << endl;
	}

	static void end_element (GMarkupParseContext *context,
			const gchar         *element_name,
			gpointer             user_data,
			GError             **error)
	{
		Parser* p = (Parser*)user_data;
		//cerr << "END " << p->fullpath() << endl;
		p->path.pop_back();
		if (p->path.size() < p->namelev)
		{
			p->name.clear();
			p->namelev = 0;
		}
	}

	static void text(GMarkupParseContext *context,
			const gchar         *text,
			gsize                text_len,
			gpointer             user_data,
			GError             **error)
	{
		Parser* p = (Parser*)user_data;
		string stext = str::trim(text);
		if (stext.empty()) return;
		// cerr << "TEXT for " << p->fullpath() << "[" << p->name << "]: " << text << endl;
		p->acquire(stext);
	}

};
}

static GMarkupParser parserfuncs = {
	Parser::start_element,
	Parser::end_element,
	Parser::text,
	NULL,
	NULL
};

void loadOld(const char* text, size_t length, const std::string& file, Config& cfg)
{
	Parser parser(cfg);
	GMarkupParseContext *context = g_markup_parse_context_new (
			&parserfuncs,
			(GMarkupParseFlags)0,
			(gpointer)&parser,	// usedata
			NULL);

	try {
		GError* err = NULL;
		g_markup_parse_context_parse (context, text, length, &err);
		if (err != NULL)
		{
			string msg = err->message;
			g_error_free(err);
			throw wibble::exception::Consistency("parsing xml file " + file, msg);
		}
	} catch (...) {
		g_markup_parse_context_free (context);
		throw;
	}

	g_markup_parse_context_free (context);

	// cfg.dump(cerr);
}

}
}

// vim:set ts=4 sw=4:
