'use strict';

require('mocha');
var match = require('./support/match');
var assert = require('assert');

describe('running extglob against minimatch tests', function() {
  var tests = [
    ['*(a|b[)', '*(a|b\\[)', false],
    ['123abc', 'ab*d+(e|f)', false],
    ['123abc', 'ab?*(e|f)', false],
    ['a', '!(a)', false],
    ['a', '(b)', false],
    ['a', '??', false],
    ['a', 'a??b', false],
    ['a', 'b?(a|b)', false],
    ['a.', '*.+(b|d)', false],
    ['a.a', '!(*.[a-b]*)', false],
    ['a.a', '!(*.a|*.b|*.c)', false],
    ['a.a', '!(*[a-b].[a-b]*)', false],
    ['a.a', '!*.(a|b)', false],
    ['a.a', '!*.(a|b)*', false],
    ['a.a', '*.!(a)', false],
    ['a.a', '*.+(b|d)', false],
    ['a.a.a', '!(*.[a-b]*)', false],
    ['a.a.a', '!(*[a-b].[a-b]*)', false],
    ['a.a.a', '!*.(a|b)', false],
    ['a.a.a', '!*.(a|b)*', false],
    ['a.a.a', '*.+(b|d)', false],
    ['a.abcd', '!(*.a|*.b|*.c)', false],
    ['a.abcd', '!(*.a|*.b|*.c)*', false],
    ['a.abcd', '*.!(a|b|c)', false],
    ['a.abcd', '*.!(a|b|c)*', false],
    ['a.b', '!(*.*)', false],
    ['a.b', '!(*.[a-b]*)', false],
    ['a.b', '!(*[a-b].[a-b]*)', false],
    ['a.b', '!*.(a|b)', false],
    ['a.b', '!*.(a|b)*', false],
    ['a.bb', '!(*.[a-b]*)', false],
    ['a.bb', '!(*[a-b].[a-b]*)', false],
    ['a.bb', '!*.(a|b)', false],
    ['a.bb', '!*.(a|b)*', false],
    ['a.ccc', '!*.(a|b)', false],
    ['a.ccc', '!*.(a|b)*', false],
    ['a.ccc', '*.+(b|d)', false],
    ['a.js', '!(*.js)', false],
    ['a.js', '*.!(js)', false],
    ['a.js.js', '!(*.js)', false],
    ['aa', '?', false],
    ['aa', '@(a)b', false],
    ['aa', 'a??b', false],
    ['aab', '?', false],
    ['aab', '??', false],
    ['aab', '@(c)b', false],
    ['ab', 'a!(@(b|B))', false],
    ['aB', 'a!(@(b|B))', false],
    ['ab', 'a(*b', false],
    ['ab', 'ab**(e|f)g', false],
    ['ab', 'ab*+(e|f)', false],
    ['ab', 'ab*d+(e|f)', false],
    ['ab', 'ab?*(e|f)', false],
    ['abcdef', '(a+|b)+', false],
    ['abcdef', 'ab**(e|f)g', false],
    ['abcdef', 'ab?*(e|f)', false],
    ['abcfef', '(a+|b)+', false],
    ['abcfef', 'ab**(e|f)g', false],
    ['abcfef', 'ab*d+(e|f)', false],
    ['abcfefg', '(a+|b)+', false],
    ['abcfefg', 'ab*d+(e|f)', false],
    ['abcfefg', 'ab?*(e|f)', false],
    ['abd', '(a+|b)+', false],
    ['abd', 'a!(@(b|B))d', false],
    ['abd', 'ab*d+(e|f)', false],
    ['abef', '(a+|b)+', false],
    ['abef', '*(a+|b)', false],
    ['abef', 'ab**(e|f)g', false],
    ['abef', 'ab*d+(e|f)', false],
    ['acd', '(a+|b)+', false],
    ['acd', 'ab*d+(e|f)', false],
    ['acd', 'ab?*(e|f)', false],
    ['ax', 'a?(b*)', false],
    ['b/a', '!(b/a)', false],
    ['baaac', '*(@(a))a@(c)', false],
    ['bb', 'a?(a|b)', false],
    ['c', '*(@(a))a@(c)', false],
    ['c.a', '!(*.[a-b]*)', false],
    ['c.a', '!*.(a|b)', false],
    ['c.a', '!*.(a|b)*', false],
    ['c.a', '*.!(a)', false],
    ['c.a', '*.+(b|d)', false],
    ['c.js', '!(*.js)', false],
    ['c.js', '*.!(js)', false],
    ['cow', '.!(*.*)', false],
    ['d.a.d', '!*.(a|b)', false],
    ['d.a.d', '!*.(a|b)*', false],
    ['egz', '@(b+(c)d|e+(f)g?|?(h)i@(j|k))', false],
    ['f', '!(f)', false],
    ['f', '*(!(f))', false],
    ['f', '+(!(f))', false],
    ['f.a', '!(*.a|*.b|*.c)', false],
    ['f.a', '*.!(a|b|c)', false],
    ['foo', '!(foo)', false],
    ['foo', '!(foo)*', false], // bash 4.3 disagrees
    ['foo', '!(foo)+', false],
    ['foo', '!(foo)b*', false],
    ['foo', '*(!(foo))', false],
    ['foo.js.js', '*.!(js)*', false],
    ['foo.js.js', '*.!(js)*.!(js)', false],
    ['foo.js.js', '*.!(js)+', false],
    ['foob', '!(foo)b*', false],
    ['foobar', '!(foo)*', false], // bash 4.3 disagrees
    ['foobar', '!(foo)b*', false],
    ['foobb', '!(foo)b*', false],
    ['foooofof', '*(f+(o))', false],
    ['foooofofx', '*(f*(o))', false],
    ['foooxfooxofoxfooox', '*(f*(o)x)', false],
    ['mad.moo.cow', '!(*.*).!(*.*)', false],
    ['mad.moo.cow', '.!(*.*)', false],
    ['Makefile.in', '!(*.c|*.h|Makefile.in|config*|README)', false],
    ['moo', '.!(*.*)', false],
    ['moo.cow', '!(*.*).!(*.*)', false], // bash 4.3 disagrees
    ['moo.cow', '.!(*.*)', false],
    ['mucca.pazza', 'mu!(*(c))?.pa!(*(z))?', false],
    ['ofooofoofofooo', '*(f*(o))', false],
    ['ofoooxoofxoofoooxoofxofo', '*(*(of*(o)x)o)', false],
    ['oxfoxfox', '*(oxf+(ox))', false],
    ['shell.c', '!(*.c|*.h|Makefile.in|config*|README)', false],
    ['xfoooofof', '*(f*(o))', false],
    ['zoot', '@(!(z*)|*x)', false],
    ['zz', '(a+|b)*', false],
    ['a', '(a)', true],
    ['a', '*(a)', true],
    ['a', '+(a)', true],
    ['a', '?', true],
    ['a', '?(a|b)', true],
    ['a', 'a?(a|b)', true],
    ['a', 'a?(x)', true],
    ['a((((b', 'a(*b', true],
    ['a((b', 'a(*b', true],
    ['a(b', 'a(*b', true],
    ['a.', '!(*.a|*.b|*.c)', true],
    ['a.', '*!(.a|.b|.c)', true],
    ['a.', '*.!(a)', true],
    ['a.', '*.!(a|b|c)', true],
    ['a.a', '(a|d).(a|b)*', true],
    ['a.a', '*!(.a|.b|.c)', true],
    ['a.a.a', '*.!(a)', true],
    ['a.abcd', '*!(*.a|*.b|*.c)*', true],
    ['a.abcd', '*!(.a|.b|.c)', true],
    ['a.b', '(a|d).(a|b)*', true],
    ['a.b', '*!(.a|.b|.c)', true],
    ['a.b', '*.!(a)', true],
    ['a.b', '*.+(b|d)', true],
    ['a.bb', '(a|d).(a|b)*', true],
    ['a.bb', '*.+(b|d)', true],
    ['a.c', '*!(.a|.b|.c)', true],
    ['a.c.d', '!(*.a|*.b|*.c)', true],
    ['a.c.d', '*!(.a|.b|.c)', true],
    ['a.c.d', '*.!(a|b|c)', true],
    ['a.ccc', '!(*.[a-b]*)', true],
    ['a.ccc', '!(*[a-b].[a-b]*)', true],
    ['a.js', '*!(.js)', true],
    ['a.js.js', '*!(.js)', true],
    ['a.js.js', '*.!(js)', true],
    ['a.md', '!(*.js)', true],
    ['a.md', '*!(.js)', true],
    ['a.md', '*.!(js)', true],
    ['aa', '!(a)', true],
    ['aaac', '*(@(a))a@(c)', true],
    ['aab', 'a??b', true],
    ['aac', '*(@(a))a@(c)', true],
    ['ab', '!(*.*)', true],
    ['ab', '(a+|b)+', true],
    ['ab', 'ab**(e|f)', true],
    ['ab]', 'a!(@(b|B))', true],
    ['abab', 'ab**(e|f)', true],
    ['abb', '!(*.*)', true],
    ['abbcd', '@(ab|a*(b))*(c)d', true],
    ['aBc', 'a!(@(b|B))', true],
    ['abcd', '?@(a|b)*@(c)d', true],
    ['abcd', '@(ab|a*@(b))*(c)d', true],
    ['abcdef', '(a+|b)*', true],
    ['abcdef', 'ab**(e|f)', true],
    ['abcdef', 'ab*+(e|f)', true],
    ['abcdef', 'ab*d+(e|f)', true],
    ['abcfef', '(a+|b)*', true],
    ['abcfef', 'ab**(e|f)', true],
    ['abcfef', 'ab*+(e|f)', true],
    ['abcfef', 'ab?*(e|f)', true],
    ['abcfefg', '(a+|b)*', true],
    ['abcfefg', 'ab**(e|f)', true],
    ['abd', '(a+|b)*', true],
    ['abd', 'a!(@(b|B))', true],
    ['abd', 'ab**(e|f)', true],
    ['abd', 'ab?*(e|f)', true],
    ['abef', '(a+|b)*', true],
    ['abef', 'ab**(e|f)', true],
    ['abef', 'ab*+(e|f)', true],
    ['abef', 'ab?*(e|f)', true],
    ['ac', '*(@(a))a@(c)', true],
    ['ac', 'a!(@(b|B))', true],
    ['acd', '(a+|b)*', true],
    ['acd', '@(ab|a*(b))*(c)d', true],
    ['acd', 'a!(@(b|B))', true],
    ['acd', 'a!(@(b|B))d', true],
    ['ax', '?(a*|b)', true],
    ['b', '(a+|b)*', true],
    ['b/b', '!(b/a)', true],
    ['b/c', '!(b/a)', true],
    ['ba', 'b?(a|b)', true],
    ['bar', '!(foo)*', true],
    ['bar', '!(foo)b*', true],
    ['baz', '!(foo)*', true],
    ['baz', '!(foo)b*', true],
    ['c.a', '!(*[a-b].[a-b]*)', true],
    ['c.c', '*!(.a|.b|.c)', true],
    ['c.ccc', '!(*.[a-b]*)', true],
    ['c.ccc', '!(*[a-b].[a-b]*)', true],
    ['c.js', '*!(.js)', true],
    ['d.a.d', '!(*.[a-b]*)', true],
    ['d.a.d', '!(*[a-b].[a-b]*)', true],
    ['d.a.d', '*.!(a)', true],
    ['d.a.d', '*.+(b|d)', true],
    ['d.d', '!(*.a|*.b|*.c)', true],
    ['d.d', '*!(.a|.b|.c)', true],
    ['d.d', '*.!(a|b|c)', true],
    ['d.js.d', '!(*.js)', true],
    ['d.js.d', '*!(.js)', true],
    ['d.js.d', '*.!(js)', true],
    ['e.e', '!(*.a|*.b|*.c)', true],
    ['e.e', '*!(.a|.b|.c)', true],
    ['e.e', '*.!(a|b|c)', true],
    ['effgz', '@(b+(c)d|e*(f)g?|?(h)i@(j|k))', true],
    ['efgz', '@(b+(c)d|e*(f)g?|?(h)i@(j|k))', true],
    ['egz', '@(b+(c)d|e*(f)g?|?(h)i@(j|k))', true],
    ['egzefffgzbcdij', '*(b+(c)d|e*(f)g?|?(h)i@(j|k))', true],
    ['f.a', '*!(.a|.b|.c)', true],
    ['f.f', '!(*.a|*.b|*.c)', true],
    ['f.f', '*!(.a|.b|.c)', true],
    ['f.f', '*.!(a|b|c)', true],
    ['fff', '!(f)', true],
    ['fff', '*(!(f))', true],
    ['fff', '+(!(f))', true],
    ['fffooofoooooffoofffooofff', '*(*(f)*(o))', true],
    ['ffo', '*(f*(o))', true],
    ['fofo', '*(f*(o))', true],
    ['fofoofoofofoo', '*(fo|foo)', true],
    ['foo', '!(f)', true],
    ['foo', '!(x)', true],
    ['foo', '!(x)*', true],
    ['foo', '*(!(f))', true],
    ['foo', '+(!(f))', true],
    ['foo.js.js', '*.!(js)', true],
    ['foobar', '!(foo)', true],
    ['foofoofo', '@(foo|f|fo)*(f|of+(o))', true],
    ['fooofoofofooo', '*(f*(o))', true],
    ['foooofo', '*(f*(o))', true],
    ['foooofof', '*(f*(o))', true],
    ['foooxfooxfoxfooox', '*(f*(o)x)', true],
    ['foooxfooxfxfooox', '*(f*(o)x)', true],
    ['foot', '@(!(z*)|*x)', true],
    ['foox', '@(!(z*)|*x)', true],
    ['Makefile', '!(*.c|*.h|Makefile.in|config*|README)', true],
    ['ofoofo', '*(of+(o))', true],
    ['ofoofo', '*(of+(o)|f)', true],
    ['ofoooxoofxo', '*(*(of*(o)x)o)', true],
    ['ofoooxoofxoofoooxoofxo', '*(*(of*(o)x)o)', true],
    ['ofoooxoofxoofoooxoofxoo', '*(*(of*(o)x)o)', true],
    ['ofoooxoofxoofoooxoofxooofxofxo', '*(*(of*(o)x)o)', true],
    ['ofxoofxo', '*(*(of*(o)x)o)', true],
    ['oofooofo', '*(of|oof+(o))', true],
    ['ooo', '!(f)', true],
    ['ooo', '*(!(f))', true],
    ['ooo', '+(!(f))', true],
    ['oxfoxoxfox', '*(oxf+(ox))', true],
    ['parse.y', '!(*.c|*.h|Makefile.in|config*|README)', true],
    ['zoox', '@(!(z*)|*x)', true]
  ];

  tests.forEach(function(test) {
    var fixture = test[0];
    var pattern = test[1];
    var expected = test[2];
    var msg = 'should ' + (expected ? '' : 'not ') + 'match ' + pattern;

    it(msg, function() {
      assert.equal(match.isMatch(fixture, pattern), expected, msg);
    });
  });
});
